#!/usr/bin/perl -w
# vim: nu
# vim: set encoding=utf-8
#
# (c) 2006-2008      DSI - Académie de Nancy-Metz
#             Jean-Christophe TOUSSAINT <jean-christophe.toussaint@ac-nancy-metz.fr>
#             Stéphane URBANOVSKI <stephane.urbanovski@ac-nancy-metz.fr>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# you should have received a copy of the GNU General Public License
# along with this program (or with Netsaint);  if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA
#
# $Id: checker.pl,v 1.14 2008-03-03 14:59:22 uid5000 Exp $

use strict;
use warnings;

use File::Basename;			# get dirname()
use lib dirname($0).'/lib';		# to get local libs
#use libEqosAgent qw(%EXIT_CODES_REV $globalname &make_log);

use Thread::Pool;
use POSIX;

use Term::ANSIColor qw(:constants);
use IO::Handle;
use IPC::Open3;
use DBI;

use Data::Dumper;

##################################################################
# Global vars
##################################################################

# Revision
my $apprevision = '$Revision: 1.14 $';

# Nagios exit codes
my %EXIT_CODES_REV = (
	-1	=> 'UNKNOWN',
	0	=> 'OK',
	1	=> 'WARNING',
	2	=> 'CRITICAL',
	3	=> 'UNKNOWN',
	4	=> 'UNKNOWN',
	);
my %EXIT_CODES_COLOR = (
	'UNKNOWN'	=> YELLOW,
	'OK'		=> BOLD.GREEN,
	'WARNING'	=> BOLD.YELLOW,
	'CRITICAL'	=> BOLD.RED,
	);

# Pretty name
my $globalname = "checker";

# Set debug level :
my $debuglevel : shared = 4;

# Plugins paths:
my $nagiosPluginPath = 'plugins';

# database file :
my $sqliteDatafile = '/var/lib/sqlite/eqos/eqosd.sqlite';

# Plugin default timeout
my $pluginTimeout = 15;

# End loop flag:
my $haltRequest = 0;

# Config required flag :
my $configRequest = 1;

$ENV{'AVAILABILITY_MODE'}='EQOS';

# Configuration :
my %checkerConf;

# array of plugin logs :
my @logs : shared = ();

# list of thread used for timeout management :
my @expired : shared = ();

# DBI handler:
my $dbh;

##################################################################
# Functions
##################################################################

# Set end loop flag :
sub signalHalt {
	&make_log (1, BOLD RED . "SIG HALT : stop signal received !");
	$haltRequest = 1;
}

# Set config reload flag :
sub signalHup {
	&make_log (1, BOLD YELLOW . "SIG HUP : reloading ...");
	$configRequest = 1;
}

# Write a dated and formated log message
sub make_log {
	my ($lvl, $str) = @_;
	if ($lvl <= $debuglevel) {
		my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday) = gmtime();
		print WHITE . sprintf("%02d/%02d %02d:%02d:%02d", $mday, $mon+1, $hour, $min, $sec) . " [$globalname] $str" . RESET . "\n";
	}
}

# Read config
sub getConf {
	&make_log (0, GREEN . "Load configuration ...");

	my $sth = $dbh->prepare("SELECT name, value FROM eqos_config");
	$sth->execute();
	my %newConfig = %{$sth->fetchall_hashref('name')};

	if ( exists($newConfig{'debuglevel'}) ) {
		$debuglevel = $newConfig{'debuglevel'}{'value'};
		&make_log (0, CYAN . "Seting debuglevel to : ".$debuglevel);
	}
	if ( exists($newConfig{'plugintimeout'}) ) {
		$pluginTimeout = $newConfig{'plugintimeout'}{'value'};
		&make_log (0, CYAN . "Seting plugin timeout to : ".$pluginTimeout);
	}

	$sth = $dbh->prepare("SELECT id, plugin, args, period FROM eqos_services");
	$sth->execute();
	my %newCheckerConf = %{$sth->fetchall_hashref('id')};

	foreach my $tid ( keys(%checkerConf) ) {
		# delete removed tasks :
		if (! exists($newCheckerConf{$tid}) ) {
			&make_log (2, "getConf() : Remove task $tid (".$checkerConf{$tid}{'plugin'}." ".$checkerConf{$tid}{'args'}.") : id.");
			delete($checkerConf{$tid});
			next;
		}
		# also delete modified tasks :
		foreach my $k (keys(%{$newCheckerConf{$tid}})) {
			if ($checkerConf{$tid}{$k} ne $newCheckerConf{$tid}{$k} ) {
				&make_log (2, "getConf() : Remove task $tid (".$checkerConf{$tid}{'plugin'}." ".$checkerConf{$tid}{'args'}.") : $k.");
				delete($checkerConf{$tid});
				last;
			}
		}
	}

	foreach my $tid ( keys(%newCheckerConf) ) {
		my $p = $nagiosPluginPath."/".$newCheckerConf{$tid}{'plugin'};
		# skip unknown plugin :
		if ( ! -x $p ) {
			&make_log (1, "getConf() : Cancel task $tid : '$p' is not executable.");
			delete($newCheckerConf{$tid});

		# add new tasks :
		} elsif ( !exists($checkerConf{$tid}) ) {
			&make_log (2, "getConf() : Add task $tid (".$newCheckerConf{$tid}{'plugin'}." ".$newCheckerConf{$tid}{'args'}.").");
			$checkerConf{$tid} = $newCheckerConf{$tid};
			$checkerConf{$tid}{'next'} = time() + int(rand($checkerConf{$tid}{'period'}));
		}
	}
	$configRequest = 0;
}

sub pushLog () {
	my %outshare : shared = (
		'tid' => $_[0],
		'time' => $_[1],
		'val' => $_[2],
		'out' => $_[3],
	);
	{
		lock @logs;
		push (@logs,\%outshare);
	}
}

# thread code
sub execThreadPlugin {
	my ($taskId,$cmd) = @_;
	&make_log (2, "execThreadPlugin END thread id=".threads->self->tid." task=$taskId.");
	&make_log (1, GREEN ."Task $taskId start : ".MAGENTA.$cmd.RESET.".");
	my $time = time();

	my $timeout_flag = 0;
	my %return;

	my @stdout = ();
	my @stderr = ();

	my $cmdstdin = IO::Handle->new();
	my $cmdstdout = IO::Handle->new();
	my $cmdstderr = IO::Handle->new();

	&make_log (2, "Execution : ".$cmd);

	$return{"pid"} = open3($cmdstdin,$cmdstdout,$cmdstderr,$cmd) ;

	my %e : shared = (
		'taskId' => $taskId,
		'start' => $time,
		'pid' => $return{"pid"},
		'timeout' => 0,
	);
	{
		lock @expired;
		push (@expired,\%e);
	}

	&make_log (4, "\$?=".$?);
	&make_log (4, "PID = ".$return{"pid"});

	if ( $return{"pid"} == -1 ) {
		push (@stderr ,sprintf(_("Enable to execute : [%s] %s"),$cmd,$!));

	} else {
		if ( $cmdstdout->opened() ) {
			@stdout = $cmdstdout->getlines();
			&make_log (4, "stdout got ".scalar(@stdout)." lines");
		}
		if ( $cmdstderr->opened() ) {
			@stderr = $cmdstderr->getlines();
			&make_log (4, "stderr got ".scalar(@stderr)." lines");
		}
		my $waitpid = waitpid($return{"pid"}, 0);
		$return{'val'} = $?;
		&make_log (4, "\$?=".$return{'val'});
		&make_log (4, "waitpid=".$waitpid);

		if ( $e{'timeout'} ) {
			@stdout = (sprintf("Command timeout after %is !",$pluginTimeout));
			$return{'val'} = 4;

		} else {
			$return{'val'} = $return{'val'} >> 8;
			chomp (@stdout);
			chomp (@stderr);
		}
	}
	if ($#stdout < 0) {
		@stdout=('');
	}

	$e{'timeout'} = 2;
	$cmdstdin->close();
	$cmdstdout->close();
	$cmdstderr->close();

	$return{'stdout'} = \@stdout;
	$return{'stderr'} = \@stderr;

	my $out = \%return;

	&pushLog ($taskId,$time, $out->{'val'},$out->{'stdout'}[0]);

	my $stateString = exists($EXIT_CODES_REV{$out->{'val'}}) ? $EXIT_CODES_REV{$out->{'val'}} : $out->{'val'};
	&make_log (2, "execThreadPlugin END thread id=".threads->self->tid." task=$taskId.");
	my $outputString = $out->{'stdout'}[0];
	$outputString =~ s/\|.*$//; # delete perf data
	&make_log (1, GREEN ."Task $taskId end : ".$EXIT_CODES_COLOR{$stateString}.$stateString.GREEN." - ".RESET.$outputString);

	return ;
}


##################################################################
# MAIN
##################################################################

&make_log (0, "Starting $globalname ($apprevision)");

# Make STDOUT and STDERR unbufferized
select(STDERR); $| = 1;
select(STDOUT); $| = 1;

# Set signal handlers :
$SIG{'INT'} = 'signalHalt';
$SIG{'QUIT'} = 'signalHalt';
$SIG{'TERM'} = 'signalHalt';
$SIG{'HUP'} = 'signalHup';

# Define a thread pool :
my $pool = Thread::Pool->new(
	{
		'optimize' => 'memory', # default: 'memory'

		'do' => \&execThreadPlugin,

		'workers' => 10,     # default: 1
		'maxjobs' => 50,     # default: 5 * workers
		'minjobs' => 5,      # default: maxjobs / 2
	}
);

$dbh = DBI->connect("dbi:SQLite:dbname=$sqliteDatafile", "", "", { 'PrintError' => 1, 'PrintWarn' => 1, 'AutoCommit' => 1 });
$dbh->{'FetchHashKeyName'} = 'NAME_lc';
$dbh->{'unicode'} = 1;
$dbh->do('PRAGMA synchronous = OFF');

#&make_log (2, GREEN ."           busy_timeout=".$dbh->func( 'busy_timeout' ));

&pushLog(-1,time(),0,"Starting $globalname ($apprevision)");

&make_log (1, GREEN . "Starting main loop");

do {
	if ($configRequest) {
		&getConf();
	}

	# Start needed checks :
	foreach my $taskId ( keys(%checkerConf) ) {
		my $dt = time() - $checkerConf{$taskId}{'next'} ;

		if ( $dt > $checkerConf{$taskId}{'period'} ) {
			&make_log (2, YELLOW ."exec $taskId: Check outdated since $dt s !");
			$checkerConf{$taskId}{'next'} = time();

		}
		if ( $dt > 0 ) {
			my $nextCheckTime = $checkerConf{$taskId}{'next'} + $checkerConf{$taskId}{'period'};
			$checkerConf{$taskId}{'next'} = $nextCheckTime;
			my $cmd = $nagiosPluginPath."/".$checkerConf{$taskId}{'plugin'}." ".$checkerConf{$taskId}{'args'};
			$checkerConf{$taskId}{'thead'} = $pool->job( $taskId,$cmd );

		}
	}

	# Check timeouted plugin :
	my $ctimeout = time() ;
	my $i = 0;
	while ( exists($expired[$i]) ) {
		if ( $ctimeout - $expired[$i]{'start'} > $pluginTimeout ) {

			if ( $expired[$i]{'timeout'} == 2 ) {
				&make_log (3, "Cleaning up task : ".$expired[$i]{'taskId'});
				if ( $i == 0 ) {
					$i--;
					lock @expired;
					shift(@expired);
				}

			} else {
				&make_log (2, YELLOW . "Plugin exec timeout : ".$expired[$i]{'taskId'});
				{
					lock @expired;
					$expired[$i]{'timeout'} = 1;
				}
				if ( $expired[$i]{'pid'} > 1) {
					my @killed = kill( 9, $expired[$i]{'pid'}) ;
					&make_log (3, @killed." process killed");
				}
			}
		} else {
			last;
		}
		$i++;
	}

#	&make_log (2, BOLD.YELLOW . "Debuglevel : ".$debuglevel);
	sleep 1;

	if ($haltRequest) {
		&make_log (1, GREEN . "Shuting down thread pool : ".$pool->todo. " jobs waiting ...");
		$pool->abort;
		&pushLog(-1,time(),0,"Stoping $globalname ($apprevision)");
	}
	if ( ($#logs >= 0) || $haltRequest) {
		# Store logs in database
		&make_log (2, 'Saving '.($#logs+1).' logs.');
		# CREATE TABLE eqos_states ( id INTEGER PRIMARY KEY, timestamp INTEGER, idService INTEGER, state TEXT, message TEXT );
		my $sqlPrepareInsertLogs = $dbh->prepare("INSERT INTO eqos_states VALUES (NULL,?,?,?,?);");

		my $loop = 1;
		while ( exists($logs[0]) && $loop ) {
			&make_log (3, 'SQL insert : task_id='.$logs[0]->{'tid'}.' time='.$logs[0]->{'time'}.' val='.$logs[0]->{'val'}.' out='.$logs[0]->{'out'});

			if ( ! $sqlPrepareInsertLogs->execute($logs[0]->{'time'},$logs[0]->{'tid'},$logs[0]->{'val'},$logs[0]->{'out'}) ) {
				&make_log (1, BOLD RED . "SQL ERROR on log insert : ".$dbh->errstr);

				if ( $#logs > 500 ) {
					# drop messages
					lock @logs;
					shift(@logs);

				} else {
					# get out of the loop if failled
					$loop = 0;
				}

			} else {
				# remove successfuly inserted logs
				lock @logs;
				shift(@logs);
			}
		}
		$sqlPrepareInsertLogs->finish;

	}
#	my $stat = `ps hvp $$`;
#	chomp($stat);
#	&make_log (2, CYAN . "Stats : ".$stat);
} while ($haltRequest == 0);

&make_log (0, GREEN . "Exiting main loop");

#print Dumper(\@logs);
