# -*- coding: utf-8 -*-
'''
Support for Linux File Access Control Lists
'''
from salt.exceptions import CommandExecutionError


def _raise_on_no_files(*args):
    if len(args) == 0:
        raise CommandExecutionError('You need to specify at least one file or directory to work with!')


def getfacl(*args, **kwargs):
    return __salt__['acl.getfacl'](*args, **kwargs)


def _acl_prefix(acl_type):
    prefix = ''
    if acl_type.startswith('d'):
        prefix = 'd:'
        acl_type = acl_type.replace('default:', '')
        acl_type = acl_type.replace('d:', '')
    if acl_type == 'user' or acl_type == 'u':
        prefix += 'u'
    elif acl_type == 'group' or acl_type == 'g':
        prefix += 'g'
    elif acl_type == 'mask' or acl_type == 'm':
        prefix += 'm'
    return prefix


def modfacl(*args, **kwargs):
    return __salt__['acl.modfacl'](*args, **kwargs)


def delfacl(acl_type, acl_name='', *args, **kwargs):
    '''
    FIXME EOLE: -R should be befor -x option
    Remove specific FACL from the specified file(s)

    CLI Examples:

    .. code-block:: bash

        salt '*' acl.delfacl user myuser /tmp/house/kitchen
        salt '*' acl.delfacl default:group mygroup /tmp/house/kitchen
        salt '*' acl.delfacl d:u myuser /tmp/house/kitchen
        salt '*' acl.delfacl g myuser /tmp/house/kitchen /tmp/house/livingroom
        salt '*' acl.delfacl user myuser /tmp/house/kitchen recursive=True
    '''
    recursive = kwargs.pop('recursive', False)

    _raise_on_no_files(*args)

    cmd = 'setfacl'
    if recursive:
        cmd += ' -R'

    cmd += ' -x'

    cmd = '{0} {1}:{2}'.format(cmd, _acl_prefix(acl_type), acl_name)

    for dentry in args:
        cmd += ' {0}'.format(dentry)
    __salt__['cmd.run'](cmd, python_shell=False)
    return True


def find(*args, **kwargs):
    return __salt__['file.find'](*args, **kwargs)
