# -*- coding: utf-8 -*-
'''
Support for Linux File Access Control Lists
'''
from salt.exceptions import CommandExecutionError as _CommandExecutionError
from creole.client import CreoleClient as _CreoleClient
from os.path import abspath as _abspath
_DIRECTORIES = _CreoleClient().get_creole('acl_directories')


def _raise_on_no_files(*args):
    if len(args) == 0:
        raise _CommandExecutionError('You need to specify at least one file or directory to work with!')


def getfacl(path, **kwargs):
    path = _abspath(path)
    for directory in _DIRECTORIES:
        if path.startswith(directory):
            break
    else:
        return
    return __salt__['acl.getfacl'](path)

def getadusers(**kwargs):
    cmd = 'wbinfo -u'
    users = __salt__['cmd.run'](cmd)
    adusers = users.splitlines()
    return adusers

def _acl_prefix(acl_type):
    prefix = ''
    if acl_type.startswith('d'):
        prefix = 'd:'
        acl_type = acl_type.replace('default:', '')
        acl_type = acl_type.replace('d:', '')
    if acl_type == 'user' or acl_type == 'u':
        prefix += 'u'
    elif acl_type == 'group' or acl_type == 'g':
        prefix += 'g'
    elif acl_type == 'mask' or acl_type == 'm':
        prefix += 'm'
    return prefix


def chmod(acl_type, rights, path, **kwargs):
    cmd = 'chmod {3} {0}={1} {2}'.format(acl_type, rights, path, '-R' if kwargs.get('recursive', False) else '')
    return __salt__['cmd.run'](cmd)

def modfacl(acl_type, acl_name, perms, path, **kwargs):
    path = _abspath(path)
    for directory in _DIRECTORIES:
        if path.startswith(directory):
            break
    else:
        return
    return __salt__['acl.modfacl'](acl_type, acl_name, perms, path, recursive=kwargs.get('recursive', False))


def delfacl(acl_type, acl_name, path, **kwargs):
    '''
    FIXME EOLE: -R should be set before -x option
    Remove specific FACL from the specified file(s)

    CLI Examples:

    .. code-block:: bash

        salt '*' acl.delfacl user myuser /tmp/house/kitchen
        salt '*' acl.delfacl default:group mygroup /tmp/house/kitchen
        salt '*' acl.delfacl d:u myuser /tmp/house/kitchen
        salt '*' acl.delfacl g myuser /tmp/house/kitchen /tmp/house/livingroom
        salt '*' acl.delfacl user myuser /tmp/house/kitchen recursive=True
    '''
    path = _abspath(path)
    for directory in _DIRECTORIES:
        if path.startswith(directory):
            break
    else:
        return
    recursive = kwargs.pop('recursive', False)

    cmd = 'setfacl'
    if recursive:
        cmd += ' -R'

    cmd += ' -x'

    cmd = '{0} {1}:{2}'.format(cmd, _acl_prefix(acl_type), acl_name)

    cmd += ' {0}'.format(path)
    __salt__['cmd.run'](cmd, python_shell=False)
    return True


def find(path, **kwargs):
    path = _abspath(path)
    for directory in _DIRECTORIES:
        if path.startswith(directory):
            break
    else:
        return
    return __salt__['file.find'](path, **kwargs)
