# -*- coding: utf-8 -*- #

from os import path, kill
import subprocess, signal
import sys

import warnings
import cjson
from flask import session, g
from tiramisu.option import *
from tiramisu.value import Multi
from tiramisu.error import PropertiesOptionError

from creole.loader import creole_loader, config_load_store, \
                          config_save_values, config_get_values
from creole.var_loader import modes, modes_level, convert_value, type_option, \
                              type_option_convert
from creole.config import configeol
from creole import eosfunc

try:
    from collections import OrderedDict
    from os import umask
    #8076
    umask(0o022)
except:
    from pyeole.odict import OrderedDict

try:
    from zephir.lib_zephir import xmlrpclib
except:
    import xmlrpclib

#return always warning (even if same warning is already returned)
warnings.simplefilter("always", ValueWarning)


MSG_ERR_IMPORT_VAR = "Problème de chargement de la variable \"{1}\"({2}) de la catégorie \"{0}\""

class ConfigEmptyError(Exception):
    pass


session_tiramisu = {}

def get_id():
    """ Should return an ID per user in order to instanciate the session """
    raise "Un gestionnaire de sessions doit être défini"

init_zephir = None

def reload_all():
    reload(eosfunc)
    eosfunc.load_funcs(force_reload=True)

def config_exists():
    """ Checks whether a config.eol file exists on the server """
    return path.isfile(configeol)

def quit_genconfig(id_):
    """ Function to kill processes associated to gen_config
    :param id_: session id
    :type  id_: `str`
    """
    p = subprocess.Popen(['ps', '-U', 'genconfig', '--no-header'], stdout=subprocess.PIPE)
    out, err = p.communicate()
    for line in out.splitlines():
        if 'uzbl-core' in line:
            pid = int(line.split(None, 1)[0])
            kill(pid, signal.SIGKILL)
        elif 'chromium' in line:
            pid = int(line.split(None, 1)[0])
            kill(pid, signal.SIGKILL)

def del_config(id_):
    """ Removes the user's current configuration
    :param id_: session id
    :type  id_: `str`
    """
    if id_ in session_tiramisu:
        del(session_tiramisu[id_]['config'])
        session_tiramisu[id_]['config'] = None

def del_session(id_):
    """ Removes the entire session
    :param id_: session id
    :type  id_: `str`
    """
    if id_ in session_tiramisu:
        del(session_tiramisu[id_])

def init_session(id_):
    """ Instanciate the user's session
    :param id_: session id
    :type  id_: `str`
    """
    if not session_tiramisu.has_key(id_):
        session_tiramisu[id_] = {'config':None, 'mode': 'normal', 'debug': False, 'zephir': { 'available': False, 'reauth': True }}

def init_zephir_session(id_, zephir):
    """ Instanciate Zéphir user's session with a dictionnary
        containing the connection information to Zéphir
    :param id_: session id
    :type  id_: `str`
    :param zephir: Zéphir connection informations
    :type  zephir: `dict`
    """
    init_session(id_)
    session_tiramisu[id_]['zephir'] = zephir

def store_session_data(id_, data_dict):
    """ Loads some dictionnary into the user's session
    :param id_: session id
    :type  id_: `str`
    :param data_dict: data to inject into the session
    :type  data_dict: `dict`
    """
    for key_name, val in data_dict.items():
        session_tiramisu[id_][key_name] = val
    set_mode(id_, session_tiramisu[id_]['mode'])
    set_debug(id_, session_tiramisu[id_]['debug'])

def get_config(id_, force_dirs=None, force_configfile=None, store=None,
               load_error=False, force_load_creole_owner=None):
    """ Load a Creole configuration
    :param id_: session id
    :type  id_: `str`
    :param force_dirs: name of the dictionnaries repository
    :type  force_dirs: `str`
    :param force_configfile: name of the config file to use
    :type  force_configfile: `str`
    :param store: raw_data config to load
    :type  store:`str`
    :param load_error: load the errors
    :type  load_error: `boolean`
    :param force_load_creole_owner: name of all variables owner
    :type  force_load_creole_owner: `str`
    :returns: current configuration
    :type   : `tiramisu`
    """
    #return session_tiramisu[session['tiramisu_id']]['tiramisu']
    #FIXME : code temporaire jusqu'au support des sessions
    init_session(id_)
    if init_zephir is not None:
        # gen_config running in Zéphir
        if session_tiramisu[id_]['config'] is None:
            if session_tiramisu[id_].get('zephir_data', None):
                # if configuration has been reset, rebuild it from source data
                zephir_data = session_tiramisu[id_]['zephir_data']
                session_data = session_tiramisu[id_]
                init_zephir(id_, session_tiramisu[id_]['zephir_data'], session_data, load_error,
                            force_load_owner=force_load_creole_owner)
        # should always be initialized in zephir mode
        assert session_tiramisu[id_]['config'] is not None, "Session invalide, relancez l'interface \
        de configuration.\nSi nécessaire, reconnectez vous à Zéphir."
    else:
        if session_tiramisu[id_]['config'] is None or force_dirs is not None \
                or force_configfile is not None:
            # new session or configuration object destroyed
            if store is not None:
                load_values = False
                rw = None
            else:
                load_values = True
                rw = True
            reload_all()
            with warnings.catch_warnings(record=True) as warns:
                config = creole_loader(rw=rw, owner='gen_config',
                                       load_values=load_values,
                                       reload_config=False,
                                       force_configeol=force_configfile,
                                       force_load_creole_owner=force_load_creole_owner,
                                       force_dirs=force_dirs)
            for warn in warns:
                config.impl_set_information(warn.message.opt, str(warn.message))
            if store is not None:
                with warnings.catch_warnings(record=True) as warns:
                    config_load_store(config, 'creole', store)
                for warn in warns:
                    config.impl_set_information(warn.message.opt, str(warn.message))
                config.read_write()
            store_session_data(id_, {'config':config})
    return session_tiramisu[id_]['config']

def get_zephir(id_):
    """ Returns Zéphir informations stored in the user's session
    :param id_: session id
    :type  id_: `str`
    :returns: Zéphir information dictionnary
    :type   : `dict`
    """
    return session_tiramisu[id_].get('zephir', { 'available': False, 'reauth': True })

def get_zephir_config(id_):
    """ Retrieves the configuration from distant server
        and loads it right away into the user's session
    :param id_: session id
    :type  id_: `str`
    :returns: configuration from Zéphir
    :type   : `tiramisu`
    """
    zephir = get_zephir(id_)
    proxy = zephir['proxy'];
    try:
        code, res = proxy.serveurs.get_status(zephir['server_id'])
        code, info_serv = proxy.serveurs.get_serveur(zephir['server_id'])

    except xmlrpclib.ProtocolError:
        print "Accès au serveur %s non autorisé" % zephir['server_id']
    except Exception, e:
        print "Erreur de lecture des informations sur Zéphir \n(%s)" % str(e)
    else:
        if res['config_ok'] == 0:
            raise ConfigEmptyError("La configuration distante (Zéphir) n'existe pas")
        libelle = info_serv[0]['libelle']
        id_mod = info_serv[0]['module_actuel']
    code, res = proxy.serveurs.get_dico(zephir['server_id'], 'modif_config')
    # chargement des valeurs
    values = eval(res[-1])
    load_errors = load_config(id_, store=values)
    config = get_config(id_)
    return config


def is_upgrade(id_):
    """ Tells whether the current configuration
        is imported from an older version of EOLE
    :param id_: session id
    :type  id_: `str`
    """
    config = get_config(id_)
    return config.impl_get_information('upgrade', False) is not False

def get_upgrade_version(id_):
    """ Get the current configuration file version
    :param id_: session id
    :type  id_: `str`
    :returns: version number
    :type   : `str`
    """
    config = get_config(id_)
    return config.impl_get_information('upgrade', '')

def get_version(id_):
    """ Get module version for current configuration
    :param id_: session id
    :type  id_: `str`
    :returns: version number
    :type   : `str`
    """
    config = get_config(id_)
    return config._getattr("creole.general.eole_version", force_permissive=True)

def get_release(id_):
    """ Get release version for current configuration
    :param id_: session id
    :type  id_: `str`
    :returns: release number
    :type   : `str`
    """
    config = get_config(id_)
    return config._getattr("creole.general.eole_release", force_permissive=True)

def load_config(id_, configfile=None, store=None):
    """ Load configuration from a file or from raw_data store
    :param id_: session id
    :type id_: str
    :param configfile: path to configfile
    :type configfile: str
    :param store: raw_data config to load
    :type  store: `str`
    :returns: return a list of loading errors
    :type   : `dict`
    """
    del_config(id_)
    config = get_config(id_, force_dirs=None, force_configfile=configfile,
                        store=store, force_load_creole_owner='import')
    set_mode(id_, session_tiramisu[id_]['mode'])
    set_debug(id_, session_tiramisu[id_]['debug'])
    return get_load_errors(id_)


def get_load_errors(id_):
    config = get_config(id_)
    setting = config.cfgimpl_get_settings()
    load_errors = []
    for path, properties in setting.get_modified_properties().items():
        if 'load_error' in properties:
            path = path.split('.')
            category_name = config.unwrap_from_path('.'.join(path[0:2])
                                ).impl_getdoc().capitalize()
            try:
                varname = config.unwrap_from_path('.'.join(path)).impl_getdoc()
                varid = path[-1]
                if isinstance(category_name, unicode):
                    category_name = unicode.encode(category_name, 'utf8')
                load_errors.append(MSG_ERR_IMPORT_VAR.format(category_name, varname, varid))
            except PropertiesOptionError:
                pass
    return load_errors


def get_load_error(id_):
    config = get_config(id_)
    if config.impl_get_information('load_error', False):
        return True
    return False


def get_categories(id_, get_nested=False, init=False, zephir_sync=False):
    """Get category's names and, if category_name is set, all variables of
    current category_name.

    :param id_: session id
    :type  id_: `str`
    :param get_nested: include nested tags & variables
    :type  get_nested: `bool`
    :param init: load config from Zéphir
    :type  init: `bool`
    :param zephir_sync: get configuration from Zéphir first
    :type  zephir_sync: `bool`
    :returns: list of categories like:
                [{'name': 'cat1, 'help': 'help1', 'mode': 'basic'}, ...]
    :type   : `list`
    """
    zephir = get_zephir(id_)

    if zephir['available'] and zephir_sync:
        config = get_zephir_config(id_)
    else:
        config = get_config(id_)
    ret = []
    for name, ownconfig in config.creole.iter_groups():
        optdesc = ownconfig.cfgimpl_get_description()
        help_ = optdesc.impl_get_information('help', '')
        icon = optdesc.impl_get_information('icon', 'tags')
        if icon is None:
            icon = 'tags'
        category = {'id': name, 'description': optdesc.impl_getdoc().capitalize(),
                    'icon': icon,
                    'help': help_}
        has_value = False
        for var in ownconfig.iter_all():
            path = get_path(name, var[0])
            option = config.unwrap_from_path(path)
            type_ = type_option[type(option)]
            if not get_nested:
                if type_ == 'optiondescription':
                    for slave in getattr(config, path):
                        has_value = True
                        break
                    if has_value:
                        break
                else:
                    has_value = True
                    break
            else:
                tags = get_tags(id_, name)
                if tags != []:
                    category['tags'] = tags
                    has_value = True
        if has_value:
            for mode in modes_level:
                if mode in config.cfgimpl_get_settings()[optdesc]:
                    category['mode'] = mode
                    break
            ret.append(category)
    return ret


def get_modes(id_):
    """ Get available modes and current active mode
    :param id_: session id
    :type  id_: `str`
    :returns: list of available modes and the selected one
    :type   : `list`
    """
    config = get_config(id_)
    cur_mode = get_mode(id_)
    ret = []
    for mode in modes.values():
        ret.append({'id':mode.name, 'level':mode.level, 'current':mode.name==cur_mode})
    return ret

def get_path(category_name, name=None, master_slave=None):
    """ Construct path string
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param name: name of target variable
    :type  name: `str`
    :param master_slave: name of targer sub_variable
    :type  master_slave: `bool`
    :returns: creole path to variable
    :type   : `str`
    """
    path = 'creole.' + category_name
    if name is not None:
        path += '.' + name
    if master_slave is not None:
        path += '.' + master_slave
    return path


def get_variable(id_, category_name, name, slave=None, slaves=None, current_tag=False,
                 force_permissive=False):
    """ Get variable from specified name
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param name: variable name
    :type  name: `str`
    :param slave: for master/slave name must be master's name
                  slave must be slave's name
                  (or master's one if slave not None)
    :type  slave: `str`
    :param slaves: list of slave variables
    :type  slaves: `list`
    :param current_tag: specify current tag
    :type  current_tag: `str`
    :param force_permissive: abstract permissions to get variable
    :type  force_permissive: `bool`
    :returns: dictionnary of variable proprieties
    :type   : `dict`
    :raises: PropertiesOptionError
    """
    def _get_variable(option, type_, vname, category_name, current_tag):
        #build variable
        setting = config.cfgimpl_get_settings()
        mandatory = 'mandatory' in setting[option]
        auto_freeze = 'auto_freeze' in setting[option]
        error = None
        multi = option.impl_is_multi()
        try:
            with warnings.catch_warnings(record=True) as warns:
                value = config._getattr(path, force_permissive=force_permissive)
            if isinstance(value, Multi):
                value = list(value)
        except ValueError, err:
            error = err
            if multi:
                value = []
            else:
                value = None
        is_default_owner = config.cfgimpl_get_values().is_default_owner(option)
        is_calculated =  option.impl_has_callback() and is_default_owner
        variable = {'id': vname, 'value': value,
            'description': option.impl_getdoc(), 'type': type_,
            'multi': multi, 'mandatory': mandatory,
            'auto_freeze': auto_freeze, 'categoryid': category_name,
            'owner': str(config.getowner(option)), 'is_calculated': is_calculated,
            'default_owner': is_default_owner}
        if error:
            variable['warning'] = "Erreur au chargement de la variable : {0}".format(error)
        else:
            variable['warning'] = config.impl_get_information(option, '')
            if 'load_error' in setting[option] and not variable['warning']:
                variable['warning'] = 'Problème d\'importation de cette variable'
        if current_tag is not False:
            variable['tagid'] = current_tag
        variable['help'] = option.impl_get_information('help', '')

        #hidden for debug mode
        cat_path = get_path(category_name)
        cat_option = config.unwrap_from_path(cat_path, force_permissive=force_permissive)
        variable['hidden'] = 'hidden' in setting[option] or 'hidden' in setting[cat_option]
        variable['editable'] = not 'frozen' in setting[option] and not variable['hidden']

        if type_ == 'choice':
            variable['choices'] = option.impl_get_values()
            variable['open'] = option.impl_is_openvalues()
        variable['mode'] = 'basic'
        for mode in modes_level:
            if mode in setting[option]:
                variable['mode'] = mode
                break
        return variable

    #FIXME _var_force_edit_state(self, var):
    #FIXME previous/default value
    config = get_config(id_)
    path = get_path(category_name, name, master_slave=slave)
    option = config.unwrap_from_path(path, force_permissive=force_permissive)
    if isinstance(option, SymLinkOption):
        opt = option._opt
    else:
        opt = option
    type_ = type_option[type(opt)]
    if type_ == 'optiondescription':
        #if option is an OptionDescription,
        #assume children are master/slave
        #first, get all slaves
        slaves = []
        #slaves are set has subvariable of master
        for _slave in [var[0] for var in config._getattr(path,
                                                         force_permissive=force_permissive).__iter__(force_permissive=force_permissive) \
                if var[0] != name]:
            try:
                #FIXME ---duplicate
                path = get_path(category_name, name, master_slave=_slave)
                opt_slave = config.unwrap_from_path(path, force_permissive=force_permissive)
                _type = type_option[type(opt_slave)]
                if current_tag is not False:
                    tag = opt_slave.impl_get_information('separator', (None, None))[0]
                    if tag is not None:
                        current_tag = tag
                #FIXME +++duplicate
                slaves.append(_get_variable(opt_slave,_type,  _slave,
                                            category_name, current_tag))
            except PropertiesOptionError:
                pass
        slave = name
        path = get_path(category_name, name, master_slave=slave)
        option = config.unwrap_from_path(path, force_permissive=force_permissive)
        type_ = type_option[type(option)]
    if slave is not None:
        vname = slave
    else:
        vname = name
    if current_tag is not False:
        tag = option.impl_get_information('separator', (None, None))[0]
        if tag is not None:
            current_tag = tag
    variable = _get_variable(option, type_, vname, category_name, current_tag)
    if slaves is not None:
        master_values = variable['value']
        _vars = []
        for idx, val in enumerate(master_values):
            var = copy(variable)
            var['id'] = var['id'] + '_' + str(idx)
            var['value'] = val
            var['groupid'] = vname
            var['index'] = idx
            var.pop('multi')
            var['slaves'] = []
            for slave in deepcopy(slaves):
                slave['id'] = slave['id'] + '_' + str(idx)
                slave['value'] = slave['value'][idx]
                slave.pop('multi')
                slave['groupid'] = vname
                slave['index'] = idx
                var['slaves'].append(slave)
            _vars.append(var)
        variable = {'name': vname, 'id': vname,
                    'description': variable['description'], 'masters': _vars,
                    'type': 'group', 'categoryid': category_name,
                    'mode': variable['mode']}
    if current_tag is not False:
        return current_tag, variable
    else:
        return variable


def is_category(id_, category_name):
    """ Tell whether a category exists or not
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :returns: True if it is a category
    :type   : `bool`
    """
    config = get_config(id_)
    if category_name not in \
            [name for name, conf in config.creole.iter_groups()]:
        return False
    return True


def get_variables(id_, category_name):
    """ Get list of variables for a specified category
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :returns: list of variables from category
    :type   : `list`
    """
    config = get_config(id_)

    variables = {}
    old_tag = None
    for child in getattr(config.creole, category_name).cfgimpl_get_description().impl_getchildren():
        try:
            name = child._name
            tag, variable = get_variable(id_, category_name, name, current_tag=old_tag)
            old_tag = tag
            variables.setdefault(tag, []).append(variable)
        except PropertiesOptionError:
            pass
    return variables


def get_tags(id_, category_name):
    """ Get list of tags for a specified category
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :returns: list of tags from category
    :type   : `list`
    """
    config = get_config(id_)

    tags = []
    current_tag = None
    old_tag = None

    for child in getattr(config.creole, category_name).cfgimpl_get_description().impl_getchildren():
        try:
            name = child._name
            tag, variable = get_variable(id_, category_name, name, current_tag=old_tag)
            if current_tag is None:
                current_tag = {"id": 'Configuration', 'variables': []}
            if tag and (current_tag is None or current_tag['id'] != tag):
                if current_tag['variables'] != []:
                    tags.append(current_tag)
                current_tag = {'id': tag, 'variables': []}
            current_tag['variables'].append(variable)
            old_tag = tag
        except PropertiesOptionError:
            tag = child.impl_get_information('separator', (None, None))[0]
            if tag is not None:
                old_tag = tag
    if current_tag != None:
        tags.append(current_tag)
    return tags


def get_variables_set(id_, variables):
    """ Get list of variable from a set
    :param id_: session id
    :type  id_: `str`
    :param variables: variables ids set
    :type  variables: `list`
    :returns: list of variables from the set
    :type   : `list`
    """
    config = get_config(id_)

    variables_set = []
    old = []
    for variable in variables:
        path = config.creole.find_first(byname=variable, type_='path')
        path = '.'.join(path.split('.')[:3])
        if path not in old:
            variables_set.append(get_variable_from_path(id_, path))
            old.append(path)
    return variables_set

def set_value_multi(id_, category_name, master_name, variable_name, value, index):
    """ Save the value of a mutli type variable
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param master_name: master variable name
    :type  master_name: `str`
    :param variable_name: for master/slave name must be master's name
                          slave must be slave's name
                          (or master's one if slave not None)
    :type  variable_name: `str`
    :param value: value to set to the variable
    :type  value: `str`
    :param index: index of the value to update
    :type  index: `int`
    """
    config = get_config(id_)
    path = get_path(category_name, master_name, variable_name)
    parent_option = config.unwrap_from_path(get_path(category_name))
    option = config.unwrap_from_path(path)
    value = convert_value(option, value)
    if master_name is not None:
        master_option = config.unwrap_from_path(get_path(category_name, master_name))
        config.cfgimpl_get_settings().setpermissive(tuple(modes_level), opt=master_option)
    config.cfgimpl_get_settings().setpermissive(tuple(modes_level), opt=parent_option)
    config.cfgimpl_get_settings().setpermissive(tuple(modes_level), opt=option)
    config.impl_set_information(option, '')
    with warnings.catch_warnings(record=True) as warns:
        getattr(config, path)[index] = value
    for warn in warns:
        config.impl_set_information(warn.message.opt, str(warn.message))
    config.cfgimpl_get_settings()[option].remove("load_error")


def set_value(id_, category_name, variable_name, value):
    """ Save the value of a standard type variable
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param variable_name: for master/slave name must be master's name
                          slave must be slave's name
                          (or master's one if slave not None)
    :type  variable_name: `str`
    :param value: value to set to the variable
    :type  value: `str`
    """
    config = get_config(id_)
    path = get_path(category_name, variable_name)
    parent_option = config.unwrap_from_path(get_path(category_name))
    option = config.unwrap_from_path(path)
    if isinstance(option, OptionDescription):
        # Cas Master sans Slaves interprétée comme multi dans genconfig
        path = get_path(category_name, variable_name, variable_name)
        option = config.unwrap_from_path(path)
    if option.impl_is_multi():
        values = []
        for val in value:
            values.append(convert_value(option, val))
        value = values
    else:
        value = convert_value(option, value)
    config.cfgimpl_get_settings().setpermissive(tuple(modes_level), opt=parent_option)
    config.cfgimpl_get_settings().setpermissive(tuple(modes_level), opt=option)
    config.impl_set_information(option, '')
    with warnings.catch_warnings(record=True) as warns:
        setattr(config, path, value)
    for warn in warns:
        config.impl_set_information(warn.message.opt, str(warn.message))
    config.cfgimpl_get_settings()[option].remove("load_error")


def pop_value(id_, category_name, master_name, variable_name, index):
    """ Removes a value from a mutli type variable
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param master_name: master variable name
    :type  master_name: `str`
    :param variable_name: for master/slave name must be master's name
                          slave must be slave's name
                          (or master's one if slave not None)
    :type  variable_name: `str`
    :param index: index of the value to remove
    :type  index: `int`
    """
    if master_name != variable_name:
        raise ValueError('master_name and variable_name must have same value',
                'not {} and {}'.format(master_name, variable_name))
    config = get_config(id_)
    path = get_path(category_name, variable_name, variable_name)
    option = config.unwrap_from_path(path)
    value = getattr(config, path)
    config.impl_set_information(option, '')
    with warnings.catch_warnings(record=True) as warns:
        value.pop(index)
    for warn in warns:
        config.impl_set_information(warn.message.opt, str(warn.message))
    config.cfgimpl_get_settings()[option].remove("load_error")


def append_value(id_, category_name, master_name, variable_name):
    """ Append an empty value in a mutli type variable
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param master_name: master variable name
    :type  master_name: `str`
    :param variable_name: for master/slave name must be master's name
                          slave must be slave's name
                          (or master's one if slave not None)
    :type  variable_name: `str`
    """
    if master_name != variable_name:
        raise ValueError('master_name and variable_name must have same value',
                'not {} and {}'.format(master_name, variable_name))
    config = get_config(id_)
    path = get_path(category_name, master_name, variable_name)
    option = config.unwrap_from_path(path)
    value = getattr(config, path)
    config.impl_set_information(option, '')
    with warnings.catch_warnings(record=True) as warns:
        value.append()
    for warn in warns:
        config.impl_set_information(warn.message.opt, str(warn.message))
    config.cfgimpl_get_settings()[option].remove("load_error")


def reset_value(id_, category_name, master_name, variable_name):
    """ Reset a variable value
    :param id_: session id
    :type  id_: `str`
    :param category_name: name of the category
    :type  category_name: `str`
    :param master_name: master variable name
    :type  master_name: `str`
    :param variable_name: for master/slave name must be master's name
                          slave must be slave's name
                          (or master's one if slave not None)
    :type  variable_name: `str`
    """
    config = get_config(id_)
    path = get_path(category_name, master_name, variable_name)
    option = config.unwrap_from_path(path)
    homeconfig, name = config.cfgimpl_get_home_by_path(path)
    homeconfig.__delattr__(name, force_permissive=True)
    config.cfgimpl_get_settings().setpermissive(tuple(), opt=option)
    if master_name is not None:
        master_option = config.unwrap_from_path(get_path(category_name, master_name))
        config.cfgimpl_get_settings().setpermissive(tuple(), opt=master_option)
    config.cfgimpl_get_settings()[option].remove("load_error")


def set_debug(id_, is_debug):
    """ Enable/Disable debug mode
    :param id_: session id
    :type  id_: `str`
    :param is_debug: True to enable debug mode
    :type  is_debug: `bool`
    :returns: session debug value
    :type   :`bool`
    """
    config = get_config(id_)
    if is_debug:
        config.cfgimpl_get_settings().remove('hidden')
    else:
        config.cfgimpl_get_settings().append('hidden')
    session_tiramisu[id_]['debug'] = not 'hidden' in config.cfgimpl_get_settings()
    return session_tiramisu[id_]['debug']


def get_debug(id_):
    """ Figure whether in debug mode or not
    :param id_: session id
    :type  id_: `str`
    :returns: session debug value
    :type   :`bool`
    """
    if id_ in session_tiramisu:
        return session_tiramisu[id_]['debug']
    else:
        return False

def set_mode(id_, mode):
    """ Define which edition mode to select
    :param id_: session id
    :type  id_: `str`
    :param mode: possible values = ['basic', 'normal', 'expert']
    :type  mode: `str`
    :returns: session mode value
    :type   :`bool`
    """
    config = get_config(id_)
    for mode_level in modes.values():
        if modes[mode] < mode_level:
            config.cfgimpl_get_settings().append(mode_level.name)
        else:
            config.cfgimpl_get_settings().remove(mode_level.name)
    # store mode in session in case config object gets reloader
    session_tiramisu[id_]['mode'] = mode


def get_mode(id_):
    """ Get current edition mode
    :param id_: session id
    :type  id_: `str`
    :returns: session mode value
    :type   :`bool`
    """
    return session_tiramisu[id_]['mode']


def get_variable_from_path(id_, path, force_permissive=False):
    """ Get variable from a path
    :param id_: session id
    :type  id_: `str`
    :param path: path to variable
    :type  path: `str`
    :param force_permissive: abstract permissions to get variable
    :type  force_permissive: `bool`
    :returns: dictionnary of variable properties
    :type   : `dict`
    """
    path = path.split('.')
    category_name = path[1]
    name = path[2]
    if len(path) == 3:
        slave = None
    else:
        name = path[2]
        slave = path[3]
    return get_variable(id_, category_name, name, slave,
                        force_permissive=force_permissive)


def valid_mandatory(id_, zephir_sync=False):
    """ Search mandatory options without value
    :param id_: session id
    :type  id_: `str`
    :param zephir_sync: synchro with Zéphir first ?
    :type  zephir_sync: `bool`
    :returns: list of mandatory variables = [{'category': 'cat1', 'variable': 'var1'}, ...]
    :type   : `dict`
    """
    zephir = get_zephir(id_)
    if zephir['available'] and zephir_sync:
        config = get_zephir_config(id_)
        return [] # dont validate mandatories if config from Zéphir
    else:
        config = get_config(id_)
    #Zéphir specific code
    source = session_tiramisu[id_].get('source', None)
    # no check on mandatories when editing default values (Zéphir)
    if source in ('variante', 'module'):
        mandatory_errors = []
    else:
        mandatory_errors = list(config.cfgimpl_get_values().mandatory_warnings(force_permissive=True))
    ret = []
    old = []
    for path in mandatory_errors:
        #for master/slave, get master
        if not path.startswith('creole'):
            continue
        path = '.'.join(path.split('.')[:3])
        if path not in old:
            ret.append(get_variable_from_path(id_, path, force_permissive=True))
            old.append(path)
    return ret


def diff_config(id_, zephir_sync=False):
    """Calculate difference between stored values and current's one
    :param id_: session id
    :type  id_: `str`
    :returns: {'var1': {'old_value': 'val1'}, 'var2': {'old_value': 'oval2',
            'new_value': 'nval2'}, 'var3': {'new_value': 'val3'}}
    :type   : `dict`
    """
    config = get_config(id_)
    if zephir_sync:
        reload_all()
        #difference avec la configuration Zéphir
        new_config = creole_loader()
        try:
            store = new_config.impl_get_information('orig_values')
        except:
            store = {}
    else:
        try:
            store = config.impl_get_information('orig_values')
        except:
            store = {}
    opt_values = config.cfgimpl_get_values()
    new_store = OrderedDict()
    source = session_tiramisu[id_].get('source', None)
    # when editing default values in Zéphir, we don't force auto_save(/freeze)
    if not source in ('variante', 'module'):
        #to force store auto_save/auto_freeze values
        get_values(id_, check_mandatory=True)

    for path, own_val in opt_values.get_modified_values().items():
        owner, value = own_val
        name = path.split('.')[-1]
        new_store[name] = {'val': value}
        new_store[name]['owner'] = owner
    #add auto_freeze not freeze
    setting = config.cfgimpl_get_settings()
    for path in config.cfgimpl_get_description().impl_getpaths():
        try:
            option = config.unwrap_from_path(path, force_permissive=True)
            name = path.split('.')[-1]
            value = config._getattr(path)
            if 'auto_freeze' in setting[option] and 'frozen' not in setting[option]:
                #option is not frozen before instance, we only check value (#7406)
                if name not in new_store or new_store[name]['val'] != value:
                    new_store[name] = {'val': value, 'owner': setting.getowner()}
        except PropertiesOptionError, err:
            if 'hidden' in err.proptype and not zephir_sync:
                #if option is hidden in current config, don't show it :
                try:
                    store.pop(name)
                except KeyError:
                    pass
                try:
                    new_store.pop(name)
                except KeyError:
                    pass
            pass
    store_set = set(store.keys())
    new_store_set = set(new_store.keys())
    diffs = []
    left_diff = store_set - new_store_set
    right_diff = new_store_set - store_set
    intersect = store_set & new_store_set

    for full_key in config.cfgimpl_get_description()._cache_paths[1]:
        key = full_key.split('.')[-1]
        if key in left_diff:
            try:
                # no new keys -> old_values
                path = config.creole.find_first(byname=key, type_='path',
                                                force_permissive=True)
                diffs.append(dict(
                    get_variable_from_path(id_, path,
                                           force_permissive=True).items() + {
                        'old_value': store[key]['val'],
                        'old_user': store[key]['owner']
                    }.items()
                ))
            except AttributeError:
                #remove older/disabled variable's
                pass
        elif key in right_diff:
            try:
                path = config.creole.find_first(byname=key, type_='path',
                                                force_permissive=True)
                diffs.append(dict(
                    get_variable_from_path(id_, path,
                                           force_permissive=True).items() + {
                        'new_value': new_store[key]['val'],
                        'new_user': new_store[key]['owner']
                    }.items()
                ))
            except AttributeError:
                #for example when imported config file
                pass
        elif key in intersect:
            if store[key] != new_store[key]:
                try:
                    path = config.creole.find_first(byname=key, type_='path',
                                                    force_permissive=True)
                    diffs.append(dict(
                        get_variable_from_path(id_, path,
                                               force_permissive=True).items() + {
                            'old_value': store[key]['val'],
                            'old_user': store[key]['owner'],
                            'new_value': new_store[key]['val'],
                            'new_user': new_store[key]['owner']
                        }.items()
                    ))
                except AttributeError:
                    pass
    return diffs


def get_values(id_, check_mandatory=True):
    """ Get a JSON dump of the values from configuration
    :param id_: session id
    :type  id_: `str`
    :returns: JSON string of the configuration
    :type   : `str`
    """
    config = get_config(id_)
    return config_get_values(config, 'creole', check_mandatory=check_mandatory)


def save_values(id_, comment, source=None):
    """ Save all values in config.eol file.
    :param id_: session id
    :type  id_: `str`
    :param comment: add comment to the save procedure
    :type  comment: `str`
    :param source: which configuration to save = ['distant','local'}
    :type  source: `str`
    """
    zephir = get_zephir(id_)
    if zephir['available'] and source == 'distant':
        config = get_zephir_config(id_)
    elif zephir['available'] and source == 'local':
        del_config(id_)
        config = get_config(id_)
    else:
        config = get_config(id_)
    config_save_values(config, 'creole', reload_config=False, eol_file=configeol)
    # send remote config
    save_zephir_config(id_)
    # return mode and debug to previous state
    # after reloading config
    mode = get_mode(id_)
    debug = get_debug(id_)
    set_mode(id_, mode)
    set_debug(id_, debug)

def save_zephir_config(id_):
    """ Sends the configuration to Zéphir server
    :param id_: session id
    :type  id_: `str`
    """
    zephir = get_zephir(id_)
    values = get_values(id_)
    try:
        code, msg = zephir['proxy'].serveurs.save_conf(zephir['server_id'], [unicode(str(values))])
    except Exception, e:
        # FIXME: Log Error messages
        code = 0
        msg = str(e)

