#!/bin/bash

#
# Enregistrement Zéphir automatique
#

source ./context.sh

ZCREDS="zcreds.sc"
ZCA="zephir-ca.crt"
BOOT_LOG_FILE="/var/log/hapy-deploy-$(date +'%d%m%Y').log"

export HTTP_PROXY=${PROXY}
export HTTPS_PROXY=${PROXY}
export http_proxy=${PROXY}
export https_proxy=${PROXY}

function lgmsg() {
  echo "$(date +%d%m%y-%H%M%S):${@}" | tee -a ${BOOT_LOG_FILE}
}

function setup_network() {
  device=$(CreoleGet nom_zone_eth0)
  ip=""
  mask=""
  gw=""
  dns=""

  if [[ -n ${ETH0_IP} ]]
  then
    ip=${ETH0_IP}
    mask=${ETH0_MASK}
    gw=${ETH0_GATEWAY}
    dns=${ETH0_DNS}
  elif [[ -n ${IP_ETH0} ]]
  then
    ip=${IP_ETH0}
    mask=${NETMASK_ETH0}
    gw=${GW_ETH0}
    dns=${DNS_ETH0}
  fi

  if [[ -n ${ip} ]]
  then
    # Up the interface
    ip link set up ${device}
    if [[ $? -ne 0 ]]
    then
      return 4
    fi
    # Setup IP for "ETH0"
    ip a add ${ip}/${mask} dev ${device} | tee -a ${BOOT_LOG_FILE}
    if [[ $? -ne 0 ]]
    then
      echo "Debug"
      echo $?
      return 2
    fi

    # Setup default gateway
    out=$(ip route add default via ${gw} onlink dev ${device} 2>&1 | tee -a ${BOOT_LOG_FILE})
    res=${?}
    if [[ "${out}" =~ ^Error.* ]]
    then
      echo "DBG ${out}"
      return 3
    fi

    echo "nameserver ${dns}" > /etc/resolv.conf
  else
    # Update dhclient configuration
    sed -e 's/timeout.*/timeout 5;/g' -i /etc/dhcp/dhclient.conf

    dhclient -v ${device} | tee -a ${BOOT_LOG_FILE}
    if [[ ${?} -ne 0 ]]
    then
      return 1
    else
      return 0
    fi
  fi
}

function enr_zeph(){
    creds=($(< ${ZCREDS}))
    addr=${creds[0]}
    user=${creds[1]}
    pass=${creds[2]}

    eole_version=$(CreoleGet eole_version)
    http_proxy=${HTTP_PROXY} https_proxy=${HTTPS_PROXY} apt-eole install expect
    /usr/bin/expect << __EOF__
spawn enregistrement_zephir
set timeout 300
log_file ${BOOT_LOG_FILE}
expect {
  "*Voulez-vous établir une configuration réseau minimale*" { send -- N\n; exp_continue }
  "*Entrez l'adresse*" { send -- ${addr}\r; exp_continue; }
  "*Entrez votre login pour l'application Zéphir*" { send -- ${user}\r; exp_continue; }
  "*Mot de passe pour l'application Zéphir pour*" { send -- ${pass}\r; exp_continue; }
  "*continuer l'enregistrement (O/N) ?*" { send -- O\n; exp_continue; }
  "*créer le serveur dans la base du serveur Zéphir*" { send -- N\r; exp_continue; }
  "*rien pour saisir directement*de serveur*" { send -- \r; exp_continue; }
  "*entrez le n° identifiant le serveur l'application*" { send -- ${ZEPHIR_ID}\n; exp_continue; }
  "*Voulez vous migrer le serveur vers le module*" { send -- O\n; exp_continue; }
  "*matériel*" { send -- \n; exp_continue; }
  "*processeur*" { send -- \n; exp_continue; }
  "*disque dur*" { send -- \n; exp_continue; }
  "*exp_continuer*" { send -- O\r; exp_continue; }
  "*Entrez le numéro de votre choix*" { send -- 2\r; exp_continue; }
  "*le numéro attribué à ce serveur sur le serveur Zéphir est : ${ZEPHIR_ID}*" { exp_continue; }
  timeout { exit }
  eof { exit }
}
__EOF__
    return $?
}



function is_registred(){
  ot=$(enregistrement_zephir -c)
  return $?
}

function is_zephir_available(){
  creds=($(< ${ZCREDS}))
  addr=${creds[0]}

  cnt=0
  timeout=900
  while [[ ${cnt} -ne ${timeout} ]]
  do
    nc -z -w2 ${addr} 7080
    if [[ ${?} -eq 0 ]]
    then
      return 0
      break
    fi
    cnt=$((cnt+1))
    sleep 1
    lgmsg "ENREGISTREMENT_ZEPHIR:RUNNING:Waiting for ${addr} port 7080 [${cnt}]"
    if [[ ${timeout} -eq ${cnt} ]]
    then
      return ${timeout}
      break
    fi
  done
}

[[ ! -e ${BOOT_LOG_FILE} ]] && touch ${BOOT_LOG_FILE}

if [[ -e ${ZCREDS} ]]
then
    if [[ -e ${ZCA} ]]
    then
        cp -v ${ZCA} /usr/local/share/ca-certificates/ | tee -a ${BOOT_LOG_FILE}
        update-ca-certificates | tee -a ${BOOT_LOG_FILE}
    fi

    is_registred
    if [[ ${?} -eq 0 ]]
    then
      lgmsg "ENREGISTREMENT_ZEPHIR:DONE:"
      exit 0
    fi

    setup_network
    case "${?}" in
      "0")
        lgmsg "SETUP_NETWORK:OK"
        ;;
      "1")
        lgmsg "SETUP_NETWORK:ERROR:ETH0_IP is empty in context"
        exit 1
        ;;

      "2")
        lgmsg "SETUP_NETWOKR:ERROR:Adding IP to ETH0 failed"
        exit 2
        ;;

      "3")
        lgmsg "SETUP_NETWORK:ERROR:Adding default gateway failed"
        exit 3
        ;;

      "4")
        lgmsg "SETUP_NETWORK:ERROR:Link up failed"
        exit 3
        ;;


      *)
        lgmsg "SETUP_NETWORK:ERROR:Unknown error"
        exit 255
        ;;
    esac

    sleep 5
    is_zephir_available
    if [[ $? -ne 0 ]]
    then
      lgmsg "ENREGISTREMENT_ZEPHIR:KO:Zephir server is not responding"
      exit 4
    fi

    enr_zeph
    r=${?}
    if [[ $r -eq 0 ]]
    then
      lgmsg "ENREGISTREMENT_ZEPHIR:DONE:DONE"
      sleep 5 # cool down things
      exit 0
    else
      exit $r
    fi
else
   lgmsg "ERROR: ${ZCREDS} file is missing !"
   exit 10
fi
