# -*- coding: utf-8 -*-
#########################################################################
# eole-password - password management for eole
# Copyright © 2012 Pôle de Compétence EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
#########################################################################

"""EOLE Password manage passwords for EOLE

To use Eole password you only have to play with the :class:`Password`
provide all the necessary informations to this class will give you
the expected result "Changing passwords"

The other classes are used by :class:`Password` and "private" (even this
don't means anything for python) so don't use it !

Usage example :

handler = Password()
handler.process()

"""

import re
#import sys
import yaml
import hashlib
import getpass
import MySQLdb
import crypt
import spwd as shadow
import random
import time
import tempfile
import filecmp
import shutil
import psycopg2
from os import remove, close, write, makedirs, listdir
from os.path import isfile, basename as bsname, dirname, join, isdir

ALPHA = "abcdefghijklmnopqrstuvwxyz0123456789"

#CONFDIR   = "/tmp/eole-password/config/eole-password"
#ACTIONLOG = "/tmp/eole-password/log/eole-password.log"
#MAIN_CONF = "/tmp/etc/eole/eolepwd.cfg"
CONFDIR = "./config/eole-password"
ACTIONLOG = "./log/eole-password.log"
MAIN_CONF_FILE = "./config/main.cfg"

MAIN_CONF = None

""" List of Supported Engines
"""
SUPPORTED_DB = ['mysql',
                'pgsql',
                'sqlite']


class EolePwdError(Exception):
    pass


class EpwdConfig():
    """ Private class to load supported database configuration
    """

    def __init__(self, configfile=MAIN_CONF_FILE):
        self.configfile = configfile
        self.config = None
        for support in SUPPORTED_DB:
            setattr(self, support + '_config', [])

    def add_conf(self, attribute, attr_name, value):
        """ Private method to add configuration elements to allready loaded
            configuration. Do not use.
        """
        ret = []
        orig = getattr(self, attr_name)
        if isinstance(orig, dict):
            ret = [orig, value]
        elif isinstance(orig, list):
            ret = orig.append(value)

        setattr(self, attr_name, ret)

    def reload(self, newfile):
        """ Private method to reload a configuration. Do not use.
        """
        self.configfile = newfile
        self.config = None

    def get_conf(self, ctype):
        """ Get a configuration for a given type
        """
        arg_name = ctype + '_config'
        config = getattr(self, arg_name)

        if config == []:
            self.load_conf()
            for conf in config:
                self.add_conf(config, arg_name, conf)

        return(getattr(self, arg_name))

    def load_conf(self):
        """ Load the configuration from yaml file
        """
        try:
            self.config = yaml.load_all(file(self.configfile))
            for elm in self.config:
                for support in SUPPORTED_DB:
                    supp = support + '-dbhost'
                    if supp in elm:
                        attrib = support + '_config'
                        setattr(self, attrib, elm)
                        break
        except IOError as err:
            print("ERROR {0}: {1} {2}".format(err.errno,
                                              err.strerror,
                                              self.configfile))
            raise EolePwdError("Configuration Load failed (%s)" % err.strerror)


class ActionLog:
    """ Private class to manage eole-password action log file
        This class provides two methods :method:`update_log` to add
        and entry on the action log file, :method:`is_in_log` to check
        if an entry allready exists

        This is to use inside of eole-password
    """

    def __init__(self, logfile):
        """
        :param logfile: the path to action log file
        :type logfile: full path
        """
        self.log = logfile

    def update_log(self, message):
        """ Add a message to the action log
        The action log is used for the support of option 'onetime'.
        You may whant to change some user password only one time.

        :param message: The formated message
        :type message: Formated String (format : tag:user:timestamp)
        """
        if not isdir(dirname(self.log)):
            try:
                makedirs(dirname(self.log))
            except OSError as err:
                print("ERROR[{0}]: {1}".format(err.errno, err.strerror))

        with open(self.log, 'a+') as logfd:
            logfd.write(message)

    def is_in_log(self, username):
        """ check if any entry contains the username in action log

        :param username: The user name
        :type username: String
        """
        expr = '(:' + username + ':)'
        try:
            with open(self.log, 'r') as logfd:
                for line in logfd.readlines():
                    if re.search(expr, line):
                        return True
                    else:
                        return False
        except:
            print("Log file is missing")


class EolePassword(object):
    """Master class for password handling in Eole

    This Class offer the common methods and attributes for all password
    manipulation classes.

    An EolePassword is represented by :
        - a username :attr:`self.user`
        - an old password :attr:`self.old_pass`
        - an new clear password :attr:`self.ncl_pass`
        - an new crypted password :attr:`self.ncr_pass`

    Do not use this class directly use the interface :class:`Password`

    """
    def __init__(self, name):
        self.user = name
        self.old_pass = ""
        self.ncl_pass = ""
        self.ncr_pass = ""
        #self.new_pass = ""

    def gen_salt(self, length=8):
        """ Generate a salt of passed length

        The generated salt is composed of alphanumeric characters.
        It picks random character in an alphanumeric list

        :param length: The salt length (default 8)
        :type length: Integer
        """

        salt = []
        for elm in range(length):
            salt.append(random.choice(ALPHA))
        return "".join(salt)

    def gen_rand_pass(self):
        """ Generate a random password

        The generated a random password composed of alphanumeric and special
        characters.

        :return: Random password
        :rtype: string
        """
        passwd_chars = ALPHA + '~!@#$%^&*-_=+?'
        passwd_length = random.sample([10, 11, 12], 1).pop()
        return ''.join(random.sample(passwd_chars, passwd_length))

    def update_conf_file(self, path, pattern):
        """ Replace a password on given file
        :param file: The file name
        :type file: string (full path)
        :param pattern: The pattern to match
        :type pattern: regexp pattern
        """

        try:
            conf_fd = open(path, 'r')
            tmpfd, tmppath = tempfile.mkstemp()
            pat = ('(?<=%s).[\w,\W]\S*' % pattern)
            for line in conf_fd.readlines():
                if line.startswith(pattern):
                    line = re.sub(pat, ("'%s'" % self.ncl_pass), line)
                write(tmpfd, line)
            conf_fd.close

            if not filecmp.cmp(path, tmppath):
                shutil.copy2(path, path + ".bck")
                shutil.copy2(tmppath, path)

            close(tmpfd)
            remove(tmppath)
        except IOError as err:
            print("WARN {0}: {1} {2}".format(err.errno,
                                             err.strerror,
                                             path))
            return False


class EoleShadow(EolePassword):
    """
    Shadow passowrd operations and manipulation

    This child of :class:`EolePassword` handle all actions for /etc/shadow
    password changing support

    This is a "private" class  used by the Interface class :class:`Password`,
    do not use !

    """

    def shadowcrypt(self, password, saltlen=8, hashalgo=6, salt=None):
        """ Crypt a string to the "shadow" format

        This is a private method used by :method:`gen_new_password`,
        do not use !

        :param saltlen: Length of salt string (default 8)
        :type saltlen: Integer
        :param hashalgo: Hash algorithm  (6 for SHA512)
        :type hashalgo: Integer
        :param salt: The salt for hashing (default None means this is
        generated by :method:`self.gen_salt`)
        :type salt: String
        """

        if salt is None:
            salt = self.gen_salt(saltlen)
        return crypt.crypt(password, '$%s$%s$' % (hashalgo, salt))

    def get_user_info(self):
        """ d user informations from Shadow
        Set the value of :attr:`old_pass`

        This is a private class used by Password interface class do not use.

        :return: True of operation success False if failed
        :rtype: Boolean
        """
        try:
            pwd = shadow.getspnam(self.user)
            self.old_pass = pwd.sp_pwd
            return True
        except:
            return False

    def gen_new_password(self, mode):
        """ Generate a new crypted password.

        Set a value for :attr:`new_pass`

        This is a private class used by Password interface class do not use.

        :param mode: The password operation type (onetime or auto)
        :type mode: String (onetime or auto)
        """
        shpwd = ""
        if mode == 'auto':
            shpwd = self.gen_rand_pass()
        elif mode == 'manual':
            print("\nChanging password :\n-------------------")
            print("Username : %s" % self.user)
            shpwd = getpass.getpass('Password : ')
            print("-------------------")
        else:
            raise EpwdConfig("FIXME ERROR")

        self.ncr_pass = self.shadowcrypt(shpwd)

    def update_password(self, operations):
        print("FIXME !! Update password")
        pass


class EoleMypwd(EolePassword):
    """
    MySQL password operations and manipulation this class don't use
    any password you need a my.cnf on /etc with root password.
    """
    def __init__(self, name, host):
        self.dbc = None
        if host == "all":
            self.host = '%'
        else:
            self.host = host
        EolePassword.__init__(self, name)

    def open_dbc(self):
        """ Open a Data Base connection (dbc)

        This is a private class used by EoleMypwd class do not use.
        """
        conf = MAIN_CONF.get_conf("mysql")
        host = conf['mysql-dbhost']
        user = conf['mysql-dbuser']
        if conf['mysql-dbpass']:
            dbpass = conf['mysql-dbpass']
        else:
            dbpass = None
        if 'mysql-port' in conf:
            port = conf['mysql-dbport']
        else:
            port = "3306"

        try:
            if dbpass:
                self.dbc = MySQLdb.connect(host=host,
                                           user=user,
                                           port=int(port),
                                           db='mysql',
                                           passwd=dbpass)
            else:
                # FIXME Ne marche pas en mode conteneur a mettre dans le
                # fichier de conf
                my_conf = "/etc/mysql/my.cnf"
                self.dbc = MySQLdb.connect(host=host,
                                           user=user,
                                           port=int(port),
                                           db='mysql',
                                           read_default_file=my_conf)

        except MySQLdb.OperationalError as err:
            print("ERROR 5 : Connection to {0}:{1} failed".format(host, port))
            raise EolePwdError("Connection to %s:%s failed" % (host, port))

    def close_dbc(self):
        """ Close a Data Base connection (dbc)

        This is a private class used by EoleMypwd class do not use.
        """
        self.dbc.close

    def run_req(self, request):
        """ Run a SQL request on opened DBCO

        This is a private class used by EoleMypwd class do not use.
        """
        try:
            cursor = self.dbc.cursor()
            affected_rows = cursor.execute(request)
            if affected_rows > 0:
                res = cursor.fetchone()
                if res:
                    return res
                else:
                    return True
            else:
                print("WARN 1 : Running SQL request failed")
                return None
        except:
            return None

    def mysql_crypt_pass(self, password):
        pass1 = hashlib.sha1(password).digest()
        pass2 = hashlib.sha1(pass1).hexdigest()
        return "*" + pass2.upper()

    def get_user_info(self):
        if not self.dbc:
            self.open_dbc()

        req_opwd = "SELECT Password "
        req_opwd += "FROM user "
        req_opwd += "WHERE User = '%s'" % self.user
        if self.user != "root":
            req_opwd += "AND Host = '%s'" % self.host

        print(req_opwd)

        self.old_pass = self.run_req(req_opwd)
        if self.old_pass:
            return True
        else:
            return False

    def gen_new_password(self, mode):
        if mode == 'auto':
            self.ncl_pass = self.gen_rand_pass()
        else:
            print("\nChanging MySQL password :\n------------------------")
            print("Username : %s" % self.user)
            self.ncl_pass = getpass.getpass('Password : ')
            print("------------------------")

        self.ncr_pass = self.mysql_crypt_pass(self.ncr_pass)

    def update_password(self, actions):
        """ Run all replacing password operations

        Pass a list of dict like {'file': 'filename', pattern: 'password='}

        :param actions: The actions to run
        :type actions: list of dict
        """
        if self.user == "root":
            req = ("UPDATE user SET password=PASSWORD(\'%s\')" % self.ncl_pass)
            req += (" WHERE User=\'%s\';" % self.user)
            req += ("flush privileges;")
        else:
            req = ("SET PASSWORD FOR \'%s\'@\'%s\'" % (self.user, self.host))
            req += (" = PASSWORD(\'%s\');" % (self.ncl_pass))

        if self.run_req(req):
            self.update_conf_file(MAIN_CONF_FILE, "mysql-dbpass: ")

        for action in actions:
            self.update_conf_file(action['file'], action['pattern'])


class EolePgPwd(EolePassword):
    """
    Postgresql password operations and manipulation.
    """
    def __init__(self, name, host):
        self.dbc = None
        EolePassword.__init__(self, name)

    def open_dbc(self):
        """ Open a Data Base connection (dbc)

        This is a private class used by EoleMypwd class do not use.
        """
        conf = MAIN_CONF.get_conf("pgsql")
        host = conf['pgsql-dbhost']
        user = conf['pgsql-dbuser']
        pwd = conf['pgsql-dbpass']

        if 'pgsql-port' in conf:
            port = conf['pgsql-dbport']
        else:
            port = "5432"

        try:
            conn_str = "dbname='template1' user=" + user
            if host != "localhost":
                conn_str += " host='" + host + "'"
            conn_str += " password='" + pwd + "'"
            self.dbc = psycopg2.connect(conn_str)
        except psycopg2.OperationalError:
            msg = "Unable to open connection to posgresql server"
            print("ERROR 10: {0} : {1}".format(msg, host))
            raise EolePwdError("%s" % (msg))

    def close_dbc(self):
        """ Close the Data Base connection (dbc)

        This is a private class used by EoleMypwd class do not use.
        """
        if self.dbc:
            self.dbc.close()

    def run_req(self, request):
        """ Run a SQL request on opened DBCO

        This is a private class used by EoleMypwd class do not use.
        """
        try:
            cursor = self.dbc.cursor()
            affected_rows = cursor.execute(request)
            if affected_rows > 0:
                res = cursor.fetchone()
                if res:
                    return res
                else:
                    return True
            else:
                print("WARN 1 : Running SQL request failed")
                return None
        except:
            return None

    def get_user_info(self):
        if not self.dbc:
            self.open_dbc()

        req_opwd = "SELECT passwd "
        req_opwd += "FROM pg_user "
        req_opwd += "WHERE username = '%s'" % self.user
#        if self.user != "root":
#            req_opwd += "AND Host = '%s'" % self.host

        print(req_opwd)

        self.old_pass = self.run_req(req_opwd)
        if self.old_pass:
            return True
        else:
            return False


class User():
    """ User is the object discribed in the configuration file.

        This class is instancied by the the configuration loader.

        A :class:`User` is discribed by :
            - a name :attr:`self.name`
            - a type :attr:`self.usertype`
            - a password :attr:`self.password` (instance of EolePassword)
            - a database type (if needed) :attr:`self.db`

    """
    def __init__(self, element):
        self.name = element['user']
        self.host = element['host']
        self.usertype = element['type']
        self.password = None
        if element['mode'] is None:
            self.mode = 'auto'
        else:
            self.mode = element['mode']

        if element['to_change'] is None:
            self.to_change = []
        else:
            self.to_change = element['to_change']
        self.password_factory()

    def password_factory(self):
        if self.usertype == "system":
            self.password = EoleShadow(self.name)
        elif self.usertype == "mysql":
            self.password = EoleMypwd(self.name, self.host)
        elif self.usertype == "pgsql":
            self.password = EolePgPwd(self.name, self.host)
        else:
            print("Unsupported type %s for user %s" % (self.usertype,
                                                       self.name))

    def add_task(self, value):
        if value['to_change']:
            self.to_change += value['to_change']

    def __repr__(self):
        return "[User {}]".format(self.name)


class UserContainer(dict):
    """ UserContainer is a child of dict class.
    This particular dict contains all the "actions" to be
    done by eole-password and stored in configuration

    This way we can manage the actions like a dict
    """

    def __setitem__(self, key, value):
        """ Add an item to this particular dict

        If the key allready exist in dict, it add a task to
        the existing element. A task is represented by the :class:`User`
        attribute :attr:`to_change`.

        :param key: The dict entry key
        :param value: The value for this dict entry
        """
        if key not in self:
            dict.__setitem__(self, key, User(value))
        else:
            self[key].add_task(value)


class Password:
    """ Password is th Interface class you can use.

    This class loads configuration and offer the methods for password changing

    """

    def __init__(self,
                 actionlog=ACTIONLOG,
                 confdir=CONFDIR,
                 mainconfig=MAIN_CONF,
                 directconfig=None):
        self.actionlog = ActionLog(actionlog)
        self.config_dir = confdir
        self.operations = UserContainer()
        self.directconfig = directconfig

        MAIN_CONF.reload(mainconfig)

    def load_conf(self, directory=None, ctype=None):
        """ Load configuration in YAML format from a directory
        and fill the :attr:`self.operations`

        Each operation is identified by a triplet : type, user, database

        If you change a single password for a user in many appliation
        (consultation accounts)
        The password is changed one time and all the configurations are filled.

        This is a private method (do not use)

        :param directory: The configuration directory
        :type directory: String
        """
        if not directory:
            directory = self.config_dir

        try:
            onlyfiles = []
            for fld in listdir(directory):
                name = join(directory, fld)
                """ Ignore hidden files """
                if fld[0] == '.':
                    continue
                # Configuration have to be a "file"
                # Can't be the MAIN_CONF_FILE
                # Can't have ".bck" extention
                ext = name.split(".")[-1]
                if (isfile(name) and
                        bsname(name) != bsname(MAIN_CONF_FILE) and
                        ext != 'bck'):
                    onlyfiles.append(fld)
        except OSError as err:
            print("ERROR {0}: Listing \"{2}\" {1}".format(err.errno,
                                                          err.strerror,
                                                          directory))
            raise EolePwdError("Listing \"%s\" failed (%s)" % (directory,
                                                               err.strerror))

        for config_file in onlyfiles:
            try:
                dico = yaml.load_all(file(directory + '/' + config_file))
                for elm in dico:
                    if not 'host' in elm:
                        elm['host'] = "%"
                    if not 'to_change' in elm:
                        elm['to_change'] = None
                    else:
                        acts = elm['to_change']
                        for act in acts:
                            if 'file' not in act or 'pattern' not in act:
                                raise yaml.scanner.ScannerError

                    if not 'mode' in elm:
                        elm['mode'] = None

                    if ctype:
                        if elm['type'] == type:
                            self.operations[(elm['type'],
                                             elm['user'],
                                             elm['host'])] = elm
                    else:
                        self.operations[(elm['type'],
                                         elm['user'],
                                         elm['host'])] = elm
            except KeyError as err:
                msg = "Invalid configuration file"
                print("ERROR 3: {0} : {1}".format(msg, config_file))
                raise EolePwdError("%s : %s" % (msg, config_file))

            except yaml.scanner.ScannerError as err:
                msg = "Configuration syntax error"
                if directory[-1] == '/':
                    cfile = directory + config_file
                else:
                    cfile = directory + '/' + config_file
                print("ERROR 4: {0} : {1}".format(msg, cfile))
                raise EolePwdError("%s : %s" % (msg, config_file))

    def renew(self, operation):
        """ Start all operations for password change.
        This method process each operation contained in
        :attr:`self.operations`.

        """
        if operation.password.get_user_info():
            is_in_log = self.actionlog.is_in_log(operation.password.user)
            if operation.mode == "onetime" and is_in_log:
                print("Nothing to do for %s" % operation.password.user)
            else:
                """ Generates new password """
                operation.password.gen_new_password(operation.mode)
                """ Update all configured files """
                operation.password.update_password(operation.to_change)
                tag = time.strftime("%d%m%Y%H%M%S")
                message = "CHANGED:"
                message += operation.password.user + ":" + tag + "\n"
                self.actionlog.update_log(message)
        else:
            msg = "User information for "
            msg += operation.password.user
            msg += " not found"
            print(msg)

    def run_operations(self):
        """ Private method to run all stored actions (operations)
        """
        for oprt in self.operations.values():
            if oprt.password:
                self.renew(oprt)

    def process(self, ptype=None):
        """ This is the called method :
            This runs configuration loading and password renew operations

        """
        if ptype:
            self.load_conf(self.config_dir, ptype)
            self.run_operations()
        else:
            if self.directconfig:
                operation = User(self.directconfig)
                self.renew(operation)
            else:
                self.load_conf(self.config_dir)
                self.run_operations()


MAIN_CONF = EpwdConfig()
