# -*- mode: python; coding: utf-8 -*-

from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData, HTMLData
from zephir.monitor.agentmanager.util import status_to_img
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.util import log

from twisted.internet.utils import getProcessOutput

from os import remove
from os.path import isfile

SMARTCTL = '/usr/sbin/smartctl'
TEST_LINE = 'overall-health'

attr_col={'id':0,
          'name':1,
          'flag':2,
          'value':3,
          'worst':4,
          'thresh':5,
          'type':6,
          'updated':7,
          'when':8,
          'raw':9,
          }

class SmartMonAlertes(Agent):
   def __init__(self, name, attr_agents, **params):
      Agent.__init__(self, name, **params)
      self.status = status.Unknown()
      # Attributs data are fetch from attributs agent
      self.attr_agents = attr_agents
      self.disabled = {}

      try:
         # recherche de l'identifiant zephir 
         # (nécessaire pour l'url du graphe de status des 'sous agents')
         from zephir.zephir_conf.zephir_conf import id_serveur
         self.id_serveur = id_serveur
      except:
         # si on est pas enregistré : 0
         self.id_serveur = 0
            
      self.status = status.Unknown()

      for attrs in self.attr_agents:
         disk = attrs.disk
         # Alert file is clear at each start
         alert_file = attrs.alert_file
         if isfile(alert_file):
            remove(alert_file)

      self.table = TableData([
            ('disk', 'Disque', {'align':'center'}, None),
            ('status', 'État', {'align':'center'}, None),
            ('type', 'Type', {'align':'left'}, None),
            ('alert', 'Alerte', {'align':'left'}, None),
            ])
      title = HTMLData("<h3>Surveillance des disques&nbsp;:</h3>")

      self.data = [ title, self.table ]

   def measure(self):
      measure=[]
      for attrs in self.attr_agents:
         # No smart, no data
         disabled = self.is_disabled(attrs)
         if disabled:
            measure.append(disabled)
            continue

         alert = {'type':'', 'msg':''}
         if not isfile(attrs.alert_file):
            measure.append( self.get_status_from_attrs(attrs) )
         else:
            self.status = status.Error()
            fic = file(attrs.alert_file)
            info = fic.read().strip()
            fic.close()

            # Parse messages
            for line in info.splitlines():
               if line.startswith('MESSAGE'):
                  msg_start = line.find(':')+1
                  measure.append({ 'disk' : self.attr_link(attrs),
                                   'status' : status.Error(),
                                   'type'   : alert['type'],
                                   'alert'  : alert['msg'] + line[msg_start:].strip(),
                                   })
                  alert = {'type':'', 'msg':''}
               elif line.startswith('FAILTYPE'):
                  alert['type'] += line.split(':')[1].strip()

      # for item in measure:
      #    log.msg(item)
      return { self.name : measure }

   def write_data(self):
      Agent.write_data(self)
      if self.last_measure is not None:
         self.table.table_data = self.last_measure.value[self.name]

   def check_status(self):
      """Remonte une erreur si un disque est en erreur"""
      if self.last_measure is not None:
         return self.status
      else:
         return status.Unknown()

   def is_disabled(self, attrs):
      if attrs.disabled is not None:
         if self.status in [ status.OK(), status.Unknown() ]:
            self.status = status.Warn()
         return { 'disk'   : self.attr_link(attrs),
                  'status' : status.Warn(),
                  'type'   : 'SMART Disabled',
                  'alert'  : attrs.disabled,
                  }

   def get_status_from_attrs(self, attrs):
      agent_status = attrs.check_status()
      if agent_status not in [ status.OK(),  status.Unknown() ]:
         self.status = agent_status
         return { 'disk'   : self.attr_link(attrs),
                  'status' : agent_status,
                  'type'   : 'attributes',
                  'alert'  : 'Vérifier les attributs SMART',
                  }
      else:
         if self.status not in [ status.Error(),  status.Warn() ]:
            self.status = status.OK()
         return { 'disk'   : self.attr_link(attrs),
                  'status' : status.OK(),
                  'type'   : '',
                  'alert'  : '',
                  }

   def attr_link(self, agent, name=""):
      if not name:
         name = agent.disk
      return '<a href="/agents/%s/%s">%s</a>' % (self.id_serveur, agent.name, name)

class SmartMonAttributes(Agent):
   def __init__(self, name, disk, disabled, **params):
      Agent.__init__(self, name, **params)
      from creole.parsedico import parse_dico
      try:
         # nouvelle version : pas de variables cachées
         d = parse_dico(True)
      except:
         d = parse_dico()

      self.dico = {
         'smartmontools_devs_attrs_fail_ignore':d['smartmontools_devs_attrs_fail_ignore'],
         'smartmontools_devs_attrs_change_ignore':d['smartmontools_devs_attrs_change_ignore'],
         'smartmontools_attr_warn': d['smartmontools_attr_warn'],
         'smartmontools_track_temp':d['smartmontools_track_temp'],
         'smartmontools_temp_diff':d['smartmontools_temp_diff'],
         'smartmontools_temp_info':d['smartmontools_temp_info'],
         'smartmontools_temp_crit':d['smartmontools_temp_crit'],
         }

      self.disk = disk
      self.alert_file = "/run/shm/smart.d/" + disk.replace('/', '_')

      # Does disk support smart ?
      if disabled is not None:
         self.status = status.Warn()
         self.disabled = disabled
      else:
         self.disabled = None
         self.status = status.Unknown()

      title = HTMLData("<h3>Surveillance du disque : <i>%s</i></h3>" % self.disk)
      self.table = TableData([
            ('id', 'ID', {'align':'center'}, None),
            ('name', 'Nom', {'align':'left'}, None),
            ('status', 'État', {'align':'center'}, None),
            ('value', 'Valeur', {'align':'center'}, None),
            ('worst', 'Maximum', {'align':'center'}, None),
            ('thresh', 'Seuil', {'align':'center'}, None),
            ('type', 'Type', {'align':'center'}, None),
            ('updated', 'Mise à jour', {'align':'center'}, None),
            ('when', 'Dernier échec', {'align':'center'}, None),
            ('raw', 'Valeur brute', {'align':'center'}, None),
            ])
      self.data = [title, self.table]

   def measure(self):
      # No smart, no data
      if self.disabled is not None:
         self.status = status.Warn()
         return { self.name : [] }

      measure = getProcessOutput(SMARTCTL,
                                 args = ['--all', self.disk],
                                 env = {'LC_ALL':'C'} )
      measure.addCallback(self.parse_health)
      return measure

   def parse_health(self, output):
      data = []

      for line in output.splitlines():
         if line.find(TEST_LINE) != -1:
            if line.find('PASSED') != -1:
               self.status = status.OK()
            else:
               self.status = status.Error()

      attrs_start = output.find('ID#')
      attrs_end   = output.find('\n\n', attrs_start)
      for line in output[attrs_start:attrs_end].splitlines():
         if line.startswith('ID#'):
            continue
         else:
            attr = line.split()
            if str(attr[attr_col['id']]) in self.dico['smartmontools_devs_attrs_fail_ignore']:
               continue
            if str(attr[attr_col['id']]) in self.dico['smartmontools_devs_attrs_change_ignore']:
               continue
            # Value <= Thresh is Error
            if int(attr[attr_col['value']]) <= int(attr[attr_col['thresh']]):
               attr_status = status.Error()
               self.status = status.Error()
            # Value <= Thresh + 10 is Warning
            elif int(attr[attr_col['value']]) <= \
                   int(attr[attr_col['thresh']]) + int(self.dico['smartmontools_attr_warn']):
               attr_status = status.Warn()
               if self.status != status.Error():
                  self.status = status.Warn()
            else:
               attr_status = status.OK()
               if self.status == status.Unknown():
                  self.status = status.OK()
            data.append({
                  'id':attr[attr_col['id']],
                  'name':attr[attr_col['name']],
                  'status':attr_status,
                  'value':attr[attr_col['value']],
                  'worst':attr[attr_col['worst']],
                  'thresh':attr[attr_col['thresh']],
                  'type':attr[attr_col['type']],
                  'updated':attr[attr_col['updated']],
                  'when':attr[attr_col['when']],
                  'raw':attr[attr_col['raw']],
                  })
      # Some debug about what data contains
      # for item in data:
      #    log.msg(item)
      return { self.name : data }

   def write_data(self):
      Agent.write_data(self)
      if self.last_measure is not None:
         self.table.table_data = self.last_measure.value[self.name]

   def check_status(self):
      """Remonte une erreur si un disque est en erreur"""
      if self.last_measure is not None:
         return self.status
      else:
         return status.Unknown()

   def get_name(self):
      """Renvois le nom de l'agent"""
      if self.name is not None:
         return self.name

   def get_attrs(self):
      """Renvois le tableau des attributs SMART"""
      if self.last_measure is not None:
         return self.last_measure.value
