# -*- coding: utf-8 -*-
"""ce module est une extension openid pour supporter les
informations supplémentaires nécessaires à Eole

Code inspiré de openid.sreg qui définit l'extension `Simple Registration`
"""

from openid.extension import Extension
from openid.message import registerNamespaceAlias

EOLE_OPENID_NS = 'http://ead.eole.com/openid/eoledata'

# apparemment seulement nécessaire pour compatibilité OpenId 1
registerNamespaceAlias(EOLE_OPENID_NS, 'eole')


def supportsEoleData(endpoint):
    """Est-ce que ce endpoint dit supporter l'extension eole ?

    @param endpoint: l'objet endpoint tel que renvoyé par openid.discover
    @type endpoint: openid.consumer.discover.OpenIDEndpoint

    @returns: si l'extension Eole est supportée ou pas
    @rtype: bool
    """
    # XXX à définir dans le document yadis ??
    return endpoint.usesExtension(EOLE_OPENID_NS)


class EoleExtension(Extension):
    ns_uri = EOLE_OPENID_NS
    ns_alias = 'eole'
    
    # Liste des champs définis dans l'extension EOLE
    data_fields = {
        'secureid'    : "Identifiant anonyme (crypté)",
        'user_groups'    : "Groupes de l'utilisateur",
        'uid'     : "Login de l'utilisateur",
        'sn'   : 'Nom de famille',
        'givenName' : 'Prénom',
        'dateNaissance' : 'Date de naissance',
        'mail'     : 'Adresse email',
        'codecivilite'    : 'Civilité',
        }

    fieldlist = frozenset(data_fields)
    

class EoleExtensionRequest(EoleExtension):
    """Cette classe encapsule une demande d'information openid sur les
    informations relatives à l'extension Eole
    """
    def __init__(self, required=(), optional=()):
        """
        @param required: la liste des informations que l'on souhaite
          obligatoirement obtenir
        @param optional: la liste des informations optionnelles 
        """
        super(EoleExtensionRequest, self).__init__()
        self.required = []
        self.optional = []
        self.addRequestFields(required, True)
        self.addRequestFields(optional, False)

    @classmethod
    def fromOpenIDRequest(cls, request):
        """Create a simple registration request that contains the
        fields that were requested in the OpenID request with the
        given arguments

        @param request: The OpenID request
        @type request: openid.server.CheckIDRequest

        @returns: The newly created simple registration request
        @rtype: C{L{SRegRequest}}
        """
        args = request.message.getArgs(cls.ns_uri)
        print 'args ==', args
        required = args.get('required')
        required = required and required.split(',') or ()
        optional = args.get('optional')
        optional = optional and optional.split(',') or ()
        return cls(required, optional)

    
    def getExtensionArgs(self):
        args = {}
        if self.required:
            args['required'] = ','.join(self.required)
        if self.optional:
            args['optional'] = ','.join(self.optional)
        return args
    

    def addRequestField(self, fieldName, required=False):
        """Request the specified field from the OpenID user

        @param fieldName: the unqualified simple registration field name
        @type fieldName: str

        @param required: whether the given field should be presented
            to the user as being a required to successfully complete
            the request

        @raise AssertionError: when the field requested is not a simple
            registration field or strict is set and the field was
            requested more than once
        """
        assert fieldName in self.data_fields, "%r n'est pas champ autorisé" % fieldName
        assert fieldName not in self.allFields(), "%r est utilisé plusieurs fois" % fieldName
        if required:
            self.required.append(fieldName)
        else:
            self.optional.append(fieldName)

            
    def addRequestFields(self, fieldNames, required=False):
        """Add the given list of fields to the request

        @param field_names: The simple registration data fields to request
        @type field_names: [str]

        @param required: Whether these values should be presented to
            the user as required

        @raise ValueError: when a field requested is not a simple
            registration field or strict is set and a field was
            requested more than once
        """
        for fieldName in fieldNames:
            self.addRequestField(fieldName, required)

    def allFields(self):
        return set(self.required + self.optional)
    
    def __iter__(self):
        return iter(self.allFields())

def proxyAttr(attr, methName):
    def meth(self, *args, **kwargs):
        return getattr(getattr(self, attr), methName)(*args, **kwargs)
    return meth

class EoleExtensionResponse(EoleExtension):
    
    def __init__(self, data=None):
        super(EoleExtensionResponse, self).__init__()
        self.data = data or {}


    @classmethod
    def extractResponse(cls, request, responseData):
        data = {}
        for fieldName in request:
            value = responseData.get(fieldName)
            if value is not None:
                data[fieldName] = value
        return cls(data)

    @classmethod
    def fromSuccessResponse(cls, response, signedOnly=True):
        """Create a C{L{SRegResponse}} object from a successful OpenID
        library response
        (C{L{openid.consumer.consumer.SuccessResponse}}) response
        message

        @param success_response: A SuccessResponse from consumer.complete()
        @type success_response: C{L{openid.consumer.consumer.SuccessResponse}}

        @param signed_only: Whether to process only data that was
            signed in the id_res message from the server.
        @type signed_only: bool

        @rtype: SRegResponse
        @returns: A simple registration response containing the data
            that was supplied with the C{id_res} response.
        """
        if signedOnly:
            args = response.getSignedNS(cls.ns_uri)
        else:
            args = response.message.getArgs(cls.ns_uri)
        data = dict( (fieldName, args[fieldName]) for fieldName in cls.fieldlist
                     if fieldName in args )
        return cls(data)

    
    def getExtensionArgs(self):
        """Get the fields to put in the simple registration namespace
        when adding them to an id_res message.

        @see: openid.extension
        """
        return self.data

    ## Read-only dictionary interface
    get = proxyAttr('data', 'get')
    items = proxyAttr('data', 'items')
    iteritems = proxyAttr('data', 'items')
    keys = proxyAttr('data', 'keys')
    iterkeys = proxyAttr('data', 'iterkeys')
    has_key = proxyAttr('data', 'has_key')
    __contains__ = proxyAttr('data', '__contains__')
    __iter__ = proxyAttr('data', '__iter__')
    __getitem__ = proxyAttr('data', '__getitem__')
    __len__ = proxyAttr('data', '__len__')
