# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# ticket.py
#
# Classe représentant un ticket CAS de type applicatif ou proxy (ST/PT)
# fonctionnement inspiré de la librairie cas :
# http://www.ja-sig.org/products/cas/
#
###########################################################################
from eolesso.util import gen_ticket_id, get_service_from_url

def checkpath(proxypath):
    """this simple decorator checks proxies list is somewhat consistent
    with the ticket type :
    - ST ticket should not have any proxy
    - PT tickets should have at least one proxy
    """
    def wrapper(self):
        proxies = proxypath(self)
        if self.ticket.startswith('ST'):
            assert len(proxies) == 0
        elif self.ticket.startswith('PT'):
            assert len(proxies) > 1
        return proxies
    return wrapper

class AppTicket(object):

    def __init__(self, ticket, session_id, service_url, cas_url, from_credentials=False, parent_ticket=None):
        """
        @param ticket: le ticket applicatif servant à la validation (PT ou ST)
        @param session_id: la session utilisateur associée au ticket
        @param service_url: l'url du service demandant l'authentification
        @param cas_url: adresse du serveur d'authentification
        @param parent_ticket: instance de ticket parent (dernier proxy traversé)
        """
        self.ticket = ticket
        self.session_id = session_id
        self.service_url = get_service_from_url(service_url)
        self.parent = parent_ticket
        self.cas_url = cas_url
        self.filter = ''
        self.pgt_url = None
        self.pgt = None
        self.pgtiou = None
        self.valid = True
        self.from_credentials = from_credentials

    def generate_pgt(self, pgt_url):
        """
        @pparam pgt_url: addresse de rappel du proxy
        génère un ticket permettant au service de servir de proxy pour
        l'authentification. après cet appel, le ticket PGT doit être stocké
        dans le dictionnaire PGT -> AppTicket
        Avant de stocker le ticket, on effectue une vérification sur pgt_url
        Si pgt_url n'est pas conforme, on renvoie False
        """
        self.pgt = gen_ticket_id('PGT', self.cas_url)
        self.pgtiou = gen_ticket_id('PGTIOU', self.cas_url)
        self.pgt_url = pgt_url.geturl()

    def reset_pgt(self):
        self.pgt_url = None
        self.pgt = None
        self.pgtiou = None

    def verif_ticket(self, ticket, service_url, keep_valid = False):
        """
        vérifie que le ticket est valide et correspond au bon service
        Le ticket ne peut être validé qu'une fois
        """
        if self.valid:
            # dans tous le cas, on invalide le ticket (valable une seule fois)
            if not keep_valid:
                self.valid = False
            return (ticket == self.ticket and
                    get_service_from_url(service_url) == self.service_url)
        return False

    @checkpath
    def proxypath(self):
        if self.pgt_url is None or self.parent is None:
            return []
        return [self.parent.pgt_url] + self.parent.proxypath()
