# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# util.py
#
# fonctions utilitaires pour la librairie sso eole
# (actuellement seulement annuaire LDAP)
#
###########################################################################
"""
    Librairies d'utilitaires pour le serveur eole-sso
"""
import urlparse, socket, re, os, IPy
import time, random, hashlib
from ConfigParser import ConfigParser
from glob import glob

from twisted.python import log
from twisted.web2.http_headers import parseCookie
from twisted.web2.http import StatusResponse, responsecode

repl_dir = '/etc/ldap/replication'
info_etabs = os.path.join(repl_dir, 'zephir', 'etabs.ini')
# définition du répertoire de l'application dispatcher
dispatcher_dir = '/var/www/html/edispatcher'

# redéfinition de la classe RedirectResponse de twisted.web2
# qui utilise le code 301 au lieu de 303
class RedirectResponse (StatusResponse):
    """
    A L{Response} object that contains a redirect to another network location.
    """
    def __init__(self, location):
        """
        @param location: the URI to redirect to.
        """
        super(RedirectResponse, self).__init__(
            responsecode.SEE_OTHER,
            "Redirecting to %s." % (location,)
        )

        self.headers.setHeader("location", location)

def urljoin(url, params):
    """ajoute des paramètres à une url
    >>> urljoin('http://test/','a=b')
    'http://test/?a=b'
    >>> urljoin('http://test/?a=b','c=d')
    'http://test/?a=b&c=d'
    """
    urllist = list(urlparse.urlsplit(url))
    if urllist[-2] == '':
        urllist[-2] = params
    else:
        urllist[-2] += '&' + params
    return urlparse.urlunsplit(urllist)

def format_err(msg):
    """
        renvoie une balise contenant une erreur
    """
    return """<h1><font color="red">%s</font></H1>""" % msg

def cas_response(resptype, body, **attrs):
    """
        convenience cas response factory
    """
    attrs = ' '.join('%s="%s"' % kv for kv in attrs.items())
    if attrs:
        attrs = ' ' + attrs
    return str("""<?xml version="1.0" encoding="UTF-8"?>
<cas:serviceResponse xmlns:cas="http://www.yale.edu/tp/cas">
  <cas:%(rtype)s%(attrs)s>
    %(body)s
  </cas:%(rtype)s>
</cas:serviceResponse>\r\n\r\n""" % \
{'rtype': resptype, 'attrs': attrs, 'body': body})

def is_true(param):
    if param is None:
        return False
    return param.lower() == 'true'

def unescape(url, quote=None):
    """
        supprime les caractères escapés dans une url
    """
    entities = [('&gt;','>'), ('&lt;','<'), ('&amp;','&')]
    if quote:
        entities.insert(0, ('&quot;','"'))
    for substr, car in entities:
        url = url.replace(substr, car)
    return url

def get_service_from_url(url):
    """
        supprime la partie requête d'une url
        pour permettre d'indentifier le service
    """
    url = unescape(url)
    u = urlparse.urlparse(url)
    service = "%s://%s%s" % (u.scheme, u.netloc, u.path)
    return service

def getCookies(ctx):
    """
        renvoie la liste  des cookies sous forme Twisted
    """
    cookiestr = ctx.headers.getRawHeaders('Cookie')
    if cookiestr:
        return parseCookie(cookiestr)
    return []

def getCookie(ctx, key):
    """renvoie un cookie Twisted"""
    for cookie in getCookies(ctx):
        if cookie.name == key:
            return cookie
    return None

def verif_cert(conn, cert, err_num, err_depth, rc):
    """fonction de vérification du certificat"""
    return not cert.has_expired()

def check_hostname_by_ip(url, addr):
    """vérification de la validité de l'hôte de l'url en fonction d'une IP
    """
    try:
        # vérification de l'adresse
        ip = IPy.IP(addr)
        try:
            hostname = socket.gethostbyname(url.hostname)
        except:
            hostname = url.hostname
        assert ip.overlaps(hostname)
        return True
    except:
        return False
    return False

def check_hostname_by_domain(url, addr):
    """
        vérification de la validité de l'hôte de l'url
        par expression régulière
    """
    #try:
    #    hostname = socket.gethostbyaddr(url.hostname)[0]
    #except:
    hostname = url.hostname
    # on essaye un match de l'expression régulière sur le nom d'hôte
    try:
        match_obj = re.match(addr, hostname)
    except:
        log.msg(_("""Error matching regexp '%s' (invalid syntax ?)""") % addr)
        return False

    return match_obj is not None

def gen_random_id(prefix=''):
    """
        Renvoie un identifiant aléatoire
        :prefix: préfixe à placer devant la string
    """
    random_id = str(time.time())
    return prefix + hashlib.sha224('%s/%s' % (
                    random_id, random.randrange(2**100))).hexdigest()

def gen_ticket_id(prefix, address):
    """
        à redéfinir dans la classe dérivée
        génère un id de ticket ex : ST-monetab.ac-acad.fr-51de51d5e1d5e1d51d5e1d5e1
        :prefix: le préfixe décrivant le type de ticket
        :address: l'adresse associée au ticket fourni
    """
    ticket_id = prefix + '-' + address + '-' + gen_random_id()
    return ticket_id

def get_replication_branches():
    """
        returns search_bases as defined in /etc/ldap/replication.conf
        if available
    """
    F_CONF_REPL = '/etc/ldap/replication.conf'
    search_bases = {}
    libel_etab = ""
    if os.path.isfile(F_CONF_REPL):
        f_repl = file(F_CONF_REPL)
        for line in f_repl:
            line = line.strip()
            if line.startswith('#') and line.endswith(')'):
                # on récupère le libellé de l'établissement qui
                # apparait en début de section
                libel_etab = line[1:].strip()
            if line.startswith("searchbase="):
                branch = line.strip()
                # on supprime searchbase= et les guillemets
                branch = branch[branch.index("=") + 2:-1]
                if not libel_etab:
                    # si libellé non trouvé, on cherche le rne
                    # (normalement, premier objet du dn de recherche)
                    libel_etab = branch.split(',')[0]
                    libel_etab = libel_etab[libel_etab.index('=') + 1:]
                search_bases[branch] = libel_etab
                libel_etab = ""
        f_repl.close()
    return search_bases

def update_etab_js(js_filename, dispatcher_dir=dispatcher_dir):
    """Mise à jour du fichier javascript de description des établissements répliqués
    Met également à jour le fichier etabs.ini du dispatcher
    """
    update_ok = True
    etab_js = ["\nvar etabs=new Array();"]
    # lecture des infos dans le fichier
    infos = ConfigParser()
    etabs_ini = ConfigParser()
    if os.path.isfile(info_etabs):
        try:
            infos.read(info_etabs)
        except:
            log.msg('Erreur lors de la lecture du fichier %s' % info_etabs)
            update_ok = False
    # si des fichiers de personnalisation existent, on les lit
    for etab_perso in glob(os.path.join(repl_dir, "etabs_*.ini")):
        try:
            # les infos sur des établissements supplémentaires ou des
            # redéfinitions des données présentes dans le .ini géré par
            # zéphir peuvent être récupérées dans les fichiers etabs_*.ini
            # dans le répertoire de réplication.
            infos.read(etab_perso)
        except:
            log.msg('Erreur lors de la prise en compte du fichier %s' % etab_perso)
    for rne in infos.sections():
        libelle=type_etab=""
        if 'libelle_etab' in infos.options(rne):
            libelle = infos.get(rne, 'libelle_etab')
        if 'type_etab' in infos.options(rne):
            type_etab = infos.get(rne, 'type_etab')
        if libelle.upper().startswith(type_etab.upper()):
            libelle = libelle.upper().replace(type_etab.upper(), '', 1).strip()
        else:
            libelle = libelle.upper()
        etab_js.append("""etabs["%s"]={libelle:"%s",type:"%s"};""" % (rne.upper(), libelle, type_etab.capitalize()))
        # écriture d'un fichier d'information (rne: libellé, portal) pour le service dispatcher
        etabs_ini.add_section(rne)
        etabs_ini.set(rne, 'libelle', infos.get(rne, 'libelle_etab'))
        if 'portail_etab' in infos.options(rne):
            etabs_ini.set(rne, 'portail', infos.get(rne, 'portail_etab'))
        else:
            etabs_ini.set(rne, 'portail', '')

    try:
        # fichier d'infos établissments du dispatcher si nécessaire
        if os.path.isdir(dispatcher_dir):
            f_ini = open(os.path.join(dispatcher_dir, 'utils', 'etabs.ini'),'w')
            etabs_ini.write(f_ini)
            f_ini.close()
    except:
        log.msg('Erreur de génération du fichier %s' % os.path.join(dispatcher_dir, 'utils', 'etabs.ini'))
    # sauvegarde du fichier d'établissements pour EoleSSO
    # (utilisé pour la liste déroulante en cas d'homonymes)
    f_etab_js = open(js_filename, 'w')
    f_etab_js.write("\n".join(etab_js)+"\n")
    f_etab_js.close()

class EoleParser(ConfigParser):
    """Sous classe de ConfigParser pour préserver la casse des options"""

    def optionxform(self, option_str):
        return option_str

def check_url(host, port):
    s = socket.socket()
    try:
        s.connect((host, int(port)))
    except:
        return False
    s.close()
    return True

def edu_user_source(user_infos):
    """Renvoie la provenance de l'utilisateur en fonction des attributs détectés
    """
    source = "UNKNOWN"
    if "FrEduVecteur" in user_infos:
        source = "TS"
    elif 'intid' in user_infos:
        source = "ENT"
    elif "FrEduRne" in user_infos:
        source = "AGENT"

def is_acad(user_infos):
    return edu_user_source(user_infos) == "AGENT"

def is_ts(user_infos):
    return edu_user_source(user_infos) == "TS"

def is_ent(user_infos):
    return edu_user_source(user_infos) == "ENT"
