# -*- coding: utf-8 -*-
"""Ce module définit un service d'authentification basé sur le
protocole openid 2.0.

Définition du protocole openid 2.0 :

http://openid.net/specs/openid-authentication-2_0-09.html


"""
import cgi
import sys
import Cookie
from datetime import datetime
from time import time
from urllib import quote

from twisted.internet import reactor
from twisted.web2 import server, static, http, channel, responsecode
from twisted.web2.http_headers import MimeType
from twisted.web2.resource import Resource, PostableResource
from twisted.web2.http_headers import Cookie as TwCookie, parseCookie

from openid.server import server as oidserver
from openid.consumer import discover
from openid.store.filestore import FileOpenIDStore

from eolesso.oidlib import simplifyQuery, HTMLResponse, getHostName
from eolesso import oideole
from eolesso.util import RedirectResponse
from config import AUTH_FORM_URL

## web resources #####################################################

class PersistencyManager(object):
    """classe responsable de gérer les états des connections et
    authentifications
    """
    def __init__(self, session_manager, baseurl):
        endpoint = baseurl + 'openidserver'
        store = FileOpenIDStore('oid_store.dat')
        self.session_manager = session_manager
        self.baseurl = baseurl
        self.oidserver = oidserver.Server(store, endpoint)
        # Ce dictionnaire liste l'ensemble des sites ("trusted_root") toujours
        # autorisés par une identité donnée
        # structure du dicionnaire (identify, trusted_root) ==> 'always'
        self.approved = {}
        self.allowed_fields = {}
        self.lastCheckIDRequest = {}
        self.lastAuthRequest = {}
        # (hard) proxy attributes
        for attr in ('get_user_info', 'get_app_ticket', 'get_user_details',
                     'verify_session_id', '_check_filter', 'filters'):
            setattr(self, attr, getattr(session_manager, attr))

    def serverYadisURL(self):
        return self.baseurl + 'serveryadis'

    def endpointURL(self):
        return self.baseurl + 'openidserver'



def setuser_and_query(func):
    """petit décorateur pour positionner self.user et self.query"""
    def newfunc(self, request):
        cookie = self.getCookie(request,'EoleSSOServer')
        self.sso_login_ticket = None
        self.user = None
        if cookie != None:
            cookie = cookie.value
            self.sso_login_ticket = cookie
            try:
                self.user = self.manager.get_user_info(self.sso_login_ticket)
            except:
                self.sso_login_ticket = None
            # print 'no user cookie found'
        # print "cookie parsed, user : %s, ticket : %s" % (self.user, self.sso_login_ticket)
        self.query = simplifyQuery(request.args)
        return func(self, request)
    return newfunc



class IPResourceMixIn(object):
    """Identity Provider MixIn : définit tous les comportements
    communs aux urls gérées par le server
    """
    def __init__(self, persistencyManager):
        self.manager = persistencyManager
        # on déclare ces deux attributs pour faire un peu plus propre
        self.user = None
        self.query = None
        super(IPResourceMixIn, self).__init__()


    def buildTwistedResponse(self, response):
        """méthode qui transforme une réponse openid en réponse
        HTTP / Twisted
        """
        # print 'response', type(response)
        try:
            webresponse = self.manager.oidserver.encodeResponse(response)
        except oidserver.EncodingError, why:
            text = why.response.encodeToKVForm()
            print 'got error while building twisted response', text
            return http.StatusResponse(500, '<pre>%s</pre>' % text,
                                       title="encoding error")
        except Exception, why:
            print 'got error while building twisted response', str(why)
            return http.StatusResponse(500, '<pre>%s</pre>' % str(why),
                                       title="error")
        # print 'building twisted response', webresponse.headers, webresponse.body
        return http.Response(code=webresponse.code,
                             stream=webresponse.body,
                             headers=webresponse.headers)

    def addAuthDataResponse(self, request, response):
        eole_req = oideole.EoleExtensionRequest.fromOpenIDRequest(request)
        # création d'un app_ticket pour l'url à autoriser
        assert self.sso_login_ticket is not None
        app_ticket = self.manager.get_app_ticket(self.sso_login_ticket, request.trust_root)
        if app_ticket:
            # récupération des infos sur le serveur d'authentification (infos filtrée)
            app_ok, eole_data = self.manager.get_user_details(app_ticket, request.trust_root)
        if app_ok:
            # supression des champs non autorisés par l'utilisateur
            eole_data = self.filterFields(eole_data, request)
        else:
            eole_data = {}
        eole_resp = oideole.EoleExtensionResponse.extractResponse(eole_req, eole_data)
        response.addExtension(eole_resp)

    def filterFields(self, login_info, request):
        # login_info = login_info.copy()
        login_filtered = {}
        key = (request.identity, request.trust_root)
        if self.manager.allowed_fields.get(key) != None:
            # choix des champs stockés dans PersistencyManager
            allowed_attrs = self.manager.allowed_fields.get(key)
        else:
            allowed_attrs = self.query.keys()
        for attribute, info in login_info.items():
            if 'attr_'+attribute in allowed_attrs:
                login_filtered[attribute] = ",".join(info)
        return login_filtered

    def grant(self, request, identifier=None):
        response = request.answer(True, identity=identifier)
        self.addAuthDataResponse(request, response)
        return response

    def deny(self, request):
        return request.answer(False)


    def cookieHeaders(self):
        if self.user is None:
            return None
        else:
            return TwCookie("user", self.user, path="/oid/",
                            expires= time() + 2*3600, discard=True, secure=True)

    def getCookies(self, ctx):
        """renvoie la liste  des cookies sous forme Twisted"""
        cookiestr = ctx.headers.getRawHeaders('Cookie')
        # print 'cookiestr' , repr(cookiestr)
        if cookiestr:
            return parseCookie(cookiestr)
        return []

    def getCookie(self, ctx, key):
        """renvoie un cookie Twisted"""
        for cookie in self.getCookies(ctx):
            if cookie.name == key:
                return cookie
        return None


class IPResource(IPResourceMixIn, Resource):
    """définition de base d'une url accessible par GET"""


class IPPostableResource(IPResourceMixIn, PostableResource):
    """définition de base d'une url accessible par POST"""


def outdate(cookie):
    """périme un cookie twisted"""
    cookie.expires = 1. #'Thu, 01-Jan-1970 00:00:01 GMT'


class OpenIdServerResource(IPPostableResource):
    """endpoint url"""

    @setuser_and_query
    def render(self, ctx):
        cookies = self.getCookies(ctx)
        for cookie in cookies:
            if cookie.name == 'auth-request':
                auth_cookie = cookie
                break
        else:
            auth_cookie = None
        response = self._render(ctx)
        if auth_cookie:
            outdate(auth_cookie)
            self.manager.lastAuthRequest.pop(tuple(auth_cookie.value.split('#')), None)
            response.headers.setHeader('Set-Cookie', cookies)
        return response

    def _render(self, ctx):
        request = self.getPendingAuthRequest(ctx)
        if request is None:
            request = self.manager.oidserver.decodeRequest(self.query)

        if request is None:
            # Display text indicating that this is an endpoint.
            return http.Response(stream=str(HTMLResponse(title='error',
                                                  content="vous vous êtes égaré (openidserver)")))

        if request.mode in ("checkid_immediate", "checkid_setup"):
            return self.handleCheckIDRequest(request)
        else:
            response = self.manager.oidserver.handleRequest(request)
            return self.buildTwistedResponse(response)

    def getPendingAuthRequest(self, ctx):
        cookie = self.getCookie(ctx,  'auth-request')
        if cookie:
            return self.manager.lastAuthRequest.get(tuple(cookie.value.split('#')))
        return None

    def isAuthorized(self, identity_url, trust_root):
        if self.user is None:
            return False

        if identity_url != self.manager.baseurl + 'id/' + self.user:
            return False

        key = (identity_url, trust_root)
        return self.manager.approved.get(key) is not None


    def handleCheckIDRequest(self, request):
        user_verified = False
        if self.user:
            # vérification de la validité de la session
            verif =  self.manager.verify_session_id(self.sso_login_ticket)
            user_verified = verif[0]
        if not user_verified:
            self.manager.lastAuthRequest[(request.identity, request.trust_root)] = request
            return self.askLogin(request)

        if self.isAuthorized(request.identity, request.trust_root):
            return self.buildTwistedResponse(self.grant(request))
        elif request.immediate:
            return self.buildTwistedResponse(self.deny(request))

        self.manager.lastCheckIDRequest[self.user] = request
        return http.Response(stream=self.buildDecidePage(request))

    def askLogin(self, request):
        response = RedirectResponse("""%s?service=%s""" % (AUTH_FORM_URL, self.manager.endpointURL()))
        response.headers.setHeader('Set-Cookie',
                                   [TwCookie('auth-request', '%s#%s' %
                                             (request.identity, request.trust_root),
                                             path="/oid/", expires=time() + 2*3600, discard=True
                                             )])
        return response


    def buildDecidePage(self, request):
        """ page d'authorisation de renvoyer des informations au trust_root """
        id_url_base = self.manager.baseurl + 'id/'
        # XXX: This may break if there are any synonyms for id_url_base,
        # such as referring to it by IP address or a CNAME.
        assert request.identity.startswith(id_url_base)
        expected_user = request.identity[len(id_url_base):]

        if request.idSelect(): # We are being asked to select an ID
            msg = '''\
            <p>Nous avons reçu une demande pour vous authentifier.
            Vous pouvez choisir un identifiant sous lequel vous voulez
            être enregistré sur ce site.
            </p>
            '''
            fdata = {
                'id_url_base': id_url_base,
                'trust_root': request.trust_root,
                 'attr_choice': self.generateAttrChoice(request),
               }
            form = '''\
            <form method="POST" action="/oid/allow">
            <table>
              <tr><td>Identité:</td>
                 <td>%(id_url_base)s<input type='text' name='identifier'></td></tr>
              <tr><td>Site de confiance:</td><td>%(trust_root)s</td></tr>
            </table>
            %(attr_choice)s
            <p>Souhaitez vous autoriser ce site à vous authentifier ?</p>
            <input type="checkbox" id="remember" name="remember" value="yes"
                /><label for="remember">mémoriser ma décision</label><br />
              <input type="submit" name="yes" value="oui" />
              <input type="submit" name="no" value="non" />
            </form>
            '''%fdata
        elif expected_user == self.user:
            msg = '<p>Un nouveau site nous a demandé de vous authentifier.</p>'

            fdata = {
                'identity': request.identity,
                'trust_root': request.trust_root,
                'attr_choice': self.generateAttrChoice(request),
                }
            form = '''\
            <table>
              <tr><td>Identifiant : </td><td>%(identity)s</td></tr>
              <tr><td>Site de confiance : </td><td>%(trust_root)s</td></tr>
            </table>
            <form method="POST" action="/oid/allow">
              %(attr_choice)s
              <p>Souhaitez vous autoriser ce site à vous authentifier ?</p>
              <input type="checkbox" id="remember" name="remember" value="yes"
                  /><label for="remember">mémoriser ma décision</label><br />
              <input type="submit" name="yes" value="oui" />
              <input type="submit" name="no" value="non" />
            </form>''' % fdata
        else:
            mdata = {
                'expected_user': expected_user,
                'user': self.user,
                }
            msg = '''\
            <p>Un site a demandé à consulter une identité %(expected_user)s
            qui n' est actuellement pas la vôtre %(user)s. Pour vous authentifier
            sous l'identité %(expected_user)s, cliquez sur OK.</p>''' % mdata

            fdata = {
                'identity': request.identity,
                'trust_root': request.trust_root,
                'expected_user': expected_user,
                }
            form = '''\
            <table>
              <tr><td>Identité:</td><td>%(identity)s</td></tr>
              <tr><td>Site de confiance:</td><td>%(trust_root)s</td></tr>
            </table>
            <p>Souhaitez vous autoriser ce site à vous authentifier ?</p>
            <form method="POST" action="/oid/allow">
              <input type="checkbox" id="remember" name="remember" value="yes"
                  /><label for="remember">mémoriser ma décision</label><br />
              <input type="hidden" name="login_as" value="%(expected_user)s"/>
              <input type="submit" name="yes" value="oui" />
              <input type="submit" name="no" value="non" />
            </form>''' % fdata

        page = HTMLResponse(title='Approuver la demande OpenID?')
        page.forms.append(form)
        page.messages.append(msg)
        return str(page)

    def generateAttrChoice(self, request):
        """
        Génère du html pour séléctionner les attributs obligatoires et
        mandataires qui sont renvoyés au site de confiance
        """
        # FIXME ce choix pourrait etre optionnel selon les utilisateurs
        # récupération des filtres si disponible
        data_filter = self.manager._check_filter(request.trust_root)[0]
        authorized_fields = []
        if data_filter:
            data_filter = self.manager.filters[data_filter]
            for section in data_filter.keys():
                for field in data_filter[section].values():
                    authorized_fields.append(field)
        html = []
        ignore_attrs = []
        eole_req = oideole.EoleExtensionRequest.fromOpenIDRequest(request)
        if authorized_fields == []:
            # pas de filtre, on prend tous les champs (à voir ?)
            authorized_fields.extend(eole_req.required)
            authorized_fields.extend(eole_req.optional)
        html.append('<p>Séléctionnez les attributs que vous voulez communiquer au site de confiance</p>')
        for attr in eole_req.required:
            if attr in authorized_fields:
                libelle = oideole.EoleExtension.data_fields[attr]
                html.append('<p><input type="checkbox" checked="checked" name="attr_%s"/> %s <b>*</b></p>' % (attr, libelle))
            else:
                ignore_attrs.append(attr)
        for attr in eole_req.optional:
            if attr in authorized_fields:
                libelle = oideole.EoleExtension.data_fields[attr]
                html.append('<p><input type="checkbox" checked="checked" name="attr_%s"/> %s</p>' % (attr, libelle))
            else:
                ignore_attrs.append(attr)
        html.append('<p><b>*</b> obligatoire</p>')
        if ignore_attrs:
            html.append('''<p><i>Les attributs suivants ont été demandés mais ignorés car la configuration du
                             provider ne les authorise pas : %s </i></p>''' % ', '.join(ignore_attrs))
        return '\n'.join(html)




class AllowResource(IPPostableResource):

    @setuser_and_query
    def render(self, args):
        # pretend this next bit is keying off the user's session or something,
        # right?
        request = self.manager.lastCheckIDRequest.get(self.user)
        query = self.query
        if 'yes' in query:
            if 'login_as' in query:
                self.user = query['login_as']

            if request.idSelect():
                identity = self.manager.baseurl + 'id/' + query['identifier']
            else:
                identity = request.identity

            trust_root = request.trust_root
            if query.get('remember', 'no') == 'yes':
                self.manager.approved[(identity, trust_root)] = 'always'
                allowed_f = []
                for attr_allowed in query.keys():
                    if attr_allowed.startswith('attr_'):
                        allowed_f.append(attr_allowed)
                self.manager.allowed_fields[(identity, trust_root)] = allowed_f

            response = self.grant(request, identity)
        elif 'no' in query:
            response = self.deny(request)
        else:
            assert False, 'strange allow post.  %r' % (query,)

        return self.buildTwistedResponse(response)



class LoginResource(IPResource):
    """page de login"""

    @setuser_and_query
    def render(self, args):
        # appel à la page d'authentification du serveur sso (eole-specific)
        response = RedirectResponse("""%s?service=%s""" % (AUTH_FORM_URL, self.manager.endpointURL()))
        # print response, dir(response)
        return response

class UserOwnPage(IPResource):

    def __init__(self, manager, nickname):
        super(UserOwnPage, self).__init__(manager)
        self.nickname = nickname

    def render(self, ctx):
        page = HTMLResponse(title="%s's identity page" % self.nickname)
        baseurl = self.manager.baseurl
        page.headers.append('<link rel="openid.server" href="%s" />' %
                            self.manager.endpointURL())
        page.headers.append('<meta http-equiv="x-xrds-location" content="%syadis/%s" />' %
                            (baseurl, self.nickname))
        ident = baseurl + 'id/' + self.nickname


        approved_trust_roots = []
        for (aident, trust_root) in self.manager.approved.keys():
            if aident == ident:
                approved_trust_roots.append('<li><tt>%s</tt></li>\n' % cgi.escape(trust_root))

        if approved_trust_roots:
            page.append('<p>sites de confiance :</p>\n<ul>')
            for trusted_root in approved_trust_roots:
                page.append('<li><tt>%s</tt></li>' % trusted_root)
            page.append('</ul>')
        page.messages.append("%s's identity page" % self.nickname)
        return http.Response(stream=str(page))


class HomeDir(IPResource):
    addSlash = False

    def render(self, ctx):
        return http.Response(stream="encore perdu ...")

    def locateChild(self, request, segments):
        if segments and segments[0]:
            return UserOwnPage(self.manager, segments[0]), ()
        return self, ()


class UserYadisResource(IPResource):
    """
    Serves the user id as a yadis file (used by YadisDir)
    """

    template = """\
<?xml version="1.0" encoding="UTF-8"?>
<xrds:XRDS
    xmlns:xrds="xri://$xrds"
    xmlns="xri://$xrd*($v*2.0)">
  <XRD>

    <Service priority="0">
      <Type>%s</Type>
      <Type>%s</Type>
      <URI>%%(endpoint_url)s</URI>
      <LocalID>%%(user_id_url)s</LocalID>
    </Service>

  </XRD>
</xrds:XRDS>
    """ % (discover.OPENID_2_0_TYPE, discover.OPENID_1_0_TYPE)

    def __init__(self, manager, nickname):
        super(UserYadisResource, self).__init__(manager)
        self.nickname = nickname

    def render(self, ctx):
        values = {'endpoint_url' : self.manager.endpointURL(),
                  'user_id_url' : self.manager.baseurl + 'id/' + self.nickname,
                  }
        return http.Response(headers={'Content-type' : MimeType.fromString('application/xrds+xml')},
                             stream=self.template % values)


class YadisDir(IPResource):
    """
    Sert l'identification sous format yadis, par l'url : /yadis/user_id
    """
    addSlash = False

    def render(self, ctx):
        return http.Response(stream="zog zog ...")

    def locateChild(self, request, segments):
        if segments and segments[0]:
            return UserYadisResource(self.manager, segments[0]), ()
        return self, ()


class ServerYadisResource(IPResource):
    template = """\
<?xml version="1.0" encoding="UTF-8"?>
<xrds:XRDS
    xmlns:xrds="xri://$xrds"
    xmlns="xri://$xrd*($v*2.0)">
  <XRD>

    <Service priority="0">
      <Type>%s</Type>
      <URI>%%(endpoint_url)s</URI>
    </Service>

    <Service priority="0">
      <Type>%s</Type>
      <URI>%%(endpoint_url)s</URI>
    </Service>

  </XRD>
</xrds:XRDS>
    """ % (discover.OPENID_2_0_TYPE, oideole.EOLE_OPENID_NS)

    def render(self, ctx):
        values = {'endpoint_url' : self.manager.endpointURL(),
                  }
        return http.Response(headers={'Content-type' : MimeType.fromString('application/xrds+xml')},
                             stream=self.template % values)



class OIDRoot(IPResource):

    addSlash = False

    body = """<p>Ceci est un serveur OpenID qui est implémenté en utilisant la
    <a href="http://openid.schtuff.com/">Librairie Python OpenID</a>.</p>
    %(usermsg)s

    <p>L'URL de ce serveur est <a href="%(baseurl)s"><tt>%(baseurl)s</tt></a>.</p>
    """

    def __init__(self, session_manager, baseurl="http://localhost:8080/oid"):
        pers_manager = PersistencyManager(session_manager, baseurl)
        super(OIDRoot, self).__init__(pers_manager)

    @setuser_and_query
    def render(self, request):
        if self.user is None:
            usermsg = self._not_logged_message()
        else:
            usermsg = self._logged_message()
        page = HTMLResponse(title='Main Page')
        page.headers.append('<meta http-equiv="x-xrds-location" content="%s">'
                            % self.manager.serverYadisURL())
        page.append(self.body  % dict(usermsg=usermsg,
                                      baseurl=self.manager.baseurl))
        return http.Response(stream=str(page))


    def _not_logged_message(self):
        return """<p>Ce serveur utilise des cookies pour se souvenir qui vous
                  etes pour émuler une experience web. Vous n'etes pas
                  <a href='/oid/login'>authentifié</a>.</p>"""

    def _logged_message(self):
        openid_url = self.manager.baseurl + 'id/' + self.user
        return """<p>Vous êtes authentifié en tant que %s. Votre
        identifiant openid est <tt><a href="%s">%s</a></tt>. Vous
        pouvez l'utiliser sur un site permettant une authentification
        openid. Vous pouvez <a href="/logout">fermer votre session</a></p>""" % (
        self.user, openid_url, openid_url)



    ## aboresence des pages servies ###########################################
    def child_login(self, request):
        return LoginResource(self.manager)

    def child_id(self, request):
        return HomeDir(self.manager)

    def child_yadis(self, request):
        return YadisDir(self.manager)

    def child_serveryadis(self, request):
        return ServerYadisResource(self.manager)

    def child_openidserver(self, request):
        return OpenIdServerResource(self.manager)

    def child_allow(self, request):
        return AllowResource(self.manager)


