#!/usr/bin/env python
#-*- coding:utf-8 -*-
"""
pages de gestion de l'authentification OTP à travers le plugin PAM securid.
On effectue une double authentification (ldap + OTP directement auprès du serveur RSA).
Le niveau d'authentificaton de la session SSO sera indiqué dans le contexte d'authentification
lors de l'envoi d'assertions SAML (vers RSA-FIM par exemple).
"""
import PAM, getpass, threading, time, urllib, datetime, os, stat, cgi
import traceback
from authserver import EoleParser

from twisted.internet import defer, reactor, threads
from twisted.web2 import http, responsecode
from twisted.web2.stream import ProducerStream
from twisted.web2.resource import PostableResource as Resource
from twisted.web2.http_headers import Cookie as TwCookie
from saml_utils import available_contexts, format_timestamp

from config import IDP_IDENTITY, LDAP_BASE, DEFAULT_CSS, SECURID_USER_DIR, OTP_TIMEOUT, REDIRECT_DESYNC, OTP_PORTAL
from eolesso.util import gen_ticket_id, getCookie, getCookies, urljoin, get_service_from_url, EoleParser, RedirectResponse
from page import gen_page, trace, log

SECURID_AUTH_FORM ="""
<input type='hidden' name='user_registered' id='user_registered' value='false'/>
<label for='securid_register' style="display:none;" id='securid_reg_label' accesskey='E'>%s</label>
<input type='checkbox' style="display:none;" tabindex='3' name='securid_register' onClick='toggle_securid()' id='securid_register'/>
<div id='register_form' style='display:none;"'<hr><table>
<tr><td><label for='securid_user' id='securid_user_label' accesskey='I'>%s</label></td>
<td><input type='text' tabindex='4' name='securid_user' id='securid_user' autocomplete='%s' title='%s' /></td></tr>
<tr><td><label for='securid_pwd' accesskey='J'>%s + %s</label></td>
<td><input type='password' tabindex='5' name='securid_pwd' id='securid_pwd' autocomplete='%s' title='%s'/></td></tr>
</table></div>
"""

NEXT_TOKEN_FORM = """<form action="/securid" method="POST" name='securid_form'>
<table>
<tr><td colspan=2>%s</td></tr>
<tr><td><label for='securid_pwd' accesskey='W'>%s</label></td><td>
<input type='password' tabindex='2' size='7' & maxlength='6'  name='securid_pwd' id='securid_pwd' autocomplete='off' title='%s' />
<input type="hidden" name="from_url" value="%s"/>
<input type="hidden" name="css" value="%s"/>
</td></tr>
</table>
<p class="formvalidation">
<input type="submit" class='btn' tabindex='3'/></p></form>
"""

DESYNCHRONIZED_FORM = """<form action="/securid" method="POST" name='desync_form'>
<div class="msg_desync">%s</div>
<input type="hidden" name="from_url" value="%s"/>
<input type="hidden" name="css" value="%s"/>
<p class="formvalidation">
<input type="submit" class='btn' tabindex='3'/></p></form>
"""

def gen_retry_form(from_url, err_msg, css):
    params = []
    if from_url and from_url != 'unspecified':
        params.append(('service', from_url))
    dest = urllib.quote_plus('/login')
    if params:
        dest += '?%s' % urllib.urlencode(params)
    header_script = """<script type="text/javascript">
    function ask_for_retry()
    {
        alert("%s");
        window.location='/logout?url=%s';
    }
    </script>""" % (err_msg, dest)
    on_load_script = """ask_for_retry()"""
    page_data = gen_page(_("OTP authentication"), "",
                    javascript=on_load_script,
                    header_script=header_script,
                    css=css)
    return page_data

class NewTokenDisabled(Exception):
    pass

class PAMChecker:

    def __init__(self, pam_service):
        self.pam_service = pam_service
        self.ev_auth = threading.Event()
        self.ev_auth.status = "init"
        self.ev_auth.user_input = None
        # valeur du cookie SSO en cas de nouvelle session
        self.ev_auth.auth_cookie = None
        self.ev_auth.css = DEFAULT_CSS

    @trace
    def callIntoPAM(self, user, conv):
        """version simplifiée de callIntoPam
        """
        self.pam = PAM.pam()
        self.pam.start(self.pam_service)
        self.pam.set_item(PAM.PAM_USER, user)
        self.pam.set_item(PAM.PAM_CONV, conv)
        self.pam.setUserData({})
        try:
            self.pam.authenticate() # these will raise
        except:
            return False
        return True

    @trace
    def pamAuthenticateThread(self, user, conv):

        def _conv(auth, items, user_data):
            # from twisted.internet import reactor
            try:
                d = conv(items)
            except NewTokenDisabled:
                return
            except Exception, e:
                log.msg(str(e))
                return
            self.ev = threading.Event()
            def cb(r):
                self.ev.r = (1, r)
                self.ev.set()
            def eb(e):
                self.ev.r = (0, e)
                self.ev.set()
            reactor.callFromThread(d.addCallbacks, cb, eb)
            self.ev.wait(20)
            # on vérifie si cette authentification est toujours en cours
            # (si par exemple fermeture navigateur ou reload de l'utilisateur)
            if self.ev.r:
                done = self.ev.r
                if done[0]:
                    return done[1]
                else:
                    raise done[1].type, done[1].value
            else:
                # sortie en timeout avant la saisie des données
                raise Exception, " ++ Aucun identifiant récupéré"

        return self.callIntoPAM(user, _conv)

    @trace
    def twisted_conv(self, items):
        """Fonction lancée dans la boucle principale et chargée
        de gérer le dialogue avec l'utilisateur
        """
        resp = []
        for i in range(len(items)):
            message, kind = items[i]
            if kind == 1: # password
                # on demande la saisie d'un mot de passe (code PIN + jeton)
                self.ev_auth.clear()
                self.ev_auth.status = 'asking'
                WAIT_TIMEOUT = 20
                if 'Wait' in message:
                    self.ev_auth.status = 'waiting'
                    if REDIRECT_DESYNC:
                        desync_msg = _("your key is desynchronized") + "<br/>"
                        if OTP_PORTAL:
                            desync_msg += _("redirecting to OTP login page")
                        else:
                            desync_msg += _("connect to OTP login page")
                        page_content = DESYNCHRONIZED_FORM % (desync_msg,
                                          str(self.ev_auth.session_data[3]) or "unspecified",
                                          str(self.ev_auth.css))
                    else:
                        # on attend que le mot de passe soit renouvelé
                        # (attente de 2 minutes maximum)
                        WAIT_TIMEOUT = OTP_TIMEOUT
                        # écriture du formulaire de saisie du mot de passe dans la requête
                        page_content = NEXT_TOKEN_FORM % (_("Second passcode required, wait for next token"),
                                                _("Enter new token"),
                                                _("Enter new token"),
                                                str(self.ev_auth.session_data[3]) or "unspecified",
                                                str(self.ev_auth.css))
                    # envoi de la réponse
                    self.ev_auth.response_stream.write(gen_page(_('New token required'), page_content, css=self.ev_auth.css))
                    self.ev_auth.response_stream.finish()
                    if REDIRECT_DESYNC:
                        raise NewTokenDisabled('new token mode disabled')
                self.ev_auth.wait(WAIT_TIMEOUT)
                if self.ev_auth.status == 'answered':
                    resp.append((self.ev_auth.user_input, 0))
                else:
                    # l'utilisateur n'a pas saisi à temps son 2ème passcode ou on ne le gère pas
                    resp.append(("", 0))
            elif kind == 2: # text
                # cas non géré (saisie en clair)
                log.msg("DEMANDE DE SAISIE EN CLAIR RECUE (Pam SecurId): %s" % msg)
                resp.append(("", 0))
            elif kind in (3,4):
                self.display_info(message)
                resp.append(("", 0))
            else:
                log.msg("REPONSE PAM NON GEREE : %s / %s" % (str(kind), message))
                return defer.fail('foo')
        d = defer.succeed(resp)
        return d

    @trace
    def callb_auth(self, result):
        """prend en compte le résultat de l'authentification PAM et continue
        la chaîne d'authentification"""
        try:
            session_id = self.ev_auth.session_id
            # login_ticket = self.manager.get_login_ticket(self.ev_auth.session_data)
            if not session_id.startswith('TGC'):
                if result:
                    # on n'avait pas de session au préalable, on en crée une nouvelle
                    username = self.ev_auth.session_data[0]
                    user_branch = self.ev_auth.session_data[1]
                    if self.ev_auth.session_data[8] != True:
                        # dans le cas ou on veut associer l'identifiant à un
                        # utilisateur local, on vérifie l'authentification LDAP
                        password = self.ev_auth.session_data[2]
                        defer_auth = self.manager.authenticate(username, password, user_branch)
                    else:
                        # utilisateur déjà enregistré, on ne vérifie pas le mot de passe ldap
                        feder_attrs = {'uid':[username]}
                        defer_auth = self.manager.authenticate_federated_user(feder_attrs, IDP_IDENTITY, time.time(), available_contexts['URN_TIME_SYNC_TOKEN'], search_branch=user_branch)
                    return defer_auth.addCallbacks(self.update_session, self.errb_auth, callbackArgs=[result])
                elif self.ev_auth.status == 'waiting':
                    # si le status est encode waiting, on a dépassé le délai
                    # maximum autorisé en mode 'next token' ou ce mode est désactivé
                    if not REDIRECT_DESYNC:
                        self.ev_auth.status = 'timeout'
                    self.ev_auth.user_input = ''
                    self.ev_auth.set()
                    return False

            # session existante: on modifie le niveau d'authentification
            return self.update_session((session_id, None), result)
        except Exception, e:
            traceback.print_exc()
            log.msg("Erreur dans le callback d'authentification OTP: %s" % str(e))

    @trace
    def update_session(self, session_infos, result_securid):
        """met en place les cookies si nécessaire et renvoie l'utilisateur sur l'url de redirection"""
        self.ev_auth.status = "failure"
        err_msg = _('Authentication failed, try again')
        # définition des urls de retour
        from_url = self.ev_auth.session_data[3]
        redirect_to = self.ev_auth.session_data[4]
        session_id, user_data = session_infos
        if session_id:
            if from_url == '':
                # pas de service spécifié, on indique que la session est ouverte, ou on redirige si demandé
                if redirect_to != '':
                    target_url = redirect_to
                else:
                    target_url = '/loggedin'
            else:
                # on redirige sur la page de l'application de destination
                app_ticket = self.manager.get_app_ticket(session_id, from_url, from_credentials=True)
                target_url = urljoin(from_url, 'ticket=%s' % app_ticket)
                log.msg("%s -- %s" % (session_id, _("Redirecting to calling app with ticket : %s") % get_service_from_url(from_url)))
            if session_id.startswith('TGC') and self.manager.user_sessions.get(session_id,(None,))[0] == None:
                # Si session délivrée par le parent, rediriger sur lui pour poser un cookie d'auth qu'il puisse lire
                req_args = (('ticket', session_id), ('return_url', target_url))
                log.msg("%s -- %s" % (session_id, _("Redirecting to parent server for session setting")))
                redirect_url = '%s/gen_cookie?%s' % (self.manager.parent_url, urllib.urlencode(req_args))
            else:
                redirect_url = target_url
            # la session a été créée après authentification sur securid
            self.ev_auth.auth_cookie = session_id
            # cas d'une session existante
            if result_securid and self.manager.set_auth_class(session_id, available_contexts['URN_TIME_SYNC_TOKEN']):
                self.ev_auth.status = "success"
                if self.ev_auth.session_data[8] != True:
                    # si besoin, on enregistre l'identifiant alternatif de l'utilisateur
                    username = self.ev_auth.session_data[0]
                    user_branch = self.ev_auth.session_data[1]
                    securid_login = self.ev_auth.session_data[6]
                    log.msg("%s -- %s" % (session_id, _("Registering identifier %s for user %s (%s)") % (str(securid_login), str(username), user_branch or LDAP_BASE)))
                    try:
                        register_ok = self.ev_auth.user_store.register_user(username, securid_login, user_branch)
                    except:
                        traceback.print_exc()
                    if not register_ok:
                        err_msg = _('Error: this Id is already associated with a different user')
                        return self.securid_failed(from_url, err_msg)
            else:
                err_msg = _('OTP authentication failed, try again')
        if self.ev_auth.status == "success":
            # script pour mise en place du cookie sso
            header_script = """<script type="text/javascript" src="scripts/mootools-core-1.4.2.js"></script>
            <script type="text/javascript">
            sso_cook = Cookie.write('EoleSSOServer', '%s', {secure: true, duration:false});
            </script>
            """ % session_id
            on_load_script = "window.location='%s'" % redirect_url
            # gestion de la redirection sur la requête
            self.ev_auth.response_stream.write(gen_page(_("OTP authentication"), "",
                                               javascript=on_load_script, header_script=header_script, css=self.ev_auth.css))
            self.ev_auth.response_stream.finish()
            # la procédure d'authentification est terminée, on invalide le checker
            self.ev_auth.status = "expired"
            return True
        else:
            return self.securid_failed(from_url, err_msg)

    @trace
    def securid_failed(self, from_url, err_msg):
        page_data = gen_retry_form(from_url, err_msg, self.ev_auth.css)
        self.ev_auth.response_stream.write(page_data)
        self.ev_auth.response_stream.finish()
        # la procédure d'authentification est terminée, on invalide le checker
        self.ev_auth.status = "expired"
        return False

    @trace
    def errb_auth(self, result):
        self.ev_auth.response_stream.write(gen_page(_("OTP authentication"), "Erreur rencontré lors de l'authentification OTP (PAM)", css=self.ev_auth.css))
        log.msg("Erreur à l'exécution de l'authentification PAM (%s) : %s" % (str(self), result.getErrorMessage()))
        # Erreur, on annule la session SSO de l'utilisateur
        self.manager.logout(self.ev_auth.session_id)
        self.ev_auth.response_stream.finish()

    @trace
    def checkCreds(self, username, response_stream, manager):
        # on passe en paramètre la requête serveur pour permettre
        # d'interagir avec l'utilisateur
        self.manager = manager
        self.ev_auth.response_stream = response_stream
        self.d = threads.deferToThread(self.pamAuthenticateThread, username, self.twisted_conv)
        self.d.addCallbacks(self.callb_auth, self.errb_auth)
        return self.d

    def display_info(self, msg):
        log.msg(msg)

class SecuridCheck(Resource):

    isLeaf = True

    def __init__(self, manager):
        self.manager = manager
        Resource.__init__(self)

    @trace
    def render(self, request):
        responseStream = ProducerStream()
        login_ticket = None
        ## Def l'id de session
        tw_cook = getCookie(request, 'EoleSSOChecker')
        if tw_cook is None:
            uniq_id = gen_ticket_id('TMP', self.manager.address)
        else:
            uniq_id = tw_cook.value

        checker = self.manager.checkers.get(uniq_id, (None, None))[0]
        status = None
        if hasattr(checker, 'ev_auth'):
            status = checker.ev_auth.status
            if status not in ("waiting", "timeout"):
                self.purge_checker(uniq_id)
                checker = None
        try:
            css = request.args['css'][0]
        except KeyError:
            css = DEFAULT_CSS
        access_allowed = False
        # le mot de passe est récupéré dans la requête si l'utilisateur
        # doit le saisir une deuxième fois (mode next token)
        passwd = request.args.get('securid_pwd',[""])[0]
        # on vérifie si on est dans le cas d'un timeout après demande d'un deuxième token
        if request.args.get('from_url', []):
            from_url = cgi.escape(request.args.get('from_url',['unspecified'])[0])
            if REDIRECT_DESYNC:
                # redirection vers le portail OTP pour synchroniser la clé
                if OTP_PORTAL:
                    redirect_url = OTP_PORTAL
                elif from_url and from_url != "unspecified":
                    redirect_url = from_url
                else:
                    redirect_url = "/"
                if checker:
                    # on force la suppression immédiate du checker
                    self.purge_checker(checker.ev_auth.session_id)
                return RedirectResponse(redirect_url)
            elif checker is None or status == "timeout":
                # cas d'une réponse trop tardive pour la saisie du deuxième passcode (clé désynchronisée)
                # on demande une nouvelle tentative de connexion
                css = cgi.escape(request.args.get('css',[''])[0])
                err_msg = _('Input delay expired, try again')
                if checker:
                    # on force la suppression immédiate du checker
                    self.purge_checker(checker.ev_auth.session_id)
                response_data=gen_retry_form(from_url, err_msg, css)
                response = http.Response(code=responsecode.OK, stream=response_data)
                return response
        if checker is None:
            if request.args.has_key('lt'):
                # on vérifie qu'on a bien été appelé apr la page d'authentification
                login_ticket = request.args.get('lt', [''])[0]
            else:
                # premier accès à la page ('pas de mot de passe saisi')
                return http.Response(stream=gen_page(_('Error'), _('Resource not Allowed'), css=css))
            # initialisation d'un nouveau checker
            if not self.manager.login_sessions.validate_session(login_ticket):
                return http.Response(stream=gen_page(_('Error'), _('Resource not Allowed'), css=css))
            # récupération des paramètres stockés avec le ticket lt
            session_data = self.manager.login_sessions.get_session_info(login_ticket)
            if css == DEFAULT_CSS:
                css = session_data[5]
            checker = self.init_checker(uniq_id)
            # récupération du login securid de l'utilisateur
            username = session_data[6]
            passwd = session_data[7]
            # on enregistre les paramètres d'origine dans l'objet checker en prévision
            # de la suite du traitement de l'authentification (from_url, redirect_to)
            checker.ev_auth.css = css
            checker.ev_auth.securid_login = username
            checker.ev_auth.session_id = uniq_id
            checker.ev_auth.checker_pool = self.manager.checkers
            checker.ev_auth.session_data = session_data
            checker.ev_auth.user_store = self.manager.securid_user_store
            checker.checkCreds(username, responseStream, self.manager)
            # on attend que ev_auth passe en status asking (10 secondes max)
            if checker.ev_auth.status == "init":
                wait_iter = 0
                while checker.ev_auth.status != 'asking' and wait_iter <= 20:
                    wait_iter += 1
                    time.sleep(0.5)
        # envoi des données saisies au thread pam + requête pour interaction
        checker.ev_auth.user_input = passwd
        checker.ev_auth.response_stream = responseStream
        checker.ev_auth.status = 'answered'
        checker.ev_auth.set()
        response = http.Response(code=responsecode.OK, stream=responseStream)
        # mise en place des nouveaux cookies si besoin
        if login_ticket != None:
            # nouveau checker créé, on met en place le cookie pour le retrouver
            cookies = []
            for cookie in getCookies(request):
                if cookie.name != 'EoleSSOChecker':
                    cookies.append(cookie)
            cookies.append(TwCookie('EoleSSOChecker', uniq_id, path = "/securid", discard=True, secure=True))
            response.headers.setHeader('Set-Cookie', cookies)
        return response

    @trace
    def purge_checker(self, uniq_id, auto=False):
        """suppression d'un checker.
        on débloque ev_auth au cas ou un thread serait en attente
        (ex : reload de la page par l'utilisateur en mode 'next token')
        si auto == False: supression manuelle, on annule le callback
        de supression automatique
        """
        try:
            if self.manager.checkers[uniq_id][0].ev_auth.status in ('waiting', 'asking'):
                self.manager.checkers[uniq_id][0].ev_auth.user_input = ''
                self.manager.checkers[uniq_id][0].ev_auth.set()
            self.manager.checkers[uniq_id][0].ev_auth.status = "expired"
            if not auto:
                self.manager.checkers[uniq_id][1].cancel()
            del(self.manager.checkers[uniq_id])
        except:
            log.msg("  + Erreur à la suppression : checker non trouvé (ne devrait pas arriver !)")

    @trace
    def init_checker(self, uniq_id):
        # on commence par purger tous les checkers qui ont expiré
        for id_session, checker_data in self.manager.checkers.items():
            checker = checker_data[0]
            if checker.ev_auth.status == "expired":
                self.purge_checker(id_session)
        checker = PAMChecker('rsa_securid')
        # fonction pour supprimer automatiquement le checker après OTP_TIMEOUT +2 minutes
        delete_cb = reactor.callLater(OTP_TIMEOUT + 120, self.purge_checker, uniq_id, True)
        self.manager.checkers[uniq_id] = (checker, delete_cb)
        return checker

class SecuridUserStore():
    """Ressource de vérification de l'association d'un utilisateur dans la base locale
    """

    store_dir = SECURID_USER_DIR
    store_file = os.path.join(SECURID_USER_DIR, 'securid_users.ini')

    def __init__(self):
        # structure pour recherche des utilisateurs locaux ayant enregistré leur identifiant
        self.user_store = EoleParser()
        self.check_store_perms()
        self.load_users()
        # structure pour recherche inversée (login local depuis login securid)
        self.securid_logins = {}
        for branch in self.user_store.sections():
            for local_user, securid_user in self.user_store.items(branch):
                self.securid_logins[securid_user] = (branch, local_user)

    def check_store_perms(self):
        if not os.path.isdir(self.store_dir):
            os.makedirs(self.store_dir)
        if not os.path.isfile(self.store_file):
            f_users = open(self.store_file, 'w')
            f_users.close()
        # vérifie que seul root peut accéder au fichier des correspondances de login securid
        dir_perms = stat.S_IRWXU #(drwx------)
        os.chmod(self.store_dir, dir_perms)
        file_perms = stat.S_IRUSR|stat.S_IWUSR #(rw-------)
        os.chmod(self.store_file, file_perms)

    def save_users(self):
        f_users = open(self.store_file, 'w')
        self.user_store.write(f_users)
        f_users.close()

    def load_users(self):
        self.user_store.read(self.store_file)

    @trace
    def check_registered(self, username, search_branch):
        if self.user_store.has_option(search_branch, username):
            return "true"
        return "false"

    @trace
    def get_local_user(self, securid_login):
        return self.securid_logins.get(securid_login, (None, None))

    @trace
    def get_securid_user(self, username, search_branch):
        if self.user_store.has_option(search_branch, username):
            return self.user_store.get(search_branch, username)
        else:
            return ''

    @trace
    def register_user(self, username, securid_name, search_branch):
        if securid_name in self.securid_logins:
            if self.securid_logins[securid_name] != (username, search_branch):
                # on ne permet pas de doublons dans les identifiants securid
                log.msg("register_user : %s (%s)" % (_("Error: this Id is already associated with a different user"), securid_name))
                return False
            else:
                # déjà enregistré, on ne fait rien
                return True
        if not self.user_store.has_section(search_branch):
            self.user_store.add_section(search_branch)
        self.user_store.set(search_branch, username, securid_name)
        self.securid_logins[securid_name] = (search_branch, username)
        self.save_users()
        return True
