#! /usr/bin/env python
# -*- coding: utf-8 -*-

###########################################################################
#
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# saml_resources.py
#
# support (limité) du protocole SAML 2.0 pour eole-sso
#
###########################################################################

import traceback
import urllib, os, socket, time

from twisted.web2 import http, responsecode
from twisted.web2.http_headers import Cookie as TwCookie, MimeType
from twisted.web2.resource import PostableResource
from twisted.internet.threads import callMultipleInThread
from twisted.python import failure

from pyeole import httprequest
from eolesso.util import (getCookies, gen_random_id, RedirectResponse,
                          format_err, get_service_from_url, is_true, http_headers)

import saml_message
from saml_crypto import sign_request
from saml_utils import (encode_request, gen_metadata, get_endpoint,
                        get_attributes, InternalError, Redirect,
                        extract_message, check_required_contexts,
                        process_auth_request, process_assertion,
                        encode_idp_cookie, split_form_arg)
from saml2 import samlp
from page import gen_page, trace, log
from config import (CERTFILE, AUTH_FORM_URL, IDP_IDENTITY, DEFAULT_CSS,
                    SEND_CAS_LOGOUT, DISPLAY_FEDERATION, DEBUG_LOG, encoding)

class SamlResource(PostableResource):

    content_type = 'text/html; charset=utf-8'
    # headers par défaut
    headers = http_headers

    def set_headers(self, resp, user_headers={}):
        """attache à la réponse HTTP les headers par défaut
        + les éventuels headers passés en paramètre
        """
        if self.content_type:
            resp.headers.setRawHeaders('Content-type', [self.content_type])
        for headers in (self.headers, user_headers):
            for h_name, h_values in headers.items():
                resp.headers.setRawHeaders(h_name, h_values)
        return resp

    def render(self, request):
        # errheaders = {'Content-type': MimeType.fromString('application/xml')}
        try:
            return self._render(request)
        except Redirect, exc:
            headers = {'Content-type': MimeType.fromString(self.content_type)}
            location = exc.location
            headers['location'] =  location
            log.msg(_('Redirecting to '), location)
            # 303 See other
            return http.Response(code = 303, headers = headers)
        except Exception, exc: # unknown error
            traceback.print_exc()
            return http.StatusResponse(code = responsecode.INTERNAL_SERVER_ERROR, description = str(exc))

    def _render(self):
        pass

def gen_page_err(err_msg, from_url, css):
    """construit une réponse d'erreur
    """
    form = format_err(err_msg)
    if from_url:
        form += """<p><FORM action="%s">
        <b>%s</b> : <br/><h2><font color="blue">%s</font></h2><br/></p>
        <p class="formvalidation"><input class="btn" type="submit" value ="%s"/></p>
        </form>""" % (from_url, _('Following page has been proposed'), from_url, _('Proceed to above url'))
    else:
        form += """<br/><form action="/login" method="post">
        <p class="formvalidation">
        <input class="btn" type="submit" value="%s"></p></form>""" % _('New session')
    return http.Response(stream = gen_page(_('Logout'), form, css))

class SamlLogout(SamlResource):
    """Resource de gestion du logout (logout cas + single logout saml)
    """
    addSlash = False

    def __init__(self, manager):
        self.manager = manager
        super(SamlLogout, self).__init__()

    @trace
    def send_logout_request(self, saml_identity, sp_meta, user_id, ticket, sso_session, from_url, css):
        """Envoi d'une requête de déconnexion à une entité partenaire et stocke les informations sur l'origine de la déconnexion"""
        try:
            binding, service_url, response_url = get_endpoint(sp_meta, 'SingleLogoutService', ticket.saml_role)
        except InternalError:
            log.msg(_('no usable endpoint for %s') % sp_meta['entityID'])
            # store error for display ?
            return None

        response_id = self.manager.gen_saml_id({'type':'LogoutRequest'})
        if hasattr(ticket, 'idp_session_index'):
            session_index = ticket.idp_session_index
        else:
            session_index = ticket.ticket
        # Envoi de la demande de logout à l'entité correspondante
        if binding == samlp.BINDING_HTTP_POST:
            sign_method, req = saml_message.gen_logout_request(response_id, session_index, \
                    user_id, IDP_IDENTITY, sp_meta['entityID'], service_url, CERTFILE)
            # réponse utilisant le binding POST
            req = encode_request(req)
            form = """
                    <FORM action="%s" method="POST">
                    <input type="hidden" name="SAMLRequest" value="%s">
                    <input type="hidden" name="SigAlg" value="%s">
                    <input type="hidden" name="RelayState" value="%s">
                    <p class="formvalidation">
                    <input class="btn" type="submit" value="%s"></p>
            """ % (service_url, split_form_arg(req), sign_method, from_url, _("Proceed to service"))
            resp = http.Response(stream = gen_page(_('Logout request to service'), form, css))
            resp = self.set_headers(resp)
        else:
            sign_method, req = saml_message.gen_logout_request(response_id, session_index, user_id, IDP_IDENTITY, \
                    sp_meta['entityID'], service_url, CERTFILE, sign = False)
            # réponse utilisant le binding REDIRECT
            req = encode_request(req, True)
            req_args = sign_request(CERTFILE, sign_method, 'SAMLRequest', req, from_url)
            if req_args != []:
                redirect_url = '%s?%s' % (service_url, urllib.urlencode(req_args))
            resp = RedirectResponse(redirect_url)

        # on stocke l'id et le ticket avant de rediriger
        ticket.valid = False
        self.manager.pending_logout[sso_session][3][response_id] = ticket.ticket

        log.msg("%s -- %s" % (sso_session, _("Sending SAML logout request to %s (%s) : %s") % (saml_identity, service_url, response_id)))
        return resp

    @trace
    def send_cas_logout_request(self, ticket, sso_session, use_SOAP = False):
        """
            envoie une demande de déconnexion à un client CAS par un requête SAML
            use_SOAP : utiliser une enveloppe SOAP pour transmettre le message si True
                       (versions futures du client CAS ?)
        """
        logoutRequest = saml_message.gen_cas_logout_request(ticket.ticket, use_SOAP)
        log.msg("%s -- %s" % (sso_session, _("Sending CAS logout request to %s") % ticket.service_url))
        # gestion du passage ou non par un proxy http si défini pour cette application
        # envoi de la requête
        req = httprequest.HTTPRequest()
        self.manager.set_urllib_proxy(ticket)
        try:
            req.request(ticket.service_url, {'logoutRequest':logoutRequest})
        except httprequest.RequestError, e:
            log.msg("%s -- %s" % (ticket.session_id, _('Error sending logout request to %s : ' \
                    % (ticket.service_url,))))
            log.msg(str(e))
        # retour en mode sans proxy
        self.manager.set_urllib_proxy()

    @trace
    def process_logout_response(self, request, sso_session):
        """traite une réponse de déconnexion et met à jour les tickets correspondants dans le gestionnaire de sessions
        """
        signature_ok, saml_resp = extract_message(request, 'SAMLResponse')
        if not signature_ok:
            raise InternalError("%s : %s" % (_('signature error'), str(saml_resp)))
        response = samlp.LogoutResponseFromString(saml_resp)
        # on commence par vérifier qu'on ne rejoue pas un message déjà traité
        msg_id = response.id
        if self.manager.replayed_saml_msg(msg_id):
            # page d'erreur
            raise InternalError("%s -- %s" % (sso_session, _('response message has been replayed (%s)') % msg_id))
        resp_issuer = response.issuer.text.encode(encoding)
        in_response_to = response.in_response_to
        # Vérification de l'existence de la requête spécifiée dans in_response_to
        if in_response_to not in self.manager.pending_logout[sso_session][3]:
            # FIXME TODO: generate a saml negative saml response
            raise InternalError("%s : %s" % (_('corresponding request not found'), in_response_to))
        orig_message_data = self.manager.get_saml_msg(in_response_to)
        if orig_message_data.get('response_id', None) is not None:
            raise InternalError((_('response already received for request %s') % in_response_to))
        #issue_instant = date_from_string(response.issue_instant)
        status_code = response.status.status_code.value
        #if response.status.status_message is not None:
        #    status_msg = response.status.status_message.text
        ticket_id = self.manager.pending_logout[sso_session][3][in_response_to]
        if status_code == samlp.STATUS_SUCCESS:
            self.manager.pending_logout[sso_session][1].append(ticket_id)
        # on indique que le message d'origine a  reçu une réponse
        self.manager.update_saml_msg(in_response_to, {'response_id':msg_id})
        log.msg("%s -- %s" % (sso_session, _('SAMLResponse (%s) received from %s in response to %s') % ('LogoutResponse', resp_issuer, in_response_to)))

    @trace
    def build_logout_error(self, err_infos, sso_session, relay_state, css):
        """envoie une réponse de déconnexion négative en réponse à une requête
        """
        # récupération des infos nécessaires
        req_id, req_issuer, err_msg = err_infos
        issue_instant = time.time()
        # génération du message d'échec d'authentification
        sp_meta = self.manager.get_metadata(req_issuer)
        # on ne sait pas si le destinataire est un idp ou un sp, on cherche le service de déconnexion
        entity_role = 'SPSSODescriptor'
        if 'SingleLogoutService' not in sp_meta[entity_role]:
            entity_role = 'IDPSSODescriptor'
        try:
            binding, service_url, response_url = get_endpoint(sp_meta, 'SingleLogoutService', entity_role)
        except InternalError:
            traceback.print_exc()
            # on ne sait pas où la réponse doit être envoyée
            err_msg = _('no usable endpoint for %s') % req_issuer
            return http.Response(stream = gen_page(_('Logout Error'), err_msg, css))
        # Envoi de la réponse à l'émetteur de la requête
        response_id = self.manager.gen_saml_id({'type':'LogoutResponse'})
        sign_method, response = saml_message.gen_logout_response(req_id,
                                                                 response_id,
                                                                 IDP_IDENTITY,
                                                                 response_url,
                                                                 CERTFILE,
                                                                 samlp.STATUS_UNKNOWN_PRINCIPAL,
                                                                 status_msg = err_msg)
        return self.send_logout_response(response, sign_method, sso_session, (binding, response_url, relay_state), css)

    @trace
    def process_logout_request(self, request, sso_session):
        """traite une demande de déconnexion et initie le processus de déconnexion
        """
        signature_ok, saml_req = extract_message(request, 'SAMLRequest')
        if not signature_ok:
            # FIXME TODO: generate a saml negative saml response
            raise InternalError("signature error : %s" % str(saml_req))
        saml_req = samlp.LogoutRequestFromString(saml_req)
        req_issuer = saml_req.issuer.text.encode(encoding)
        req_id = saml_req.id
        # name_id de l'utilisateur
        req_name_id = saml_req.name_id or saml_req.base_id or saml_req.encrypted_id
        req_name_id = req_name_id.text.encode(encoding)
        try:
            if not sso_session:
                # pas de session valide !
                raise InternalError(_('Invalid session'))
            # détection de la session en cas d'index de session fourni dans la requête
            app_ticket = None
            if saml_req.session_index:
                app_ticket = saml_req.session_index.text.encode(encoding)
            else:
                # on recherche un ticket correspondant à l'émetteur et au nameid fourni
                for ticket in self.manager.user_app_tickets.get(sso_session, []):
                    if ticket.saml_ident == req_issuer and ticket.nameid == req_name_id:
                        # on a trouvé un ticket qui correspond à l'utilisateur indiqué dans la requête
                        app_ticket = ticket.ticket
                        break
            if app_ticket in self.manager.saml_sessions:
                # session définie par l'idp distant -> on récupère l'app_ticket local correspondant
                app_ticket = self.manager.saml_sessions[app_ticket]
            if app_ticket in self.manager.app_sessions:
                ticket = self.manager.app_sessions[app_ticket]
                if saml_req.session_index:
                    if ticket.session_id == sso_session:
                        if req_issuer != ticket.saml_ident:
                            raise InternalError(_('logout requester (%s) does not own session') % req_issuer)
                    # vérification du nameid
                    if (ticket.name_id != req_name_id):
                        raise InternalError(_('logout request for invalid session'))
                    # début d'un logout initié par un participant de la session
                    self.manager.pending_logout[sso_session] = [samlp.STATUS_SUCCESS, [app_ticket], (app_ticket, req_id, req_issuer), {}, [], None]
                    # on définit le ticket comme non valide
                    ticket.valid = False
                else:
                    raise InternalError(_('logout request for invalid session'))
            else:
                raise InternalError(_('Invalid session'))
            log.msg("%s -- %s" % (sso_session, _('SAMLRequest (%s) received from %s') % ('LogoutRequest', req_issuer)))
        except InternalError, msg:
            # On stocke les informations nécessaires à la génération du message d'erreur
            log.msg("%s -- %s" % (sso_session, _('Invalid SAMLRequest (%s) received from %s') % ('LogoutRequest', req_issuer)))
            return False, (req_id, req_issuer, str(msg))
        return True , sso_session

    @trace
    def check_attached_sessions(self, sso_session, closed_services, failed_services, from_url, css):
        """Vérifie les tickets rattachés à la session en cours et envoie les demandes de déconnexion nécessaires
        maintient la liste échecs/succés de chaque demande
        """
        # recherche des sessions non traitées lors du logout
        self.manager.pending_logout[sso_session][0] = samlp.STATUS_SUCCESS
        services_done = self.manager.pending_logout[sso_session][4]
        if from_url not in services_done:
            services_done.append(from_url)
        logout_call_list = []
        # on regarde si le logout a été initié par requête d'une entité partenaire
        if self.manager.pending_logout[sso_session][2]:
            req_ticket, req_id, req_issuer  = self.manager.pending_logout[sso_session][2]
        else:
            req_ticket = req_id = req_issuer = None

        for ticket in self.manager.user_app_tickets[sso_session]:
            if hasattr(ticket,'saml_ident'):
                if ticket.ticket != req_ticket:
                    if ticket.valid:
                        # ticket correspondant à une session 'saml' valide
                        # -> Envoi d'une demande de déconnexion auprès de l'entité partenaire
                        saml_identity = ticket.saml_ident
                        saml_role = ticket.saml_role
                        # Lecture des métadonnées du partenaire
                        sp_meta = self.manager.get_metadata(saml_identity)
                        user_id = ticket.name_id
                        return self.send_logout_request(saml_identity, sp_meta, user_id, ticket, sso_session, from_url, css)
                    else:
                        # Une déconnexion a déjà été demandée pour ce ticket
                        # -> On met à jour le status du logout en cours
                        if ticket.ticket in self.manager.pending_logout[sso_session][1]:
                            closed_services.append(ticket.saml_ident)
                        else:
                            failed_services.append(ticket.saml_ident)
                            # Au moins une des demandes envoyées n'a pas reçu de réponse favorable (logout partiel)
                            self.manager.pending_logout[sso_session][0] = samlp.STATUS_PARTIAL_LOGOUT
            else:
                if SEND_CAS_LOGOUT:
                    # ticket CAS standard, on envoie une requête de logout
                    # au client CAS (sur l'url de service du ticket).
                    if ticket.service_url not in services_done:
                        services_done.append(ticket.service_url)
                        if not ticket.ticket.startswith('PT-'):
                            # on n'envoie pas de demande de déconnexion
                            # sur les services authentifiés en mode proxy (imap, ftp, ...)
                            logout_call = (self.send_cas_logout_request,
                                        [ticket, sso_session, False],
                                        {})
                            logout_call_list.append(logout_call)
        if logout_call_list:
            # On appelle l'ensemble des logouts cas dans un thread
            callMultipleInThread(logout_call_list)
        return None


    @trace
    def build_logout_response(self, sso_session, relay_state, css):
        """construit une réponse à destination de l'entité ayant demandé la déconnexion
        """
        # Lecture des informations sur l'initiateur de la déconnexion
        app_ticket, req_id, req_issuer  = self.manager.pending_logout[sso_session][2]
        # recherche du role du partenaire (fournisseur de service/d'identité)
        saml_role = self.manager.app_sessions[app_ticket].saml_role
        # Si le logout est effectué suite à une LogoutRequest, on envoie une réponse au demandeur
        sp_meta = self.manager.get_metadata(req_issuer)
        try:
            binding, service_url, response_url = get_endpoint(sp_meta, 'SingleLogoutService', saml_role)
        except InternalError:
            traceback.print_exc()
            #XXX FIXME
            err_msg = _('no usable endpoint for %s') % req_issuer
            return http.Response(stream = gen_page(_('Logout Error'), err_msg, css))
        # Envoi de la réponse finale à l'émetteur de la requête
        response_id = self.manager.gen_saml_id({'type':'LogoutResponse'})
        sign_method, response = saml_message.gen_logout_response(req_id, response_id, IDP_IDENTITY, response_url, CERTFILE, \
                self.manager.pending_logout[sso_session][0])
        return self.send_logout_response(response, sign_method, sso_session, (binding, response_url, relay_state), css)

    def send_logout_response(self, response, sign_method, sso_session, dest_info, css):
        binding, response_url, relay_state = dest_info
        if binding == samlp.BINDING_HTTP_REDIRECT:
            # Réponse avec le binding REDIRECT
            response = encode_request(response, True)
            # la signature doit se faire sur la chaine suivante : SAMLResponse=value&RelayState=value&SigAlg=value -> encodé dans location header
            req_args = sign_request(CERTFILE, sign_method, 'SAMLResponse', response, relay_state)
            if req_args != []:
                redirect_url = '%s?%s' % (response_url, urllib.urlencode(req_args))
            else:
                redirect_url = response_url
            resp = RedirectResponse(redirect_url)
        else:
            # Réponse avec le binding POST (default fallback)
            response = encode_request(response)
            relay_info = ""
            if relay_state:
                relay_info = """<input type="hidden" name="RelayState" value="%s">""" % relay_state
            form = """
                    <FORM action="%s" method="POST">
                    <input type="hidden" name="SAMLResponse" value="%s">
                    %s
                    <input type="hidden" name="SigAlg" value="%s">
                    <p class="formvalidation">
                    <input class="btn" type="submit" value="%s"></p>
            """ % (response_url, split_form_arg(response), relay_info, sign_method, _('Logout confirmation to service'))
            resp = http.Response(stream = gen_page(_('Logout confirmation to service'), form, css))
            resp = self.set_headers(resp)
        log.msg("%s -- %s" % (sso_session, _("SAMLResponse (%s) sent to %s") % ('LogoutResponse', response_url)))
        return resp

    @trace
    def build_logout_page(self, sso_session, closed_services, failed_services, from_url, css):
        """construit la page à afficher en cas de déconnexion initiée localement
        """
        logout_msg = ""
        default_url, force_default = self.manager.get_default_logout_url(sso_session)
        if not from_url or force_default:
            from_url = default_url
        if closed_services != []:
            logout_msg += """<b><font color="green">%s</font></b>""" % _("Following service providers have been disconnected")
            logout_msg += "<br/><table border=0><tr><td>%s</td></tr></table>" % "</td></tr><tr><td>".join(closed_services)
        if failed_services != []:
            logout_msg += """<b><font color="orange">%s</font></b>""" % _("Following service providers did not report successful logout")
            logout_msg += "<br/><table border=0><tr><td>%s</td></tr></table>" % "</td></tr><tr><td>".join(failed_services)
        if from_url:
            logout_msg += """
            <br/><b>%s</b> : <br/><h2><font color="blue">%s</font></h2><br/>
            </p><p class="formvalidation">
            <button type="button" class="btn" onclick="javascript:location.href='%s'">%s</button></p>
            """ % (_('Following page has been proposed'), from_url, from_url, _('Proceed to above url'))
        else:
            logout_msg += """<br/><form action="/login" method="POST">
            <p class="formvalidation">
            <input class="btn" type="submit" value="%s"></p></form>""" % _('New session')
        if self.manager.pending_logout[sso_session][0] == samlp.STATUS_PARTIAL_LOGOUT:
            logout_status = _("Partial single logout success")
        else:
            logout_status = _("Single logout success")
        resp = http.Response(stream = gen_page(logout_status, str(logout_msg), css))
        return self.set_headers(resp)

    def _render(self, request):
        """Gestion de la déconnexion des sessionss SSO
        """
        # vérification de la session en cours
        cookie = None
        authenticated = False
        cookies = getCookies(request)
        sso_session = None
        for cookie in cookies:
            if cookie.name == 'EoleSSOServer':
                sso_session = cookie.value
                authenticated = self.manager.validate_session(sso_session)
                break

        css = DEFAULT_CSS
        # vérifie si on doit rediriger après la déconnexion
        from_url = None
        relay_state = None
        if 'RelayState' in request.args:
            relay_state = request.args['RelayState'][0]
        if request.args.has_key('url'):
            from_url = request.args['url'][0]
        elif request.args.has_key('service'):
            from_url = request.args['service'][0]
        elif relay_state:
            # XXX FIXME : vérifier que relay_state est une url ?
            from_url = relay_state
        # recherche d'une eventuelle css associée
        if from_url:
            service_filter = self.manager._check_filter(from_url)[0]
            if service_filter not in ['default', '']:
                if os.path.exists(os.path.join('interface', '%s.css' % service_filter)):
                    css = service_filter
        try:
            css = request.args['css'][0]
        except KeyError:
            pass
        closed_services = []
        failed_services = []
        resp = None
        saml_resp = False

        try:
            ########################
            # Gestion du logout SAML
            if not authenticated:
                # pas d'authentification détectée
                sso_session = None
                if not 'SAMLRequest' in request.args:
                    # logout initié localement, on devrait être authentifié
                    raise InternalError(_('Invalid session'))
            # traitement d'une requête reçue : LogoutRequest ou LogoutResponse
            if 'SAMLResponse' in request.args and sso_session in self.manager.pending_logout:
                # test sur pending logout : hack car dans certains cas firefox semble rejouer une requête de logout
                # vers le sp sans que l'idp l'aie envoyée
                self.process_logout_response(request, sso_session)
            if 'SAMLRequest' in request.args:
                req_ok, err_infos = self.process_logout_request(request, sso_session)
                if not req_ok:
                    # requête non valide, on renvoie un échec de déconnexion
                    return self.build_logout_error(err_infos, sso_session, relay_state, css)
            elif sso_session not in self.manager.pending_logout:
                # Logout initié localement (idp initiated), on conserve une trace de relay_state et from_url
                # au cas où on aurait des redirections à faire (requêtes SAML)
                self.manager.pending_logout[sso_session] = [samlp.STATUS_SUCCESS, [], None, {}, [], (from_url, relay_state)]
            # envoi de requêtes de déconnexion pour chaque session attachée à la session SSO
            resp = self.check_attached_sessions(sso_session, closed_services, failed_services, from_url, css)
            if resp is not None:
                return resp
            # Tous les participants à la session sont prévenus, calcul du résultat final
            # on regarde si une url de redirection était présente dans la requête d'origine
            if self.manager.pending_logout[sso_session][5] is not None:
                from_url, relay_state = self.manager.pending_logout[sso_session][5]
            if self.manager.pending_logout[sso_session][2]:
                # Envoi d'une réponse à la demande de déconnexion
                saml_resp = True
                resp = self.build_logout_response(sso_session, relay_state, css)
            else:
                # affichage d'un message (déconnexion locale)
                resp = self.build_logout_page(sso_session, closed_services, failed_services, from_url, css)

            ############################
            # Logout SSO Session Globale

            default_url, force_default = self.manager.get_default_logout_url(sso_session)
            # si la session vient d'une fédération avec un idp distant,
            # on regarde si une url de retour est configurée (et obligatoire)
            if force_default:
                from_url = default_url
            # suppression finale de la session sso et des données associées
            self.manager.logout(sso_session)
        except InternalError, err:
            log.msg(err)
        if not saml_resp:
            # logout non initié par une requête saml, on affiche la page de compte rendu de déconnexion
            # ou on redirige vers l'url spécifiée par 'url' ou 'service'
            if from_url and failed_services == []:
                # Dans le cas ou on doit rediriger et aucune erreur n'est survenue -> pas d'affichage
                # XXX FIXME : afficher un formulaire d'avertissement dans tous les cas ?
                resp = RedirectResponse(from_url)
                log.msg(_("Logged out : redirecting to %s") % from_url)
            elif resp is None:
                resp = RedirectResponse('/')

        # expiration du cookie EoleSSO
        if sso_session and cookie:
            cookie.expires = 1
            resp.headers.setHeader('Set-Cookie', cookies)
        return resp

class SamlMetadata(SamlResource):

    def __init__(self, manager):
        self.manager = manager
        super(SamlMetadata, self).__init__()

    def _render(self, request):
        """Affiche les métadonnées SAML du serveur local"""
        data = gen_metadata(self.manager, AUTH_FORM_URL, CERTFILE)
        resp = http.Response(stream = data.encode(encoding))
        return self.set_headers(resp, {'Content-type': ('application/xml',)})

class SamlErrorPage(SamlResource):
    """
    page d'erreur http
    """
    addSlash = False

    def __init__(self, code, description):
        self.code = code
        self.description = description
        title = ('EoleSSO : Erreur')
        super(SamlErrorPage, self).__init__()

    def _render(self, request):
        return http.StatusResponse(code = self.code, description = self.description)

class SamlConsumer(SamlResource):
    """Resource de traitement des demandes d'accès aux service tiers
    (Fournisseur de service)
    """
    addSlash = False

    def __init__(self, manager):
        self.manager = manager
        super(SamlConsumer, self).__init__()

    def _render(self, request):
        """Assertion Consuming processing
        """
        # vérification d'une éventuelle session en cours
        authenticated = False
        cookies = getCookies(request)
        sso_session = None
        for cookie in cookies:
            if cookie.name == 'EoleSSOServer':
                sso_session = cookie.value
                # On vérifie si la session est valide
                authenticated = self.manager.validate_session(sso_session)
                break
        # XXX FIXME : si déjà authentifié: supprimer le cookie actuel ou vérifier qu'il correspond au bon utilisateur ?
        # pb: single logout de l'ancienne session ?
        css = DEFAULT_CSS
        # vérification de l'adresse où l'utilisateur doit être redirigé
        return_url = None
        relay_state = None
        if 'RelayState' in request.args:
            relay_state = request.args['RelayState'][0]
        # recherche d'une eventuelle css associée
        if relay_state:
            # on regarde si RelayState provient d'une requête émise précédemment
            return_url = self.manager.get_relay_data(relay_state)
            if return_url is None:
                # relay_state fourni par le fournisseur d'identité
                return_url = relay_state
            service_filter = self.manager._check_filter(return_url)[0]
            if service_filter not in ['default', '']:
                if os.path.exists(os.path.join('interface', '%s.css' % service_filter)):
                    css = service_filter
        try:
            css = request.args['css'][0]
        except KeyError:
            pass
        try:
            ###########################
            # traitement de l'assertion
            if authenticated:
                # l'utilisateur est déjà authentifié
                pass
                # XXX FIXME vérifier que l'utilisateur de l'assertion correspond ?
            if 'SAMLResponse' not in request.args:
                # pas d'authentification détectée
                raise InternalError(_('SAMLResponse expected'))
            else:
                # Traitement de la requête reçue (AuthnStatement)
                success, infos =  process_assertion(self.manager, request)

        except InternalError, e:
            traceback.print_exc()
            return self.set_headers(gen_page_err(str(e), return_url, css))
        # l'assertion reçue est valide, on la traite
        if not success:
            content = _('Federation : No local user does match specified attributes')
            return self.set_headers(gen_page_err(content, return_url, css))
        else:
            for assertion_id, data in infos.items():
                # si des attributs sont remontés et valides, on cherche l'utilisateur correspondant (fédération par attribut commun)
                session_index, assertion_issuer, resp_attrs, name_id, auth_instant, class_ref = data
                # si pas de destination spécifiée (dans RelayState ou dans la requête d'origine),
                # on regarde si une destination par défaut existe pour ce fournisseur d'identité
                if not return_url:
                    idp_opts = self.manager.get_federation_options(assertion_issuer)
                    if 'default_service' in idp_opts:
                        return_url = idp_opts['default_service']
                content = u"<h2>%s</h2>" % (_("you have been authenticated by identity provider %s") % assertion_issuer,)
                # On essaye de créer une session pour l'utilisateur local correspondant (ou on reprend l'existante)
                if not authenticated:
                    sso_session = None
                # on regarde si la branche de recherche de l'utilisateur peut être déduite des attributs
                # (ex : n°rne présent)
                defer_federation = self.manager.authenticate_federated_user(resp_attrs, assertion_issuer, auth_instant, class_ref, sso_session)
                return defer_federation.addBoth(self.callb_federation, return_url, assertion_id, data, content, css, request, cookies)
            return self.set_headers(gen_page_err(_("NO USER FOUND"), return_url, css))

    @trace
    def callb_federation(self, result_fed, return_url, assertion_id, data, content, css, request, cookies):
        session_id, user_data = result_fed
        session_index, assertion_issuer, resp_attrs, name_id, auth_instant, class_ref = data
        if session_id and not isinstance(session_id, failure.Failure):
            # Mise en place de cookie SSO pour la session locale
            cookies = []
            for cookie in getCookies(request):
                if cookie.name != 'EoleSSOServer':
                    cookies.append(cookie)
            cookies.append(TwCookie("EoleSSOServer", session_id, path = "/", discard=True, secure=True))
            # Si une session existe pour ce fournisseur de service, on le supprime
            self.manager.remove_old_ticket(session_id, assertion_issuer)
            # On stocke les informations de session dans un ticket fictif (pour permettre le logout)
            sp_meta = self.manager.get_metadata(assertion_issuer)
            # recherche d'un éventuel index pour le choix du binding
            endpoint_index = None
            if 'index' in request.args:
                endpoint_index = request.args['index'][0]
            try:
                binding, service_url, response_url = get_endpoint(sp_meta, 'SingleSignOnService',
                                                                  ent_type = 'IDPSSODescriptor',
                                                                  index=endpoint_index)
            except InternalError:
                # XXX FIXME
                err_msg = _('no usable endpoint for %s') % assertion_issuer
                log.msg(err_msg)
                return self.set_headers(http.Response(stream = gen_page(_('Assertion consuming error'), err_msg, css)))
            app_ticket = self.manager.get_app_ticket(session_id, response_url, idp_ident=assertion_issuer)
            if app_ticket:
                # pour les applications saml, on utilise le filtre saml par défaut (pour avoir au moins l'attribut de federation)
                if self.manager.app_sessions[app_ticket].filter == 'default':
                    self.manager.app_sessions[app_ticket].filter = 'saml'
                # application de la css correspondante si existante
                service_filter = self.manager.app_sessions[app_ticket].filter
                if service_filter not in ['default', '']:
                    if os.path.exists(os.path.join('interface', '%s.css' % service_filter)):
                        css = service_filter
                # information supplémentaires à conserver
                self.manager.app_sessions[app_ticket].saml_ident = assertion_issuer
                self.manager.app_sessions[app_ticket].saml_role = 'IDPSSODescriptor'
                self.manager.app_sessions[app_ticket].response_id = assertion_id
                self.manager.app_sessions[app_ticket].name_id = name_id
                self.manager.app_sessions[app_ticket].idp_session_index = session_index
                if 'uaj' in sp_meta:
                    # ajout de l'uaj établissement si disponible dans les métadonnées (spécifique Education/Eole)
                    self.manager.app_sessions[app_ticket].uaj = sp_meta.get('uaj')
                if self.manager.app_sessions[app_ticket].timeout_callb.cancelled == 0:
                    # ce ticket expirera seulement à la fermeture de la session SSO
                    self.manager.app_sessions[app_ticket].timeout_callb.cancel()
                self.manager.saml_sessions[session_index] = app_ticket
            # XXX FIXME : Debug
            if return_url is None:
                if DEBUG_LOG:
                    content += """assertion_id = %s<br/><hr/>
        idp session_index = %s<br/>
        attributes = %s<br/>
        authentication context = %s<br/><hr/>
        local session created<br/>""" % (assertion_id, session_index, str(resp_attrs), class_ref)
                else:
                    content += "<br>%s" % _("no destination service given")
            # content += """%s: %s<br/>""" % (_('Redirecting to '), return_url)
        else:
            if isinstance(session_id, failure.Failure):
                log.msg('Failure searching federated_user : %s' % session_id.getErrorMessage())
            content = _('Federation : No local user does match specified attributes')
            resp = gen_page_err(content, return_url, css)
            return self.set_headers(gen_page_err)
        return self.redirect_after_federation(return_url, content, css, cookies)

    @trace
    def redirect_after_federation(self, return_url, content, css, cookies):
        if return_url:
            resp = RedirectResponse(return_url)
        else:
            resp = http.Response(stream = gen_page('Request processed', content, css))
            resp = self.set_headers(resp)
        resp.headers.setHeader('Set-Cookie', cookies)
        return resp

class SamlResponse(SamlResource):
    """ressource principale
    traitement de l'envoi d'assertions SAML
    (Fournisseur d'identité)
    """

    def __init__(self, manager):
        self.manager = manager
        super(SamlResponse, self).__init__()

    def _render(self, request):
        """Génère une réponse SAML2 correspondant au profil HTTP-POST

        - verifie qu'une session SSO est déjà en cours
        - sinon, essaie d'en établir une
        - construit une Reponse SAML (contenant une assertion ou non) et l'envoie au fournisseur de service (entityID)
        """
        app_ticket = None
        relay_state = None
        app_ticket = None
        req_id = None
        sp_ident = None
        passive = False
        force_auth = False
        requested_contexts = []
        comparison = ""
        login_ticket = None
        css = DEFAULT_CSS
        compressed = True
        if 'ticket' in request.args:
            #compressed = False
            app_ticket = request.args['ticket'][0]
        # recherche d'un éventuel index pour le choix du binding
        endpoint_index = None
        if 'index' in request.args:
            endpoint_index = request.args['index'][0]
        if 'SAMLRequest' in request.args:
            # requête SAML AuthnRequest reçue
            request_data = process_auth_request(self.manager, request, compressed=compressed)
            # stockage des données de la requête pour gérer la réponse (on génère un ticket 'LT' avec les données associées)
            sp_ident, response_url, binding, req_id, passive, force_auth, comparison, requested_contexts = request_data
            login_ticket = self.manager.get_login_ticket(req_id)
            # stockage temporaire du résultat pour conserver les informations d'origine lorsqu'une redirection
            # sur la page d'authentification est nécessaire
            req_ticket = self.manager.gen_relay_state(request_data)
            log.msg("%s : %s" % (_('SAMLRequest (%s) received from %s') % ('AuthnRequest', sp_ident), req_id))
            sp_meta = self.manager.get_metadata(sp_ident)
            # on vérifie que l'on est capable de gérer le niveau de sécurité demandé (requested_context)
            if requested_contexts:
                ctx_ok, required_ctx = check_required_contexts(comparison, requested_contexts)
                if not ctx_ok:
                    # XXX FIXME TODO: generate a saml negative saml response or display local error page (invalid context) ?
                    resp = http.Response(stream = gen_page(_('Unreachable resource'), required_ctx, css))
                    return self.set_headers(resp)
        elif 'ReqInfos' in request.args:
            # reprise des informations de la requête après nouvelle authentification
            req_ticket = request.args['ReqInfos'][0]
            try:
                request_data = self.manager.get_relay_data(req_ticket)
                sp_ident, response_url, binding, req_id, passive, force_auth, comparison, requested_contexts = request_data
                sp_meta = self.manager.get_metadata(sp_ident)
            except:
                # XXX FIXME TODO: generate a saml negative saml response or display local error page (invalid context) ?
                resp = http.Response(stream = gen_page(_('Unreachable resource'), 'ReqInfos=%s' % req_ticket, css))
                return self.set_headers(resp)
        else:
            # Authentification initiée par l'IDP
            if 'sp_ident' in request.args:
                sp_ident = request.args['sp_ident'][0]
                # recherche des metadonnées du fournisseur de services
                sp_meta = self.manager.get_metadata(sp_ident)
                sp_ident = sp_meta.get('entityID', sp_ident)
                # recherche de l'adresse à laquelle envoyer les assertions
                try:
                    binding, service_url, response_url = get_endpoint(sp_meta, 'AssertionConsumerService', index=endpoint_index)
                except InternalError:
                    traceback.print_exc()
                    err_msg = _('no usable endpoint for %s') % sp_ident
                    resp = http.Response(stream = gen_page(_('Unreachable resource'), err_msg, css))
                    return self.set_headers(resp)
            else:
                # XXX FIXME TODO: generate a saml negative saml response or display local error page (invalid issuer) ?
                err_msg = _("Missing parameter")
                resp = http.Response(stream = gen_page(_('Unreachable resource'), err_msg, css))
                return self.set_headers(resp)

        # verification de l'authentification
        authenticated = False
        session_id = None
        user_id = ""
        user_data = {}
        cookies = getCookies(request)
        from_credentials = False
        for cookie in cookies:
            if cookie.name == 'EoleSSOServer':
                session_id = cookie.value
                # vérification de la validité de la session
                authenticated = self.manager.validate_session(session_id)
        # cas ou une (ré)authentification auprès de l'utilisateur est nécessaire
        if not authenticated or (force_auth and not app_ticket):
            if not passive:
                redirect_args = []
                for argname, argvalue in request.args.items():
                    if argname == "SAMLRequest":
                        # requête d'authentification : on conserve en cache le résultat du processing
                        # pour ne pas l'effectuer une 2ème fois.
                        argname = "ReqInfos"
                        argvalue = [req_ticket]
                    redirect_args.append((argname, argvalue[0]))
                req_args = [('service', "%s/saml?%s" % (AUTH_FORM_URL, urllib.urlencode(redirect_args)))]
                if force_auth:
                    req_args.append(('renew', 'true'))
                if login_ticket:
                    req_args.append(('lt', login_ticket))
                redirect_url = '%s?%s' % (AUTH_FORM_URL, urllib.urlencode(req_args))
                resp = RedirectResponse(redirect_url)
                return resp
            elif req_id:
                # mode passif : on continue sans authentifier (réponse négative)
                sign_method, response = saml_message.gen_status_response(CERTFILE, IDP_IDENTITY, response_url, samlp.STATUS_NO_PASSIVE, req_id, _("no previous authentication"))

        # vérification des paramètres passés dans la requête
        try:
            css = request.args['css'][0]
        except KeyError:
            css = DEFAULT_CSS
        if 'RelayState' in request.args:
            relay_state = request.args['RelayState'][0]
        # La session est simulée à travers un ticket CAS factice
        # on utilise l'url de traitement des assertions pour permettre un filtrage sur le service de destination
        if app_ticket:
            # On regarde si le ticket a été délivré au moment de l'authentification
            ticket = self.manager.app_sessions[app_ticket]
            if ticket.session_id != session_id or \
               ticket.from_credentials != True or \
               ticket.service_url != "%s/saml" % (AUTH_FORM_URL):
                app_ticket = None
        if not app_ticket and authenticated:
            # Si une session existe déjà pour ce fournisseur de service, on la supprime
            self.manager.remove_old_ticket(session_id, sp_ident)
            # on génère un nouveau ticket pour ce fournisseur de service
            app_ticket = self.manager.get_app_ticket(session_id, response_url)
            ticket = self.manager.app_sessions[app_ticket]
        if app_ticket:
            # modification du service du ticket pour correspondre à l'entité partenaire
            ticket.service_url = get_service_from_url(response_url)
            if sp_ident in self.manager.apps['sp_ident']:
                ticket.filter = self.manager.apps['sp_ident'][sp_ident][0]
            else:
                id_filter = self.manager._check_filter(ticket.service_url)[0]
                ticket.filter = id_filter
            # pour les applications saml, on utilise le filtre saml par défaut (pour avoir au moins l'attribut de federation)
            if self.manager.app_sessions[app_ticket].filter == 'default':
                self.manager.app_sessions[app_ticket].filter = 'saml'
            # application de la css correspondant au filtre si existante
            service_filter = self.manager.app_sessions[app_ticket].filter
            if service_filter not in ['default', ''] and css == DEFAULT_CSS:
                if os.path.exists(os.path.join('interface', '%s.css' % service_filter)):
                    css = service_filter
            # Génrération d'un attribut nameid aléatoire pour cette session
            # Le fournisseur de service peut spécifier les attributs requis dans leurs métadonnées
            # (AttributeConsumingService/RequestedAttribute dans le xml)
            # Tous les attributs définis dans le filtre d'appplication seront envoyés
            user_id = gen_random_id('NAMEID_')
            response_id = self.manager.gen_saml_id({'type':'SamlResponse'})
            # Stockage d'informations supplémentaires dans le ticket d'application
            # sp identifier
            # response id
            # id d'authnRequest si present ?
            self.manager.app_sessions[app_ticket].saml_ident = sp_ident
            self.manager.app_sessions[app_ticket].saml_role = 'SPSSODescriptor'
            self.manager.app_sessions[app_ticket].response_id = response_id
            self.manager.app_sessions[app_ticket].name_id = user_id
            if 'uaj' in sp_meta:
                # ajout de l'uaj établissement si disponible dans les métadonnées (spécifique Education/Eole)
                self.manager.app_sessions[app_ticket].uaj = sp_meta.get('uaj')
            # ticket de type SAML : expire seulement si demandé
            if self.manager.app_sessions[app_ticket].timeout_callb.cancelled == 0:
                self.manager.app_sessions[app_ticket].timeout_callb.cancel()
            # récupération des données disponibles pour le service à atteindre
            app_ok, user_data = self.manager.get_user_details(app_ticket, response_url, renew=False, sections=True, keep_valid=True)
            data, filter_data = user_data
            user_data = {}
            # modification du nom des attributs en fonction du filtre
            for attrs in filter_data.values():
                for name, label in attrs.items():
                    user_data[name] = data[label]
            from_credentials = self.manager.app_sessions[app_ticket].from_credentials

            attr_table = []
            for attr, val in user_data.items():
                if type(val) == list:
                    val = ', '.join(val)
                if val == "":
                    val = "&nbsp;"
                attr_table.append('<tr><td>%s</td><td>%s</td></tr>' % (attr, val))
            attr_msg = _('Following attributes will be sent')
            # récupération de la date d'authentification
            auth_instant = self.manager.get_auth_instant(app_ticket)
            # on récupère le type d'authentification de la session SSO
            auth_class = self.manager.get_auth_class(app_ticket)
            # récupération de l'adresse du client effectuant la demande
            addr_client = request.chanRequest.getRemoteHost().host
            try:
                dns_client = socket.gethostbyaddr(addr_client)[0]
            except:
                # pas de résolution dns possible
                dns_client = None
            sign_method, response = saml_message.gen_response(response_id, authenticated, auth_instant, \
                                    app_ticket, from_credentials, user_id, user_data, req_id, response_url, \
                                    IDP_IDENTITY, sp_ident, CERTFILE, auth_class, addr_client, dns_client)

        if request.args.has_key('show') and DEBUG_LOG:
            resp = http.Response(stream = response.encode(encoding))
            return self.set_headers(resp, {'Content-type': ('application/xml',)})
        if DISPLAY_FEDERATION == True and user_data:
            # affichage des attributs envoyés
            msg = """<table border=1><tr><th colspan=2>%s</th></tr>%s</table>""" % (attr_msg, '\n'.join(attr_table))
            submit_input = """<p class="formvalidation"><input class="btn" type="Submit" value="%s"></p>""" % _('Proceed to service')
        else:
            msg = _('Please wait, accessing requested service...')
            submit_input = ""
        if binding == samlp.BINDING_HTTP_REDIRECT:
            # REDIRECT response
            response = encode_request(response, True)
            req_args = sign_request(CERTFILE, sign_method, 'SAMLResponse', response, relay_state)
            if req_args != []:
                redirect_url = '%s?%s' % (response_url, urllib.urlencode(req_args))
            else:
                redirect_url = response_url
            resp = RedirectResponse(redirect_url)
        else:
            # POST response (fallback par défaut)
            response = encode_request(response)
            relay_info = ""
            if relay_state:
                relay_info = """<input type="hidden" name="RelayState" value="%s">""" % relay_state

            form = """%s<br/><div class='ressource' id="wait_msg"></div><br/>
<FORM name="federate_user" action="%s" method="POST" onSubmit="return aff_wait()">
<input type="hidden" name="SAMLResponse" value="%s">
%s
<input type="hidden" name="SigAlg" value="%s">
%s
</FORM>""" % (msg, response_url, response, relay_info, sign_method, submit_input)
            # recherche du nom 'humain' de l'entité si disponible
            entity_name = sp_meta.get('entity_local_id', '')
            wait_msg = _("Awaiting response from ")
            header_script = """<script type="text/javascript">
function aff_wait()
{
    div_msg = document.getElementById('wait_msg');
    div_msg.innerHTML = "%s<span>%s</span>";
    return true;
}
</script>
""" % (wait_msg, response_url)
            head = "%s : <br/>%s" % (_('You are being Authenticated to entity'), entity_name or sp_ident)
            if DISPLAY_FEDERATION == False or user_data == {}:
                head = _('Accessing to service')
                form += """<script type="text/javascript">aff_wait();document.federate_user.submit();</script>"""
            resp = http.Response(stream = gen_page(head, form, css, header_script=header_script))
            resp = self.set_headers(resp)
        if session_id:
            log_prefix = "%s -- " % session_id
        else:
            log_prefix = ""
        log.msg("%s%s" % (log_prefix, _('SAMLResponse (%s) sent to %s') % ('AuthnStatement', sp_ident)))

        return resp

    def locateChild(self, request, segments):
        if segments and segments[0] == 'metadata':
            return SamlMetadata(self.manager), ()
        if segments and segments[0] == 'acs':
            return SamlConsumer(self.manager), ()
        if segments and segments[0] == 'logout':
            return SamlLogout(self.manager), ()
        if segments and segments[0] == 'discovery':
            return IDPDiscoveryService(self.manager), ()
        if segments == ():
            return self, ()
        return SamlErrorPage(responsecode.NOT_FOUND, "%s : %s" % (_('Unreachable resource'),  '/'.join(segments))), ()

class IDPDiscoveryService(SamlResource):

    isLeaf = True

    def __init__(self, manager):
        self.manager = manager
        SamlResource.__init__(self)

    @trace
    def render(self, request):
        try:
            css = request.args['css'][0]
        except KeyError:
            css = DEFAULT_CSS
        # récupération de l'id du fournisseur de service à contacter
        try:
            idp_ident = request.args['idp_ident'][0]
        except:
            return SamlErrorPage(responsecode.INTERNAL_SERVER_ERROR, _("missing parameter: Identity Provider"))
        # XXx FIXME : obligatoire / donner juste RelayState ?
        # url de retour après authentification
        try:
            # https%3A%2F%2Fbe1d.ac-dijon.fr%2Fsso%2FSSO%3FSPEntityID%3Durn%253Afi%253Aac-dijon%253Aet-bb.in.ac-dijon.fr%253A1.0%26TARGET%3Dhttps%253A%252F%252Fbb.in.ac-dijon.fr%253A8443%252Floggedin
            return_url = request.args['return_url'][0]
        except:
            return_url = ""
        # recherche d'un éventuel index pour le choix du binding
        endpoint_index = None
        if 'index' in request.args:
            endpoint_index = request.args['index'][0]
        # recherche des urls de réception pour la requête d'auth.
        must_sign_request = False
        try:
            sp_meta = self.manager.get_metadata(idp_ident)
            idp_ident = sp_meta.get('entityID', idp_ident)
            must_sign_request = is_true(sp_meta['IDPSSODescriptor'].get('WantAuthnRequestsSigned', 'false'))
            binding, service_url, consumer_url = get_endpoint(sp_meta, 'SingleSignOnService',
                                                              ent_type='IDPSSODescriptor', allowed_bindings=[samlp.BINDING_HTTP_REDIRECT, samlp.BINDING_HTTP_POST],
                                                              index=endpoint_index)
        except:
            err_msg = _('no usable endpoint for %s') % idp_ident
            return SamlErrorPage(responsecode.INTERNAL_SERVER_ERROR, err_msg)
        # si ce fournisseur d'identité est désactivé, on n'envoie pas de requête
        # on regarde si la fédération est autorisée pour ce partenaire
        if not self.manager.check_federation_allowed(idp_ident):
            return SamlErrorPage(responsecode.UNAUTHORIZED, "%s : %s" % (idp_ident, _('no federation allowed for this entity')))
        # récupération de l'index du jeu d'attributs associé
        attr_service_index = self.manager.get_attribute_service_index(idp_ident)
        # stockage de l'id de requête pour vérification du message de réponse du FI (in_reponse_to)
        request_id = self.manager.gen_saml_id({'type':'SamlRequest'})
        # récupération des eventuelles options pour ce fournisseur d'identité
        idp_options = self.manager.get_federation_options(idp_ident)
        force_auth = is_true(idp_options.get('force_auth', 'false'))
        is_passive = is_true(idp_options.get('passive', 'false'))
        must_sign_request = idp_options.get('sign_request', None) or must_sign_request
        # le niveau d'authentification demandé par défaut est : >= mot de passe protégé (https)
        comparison = idp_options.get('comparison', 'minimum')
        class_ref = idp_options.get('req_context', saml_message.URN_PROTECTED_PASSWORD)
        # génération d'une requête d'authentification
        sign_method, saml_request = saml_message.gen_request(self.manager, request_id, IDP_IDENTITY,
                                                             idp_ident, consumer_url, attr_service_index, CERTFILE,
                                                             sign_request=must_sign_request,
                                                             is_passive=is_passive, force_auth=force_auth,
                                                             comparison=comparison, class_ref=class_ref)
        # on stocke l'adresse de destination après authentification
        if request.args.has_key('show') and DEBUG_LOG:
            resp = http.Response(stream = saml_request.encode(encoding))
            return self.set_headers(resp, {'Content-type': ('application/xml',)})
        if return_url:
            # on génère un relay_state opaque pour retrouver l'url de retour
            # (permet d'utiliser des adresses de taille > 80 caractères)
            relay_state = self.manager.gen_relay_state(return_url)
            # on conserve le lien dans le message saml pour validation au traitement de la réponse
            self.manager.update_saml_msg(request_id, {'relay_state':relay_state})
        else:
            relay_state = None
        if binding == samlp.BINDING_HTTP_REDIRECT:
            # REDIRECT response
            saml_request = encode_request(saml_request, True)
            if must_sign_request:
                req_args = sign_request(CERTFILE, sign_method, 'SAMLRequest', saml_request, relay_state)
                if req_args != []:
                    redirect_url = '%s?%s' % (consumer_url, urllib.urlencode(req_args))
                else:
                    redirect_url = consumer_url
            else:
                req_args = []
                req_args.append(('SAMLRequest', saml_request))
                if relay_state:
                    req_args.append(('RelayState', relay_state))
                redirect_url = '%s?%s' % (consumer_url, urllib.urlencode(req_args))
            resp = RedirectResponse(redirect_url)
        else:
            # Réponse avec le binding POST (default fallback)
            saml_request = encode_request(saml_request)
            sign_method_info = ""
            relay_info = ""
            if relay_state:
                relay_info = """
                    <input type="hidden" name="RelayState" value="%s">""" % relay_state
            if must_sign_request:
                sign_method_info = """
                    <input type="hidden" name="SigAlg" value="%s">""" % sign_method
            form = """%s<br/>
                    <FORM name="form_request" action="%s" method="POST">
                    <input type="hidden" name="SAMLRequest" value="%s">%s%s
                    <script type="text/javascript">document.form_request.submit();</script>
                    </FORM>
            """ % (_('Contacting authentication server'), consumer_url, split_form_arg(saml_request), sign_method_info, relay_info)
            resp = self.set_headers(http.Response(stream = gen_page(_('IDPDiscovery'), form, css)))
        log.msg(_("Sending SAML authentication request to %s (%s) : %s") % (idp_ident, consumer_url, request_id))
        return resp
