#! /usr/bin/env python
# -*- coding: utf-8 -*-

###########################################################################
#
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# oidc_resources.py
#
# support du protocole OPENID Connect pour eole-sso
#
###########################################################################

import traceback
import urllib
import hashlib

from cgi import escape
from twisted.web2 import http
from twisted.web2.http_headers import Cookie as TwCookie

from eolesso.util import RedirectResponse, getCookies, getCookie

from page import log
from config import (AUTH_FORM_URL)
from saml_resources import SamlResource
from oidc_utils import request_auth, get_user_identifier, check_state
from oidc_utils import AccessDeniedError

class OIDConnect(SamlResource):
    """Resource used to initiate an Openid Connect authorization Request
    """

    required_parameters = ('provider',)

    def __init__(self, manager):
        self.manager = manager
        super(OIDConnect, self).__init__()

    def _render(self, request):
        try:
            provider = escape(request.args.get('provider', [None])[0])
            assert provider and provider in self.manager.external_providers, _("Unknown OpenID Provider")
            login_ticket = escape(request.args.get('lt', [None])[0])
            assert login_ticket and self.manager._DBLoginSessionsValidate(login_ticket), _("Access denied")
            orig_args = self.manager.login_sessions.get_session_info(login_ticket)
        except AssertionError, err:
            log.msg(_("OIDConnect : Error retrieving parameters : {0}").format(str(err)))
            response = http.Response(code=http.responsecode.BAD_REQUEST, stream = str(err))
            return self.set_headers(response)
        except Exception:
            traceback.print_exc()
            response = http.Response(code=http.responsecode.INTERNAL_SERVER_ERROR, stream = str(_('Access denied')))
            return self.set_headers(response)
        # récupération de la session twisted pour association du 'state' OpenID Connect
        connect_url, temp_cookie = request_auth(provider, self.manager, orig_args)
        # on créée un cookie de session temporaire pour vérifier que STATE correspond au retour (callback)
        cookies = []
        for cookie in getCookies(request):
            if cookie.name != 'EoleSSO_OIDC':
                cookies.append(cookie)
        cookies.append(TwCookie('EoleSSO_OIDC', temp_cookie, path = "/oidcallback", discard=True, secure=True))
        response = RedirectResponse(connect_url)
        response.headers.setHeader('Set-Cookie', cookies)
        return response

class OIDCallback(SamlResource):
    """Resource used to process Openid Connect authorization Response and get user identifier
    """

    required_parameters = ('provider',)

    def __init__(self, manager):
        self.manager = manager
        super(OIDCallback, self).__init__()

    def _render(self, request):
        try:
            state = escape(request.args.get('state', [None])[0])
            temp_cookie = getCookie(request, 'EoleSSO_OIDC')
            # check for state consistency against temporary cookie
            assert check_state(temp_cookie, state), _("Access denied")
            state_data  = self.manager.relay_state_cache.get_msg(state)
            assert state_data and state_data.get('provider', None) in self.manager.external_providers, _("Unknown OpenID Provider")
            provider = state_data['provider']
            orig_args = state_data['orig_args']
            nonce = state_data['nonce']
        except AssertionError, err:
            response = http.Response(code=http.responsecode.BAD_REQUEST, stream = str(err))
            return self.set_headers(response)
        except Exception:
            traceback.print_exc()
            response = http.Response(code=http.responsecode.INTERNAL_SERVER_ERROR, stream = str(_('Access denied')))
            return self.set_headers(response)
        # retrieve provider name in state cache
        client = self.manager.external_providers[provider][0]
        user_store = self.manager.external_providers[provider][1]
        # get user Id (sub) from response
        response = request.querystring
        reason = "error"
        try:
            id_token_hint, access_token, user_id = get_user_identifier(response, client, state, nonce)
        except AccessDeniedError:
            log.msg(_('User denied access to account information'))
            reason = "denied"
            user_id = None
        except Exception, err:
            log.msg(_('Error when authenticating to provider {0} : {1}')\
                    .format(self.manager.external_providers[provider][2], str(err)))
            user_id = None
        if user_id:
            # store user information (manager.session_info) and redirect on login page with provider name
            session_data = {'username':user_id,
                            'orig_args':orig_args,
                            'provider':provider,
                            'id_token_hint':id_token_hint,
                            'state':state,
                            'access_token':access_token
                           }
            lt = self.manager.get_login_ticket(session_data)
            req_args = (('from_provider', "1"), ('lt', lt))
            log.msg(_("user identified by provider {0}, looking for local user").format(provider))
            redirect_url = '%s?%s' % (AUTH_FORM_URL, urllib.urlencode(req_args))
        else:
            # provider did not return user identifier, return to login page with auth error ?
            req_args = {'failed':'1', 'reason':reason, 'provider':provider}
            req_args.update(orig_args)
            redirect_url = '%s?%s' % (AUTH_FORM_URL, urllib.urlencode(tuple(req_args.items())))
        return RedirectResponse(redirect_url)
