#! /usr/bin/env python
# -*- coding: utf-8 -*-

###########################################################################
#
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# cas_resources.py
#
# resssources pour le formulaire d'authentification du serveur EoleSSO
#
# compatible avec les versions 1 et 2 de la librairie CAS
# (http://www.ja-sig.org/products/cas/)
#
###########################################################################

import re
import os
import urlparse
from urllib2 import urlopen
import urllib
import SOAPpy
import traceback
from cgi import escape, parse_qsl
try:
    import json
    json_dump = json.dumps
except:
    import simplejson as json
    if not hasattr(json, 'dumps'):
        # ancienne version, write est 'deprecated' dans les versions récentes
        json_dump = json.write
    else:
        json_dump = json.dumps

from twisted.web2 import static, http, responsecode
from twisted.web2.resource import PostableResource as Resource
from twisted.web2.http_headers import Cookie as TwCookie, MimeType
from twisted.web2.xmlrpc import XMLRPC

# Imports pour la gestion de SSL via M2Crypto
from M2Crypto import SSL

# Imports EoleSSO
from eolesso.util import *
from eolesso.libsecure import ClientContextFactory, getPageM2
from eolesso.errors import (Redirect, CasError, MissingParameters,
                            InvalidTicket, InternalError)

import config
from page import gen_page, trace, log
from authserver import SSOSessionManager

# OPENID
from oidprovider import OIDRoot
# SAML
import saml_resources, saml_message, saml_utils


# Lecture des templates de la page d'authentification
AUTH_FORM = open(os.path.join('interface', 'authform.tmpl')).read()
# SECURID
if config.USE_SECURID or config.SECURID_PROVIDER:
    from securid_utils import SecuridCheck, SECURID_AUTH_FORM
else:
    SECURID_AUTH_FORM = ""

# initialisation d'un contexte client permettant de vérifier la validité
# des certificat (mode proxy)
client_ctx = ClientContextFactory(certfile = config.CERTFILE, keyfile = config.KEYFILE,
                           mode = SSL.m2.SSL_VERIFY_PEER|SSL.m2.SSL_VERIFY_FAIL_IF_NO_PEER_CERT,
                           ca_location = config.CA_LOCATION)

class XMLRPCManager(XMLRPC):

    def __init__(self,session_manager):
        self.manager = session_manager
        XMLRPC.__init__(self)

    def xmlrpc_reload_configuration(self, post_data):
        log.msg('* {0}'.format(_('reloading server configuration')))
        return self.manager.reload_conf()

    def xmlrpc_authenticate(self, post_data, username, password, search_branch):
        log.msg('* xmlrpc authenticate')
        return self.manager.authenticate(username, password, search_branch)

    def xmlrpc_validate_session(self, post_data, session_id):
        log.msg('* xmlrpc validate_session')
        return self.manager.validate_session(session_id)

    def xmlrpc_verify_session_id(self, post_data, session_id):
        log.msg('* xmlrpc verify_session_id')
        return self.manager.verify_session_id(session_id)

    def xmlrpc_logout(self, post_data, session_id):
        log.msg('* xmlrpc logout')
        return self.manager.logout(session_id)

    def xmlrpc_verify_app_ticket(self, post_data, app_ticket, appurl):
        log.msg('* xmlrpc verify_app_ticket')
        return self.manager.verify_app_ticket(app_ticket, appurl)

    def xmlrpc_get_user_details(self, post_data, app_ticket, appurl, sections=False, renew=False):
        log.msg('* xmlrpc get_user_details')
        return self.manager.get_user_details(app_ticket, appurl, sections, renew)

    def xmlrpc_get_user_info(self, post_data, session_id, details=False):
        log.msg('* xmlrpc get_user_infos')
        return self.manager.get_user_info(session_id, details)

    def xmlrpc_get_auth_class(self, post_data, app_ticket):
        log.msg('* xmlrpc get_auth_class')
        return self.manager.get_auth_class(app_ticket)

    def xmlrpc_get_auth_instant(self, post_data, app_ticket):
        log.msg('* xmlrpc get_auth_instant')
        return self.manager.get_auth_instant(app_ticket)

    def xmlrpc_check_securid_login(self, post_data, securid_login):
        # fonction de détection des utilisateurs enregistrés pour le mode Securid
        return self.manager.securid_user_store.check_securid_login(securid_login)


class CasResource(Resource):
    content_type = 'text/html; charset=utf-8'
    required_parameters = ()
    headers = http_headers

    def set_headers(self, resp, user_headers={}):
        """attache à la réponse HTTP les headers par défaut
        + les éventuels headers passés en paramètre
        """
        if self.content_type:
            resp.headers.setRawHeaders('Content-type', [self.content_type])
        for headers in (self.headers, user_headers):
            for h_name, h_values in headers.items():
                resp.headers.setRawHeaders(h_name, h_values)
        return resp

    def render(self, request):
        headers = {'Content-type': MimeType.fromString(self.content_type)}
        errheaders = {'Content-type': MimeType.fromString('application/xml')}
        try:
            self.check_required_parameters(request)
            return self._render(request)
        except Redirect, exc:
            traceback.print_exc()
            location = exc.location
            headers['location'] =  location
            log.msg(_("Redirecting to "), location)
            # 303 See other
            return http.Response(code=303, headers=headers)
        except CasError, exc:
            traceback.print_exc()
            return http.Response(stream=exc.as_xml(), headers=errheaders)
        except Exception, exc: # unknown error
            traceback.print_exc()
            log.msg(_("An error occured while rendering content for %s : ") % str(request.path) , str(exc))
            error = InternalError(str(exc))
            return http.Response(stream=error.as_xml(), headers=errheaders)

    def check_required_parameters(self, request):
        """raises a `MissingParameters` exception if all required parameters
        are not found in `request.args`.
        """
        missing = [param for param in self.required_parameters
                   if param not in request.args]
        if missing:
            raise MissingParameters(missing)

class CASCompliantResponse(CasResource):
    """implémentation du protocole CAS pour la validation de ticket
    """

    content_type = 'application/xml'

    authorized_codes=[responsecode.OK,
                      responsecode.MULTIPLE_CHOICE,
                      responsecode.MOVED_PERMANENTLY,
                      responsecode.FOUND,
                      responsecode.SEE_OTHER,
                      responsecode.NOT_MODIFIED,
                      responsecode.USE_PROXY,
                      responsecode.TEMPORARY_REDIRECT,
                      ]

    required_parameters = ('service','ticket')

    def __init__(self, manager, serve_proxy = False):
        self.manager = manager
        self.serve_proxy = serve_proxy
        self.serve_saml=False
        super(CASCompliantResponse, self).__init__()

    def _gen_user(self, infos, data_filter):
        """génération de la partie données utilisateur de la réponse CAS
        """
        data = ""
        if data_filter != "":
            if self.serve_saml:
                # génération des attributs au format SAML 1
                for attrs in data_filter.values():
                    for name, data_key in attrs.items():
                        if infos[data_key]:
                            data += """        <Attribute AttributeName="%s" AttributeNamespace="http://www.ja-sig.org/products/cas/">\n""" % name
                            for val in infos[data_key]:
                                if type(val) not in (str, unicode):
                                    val = str(val)
                                data += """          <AttributeValue>%s</AttributeValue>\n""" % escape(val)
                            data += """        </Attribute>\n"""
            else:
                # format xml CAS
                for section, attrs in data_filter.items():
                    if section == "default":
                        # pas de section, attributs au niveau le plus bas
                        data += "%s\n" % (self._gen_section(attrs, infos))
                    else:
                        data += "<cas:%s>\n%s    </cas:%s>\n" % (section, self._gen_section(attrs, infos), section)
                    data = data.rstrip()
        else:
            if self.serve_saml:
                # mode saml :pas d'attributs supplémentaires (user est déjà défini dans l'assertion)
                data = ""
            else:
                data = "    <cas:user>%s</cas:user>" % infos['uid'][0]
        return data

    def _gen_section(self,attrs,infos):
        """génère une section dans la réponse
        """
        section = ""
        for name, datakey in attrs.items():
            for val in infos[datakey]:
                if type(val) not in (str, unicode):
                    val = str(val)
                section += "      <cas:%s>%s</cas:%s>\n" % (name, escape(val), name)
        return section

    def check_proxy_callback(self, cb_url, orig_parameters):
        """vérifie la validité d'une url de callback founie par un service proxy:
        - l'url doit être en https
        - le certificat SSL doit être valide
        - le nom du certificat doit correspondre au service
        """
        proxy_ticket = orig_parameters[3]
        if proxy_ticket in self.manager.app_sessions:
            ticket = self.manager.app_sessions[proxy_ticket]
            log_prefix = "%s -- " % ticket.session_id
        else:
            log_prefix = ""
        # recherche de la session utilisateur associée à la demande
        log.msg("%s%s" % (log_prefix, _('PGT asked for service %s') % orig_parameters[2]))
        url = urlparse.urlparse(cb_url)
        if url.scheme == 'https':
            # connexion à l'url de callback pour vérifier le certificat (port 443 si pas de port spécifié)
            d = getPageM2(cb_url, checker = client_ctx._cert_verify, contextFactory = client_ctx, timeout = 10)
            return d.addCallbacks(self.proxy_cert_ok, self.proxy_errb, callbackArgs = [client_ctx, orig_parameters, url], errbackArgs = [orig_parameters,url])
        # url de proxy non valide
        log.msg(_("!! Invalid proxy url (https is mandatory) !!"))
        return self.validate_st(orig_parameters)

    def proxy_cert_ok(self, stream, ctx, orig_parameters, url):
        """gestion de la demande de PGT si proxy valide
        - envoi d'une requête GET sur l'url de callback du proxy avec les paramètres pgtId et pgtIou
        - vérification du retour : OK si code 200 (ou 3xx si redirection)
        - la validation du ticket d'origine (ST ou PT) continue, avec ajout de pgtIou si vérification OK
        """
        service = orig_parameters[2]
        ticket = orig_parameters[3]
        service_host = url.hostname
        # vérification de la concordance service / certificat
        service_host = config.ALTERNATE_IPS.get(service_host, service_host)
        if not is_dn_in_hostname(ctx.peer_cert_data['subject'], service_host):
            log.msg(_("!! Certificate error : certificate name does not match service !!"))
            # XXX FIXME : remettre en place la validation
        else:
            pgt_iou = None
            # Création d'un pgt temporaire (et pgtiou associé) en attendant la validation ?
            app_ticket = self.manager.get_proxy_granting_ticket(ticket, service, url)
            if app_ticket is not None:
                # préparation de la requête à envoyer sur le callback du proxy
                # ajout de pgtId et pgtIou dans les arguments
                callb_url, callb_args = urllib.splitquery(url.geturl())
                if callb_args:
                    req_args = parse_qsl(callb_args)
                else:
                    req_args = []
                req_args.append(('pgtId', app_ticket.pgt))
                req_args.append(('pgtIou', app_ticket.pgtiou))
                # si besoin, on déclare un proxy http pour y accéder
                self.manager.set_urllib_proxy(app_ticket)
                # envoi de la requête et lecture de la réponse
                u = urlopen("%s?%s" % (callb_url, urllib.urlencode(req_args)))
                data = u.read()
                self.manager.set_urllib_proxy()
                if u.code in self.authorized_codes:
                    log.msg(_("** Proxy response code OK : %s") % u.code)
                    # le proxy doit retourner un code 200 ou 3xx après envoi du pgtId et pg_tiou
                    return self.validate_st(orig_parameters, app_ticket)
                # réponse du proxy invalide, on n'envoie pas de pgt iou à la validation du ticket
                log.msg(_("** Proxy response code ERROR : %s") % u.code)
            else:
                log.msg(_("!! Unable to obtain PGT for service %s !!"))
        return self.validate_st(orig_parameters)

    def proxy_errb(self, err, orig_parameters, url):
        log.msg(_("!! Proxy rejected : %s !!") % url.geturl())
        log.msg("\n!! --> %s !!\n" % err.getErrorMessage())
        return self.validate_st(orig_parameters)

    def _render(self, request):
        error = {'code':'INTERNAL_ERROR','detail':''}
        data = error, None
        ok = False

        from_url = request.args['service'][0]
        ticket = request.args['ticket'][0]
        if not self.serve_proxy and ticket.startswith('PT'):
            raise InvalidTicket(ticket, _("Received PT ticket instead of ST"))
        # Récupération du paramètre renew si présent
        if request.args.get('renew', None) is not None:
            renew = True
        else:
            renew = False
        # si validation par messages SAML 1.1, on récupère la requête
        if request.args.get('Request', None):
            saml_request = request.args['Request'][0]
        # CAS 2 : récupération de l'url de callback pour les proxys (pgtURl : optionnel)
        pgt_url = None
        if config.CAS_VERSION == 2:
            try:
                pgt_url = request.args['pgtUrl'][0]
                return self.check_proxy_callback(pgt_url, [ok, data, from_url, ticket, renew])
            except KeyError:
                # Cas sans pgtUrl, on continue la validation du ticket normalement
                pass
        return self.validate_st([ok, data, from_url, ticket, renew])

    def _traversed_proxies(self, ticket):
        """return the <cas:proxies> node listing traversed proxies

        NOTE: /serviceValidate and /proxyValidate could share the same
        implementation since ticket.proxypath() should return an empty
        list in case of ST apptickets, but just in case, we return
        an hard-coded empty node for ST tickets
        """
        proxies = '\n'.join('    <cas:proxy>%s</cas:proxy>' % url
                            for url in ticket.proxypath())
        return u'\n  <cas:proxies>\n%s\n  </cas:proxies>' % proxies

    def validate_st(self, validate_parameters, pgt=None):
        ok, data, from_url, ticket, renew = validate_parameters
        if ticket is not None:
            if pgt is not None:
                pgt_data = "\n  <cas:proxyGrantingTicket>%s</cas:proxyGrantingTicket>" % pgt.pgtiou
                # on renvoie les proxys traversés si besoin
                if pgt.proxypath() != []:
                    pgt_data += self._traversed_proxies(pgt)
            else:
                # si on n'a pas de paramètre pgt_iou, on considère que d'éventuels tests
                # sur une url de proxy ont échoué
                self.manager.invalidate_proxy(ticket)
                pgt_data = ""
            # si ticket inconnu du serveur, on va tenter sa validation sur tous les serveurs d'auth
            ok, data = self.manager.get_user_details(ticket, from_url, True, renew)
        infos, filter_data = data
        if ok:
            # authentification distante (SSO) reussie
            # génération des informations utilisateur
            user_infos = self._gen_user(infos, filter_data)
            # formatting response
            if self.serve_saml:
                # on récupère la date d'authentification de l'utilisateur
                auth_instant = self.manager.get_auth_instant(ticket)
                # Réponse au format SAML 1.1
                response = saml_message.gen_saml11_response(ticket, auth_instant, from_url, config.IDP_IDENTITY, infos['uid'][0], user_infos)
                response = response.encode(config.encoding)
            elif config.CAS_VERSION == 2:
                # Réponse CAS v2
                resp_body = user_infos + pgt_data
                response = cas_response('authenticationSuccess', resp_body)
            else:
                # Réponse CAS v1
                response = 'yes\n%s\r\n\r\n' % infos['uid'][0]
        else:
            error, filter_data  = data
            # aucun des serveurs n'a authentifié l'utilisateur (meme l'authentification locale), on revient
            # sur la page de saisie de username/password
            if self.serve_saml and error['code'] != 'INTERNAL ERROR':
                # issue_instant, recipient, response_id
                response = saml_message.gen_saml11_error(error, from_url)
                response = response.encode(config.encoding)
            elif config.CAS_VERSION == 2 or self.serve_saml:
                response = cas_response('authenticationFailure', escape(error['detail']), code=error['code'])
            else:
                response = 'no\r\n\r\n'
        if config.CAS_VERSION == 1 and not self.serve_saml:
            self.content_type = 'text/html; charset=utf-8'
        cas_resp = http.Response(stream=response)
        if self.serve_saml:
            # SOAP additionnal headers for SAML 1.1
            self.set_headers(cas_resp, {'SOAPAction':['http://www.oasis-open.org/committees/security']})
        else:
            self.set_headers(cas_resp)
        return cas_resp

class SAMLCompliantResponse(CASCompliantResponse):

    content_type = 'application/xml'
    required_parameters = ('TARGET',)

    def __init__(self, manager, serve_proxy = False):
        super(SAMLCompliantResponse, self).__init__(manager, serve_proxy)
        self.serve_saml = True
        # on utilise render pour renderHTTP
        self.renderHTTP = super(SAMLCompliantResponse, self).render

    def _render(self, request):
        target = request.args['TARGET'][0]
        # Récupération de l'artefact dans la requête SAML (correspond au ticket à valider).
        return request.stream.read().addCallbacks(self.process_saml11_request, log.msg, callbackArgs = [target])

    def process_saml11_request(self, xml_stream, target):
        """procceses a saml1 request and extracts Artifact value
        """
        error = {'code':'INTERNAL_ERROR','detail':''}
        # extraction de la requête SAML1
        try:
            artifact = saml_utils.get_saml11_artifact(xml_stream)
        except SOAPpy.Error:
            # erreur de parsing de l'entête SOAP, on retourne une erreur
            error = {'code':'INTERNAL ERROR', 'detail':_('Invalid SOAP Headers')}
            artifact = None
        except:
            error = {'code':'INTERNAL ERROR', 'detail':_('No ticket in SAML Request')}
            artifact = None
        data = error, None
        return self.validate_st([False, data, target, artifact, False])

class ProxyResource(CasResource):
    """
    page de génération d'un ticket de type PT
    """
    addSlash = False
    content_type = 'application/xml'

    def __init__(self, manager):
        self.manager = manager
        super(ProxyResource, self).__init__()

    def _render(self, request):
        try:
            pgt = request.args['pgt'][0]
            target_service = request.args['targetService'][0]
        except:
            error = {'code':'INVALID_REQUEST','detail':_("Missing parameter")}
        # vérification du pgt
        if pgt in self.manager.proxy_granting_sessions:
            # génération du ticket PT
            ticket = self.manager.get_proxy_ticket(pgt, target_service)
            response_body = "<cas:proxyTicket>%s</cas:proxyTicket>" % ticket
            response = cas_response('proxySuccess', response_body)
            return self.set_headers(http.Response(stream=response))
        else:
            error = {'code':'BAD_PGT', 'detail':_("PGT not recognized by server : %s") % pgt}
        response = cas_response('proxyFailure', escape(error['detail']), code=error['code'])
        return self.set_headers(http.Response(stream=response))


class LoggedInForm(CasResource):
    """
    page de confirmation d'ouverture de session (si pas d'url de redirection fournie)
    """
    addSlash = False

    def __init__(self, manager):
        self.manager = manager
        super(LoggedInForm, self).__init__()

    def _render(self,request):
        try:
            css = escape(request.args['css'][0])
        except KeyError:
            css = config.DEFAULT_CSS
        sso_cookie = getCookie(request, 'EoleSSOServer')
        if sso_cookie:
            user_session = sso_cookie.value
            sessions = ""
            local_sessions = []
            for app_t in self.manager.user_app_tickets[user_session]:
                if hasattr(app_t,'saml_ident'):
                    sessions += """<li>session distante sur %s</li>""" % app_t.saml_ident
                else:
                    if app_t.service_url not in local_sessions:
                        local_sessions.append(urllib.quote(app_t.service_url, safe=uri_reserved_chars))
            if local_sessions:
                sessions += "<li>%s</li>" % "</li><li>".join(local_sessions)
            if sessions:
                content = """<h1>Liste de vos sessions ouvertes :</h1><br/>
                <ul>%s</ul>""" % sessions
            else:
                content = """<h1>Aucune session d'application trouvée</h1>"""
            content += """<form action="/logout" method="post">
            <p class=formvalidation><input class="btn" type="Submit" value="%s">
            </p></form>""" % _("Disconnect")
        else:
            content = _('Invalid session')
        response = http.Response(stream=gen_page(_("Valid SSO session"), content, css))
        return self.set_headers(response)

class ErrorPage(CasResource):
    """
    page d'erreur http
    """
    addSlash = False

    def __init__(self, code, description):
        self.code = code
        self.description = description
        title = (_("EoleSSO : Error"))
        super(ErrorPage, self).__init__()

    def _render(self,request):
        return http.StatusResponse(code=self.code, description=self.description)

class UnauthorizedService(ErrorPage):
    """
    Page affichée en cas de refus d'envoi d'une réponse CAS à un service
    renvoie une erreur http UNAUTHORIZED par défaut.
    Il est possible d'afficher une page personnalisée à la place en créant un template
    /usr/share/sso/interface/invalid_service.tmpl (fichier d'exemple dans le répertoire)
    """
    def __init__(self):
        tmpl_file = os.path.join(config.SSO_PATH, 'interface', 'unauthorized_service.tmpl')
        if os.path.isfile(tmpl_file):
            self.title = _("EoleSSO : Error")
            self.tmpl_data = file(tmpl_file).read()
        else:
            self.tmpl_data = None
            description = _('unauthorized service url : %s')
            ErrorPage.__init__(self, responsecode.UNAUTHORIZED, description)

    def _render(self,request):
        from_url = escape(request.args['service'][0])
        if self.tmpl_data:
            css = config.DEFAULT_CSS
            try:
                css = escape(request.args['css'][0])
            except KeyError:
                pass
            response = http.Response(stream=gen_page(self.title, self.tmpl_data.format(from_url), css))
            return self.set_headers(response)
        else:
            return http.StatusResponse(code=self.code, description=self.description % from_url)

class LogoutForm(CasResource):
    """
    page de confirmation de logout (si pas d'url de redirection fournie)
    """
    addSlash = False

    def __init__(self, manager):
        self.manager = manager
        super(LogoutForm, self).__init__()

    def _render(self,request):
        # invalidation de la session sur le serveur d'authentification
        cookies = getCookies(request)
        for cookie in cookies:
            if cookie.name == 'EoleSSOServer':
                session_id = cookie.value
                if session_id is not None:
                    # suppression de la session sur le manager
                    self.manager.logout(session_id.encode(config.encoding))
                    cookie.expires = 1
                break
        try:
            css = escape(request.args['css'][0])
        except KeyError:
            css = config.DEFAULT_CSS
        if request.args.has_key('service'):
            url_from = request.args['service'][0]
            response = RedirectResponse(url_from)
            log.msg(_("Logged out : redirecting to %s") % url_from)
        else:
            #si pas de parametre from, on affiche la page de logout en local
            content = """<form action="/login" method="post">
            <p class=formvalidation>
            <input class="btn" type="Submit" value="%s"></p></form>""" % _("Connect")
            response = http.Response(stream=gen_page(_('SSO session closed'), content, css))
            self.set_headers(response)
        # mise en place des cookies
        if cookies:
            response.headers.setHeader('Set-Cookie', cookies)
        return response

class UserChecker(Resource):
    """ressource utilisée pour détecter certains aspects de l'utilisateur
    - vérifie si l'utilisateur est sujet à des problèmes de doublon (annuaire multi établissement)
    - Si l'accès OTP est configuré, vérifie si l'utilisateur a enregistré son identifiant OTP
    """

    isLeaf = True
    content_type = 'application/json; charset=utf-8'

    def __init__(self, manager):

        self.manager = manager
        super(UserChecker, self).__init__()

    def render(self, request):
        """méthode de rendu http (renvoie si un utilisateur est déjà enregistré)
        Utilisé par le fomulaire depuis une requête AJAX
        le nom d'utilisateur à vérifier doit être passé dans la requête (argument username)
        """
        username = ''
        current_branch = request.args.get('user_branch', ['default'])[0]
        if 'username' in request.args:
            username = request.args.get('username', [''])[0]
            # vérification des doublons
            if self.manager._data_proxy.use_branches:
                if 'check_branches' in request.args and is_true(request.args.get('check_branches', [''])[0]):
                    defer_branches = self.manager._data_proxy.get_user_branches(username)
                    return defer_branches.addCallbacks(self.callb_render, log.msg, callbackArgs=[username, current_branch])
        return self.callb_render([], username, current_branch)

    @trace
    def callb_render(self, search_branches, username, current_branch):
        #search_branches = []
        #for branche in branches:
        #    br_dn = str(branche)
        #    for search_br, libelle in self.manager._data_proxy.search_branches.items():
        #        if search_br in br_dn:
        #            search_branches.append((br_dn, libelle))
        #            break
        # tri de la liste sur le libellé établissement
        search_branches.sort(key  = lambda branche: branche[1])
        user_infos = {'search_branches':search_branches}
        if username and self.manager.securid_user_store:
            # vérification de l'enregistrement de login OTP
            otp_login = self.manager._data_proxy.check_otp_config(current_branch)
            if otp_login == "identiques":
                # les logins sont identiques au login LDAP pour cet annuaire
                user_infos['securid_registered'] = 'true'
            elif otp_login == "configurables":
                user_infos['securid_registered'] = self.manager.securid_user_store.check_registered(username, current_branch)
        response = http.Response(code=responsecode.OK, stream=json_dump(user_infos))
        response.headers.setRawHeaders('Content-type', [self.content_type])
        return response

class LocalCookie(CasResource):
    """
    page de génération d'un cookie d'authentification local
    """
    addSlash = False
    required_parameters = ('return_url','ticket')

    def __init__(self, manager):
        self.manager = manager
        super(LocalCookie, self).__init__()

    def _render(self, request):
        try:
            ticket = request.args['ticket'][0]
            return_url = request.args['return_url'][0]
        except:
            error = {'code':'INVALID_REQUEST','detail':_("Missing parameter")}
            log.msg(error)
        log.msg(_("Local cookie requested"))
        # on vérifie que la session est bien existante et valide
        response = RedirectResponse(return_url)
        if self.manager.verify_session_id(ticket):
            ticket = ticket.encode(config.encoding)
            # XXX FIXME : gérer plusieurs niveaux de parents (redirection sur parent si ticket non local)
            if self.manager.user_sessions.get(ticket,(None,))[0] == None:
                print " not implemented : redirect one level higher !"
            cookies = []
            for cookie in getCookies(request):
                if cookie.name != 'EoleSSOServer':
                    cookies.append(cookie)
            cookies.append(TwCookie("EoleSSOServer", ticket, path = "/", discard=True, secure=True))
            if cookies:
                log.msg(_("Local cookie sent, redirecting to child server"))
                response.headers.setHeader('Set-Cookie', cookies)
        return response

class InfoEtabs(CasResource):
    """
    page de récupération des données des établissements (eole-dispatcher)
    """
    addSlash = False
    content_type = 'application/json'

    def _render(self,request):
        info_etabs = get_etabs(os.path.join(config.SSO_PATH,'interface','scripts','etabs.js'), sso_dir=config.SSO_PATH)
        response = http.Response(code=responsecode.OK, stream=json_dump(info_etabs._sections))
        return self.set_headers(response)

class LoginForm(CasResource):
    """The resource that is returned when you are not logged in"""
    addSlash = False

    def __init__(self, manager, auth_server, oidserver, *args):
        self.manager = manager
        self.auth_server = auth_server
        self.oidserver = oidserver
        # pré-calcul du select des établissements (si annuaire multi-établissements)
        self.select_etab = self.calc_select_etab()
        self.content = ''
        # utilisation de la favicon fournie dans le thème ou celle par défaut
        if os.path.isfile('interface/theme/image/favicon.ico'):
            self.favicon = 'interface/theme/image/favicon.ico'
        elif os.path.isfile('interface/theme/image/icon.ico'):
            self.favicon = 'interface/theme/image/icon.ico'
        else:
            self.favicon = 'interface/images/icon.ico'
        Resource.__init__(self)

    def calc_select_etab(self):
        select_etab = ""
        if self.manager._data_proxy.use_branches:
            select_etab = """<tr id="row_etab" style="display:none;">
            <td><label for='select_etab' accesskey='e'>%s</label></td>
            <td><select name="select_etab" id="select_etab" onchange="check_user_options('false')">
            <option value=""></option>""" % _('User Origin')
            #for search_base, libelle in self.manager._data_proxy.search_branches.items():
            #    select_etab += "<option value=%s>%s</option>" % (search_base, libelle)
            select_etab += "</select></td></tr>"
        return select_etab

    def get_css_from_service(self, from_url):
        css = config.DEFAULT_CSS
        # recherche d'une eventuelle css associée
        service_filter = self.manager._check_filter(from_url)[0]
        if service_filter not in ['default', '']:
            if os.path.exists(os.path.join('interface','%s.css' % service_filter)):
                css = service_filter
        return css

    def render_content(self, request):
        """
        protocole CAS:
         1. service : l'URL d'où on vient
         2. renew=true : on force à se réauthentifier
         3. gateway=true : si pas authentifié, on revient à l'URL
            définie par service sans ticket d'application
        ajout Eole
         4. redirect_to : permet de définir une page ou revenir après authentification
            (utile pour l'authentification depuis des protocoles non CAS)
        """
        self.content = ""
        gateway = is_true(request.args.get('gateway', ['false'])[0])
        renew = is_true(request.args.get('renew', ['false'])[0])
        warn = is_true(request.args.get('warn', ['false'])[0])
        css = config.DEFAULT_CSS
        try:
            from_url = request.args['service'][0]
            css = self.get_css_from_service(from_url)
        except KeyError, e:
            # nécessaire car les ressources ne sont instanciées qu'une
            # seule fois
            from_url = ''
        try:
            css = escape(request.args['css'][0])
        except KeyError:
            pass
        try:
            redirect_to = request.args['redirect_to'][0]
        except KeyError:
            redirect_to = ''
        # si demandé, on vérifie que l'application de destination est reconnue (filtre défini)
        if from_url and config.VERIFY_APP:
            if self.manager.get_app_infos(from_url) is None:
                return '/unauthorizedService?service=%s' % from_url
        #on va verifier si on a un identifiant de session dans le cookie
        user_session = getCookie(request, 'EoleSSOServer')
        # XXX: IE et Opera ne gerent pas les cookies de la meme
        # maniere que Firefox sous firefox, quand un cookie devient
        # invalide (au niveau du temps), il est supprimé sous IE et
        # Opera, ils sont gardés, on doit donc tester si ils ont une
        # valeur spéciale, ici foo
        # si on en a bien un, on le verifie auprès du serveur d'authentification
        if not renew and user_session and user_session != 'foo':
            session_id = user_session.value.encode(config.encoding)
            verif =  self.manager.verify_session_id(session_id)
            # si l'ID de session est verifie, on génère le ticket applicatif,
            # et on redirige vers l'URL de base
            if verif[0]:
                if from_url:
                    app_ticket = self.manager.get_app_ticket(session_id, from_url)
                    url = urljoin(from_url, 'ticket=%s' % app_ticket)
                    if warn:
                        self.content = str("""<html>%s<body onload="alert('%s'); document.location.href='%s'"></body></html>""" % (head % css, _("You are being authenticated by Single Sign On service"), urllib.quote(url, safe=uri_reserved_chars)))
                        return None
                    return url
                return '/loggedin'

        if gateway and from_url:
            return urljoin(from_url, 'ticket=')
        self.auth_msg = _("Authentication needed")
        if request.args.has_key('failed'):
            if request.args['failed'][0] == '1':
                self.auth_msg = _("Authentication failed, try again")
        # création d'un login ticket jouable une seule fois pour cet essai d'authentification
        login_ticket = self.manager.get_login_ticket()
        if from_url != "":
            service_input = """<input type="hidden" name="service" value="%s"/>""" % urllib.quote(from_url, safe=uri_reserved_chars)
        else:
            service_input = ""
        if redirect_to != "":
            redirect_input =  """<input type="hidden" name="redirect_to" value="%s"/>""" % urllib.quote(redirect_to, safe=uri_reserved_chars)
        else:
            redirect_input = ""

        #############################################
        # Calcul du formulaire d'authentification
        #############################################
        additional_form = ""
        passwd_check_func = ""
        if config.DEBUG_LOG:
            autocomplete = "on"
        else:
            autocomplete = "off"
        if config.USE_SECURID: #  or config.SECURID_PROVIDER:
            additional_form += SECURID_AUTH_FORM % (_("label_unregistered"),
                                                    _("Securid user"), autocomplete, _("Securid user"),
                                                    _("OTP PIN number"), _("Current token"), autocomplete,
                                                    _("Current token"),
                                                   )
            passwd_check_func = 'onkeyup="checkotp()"'
        # avertissement legal (en dessous du cadre)
        theme_txt = os.path.join(config.SSO_PATH, 'interface', 'theme', 'avertissement.txt')
        interf_txt = os.path.join(config.SSO_PATH, 'interface', 'avertissement.txt')
        if os.path.exists(theme_txt):
            avertissement = open(theme_txt).read()
        elif os.path.exists(interf_txt):
            avertissement = open(interf_txt).read()
        else:
            avertissement = ""
        # message en fonction du navigateur
        theme_ie = os.path.join(config.SSO_PATH, 'interface', 'theme', 'alert_ie.tmpl')
        interf_ie = os.path.join(config.SSO_PATH, 'interface', 'alert_ie.tmpl')
        if os.path.exists(theme_ie):
            avertissement_ie = open(theme_ie).read()
        elif config.ACTIVER_ENVOLE_INFOS and os.path.exists('/var/www/html/envole-infos/informations_ie.php'):
            avertissement_ie = open(interf_ie).read() % config.POSH_URL
        else:
            avertissement_ie = ""
        # aide utilisateur pour le choix du login
        theme_login = os.path.join(config.SSO_PATH, 'interface', 'theme', 'login_help.tmpl')
        interf_login = os.path.join(config.SSO_PATH, 'interface', 'login_help.tmpl')
        if os.path.exists(theme_login):
            login_help = open(theme_login).read()
        elif os.path.exists(interf_login):
            login_help = open(interf_login).read()
        else:
            login_help = ""
        if config.DEBUG_LOG:
            autocomplete = 'on'
        else:
            autocomplete = 'off'
        # création de la page
        content = AUTH_FORM % (login_ticket,
                service_input,
                redirect_input,
                _("Login"), _("Login"), autocomplete,
                _("Password"), _("Password"), autocomplete,
                passwd_check_func,
                self.select_etab,
                additional_form,
                login_help,
                _("Submit"),
                avertissement_ie, avertissement)
        # scripts dynamiques
        javascript = """document.forms['cas_auth_form'].username.focus();setTimeout(callb_onload, 200);"""
        header_script = """<script type="text/javascript" src="scripts/mootools-core-1.4.2.js"></script>
<script type="text/javascript" src="scripts/tools.js?v=2.0"></script>
<script type="text/javascript" src="scripts/etabs.js"></script>
<script type="text/javascript" src="scripts/homonymes.js"></script>
<script type="text/javascript">
"""
        otp_enabled = 'false'
        if config.USE_SECURID:
            # chargement des scripts pour gestion de la saisie de mots de passe OTP
            otp_enabled = 'true'
            header_script += open('interface/scripts/authform_otp.js').read() %\
            (config.OTPPASS_MINSIZE,
             config.OTPPASS_MAXSIZE,
             config.OTPPASS_REGX)
        header_script += open('interface/scripts/authform_ajax.js').read() %\
        (otp_enabled,
         _('label_password_OTP'),
         _('label_registered'),
         _('label_password'),
         _('label_unregistered'),
         _('label_password'),
         _('select your origin'),
         _('academic'),
         _('several users correspond to '),
         _('please select your authentication source'),
)
        header_script += open('interface/scripts/authform_ie.js').read().strip()
        header_script += "\n</script>"
        self.content = gen_page(self.auth_msg, content, css, javascript, header_script)
        return None

    def render_verify(self, request):
        try:
            username = unicode(request.args['username'][0],config.encoding)
        except:
            return ErrorPage(_("Authentication"),_("Access denied"))
        try:
            search_branch = unicode(request.args['select_etab'][0],config.encoding).encode(config.encoding)
        except:
            search_branch = "default"
        try:
            password = unicode(request.args['password'][0],config.encoding)
        except:
            password = ""
        try:
            from_url = request.args['service'][0]
        except:
            from_url = ""
        try:
            redirect_to = request.args['redirect_to'][0]
        except KeyError:
            redirect_to = ''
        try:
            securid_pwd = request.args['securid_pwd'][0]
        except KeyError:
            securid_pwd = ''
        securid_register = False
        if request.args.has_key('securid_register'):
            # l'utilisateur a validé l'utilisation de securid
            securid_register = True
        already_registered = False
        if is_true(request.args.get('user_registered', [''])[0]):
            # l'utilisateur a déjà enregistré son identifiant securid
            already_registered = True
        try:
            securid_login = request.args['securid_user'][0]
        except KeyError:
            securid_login = ''
        # vérification du login ticket
        try:
            ticket = request.args['lt'][0]
            assert self.manager.login_sessions.validate_session(ticket)
        except:
            # on n'est pas passé par /login : on renvoie dessus
            if from_url:
                response = RedirectResponse('/login?service=%s' % from_url)
            else:
                response = RedirectResponse('/login')
            return response
        if config.SECURID_PROVIDER and securid_register:
            # on lance une authentification SecurID distante si demandé
            try:
                sp_meta = self.manager.get_metadata(config.SECURID_PROVIDER)
            except:
                log.msg(_('no metadata available for %s') % config.SECURID_PROVIDER)
                sp_meta = None
            if securid_pwd and sp_meta:
                # on stocke les informations utiles pour la création de session et le retour
                # vers la ressource demandée si nécessaire (après confirmation par le fournisseur
                # d'identité gérant l'authentification OTP
                session_id = self.manager.securid_sessions.add_session((None, None, None, from_url, redirect_to, username, password, True, cookies))
                # recherche du login RSA associé à l'utilisateur local
                return self.send_securid_request(session_id, username, passwd, sp_meta, request)

        # création d'un login ticket avec toutes les informations nécessaires pour continuer la procédure
        cookies = getCookies(request)
        # on lance une authentification SecurID locale si demandé
        if config.USE_SECURID and securid_register:
            otp_login = self.manager._data_proxy.check_otp_config(search_branch)
            if otp_login == 'identiques':
                registered_user = username
            else:
                registered_user = self.manager.securid_user_store.get_securid_user(username, search_branch)
            if already_registered and registered_user:
                securid_login = registered_user
                securid_pwd = password
            css = config.DEFAULT_CSS
            # on regarde si une css est disponible pour ce service
            if from_url:
                css = self.get_css_from_service(from_url)
            login_ticket = self.manager.get_login_ticket((username, search_branch, password, from_url, redirect_to, css, securid_login, securid_pwd, already_registered, cookies))
            # redirection sur la ressource /securid pour validation
            # on passe les paramètres from_url et redirect_to pour continuer
            # la procédure depuis la ressource securid
            req_args = (('lt', login_ticket),)
            response = RedirectResponse('/securid?%s' % urllib.urlencode(req_args))
            return response
        # on va tenter l'authentification sur tous les serveurs d'auth
        defer_auth = self.manager.authenticate(username, password, search_branch)
        return defer_auth.addCallback(self.callb_auth, from_url, redirect_to, cookies)

    def callb_auth(self, res_auth, from_url, redirect_to, request_cookies):
        """finalise la mise en place de la session après vérification des identifiants et renvoie l'url sur laquelle rediriger (+ cookies à mettre en place)
        res_auth : id de la session créée et données utilisateur (ou None)
        """
        session_id, user_data = res_auth
        if session_id != "":
            # on stocke un cookie dans le navigateur
            # chaine = '%s,%s'%(session_id), username.encode(config.encoding))
            session_id = session_id.encode(config.encoding)
            cookies = []
            for cookie in request_cookies:
                if cookie.name != 'EoleSSOServer':
                    cookies.append(cookie)
            cookies.append(TwCookie("EoleSSOServer", session_id, path = "/", discard=True, secure=True))
            if from_url == '':
                # pas de service spécifié, on indique que la session est ouverte, ou on redirige si demandé
                if redirect_to != '':
                    target_url = redirect_to
                else:
                    target_url = '/loggedin'
            else:
                # on redirige sur la page de l'application de destination
                app_ticket = self.manager.get_app_ticket(session_id, from_url, from_credentials=True)
                target_url = urljoin(from_url, 'ticket=%s' % app_ticket)
                log.msg(_("Redirecting to calling app with ticket : %s") % get_service_from_url(from_url))
            if self.manager.user_sessions.get(session_id,(None,))[0] == None:
                # Si session délivrée par le parent, rediriger sur lui pour poser un cookie d'auth qu'il puisse lire
                req_args = (('ticket', session_id), ('return_url', target_url))
                response_url = '%s/gen_cookie?%s' % (self.manager.parent_url, urllib.urlencode(req_args))
                log.msg(_("Redirecting to parent server for session setting"))
            else:
                response_url = target_url
            response = RedirectResponse(response_url)
            if cookies:
                response.headers.setHeader('Set-Cookie', cookies)
            return response

        # aucun des serveurs n'a authentifié l'utilisateur (meme l'authentification locale), on revient
        # sur la page de saisie de username/password
        if from_url:
            return RedirectResponse('/?service=%s&failed=1' %(from_url))
        return RedirectResponse('/?failed=1')

    def send_securid_request(self, request_id, username, password, sp_meta, req_args):
        print "SEND_SECURID_REQUEST CALLED : ", request_id, username, password, sp_meta, req_args
        # demande d'authentification auprès du serveur
        # securid_login = self.manager.securid_user_store.get_securid_user(username, search_branch)
        idp_ident = config.SECURID_PROVIDER
        try:
            binding, service_url, response_url = saml_utils.get_endpoint(sp_meta, 'SingleSignOnService', ent_type='IDPSSODescriptor')
        except InternalError:
            # XXX FIXME : CSS / retourner err_msg en fin de procédure ?
            traceback.print_exc()
            err_msg = _('no usable endpoint for %s') % idp_ident
            return http.Response(stream = gen_page(_('Unreachable resource'), err_msg))
        # Génération d'une requête d'authentification
        # Le login ticket généré précédemment servira d'identifiant de requête
        # Il permettra de retrouver les informations d'origine après réponse
        # du fournisseur d'identité (in_response_to dans la réponse)
        # Envoi de la demande de logout à l'entité correspondante
        sign_method, req = saml_message.gen_request(request_id, username, password, config.IDP_IDENTITY, service_url, config.CERTFILE)
        # réponse utilisant le binding POST
        req = saml_utils.encode_request(req)
        form = """<FORM action="%s" method="POST">
        <input type="hidden" name="SAMLRequest" value="%s">
        <input type="hidden" name="SigAlg" value="%s">
        <input type="hidden" name="RelayState" value="%s">
        <input type="Submit" value="%s">
        """ % (service_url, req, sign_method, 'ADRESSE_DE_RETOUR', _("Proceed to service"))
        # XXX stocker arguments nécessaires à la suite de création de session avant redirection (dico {req_id:data})
        return http.Response(stream = gen_page(_('Contacting authentication server'), form))

    def _render(self, request):
        # retour après authentification
        if ('username' in request.args) and (('password' in request.args) or ('securid_pwd' in request.args)):
            return self.render_verify(request)
        else:
            redirect_to = self.render_content(request)
            if redirect_to == None:
                return self.set_headers(http.Response(stream=self.content))
            else:
                if redirect_to != '/loggedin' and config.DEBUG_LOG:
                    log.msg("Authform, %s : %s" % (_("Redirecting to "), get_service_from_url(redirect_to)))
                return RedirectResponse(redirect_to)

    def locateChild(self, request, segments):
        # LOGOUT GENERAL
        if segments[0] == 'logout':
            return saml_resources.SamlLogout(self.manager), ()
        # ERADICATION DES SESSIONS FANTOMES
        user_session = getCookie(request, 'EoleSSOServer')
        if user_session and user_session.value in self.manager.pending_logout:
            log.msg(_("logout interrupted for session %s : terminating") % user_session.value.encode(config.encoding))
            # REDIRECTION SUR LA PAGE DE LOGOUT
            self.manager.logout(user_session.value.encode(config.encoding))
        if segments:
            # URLS CAS
            if segments[0] in ('login','','verify'):
                return self, ()
            elif segments[0] in ['validate','serviceValidate']:
                return CASCompliantResponse(self.manager), ()
            elif segments[0] == 'proxyValidate':
                return CASCompliantResponse(self.manager, True), ()
            elif segments[0] == 'samlValidate':
                return SAMLCompliantResponse(self.manager, True), ()
            elif segments[0] == 'proxy':
                return ProxyResource(self.manager), ()
            elif segments[0] == 'loggedin':
                return LoggedInForm(self.manager), ()
            elif segments[0] == 'gen_cookie':
                return LocalCookie(self.manager), ()
            elif segments[0] == 'unauthorizedService':
                return UnauthorizedService(), ()
            # URLS XMLRPC
            elif segments[0] == 'xmlrpc':
                return self.auth_server, ()
            # URLS OPENID
            elif segments[0] == 'oid':
                return self.oidserver, segments[1:]
            # URLS SAML
            elif segments[0] == 'saml':
                return saml_resources.SamlResponse(self.manager), segments[1:]
            # URL d'IDPDiscovery
            elif segments[0] == 'discovery':
                return saml_resources.IDPDiscoveryService(self.manager), ()
            # URLS POUR VERIFICATION SECURID
            elif config.USE_SECURID and segments[0] == 'securid':
                return SecuridCheck(self.manager), ()
            elif segments[0] == 'check_user_options':
                return UserChecker(self.manager), ()
            # URL DE RECUPERATION DES ETABLISSEMENTS REPLIQUES (DISPATCHER)
            elif segments[0] == 'etabs':
                return InfoEtabs(), ()
            # URLS UTILITAIRES
            elif segments[0] == 'favicon.ico':
                return static.File(self.favicon), ()
            elif segments[0] == 'css':
                return static.File('interface'), segments[1:]
            elif segments[0] == 'scripts':
                return static.File('interface/scripts'), segments[1:]
            elif segments[0] == 'images':
                return static.File('interface/images'), segments[1:]
        return ErrorPage(responsecode.NOT_FOUND, "%s : %s" % (_('Unreachable resource'),  '/'.join(segments))), ()

