# -*- coding: utf-8 -*-
"""Ce module fournit une implémentation du protcole d'exploration
et d'identification openID (partie cliente)

"""
import cgi
import sys
import Cookie
from urllib import quote

from twisted.internet import reactor
from twisted.web2 import server, static, http, channel, responsecode
from twisted.web2.resource import Resource
from twisted.web2.http_headers import Cookie as TwCookie

from openid.consumer import consumer
from openid.store.memstore import MemoryStore
from openid.cryptutil import randomString
from openid import version_info

from eolesso.oidlib import simplifyQuery, HTMLResponse, getHostName
from eolesso import oideole
from eolesso.util import RedirectResponse

# test de la version de python openid
OID_VERSION = int(''.join([str(v) for v in version_info]))

SESSION_COOKIE_NAME = 'ead_openid'

class PersistencyManager(object):
    """classe responsable de gérer les états des connections et
    authentifications
    """
    def __init__(self, store, baseurl):
        self.store = store
        self.sessions = {}
        self.baseurl = baseurl



OPENID_PROVIDER_NAME = 'Eole'
OPENID_PROVIDER_URL = 'https://ead.eole.com/signup'


class IPResource(Resource):
    """Identity Provider MixIn : définit tous les comportements
    communs aux urls gérées par le server
    """

    title = "Exemple de page acceptant l'authentification OpenId"
    content = """<p>Ceci est un exemple de site web acceptant
l'authentification OpenId.</p>
    <p>%(msg)s</p>
    <div id="verify-form">
      <form method="get" accept-charset="UTF-8" action="/verify">
        Identifiant openid :
        <input type="text" size="60" name="openid_url" value="" />
        <input type="submit" value="Verify" />
        <img src="css/images/login-bg.gif"/><br />
<!--
  <input type="checkbox" name="immediate" id="immediate" /><label for="immediate">Use immediate mode</label>
  <input type="checkbox" name="use_eole_ext" id="use_eole_ext" /><label for="use_eole_ext">Request registration data</label>
-->
      </form>
    </div>
    """

    def __init__(self, persistencyManager):
        self.manager = persistencyManager
        super(IPResource, self).__init__()
        self._session = None

    def buildURL(self, path, **kwargs):
        return self.manager.baseurl + path + '?' + '&'.join('%s=%s' % (k, quote(v))
                                                            for k,v in kwargs.items())
    def extract_response(self, ctx):
        oidconsumer = self.getConsumer()

        # Ask the library to check the response that the server sent
        # us.  Status is a code indicating the response type. info is
        # either None or a string containing more information about
        # the return type.
        if OID_VERSION >= 202:
            url = 'http://%s:%s' % (ctx.host, ctx.port) + ctx.path
            info = oidconsumer.complete(self.query, url)
        else:
            info = oidconsumer.complete(self.query)
        eole_resp = None
        if info.status == consumer.SUCCESS:
            # This is a successful verification attempt. If this
            # was a real application, we would do our login,
            # comment posting, etc. here.
            eole_resp = oideole.EoleExtensionResponse.fromSuccessResponse(info)
        return info, eole_resp


    def renderEoleData(self, eole_resp, show_not_logged=True):
        if not eole_resp and show_not_logged:
            return '<div class="alert">No registration data was returned</div>'
        elif not eole_resp:
            return ' '
        html = []
        eole_list = sorted(eole_resp.items())
        html = ['<h2>Registration Data</h2><table class="sreg">'
                '<thead><tr><th>Field</th><th>Value</th></tr></thead>'
                '<tbody>']

        odd = ' class="odd"'
        for k, v in eole_list:
            field_name = eole_resp.data_fields.get(k, k)
            value = cgi.escape(v)
            html.append('<tr%s><td>%s</td><td>%s</td></tr>' % (odd, field_name, value))
            if odd:
                odd = ''
            else:
                odd = ' class="odd"'

        html.append('</tbody></table>')
        return '\n'.join(html)


    def render(self, request):
        self.query = simplifyQuery(request.args)
        self.cookies = request.headers.getRawHeaders('Cookie')
        return self._render(request)

    def _render(self):
        raise NotImplementedError()


    @property
    def session(self):
        """Return the existing session or a new session"""
        if self._session is not None:
            return self._session
        # Get value of cookie header that was sent
        if self.cookies:
            #FIXME use twisted cookies
            cookie_obj = Cookie.SimpleCookie(self.cookies[0])
            sid_morsel = cookie_obj.get(SESSION_COOKIE_NAME, None)
            if sid_morsel is not None:
                sid = sid_morsel.value
            else:
                sid = None
        else:
            sid = None

        # If a session id was not set, create a new one
        if sid is None:
            sid = randomString(16, '0123456789abcdef')
            session = None
        else:
            session = self.manager.sessions.get(sid)

        # If no session exists for this session ID, create one
        if session is None:
            session = self.manager.sessions[sid] = {}

        session['id'] = sid
        self._session = session
        return session


    def cookieHeaders(self):
        """est appelé avant chaque réponse et permet de s'assurer que le
        cookie de session est toujours correctement positionné.
        """
        sid = self.session['id']
        return {'Set-Cookie' : [TwCookie(SESSION_COOKIE_NAME, sid)]}


    def requestRegistrationData(self, oidreq):
        eole_request = oideole.EoleExtensionRequest(required=['secureid', 'uid', 'user_groups'],
                                                    optional=['mail', 'givenName', 'sn','codecivilite','dateNaissance'])
        oidreq.addExtension(eole_request)

    def getConsumer(self):
        """raccourci pour créer un ``consumer`` openid à partir du couple
        (session, openid_store)
        """
        return consumer.Consumer(self.session, self.manager.store)

    def buildErrorResponse(self, error):
        return RedirectResponse('/?errmsg=%s' % quote(error))
    # headers=self.cookieHeaders())



class VerifyResource(IPResource):
    """Cette ressource est appellée lorsqu'on a rentré une URL openid pour
    s'authentifier.
    C'est elle qui va lancer la procédure de ``discover`` et générer le formulaire
    d'autosubmit à envoyer au fournisseur d'identité.
    """
    def _render(self, ctx):
        # First, make sure that the user entered something
        openid_url = self.query.get('openid_url')
        if not openid_url:
            return self.buildErrorResponse('Renseignez un identifiant OpenID à verifier.')

        immediate = False # 'immediate' in self.query
        use_eole_ext = True

        oidconsumer = self.getConsumer()
        try:
            oidrequest = oidconsumer.begin(openid_url)
        except consumer.DiscoveryFailure, exc:
            fetch_error_string = 'Error in discovery: %s' % (
                cgi.escape(str(exc[0])))
            return self.buildErrorResponse(fetch_error_string)
        else:
            if oidrequest is None:
                msg = 'No OpenID services found for <code>%s</code>' % (
                    cgi.escape(openid_url),)
                return self.buildErrorResponse(msg)
            else:
                # Then, ask the library to begin the authorization.
                # Here we find out the identity server that will verify the
                # user's identity, and get a token that allows us to
                # communicate securely with the identity server.
                if use_eole_ext:
                    self.requestRegistrationData(oidrequest)

                trust_root = self.manager.baseurl
                return_to = self.buildURL('process')
                if oidrequest.shouldSendRedirect():
                    redirect_url = oidrequest.redirectURL(
                        trust_root, return_to, immediate=immediate)
                    return RedirectResponse(redirect_url)
                # headers=self.cookieHeaders())
                form = oidrequest.formMarkup(
                    trust_root, return_to,
                    form_tag_attrs={'id':'openid_message'},
                    immediate=immediate)
                return self.buildAutoSubmitPage(form, 'openid_message')



    def buildAutoSubmitPage(self, form, id):
        """Send a page containing an auto-submitting form."""
        response = """\
<html><head><title>Transaction OpenID en cours</title></head>
<body onload='document.getElementById("%s").submit()'>
Transaction OpenID en cours %s
</body></html>
"""% (id, form)
        return http.Response(stream=response, headers=self.cookieHeaders())




class ProcessResource(IPResource):
    """cette ressource gère la réponse en retour du fournisseur d'identité
    """
    def _render(self, ctx):
        """Handle the redirect from the OpenID server.
        """
        info, eole_resp = self.extract_response(ctx)
        if info.status == consumer.FAILURE and info.identity_url:
            # In the case of failure, if info is non-None, it is the
            # URL that we were verifying. We include it in the error
            # message to help the user figure out what happened.
            fmt = "Verification de %s échouée: %s"
            message = fmt % (cgi.escape(info.identity_url),
                             info.message)
        # XXX tester les attributs d'extensions requis ici
        elif info.status == consumer.SUCCESS:
            # This is a successful verification attempt. If this
            # was a real application, we would do our login,
            # comment posting, etc. here.
            fmt = "Vous avez verifé que %s est votre identité."
            message = fmt % (cgi.escape(info.identity_url),)

            #FIXME add a persistent session logged in info
            if info.endpoint.canonicalID:
                # You should authorize i-name users by their canonicalID,
                # rather than their more human-friendly identifiers.  That
                # way their account with you is not compromised if their
                # i-name registration expires and is bought by someone else.
                message += ("  This is an i-name, and its persistent ID is %s"
                            % (cgi.escape(info.endpoint.canonicalID),))
        elif info.status == consumer.CANCEL:
            # cancelled
            message = 'Verification annulée'
        elif info.status == consumer.SETUP_NEEDED:
            if info.setup_url:
                message = '<a href="%s">Enregistrement requis</a>' % info.setup_url
            else:
                # This means auth didn't succeed, but you're welcome to try
                # non-immediate mode.
                message = 'Enregistrement requis'
        else:
            # Either we don't understand the code or there is no
            # openid_url included with the error. Give a generic
            # failure message. The library should supply debug
            # information in a log.
            message = 'Verification échouée.'

        page = HTMLResponse(self.content % {'msg' : message},
                            title="Statut de l'authentification")
        # FIXME on dit que ca a marché meme quand le provider ne renvoie pas les
        # éléments "required" des extensions eole
        #if eole_resp.required:
        #    page.errors.append(errmsg)
        page.append(self.renderEoleData(eole_resp))
        return http.Response(stream=str(page), headers=self.cookieHeaders())




class AffiliateResource(IPResource):
    """ressource qui permet de se créer un compte openid"""

    def _render(self, ctx):
        eole_req = oideole.EoleExtensionRequest(required=['secureid', 'uid', 'user_groups'],
                                                optional=['mail', 'givenName', 'sn', 'codecivilite', 'dateNaissance'])
        href = eole_req.toMessage().toURL(OPENID_PROVIDER_URL)

        message = """Créez vous un identifiant openid: <a href="%s">%s</a>""" % (
            href, OPENID_PROVIDER_NAME)
        page = HTMLResponse(message, 'inscrivez vous')
        return http.Response(stream=str(page), headers=self.cookieHeaders())


class RootResource(IPResource):
    child_css = static.File('interface')
    addSlash = True

    def __init__(self, baseurl="http://localhost:8001/"):
        super(RootResource, self).__init__( PersistencyManager(MemoryStore(),
                                                               baseurl) )

    def _render(self, ctx):
        print 'requesting', ctx.uri
        errmsg = self.query.get('errmsg')
        page = HTMLResponse(self.content % {'msg' : ''}, title=self.title)
        if errmsg:
            page.errors.append(errmsg)
        info, eole_resp = self.extract_response(ctx)
        print 'info.status', info.status
        print 'eole_resp', eole_resp
        page.append(self.renderEoleData(eole_resp, show_not_logged = False))
        return http.Response(stream=str(page), headers=self.cookieHeaders())

    ## liste des segments autorisés ###########################################
    def child_verify(self, request):
        return VerifyResource(self.manager)

    def child_process(self, request):
        return ProcessResource(self.manager)

    def child_affiliate(self, request):
        return AffiliateResource(self.manager)



## main ##############################################################
def run(port=8001):
    baseurl = "http://192.168.230.26:%s/" % (port)
    # baseurl = "http://%s:%s/" % (getHostName(), port)
    site = server.Site(RootResource(baseurl))
    reactor.listenTCP(port, channel.HTTPFactory(site))
    print "c'est parti (port=%s)" % port
    reactor.run()


if __name__ == '__main__':
    if len(sys.argv) > 1:
        port = int(sys.argv[1])
    else:
        port = 8001
    run(port)
