#!/usr/bin/env python
#-*- coding:utf-8 -*-
"""
pages de gestion de l'authentification OTP à travers le plugin PAM securid.
On effectue une double authentification (ldap + OTP directement auprès du serveur RSA).
Le niveau d'authentificaton de la session SSO sera indiqué dans le contexte d'authentification
lors de l'envoi d'assertions SAML (vers RSA-FIM par exemple).
"""
import PAM, getpass, time, urllib, datetime, os, stat, cgi, re
import codecs
import traceback

from twisted.internet import defer, reactor, threads, protocol
from twisted.python import failure
from twisted.web2 import http, responsecode
from twisted.web2.stream import ProducerStream
from twisted.web2.http_headers import Cookie as TwCookie
from saml_utils import available_contexts, format_timestamp

from config import IDP_IDENTITY, LDAP_BASE, DEFAULT_CSS, encoding, \
                   OTP_TIMEOUT, REDIRECT_DESYNC, OTP_PORTAL, DEBUG_LOG, COOKIE_NAME
from eolesso.util import gen_ticket_id, getCookie, getCookies, urljoin
from eolesso.util import get_service_from_url, EoleParser, RedirectResponse
from saml_resources import SamlResource
from userstore import UserStore
from page import gen_page, trace, log

def get_otp_templates(load=True):
    """load local templates for otp next_token/redirect
    @param load: if False, only check validity
    """
    err_check = False
    if os.path.isfile('templates/next_token.tmpl'):
        try:
            form_data = codecs.open('templates/next_token.tmpl', 'r', encoding).read()
            # on vérifie que le formulaire comporte le bon nombre de 'variables'
            form_data % ("","","","","")
            if load:
                global NEXT_TOKEN_FORM
                NEXT_TOKEN_FORM = form_data
            else:
                log.msg(_('* Using custom template : {0}').format('next_token.tmpl'))
        except:
            if not load:
                log.msg(_('Invalid Form {0}, using default version').format('next_token.tmpl'))
            err_check = True
    if os.path.isfile('templates/desynchronized.tmpl'):
        try:
            form_data = codecs.open('templates/desynchronized.tmpl', 'r', encoding).read()
            # on vérifie que le formulaire comporte le bon nombre de 'variables'
            form_data % ("","","")
            if load:
                global DESYNCHRONIZED_FORM
                DESYNCHRONIZED_FORM = form_data
            else:
                log.msg(_('* Using custom template : {0}').format('desynchronized.tmpl'))
        except:
            if not load:
                log.msg(_('Invalid Form {0}, using default version').format('desynchronized.tmpl'))
            err_check = True
    if err_check and not load:
        log.msg(_('service restart needed to reload templates'))

SECURID_AUTH_FORM = u"""
<input type='hidden' name='user_registered' id='user_registered' value='false'/>
<label for='securid_register' style="display:none;" id='securid_reg_label' accesskey='E'>%s</label>
<input type='checkbox' style="display:none;" tabindex='3' name='securid_register' onClick='toggle_securid()' id='securid_register'/>
<div id='register_form' style='display:none;"'<hr><table>
<tr><td><label for='securid_user' id='securid_user_label' accesskey='I'>%s</label></td>
<td><input type='text' tabindex='4' name='securid_user' id='securid_user' autocomplete='%s' title='%s' /></td></tr>
<tr><td><label for='securid_pwd' accesskey='J'>%s + %s</label></td>
<td><input type='password' tabindex='5' name='securid_pwd' id='securid_pwd' autocomplete='%s' title='%s'/></td></tr>
</table></div>
"""

NEXT_TOKEN_FORM = u"""<form action="/securid" method="POST" name='securid_form'>
<table>
<tr><td colspan=2>%s</td></tr>
<tr><td><label for='securid_pwd' accesskey='W'>%s</label></td><td>
<input type='password' tabindex='2' size='7' & maxlength='6'  name='securid_pwd' id='securid_pwd' autocomplete='off' title='%s' autofocus />
<input type="hidden" name="from_url" value="%s"/>
<input type="hidden" name="css" value="%s"/>
</td></tr>
</table>
<p class="formvalidation">
<input type="submit" class='btn' tabindex='3'/></p></form>
"""

DESYNCHRONIZED_FORM = u"""<form action="/securid" method="POST" name='desync_form'>
<div class="msg_desync">%s</div>
<input type="hidden" name="from_url" value="%s"/>
<input type="hidden" name="css" value="%s"/>
<p class="formvalidation">
<input type="submit" class='btn' tabindex='3'/></p></form>
"""

def gen_retry_form(from_url, err_msg, css):
    params = []
    if from_url and from_url != 'unspecified':
        params.append(('service', from_url))
    dest = urllib.quote_plus('/login')
    if params:
        dest += '?%s' % urllib.urlencode(params)
    header_script = """<script type="text/javascript">
    function ask_for_retry()
    {
        alert("%s");
        window.location='/logout?url=%s';
    }
    </script>""" % (err_msg, dest)
    on_load_script = """ask_for_retry()"""
    page_data = gen_page(_("OTP authentication"), "",
                    javascript=on_load_script,
                    header_script=header_script,
                    css=css)
    return page_data

get_otp_templates()

class NewTokenDisabled(Exception):
    pass

class SecuridProtocol(protocol.ProcessProtocol):
    """protocol used to run pam_securid authentication in separate process
    """
    def __init__(self, checker):
        self.data = ""
        self.checker = checker
        self.checker.otp_proto = self
        self.deferred = defer.Deferred()
    def connectionMade(self):
        if DEBUG_LOG:
            log.msg('OTP AUTHENTICATION STARTED')
    @trace
    def outReceived(self, data):
        self.data += data
        WAIT_TIMEOUT = 20
        if 'DESYNCHRONIZED' in self.data and self.checker.status != 'waiting':
            self.checker.status = 'waiting'
            # le mode 'new token' est désactivé, l'utilisateur doit être redirigé
            desync_msg = _("your key is desynchronized") + "<br/>"
            if OTP_PORTAL:
                desync_msg += _("redirecting to OTP login page")
            else:
                desync_msg += _("connect to OTP login page")
            page_content = DESYNCHRONIZED_FORM % (desync_msg,
                              str(self.checker.session_data[3]) or "unspecified",
                              str(self.checker.css))
            # envoi de la réponse
            page_data = gen_page(_('New token required'), page_content, css=self.checker.css)
            self.checker.response_stream.write(page_data)
            self.checker.response_stream.finish()
            if DEBUG_LOG:
                log.msg('new token mode disabled, user redirection needed')
            self.deferred.callback(None)
        elif self.data.endswith('NEW TOKEN NEEDED : ') and self.checker.status != 'waiting':
            self.checker.status = 'waiting'
            # on attend que le mot de passe soit renouvelé
            # (attente de 2 minutes maximum)
            WAIT_TIMEOUT = OTP_TIMEOUT
            # écriture du formulaire de saisie du mot de passe dans la requête
            page_content = NEXT_TOKEN_FORM % (_("Second passcode required, wait for next token"),
                                    _("Enter new token"),
                                    _("Enter new token"),
                                    str(self.checker.session_data[3]) or "unspecified",
                                    str(self.checker.css))
            # envoi de la réponse
            page_data = gen_page(_('New token required'), page_content, css=self.checker.css)
            self.checker.response_stream.write(page_data)
            self.checker.response_stream.finish()
            if DEBUG_LOG:
                log.msg('desynchronised key, waiting for new token')
            self.deferred.callback(None)
        elif self.data.endswith('PASSCODE NEEDED : ') and self.checker.status != 'asking':
            # première demande de mot passe, on renvoie le passcode saisi
            self.checker.status = 'asking'
            reactor.callFromThread(self.transport.write, "{0}\n".format(self.checker.user_input))
    @trace
    def processEnded(self, reason):
        if DEBUG_LOG:
            log.msg("OTP PROCESS ENDED WITH STATUS %s" % (str(reason.value.exitCode),))
        if self.checker.status == 'waiting' and not REDIRECT_DESYNC:
            self.checker.status = 'timeout'
            self.checker.user_input = ''
        self.deferred.callback(reason.value.exitCode == 0)

class PAMChecker:

    @trace
    def __init__(self, pam_service):
        self.pam_service = pam_service
        self.status = "init"
        self.user_input = None
        # instance de protocole de contrôle du processus lancé
        self.otp_proto = None
        # valeur du cookie SSO en cas de nouvelle session
        self.auth_cookie = None
        self.css = DEFAULT_CSS

    @trace
    def checkCreds(self, username, response_stream, manager):
        # on passe en paramètre la requête serveur pour permettre
        # d'interagir avec l'utilisateur
        self.manager = manager
        self.response_stream = response_stream
        # spawn authentication process
        otp_proto = SecuridProtocol(checker = self)
        auth_process = "/usr/share/sso/securid_check"
        if type(username) == unicode:
            username = username.encode(encoding)
        cmd_args = [auth_process, username]
        if not REDIRECT_DESYNC:
            cmd_args.append("--no-redirect")
        reactor.spawnProcess(otp_proto, auth_process, cmd_args, env=os.environ, usePTY=True)

    @trace
    def callb_auth(self, result):
        """prend en compte le résultat de l'authentification PAM et continue
        la chaîne d'authentification"""
        try:
            session_id = self.session_id
            if not session_id.startswith('TGC'):
                if result:
                    # on n'avait pas de session au préalable, on en crée une nouvelle
                    username = self.session_data[0]
                    user_branch = self.session_data[1]
                    if self.session_data[8] != True:
                        # dans le cas ou on veut associer l'identifiant à un
                        # utilisateur local, on vérifie l'authentification LDAP
                        password = self.session_data[2]
                        defer_auth = self.manager.authenticate(username, password, user_branch)
                    else:
                        # utilisateur déjà enregistré, on ne vérifie pas le mot de passe ldap
                        feder_attrs = {'uid':[username]}
                        defer_auth = self.manager.authenticate_federated_user(feder_attrs, IDP_IDENTITY, time.time(), available_contexts['URN_TIME_SYNC_TOKEN'], search_branch=user_branch)
                    return defer_auth.addCallbacks(self.update_session, self.errb_auth, callbackArgs=[result])
                elif self.status == 'waiting':
                    # si le status est encore waiting, on a dépassé le délai
                    # maximum autorisé en mode 'next token' ou ce mode est désactivé
                    if not REDIRECT_DESYNC:
                        self.status = 'timeout'
                    self.user_input = ''
                    return False

            # session existante: on modifie le niveau d'authentification
            return self.update_session((session_id, None), result)
        except Exception, e:
            traceback.print_exc()
            log.msg("Erreur dans le callback d'authentification OTP: %s" % str(e))

    @trace
    def update_session(self, session_infos, result_securid):
        """met en place les cookies si nécessaire et renvoie l'utilisateur sur l'url de redirection"""
        self.status = "failure"
        err_msg = _('Authentication failed, try again')
        # définition des urls de retour
        from_url = self.session_data[3]
        redirect_to = self.session_data[4]
        session_id, user_data = session_infos
        if session_id:
            if from_url == '':
                # pas de service spécifié, on indique que la session est ouverte, ou on redirige si demandé
                if redirect_to != '':
                    target_url = redirect_to
                else:
                    target_url = '/loggedin'
            else:
                # on redirige sur la page de l'application de destination
                app_ticket = self.manager.get_app_ticket(session_id, from_url, from_credentials=True)
                target_url = urljoin(from_url, 'ticket=%s' % app_ticket)
                log.msg("%s -- %s" % (session_id, _("Redirecting to calling app with ticket : {0}").format(get_service_from_url(from_url))))
            if session_id.startswith('TGC') and self.manager.user_sessions.get(session_id,(None,))[0] == None:
                # Si session délivrée par le parent, rediriger sur lui pour poser un cookie d'auth qu'il puisse lire
                req_args = (('ticket', session_id), ('return_url', target_url))
                log.msg("%s -- %s" % (session_id, _("Redirecting to parent server for session setting")))
                redirect_url = '%s/gen_cookie?%s' % (self.manager.parent_url, urllib.urlencode(req_args))
            else:
                redirect_url = target_url
            # la session a été créée après authentification sur securid
            self.auth_cookie = session_id
            # cas d'une session existante
            if result_securid and self.manager.set_auth_class(session_id, available_contexts['URN_TIME_SYNC_TOKEN']):
                self.status = "success"
                if self.session_data[8] != True:
                    # si besoin, on enregistre l'identifiant alternatif de l'utilisateur
                    username = self.session_data[0]
                    user_branch = self.session_data[1]
                    securid_login = self.session_data[6]
                    log.msg("%s -- %s" % (session_id, _("Registering identifier {0} for user {1} ({2})").format(str(securid_login), str(username), user_branch or LDAP_BASE)))
                    try:
                        register_ok = self.user_store.register_user(username, securid_login, user_branch)
                    except:
                        traceback.print_exc()
                    if not register_ok:
                        err_msg = _('Error: this Id is already associated with a different user')
                        return self.securid_failed(from_url, err_msg)
            else:
                err_msg = _('OTP authentication failed, try again')
        if self.status == "success":
            # script pour mise en place du cookie sso
            header_script = """<script type="text/javascript" src="scripts/mootools-core-1.4.2.js"></script>
            <script type="text/javascript">
            sso_cook = Cookie.write('%s', '%s', {secure: true, duration:false});
            </script>
            """ % (COOKIE_NAME,session_id)
            on_load_script = "window.location='%s'" % redirect_url
            # gestion de la redirection sur la requête
            self.response_stream.write(gen_page(_("OTP authentication"), "",
                                               javascript=on_load_script, header_script=header_script, css=self.css))
            self.response_stream.finish()
            # la procédure d'authentification est terminée, on invalide le checker
            self.status = "expired"
            return True
        else:
            return self.securid_failed(from_url, err_msg)

    @trace
    def securid_failed(self, from_url, err_msg):
        page_data = gen_retry_form(from_url, err_msg, self.css)
        self.response_stream.write(page_data)
        self.response_stream.finish()
        # la procédure d'authentification est terminée, on invalide le checker
        self.status = "expired"
        return False

    @trace
    def errb_auth(self, result):
        page_data = gen_page(_("OTP authentication"), "Erreur rencontré lors de l'authentification OTP (PAM)", css=self.css)
        self.response_stream.write(page_data)
        self.response_stream.finish()
        log.msg("Erreur à l'exécution de l'authentification PAM (%s) : %s" % (str(self), result.getErrorMessage()))
        # Erreur, on annule la session SSO de l'utilisateur
        self.manager.logout(self.session_id)
        self.status = "failure"
        return False

class SecuridCheck(SamlResource):

    isLeaf = True

    def __init__(self, manager):
        self.manager = manager
        SamlResource.__init__(self)

    @trace
    def _render(self, request):
        responseStream = ProducerStream()
        login_ticket = None
        ## Def l'id de session
        tw_cook = getCookie(request, 'EoleSSOChecker')
        if tw_cook is None:
            uniq_id = gen_ticket_id('TMP', self.manager.address).encode(encoding)
        else:
            uniq_id = tw_cook.value

        checker = self.manager.checkers.get(uniq_id, (None, None))[0]
        status = None
        if checker:
            status = checker.status
            if status not in ("waiting", "timeout"):
                self.purge_checker(uniq_id)
                checker = None
        try:
            css = request.args['css'][0]
        except KeyError:
            css = DEFAULT_CSS
        access_allowed = False
        # le mot de passe est récupéré dans la requête si l'utilisateur
        # doit le saisir une deuxième fois (mode next token)
        passwd = request.args.get('securid_pwd',[""])[0]
        if type(passwd) == unicode:
            # pas d'envoi d'unicode au processus externe
            passwd = passwd.encode(encoding)
        # on vérifie si on est dans le cas d'un timeout après demande d'un deuxième token
        if request.args.get('from_url', []):
            from_url = cgi.escape(request.args.get('from_url',['unspecified'])[0])
            if REDIRECT_DESYNC:
                # redirection vers le portail OTP pour synchroniser la clé
                if OTP_PORTAL:
                    redirect_url = OTP_PORTAL
                elif from_url and from_url != "unspecified":
                    redirect_url = from_url
                else:
                    redirect_url = "/"
                if checker:
                    # on force la suppression immédiate du checker
                    self.purge_checker(checker.session_id)
                return RedirectResponse(redirect_url)
            elif checker is None or status == "timeout":
                # cas d'une réponse trop tardive pour la saisie du deuxième passcode (clé désynchronisée)
                # on demande une nouvelle tentative de connexion
                css = cgi.escape(request.args.get('css',[''])[0])
                err_msg = _('Input delay expired, try again')
                if checker:
                    # on force la suppression immédiate du checker
                    self.purge_checker(checker.session_id)
                response_data=gen_retry_form(from_url, err_msg, css)
                response = http.Response(code=responsecode.OK, stream=response_data)
                return self.set_headers(response)
        if checker is None:
            if request.args.has_key('lt'):
                # on vérifie qu'on a bien été appelé apr la page d'authentification
                login_ticket = request.args.get('lt', [''])[0]
            else:
                # premier accès à la page ('pas de mot de passe saisi')
                return self.set_headers(http.Response(stream=gen_page(_('Error'), _('Resource not Allowed'), css=css)))
            # initialisation d'un nouveau checker
            if not self.manager._DBLoginSessionsValidate(login_ticket):
                return self.set_headers(http.Response(stream=gen_page(_('Error'), _('Resource not Allowed'), css=css)))
            # récupération des paramètres stockés avec le ticket lt
            session_data = self.manager.login_sessions.get_session_info(login_ticket)
            if css == DEFAULT_CSS:
                css = session_data[5]
            checker = self.init_checker(uniq_id)
            # récupération du login securid de l'utilisateur
            username = session_data[6]
            passwd = session_data[7]
            # on enregistre les paramètres d'origine dans l'objet checker en prévision
            # de la suite du traitement de l'authentification (from_url, redirect_to)
            checker.css = css
            checker.securid_login = username
            checker.session_id = uniq_id
            checker.checker_pool = self.manager.checkers
            checker.session_data = session_data
            checker.user_store = self.manager.securid_user_store
            checker.user_input = passwd
            checker.checkCreds(username, responseStream, self.manager)
        def_proto = checker.otp_proto.deferred.addBoth(self._check_status, request, checker, login_ticket, responseStream)
        if checker.status == 'waiting':
            # envoi des données saisies au thread pam + requête pour interaction
            checker.user_input = passwd
            checker.response_stream = responseStream
            checker.status = 'answered'
            checker.otp_proto.transport.write("{0}\n".format(passwd))
        return def_proto

    @trace
    def _check_status(self, status, request, checker, login_ticket, responseStream):
        uniq_id = checker.session_id
        if status is None:
            # le processus est en attente d'un nouveau mot de passe. On réinitialise le deferred
            # du protocole et on affiche le formulaire de saisie
            del(checker.otp_proto.deferred)
            checker.otp_proto.deferred = defer.Deferred()
            return self._render_callb(status, request, uniq_id, login_ticket, responseStream)
        else:
            # le processus s'est terminé, on appelle les callbacks d'authentification
            def_result = defer.Deferred()
            if isinstance(status, failure.Failure):
                def_result.addCallback(checker.errb_auth)
            else:
                def_result.addCallbacks(checker.callb_auth, checker.errb_auth)
            def_result.addBoth(self._render_callb, request, uniq_id, login_ticket, responseStream)
            def_result.callback(status)
            return def_result

    @trace
    def _render_callb(self, result, request, uniq_id, login_ticket, responseStream):
        # on supprime le checker si le processus n'est plus en attente
        if result != None or REDIRECT_DESYNC:
            if uniq_id in self.manager.checkers:
                self.purge_checker(uniq_id)
        response = http.Response(code=responsecode.OK, stream=responseStream)
        response = self.set_headers(response)
        # mise en place des nouveaux cookies si besoin
        if login_ticket != None:
            # nouveau checker créé, on met en place le cookie pour le retrouver
            cookies = []
            for cookie in getCookies(request):
                if cookie.name != 'EoleSSOChecker':
                    cookies.append(cookie)
            cookies.append(TwCookie('EoleSSOChecker', uniq_id, path = "/securid", discard=True, secure=True))
            response.headers.setHeader('Set-Cookie', cookies)
        return response

    @trace
    def purge_checker(self, uniq_id, auto=False):
        """suppression d'un checker.
        on débloque ev_auth au cas ou un thread serait en attente
        (ex : reload de la page par l'utilisateur en mode 'next token')
        si auto == False: supression manuelle, on annule le callback
        de supression automatique
        """
        if DEBUG_LOG:
            log.msg("PURGING CHECKER %s" % uniq_id)
        try:
            if self.manager.checkers[uniq_id][0].status in ('waiting', 'asking'):
                self.manager.checkers[uniq_id][0].user_input = ''
            if self.manager.checkers[uniq_id][0].otp_proto:
                try:
                    # on ferme les entrées/sorties du processus OTP pour l'arrêter
                    self.manager.checkers[uniq_id][0].otp_proto.transport.loseConnection()
                except:
                    # déjà arrêté ?
                    pass
            self.manager.checkers[uniq_id][0].status = "expired"
            if not auto:
                self.manager.checkers[uniq_id][1].cancel()
            del(self.manager.checkers[uniq_id])
        except:
            if DEBUG_LOG:
                log.msg("  + Erreur à la suppression : checker non trouvé (ne devrait pas arriver !)")

    @trace
    def init_checker(self, uniq_id):
        # on commence par purger tous les checkers qui ont expiré
        for id_session, checker_data in self.manager.checkers.items():
            checker = checker_data[0]
            if checker.status == "expired":
                self.purge_checker(id_session)
        checker = PAMChecker('rsa_securid')
        # fonction pour supprimer automatiquement le checker après OTP_TIMEOUT +2 minutes
        delete_cb = reactor.callLater(OTP_TIMEOUT + 120, self.purge_checker, uniq_id, True)
        self.manager.checkers[uniq_id] = (checker, delete_cb)
        return checker

