# -*- coding: utf-8 -*-
#
##########################################################################
# eoleauth - authclient.py: utilies for eoleauth client applications
# Copyright © 2013 Pôle de compétences EOLE <eole@ac-dijon.fr>
#
# License CeCILL:
#  * in french: http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
#  * in english http://www.cecill.info/licences/Licence_CeCILL_V2-en.html
##########################################################################
"""Module d'authentification d'eole-aaa
"""
import urllib, shutil, datetime, os
from functools import wraps
from flask import request, session, redirect, current_app, url_for, flash
from eoleflask.util import make_error_response, get_proxy_url

from eoleauthlib.plugins import PLUGINS
from eoleauthlib.client import UnauthorizedError
from eoleauthlib.i18n import _

def get_active_client(reset=False):
    """Defines plugin to use for authentication.
    @arg reset: if True, always destroy previous session

    client selection scheme:
    - default client uses PAM local auth.
    - if EOLEAUTH_PLUGIN is defined in current_app.config, cheks that
      check_authsource function of the plugin returns True before using it.
    """
    # default clients : PAMClient (or CASClient if CAS_URL defined)
    conf_client = 'PAMClient'
    client = PLUGINS[conf_client]
    if current_app.config.get('EOLEAUTH_PLUGIN', ''):
        try:
            app_plugin = current_app.config.get('EOLEAUTH_PLUGIN')
            assert app_plugin in PLUGINS
            conf_client = PLUGINS.get(app_plugin)
            # check if authentication source is available
            if conf_client().check_authsource():
                client = conf_client
            else:
                current_app.logger.error(_("authentication source not available for plugin {0}").format(app_plugin))
        except AssertionError:
            current_app.logger.error(_("unknown authentication plugin : {0}").format(app_plugin))
    mode = current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper()
    return client(active=not reset, mode=mode)

def _init_session_interface():
    """returns current session manager used by eoleauth
    - by default, store sessions in local Redis server
    - if Redis not configured in socket mode, use local storage in root/.eoleauth
      (for example : server freshly installed and not yet configured)
    local storage will be deleted once redis mode is available
    """
    if hasattr (current_app, 'eoleauth_session_prefix'):
        # session management is already initialized
        return current_app.eoleauth_session_prefix
    local_dir = '/root/.eoleauth'
    if current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper() == 'LOCAL':
        session_prefix = current_app.name
    else:
        session_prefix='eoleauth'
    # used to check if application already configured
    current_app.eoleauth_session_prefix = session_prefix
    current_app.session_cookie_name = "{0}_session".format(session_prefix)
    current_app.logger.info('session/cookie prefix set to <{0}>'.format(session_prefix))
    try:
        import redis
        r_client = redis.Redis(unix_socket_path='/var/run/redis/redis.sock')
        server_info = r_client.info()
        # redis available, store sessions in local database (socket mode)
        from eoleauthlib.redissession import RedisSessionInterface
        session_interf = RedisSessionInterface(prefix="{0}:".format(session_prefix))
        current_app.logger.info(_('Session storage set to Redis'))
        if os.path.exists(local_dir):
            try:
                shutil.rmtree(local_dir)
            except:
                current_app.logger.warning(_('eoleauth: could not purge previous session storage ({0})').format(local_dir))
    except:
        # redis unavailable or not configured to listen on socket, use file storage instead (1000 sessions cached in memory)
        from eoleauthlib.cachedsession import ManagedSessionInterface, CachingSessionManager, FileBackedSessionManager
        session_storage = FileBackedSessionManager('/root/.eoleauth', current_app.config['SECRET_KEY'], "{0}:".format(session_prefix))
        session_cache = CachingSessionManager(session_storage, 1000)
        session_interf = ManagedSessionInterface(session_cache, ['/static'], datetime.timedelta(seconds=10))
        current_app.logger.info(_('Session storage set to local (Redis not configured properly)'))
    current_app.session_interface = session_interf
    return session_prefix

def init_authentication(managed_app):
    """
    initializes session management and login/logout url for application managed_app
    root_url : specifies an entry point url to redirect to after logout
    """
    @managed_app.route("/login", methods=["GET", "POST"])
    def login():
        """Checks authentication and send to authentication process if needed
        """
        try:
            client = get_active_client()
            return client.authenticate()
        except Exception, e:
            current_app.logger.error(_('Internal error during PAM Authentication : {0}').format(e))
            flash(_(u'Erreur : {0}').format(e) , 'error')
            mode = current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper()
            if mode == 'LOCAL':
                app_url = get_proxy_url(request, request.url)
                return redirect(get_proxy_url(request, url_for('login', app_url=app_url, _external=True)))
            else:
                return redirect('/eoleauth/login?app_url={0}'.format(urllib.quote(request.url)))

    @managed_app.route('/logout')
    def logout():
        """ends user session and display logged out message
        if return_url is passed in request args, redirect there
        (typically, return to application start page)
        """
        if 'username' in session:
            # remove the username from the session if it's there
            user = get_active_client()
            return user.logout()
        # déconnexion ok, on redirige sur l'url demandée ou on affiche un message
        if request.args.get('return_url',''):
            return redirect(request.args.get('return_url'))
        return _('You are disconnected')

    # This ensures that eoleflask configuration is loaded before setting session
    # interface and cookie name for the application (not avalaible when app initializes)
    managed_app.before_first_request(_init_session_interface)

def login_required(f):
    @wraps(f)
    def decorated_function(*args, **kwargs):
        mode = current_app.config.get('EOLEAUTH_MODE', 'GLOBAL').upper()
        current_app.logger.debug('In authentication decorator for {0}'.format(current_app.name))
        current_app.logger.debug('SESSION:  {0}'.format(session))
        current_app.logger.debug('  mode: {0}, current_url: {1}'.format(mode, request.url))
        try:
            assert session and session.get("username", None)
            # always check for local user restrictions even if global mode set
            if current_app.config.get('ALLOWED_USERS', []):
                try:
                    # use app.flash to display message on login page if failed ?
                    get_active_client().check_allowed_users(session)
                except UnauthorizedError, e:
                    return make_error_response(unicode(e), 401)
        except Exception, e:
            # no valid session or invalid user, redirect to eoleauth login page with current
            # request url passed as app_url
            if mode == 'LOCAL':
                app_url = get_proxy_url(request, request.url)
                return redirect(get_proxy_url(request, url_for('login', app_url=app_url, _external=True)))
            else:
                return redirect('/eoleauth/login?app_url={0}'.format(urllib.quote(request.url)))
        return f(*args, **kwargs)
    return decorated_function
