#!/usr/bin/python
#-*-coding:utf-8-*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# update_databases.py
#
# Mise à niveau des base de données sur Scribe
#
###########################################################################
"""
    Outil de mise à jour de base de données

"""
from os.path import isfile

from creole.eosfunc import gen_random

from eolesql.config import ROOT_USERNAME
from eolesql.db_layer import DB
from eolesql.log import log_console
from eolesql.collect_update import load_apps_conf
from eolesql.db_cmd import dump_sql_file
from eolesql.db_test import db_exists

from mysql_pwd import mysql_root_passwd, mod_passwords

def get_db_handler(update, new_root_passwd):
    """
        Renvoie un connecteur pour la base de données à mettre à jour
    """
    connect_dict = dict(host=update.get('host'),
                        db=update.get('db'),
                        passwd=new_root_passwd,
                        user=ROOT_USERNAME)
    return DB(connect_dict)

def necessary_to_update(db_handler, expected_response, condition_query):
    """
        Renvoie True si il est nécessaire de mettre à jour ou non
    """
    if condition_query:
        result = db_handler.query(condition_query)
    else:
        result = None
    if expected_response:
        return expected_response(result)
    else:
        return True

def update_databases(dbs_toupdate):
    """
        parcours les tables à mettre à jour,
        si la table existe,
        teste la condition,
        si elle est validée,
        on update avec le(s) fichier(s) passé(s) en paramètre
    """
    # réinitialisation du mot de passe root
    new_root_passwd = gen_random(10)
    mysql_root_passwd(new_root_passwd)
    #met dans l'ordre alphabétique (#3943)
    app_names = dbs_toupdate.keys()
    app_names.sort()
    for db_name in app_names:
        updates = dbs_toupdate[db_name]
        log_console("# Mise à jour de la base de données %s" % db_name)
        if not db_exists(db_name):
            log_console(" - Erreur : La base %s n'existe pas" % db_name)
            continue
        for update in updates:
            version = update.get('version')
            if version:
                log_console(" + Mis à jour vers la version %s" % (
                                          update.pop('version'),), 2)

            db_handler = get_db_handler(update, new_root_passwd)

            condition_query = update.get('condition_query')
            expected_response = update.get('expected_response')

            if necessary_to_update(db_handler, expected_response, \
                                                    condition_query):
                log_console(" + La condition proposée est validée => \
 On met à jour la table.", 1)
                filenames = update.get('filenames')
                for filename in filenames:
                    if not isfile(filename):
                        log_console(" - Erreur : Il manque le fichier %s" % filename)
                        continue
                    if dump_sql_file(new_root_passwd, filename) != 0:
                        log_console(" - Erreur : Erreur lors de l'intégration du fichier %s" % filename)
            else:
                if version:
                    log_console(" + La base de données %s est à jour en \
version %s." % (db_name, version), 1)
                else:
                    log_console(" + La base de données %s est à jour." % (
                                                             db_name,), 1)
    mod_passwords(new_root_passwd)

def main():
    """
        Lancement de l'outil de mise à jour de base de données
    """
    log_console("## Mise à jour de base de données ##")
    dbs_toupdate = load_apps_conf()
    if dbs_toupdate:
        update_databases(dbs_toupdate)
    else:
        log_console("Aucune base à mettre à jour")

if __name__ == '__main__':
    main()
