# -*- coding: utf-8 -*-
"""
Support for era matrix format
"""

from subprocess import call as _call
import json as _json

_ACTIVE_TAGS = '/var/lib/eole/config/regles.csv'
_ERA_MODEL_PATH = '/var/lib/eole/config/era_matrix.json'

class _OptionalRule(object):
    OPTIONAL_MASK = 1
    ACTIVE_MASK = 2
    LOG_MASK = 4
    IPSET_MASK = 8
    HIDDEN_MASK = 16

    def __init__(self, label, default_state, count, interface):
        self.label = label
        self.default_state = int(default_state)
        self.state = int(default_state)
        self.count = count
        self.interface = interface

    def set_state(self, value):
        self.state = int(value)

    def is_optional(self):
        return self.OPTIONAL_MASK & self.state == self.OPTIONAL_MASK

    def is_hidden(self):
        return self.HIDDEN_MASK & self.state == self.HIDDEN_MASK

    def is_active(self):
        return self.ACTIVE_MASK & self.state == self.ACTIVE_MASK

    def activate(self):
        self.state = self.state | self.ACTIVE_MASK

    def deactivate(self):
        self.state = self.state ^ self.ACTIVE_MASK

    def output(self):
        return ','.join([self.label, str(self.state), str(self.count), self.interface])


def _update_directive_states(directives):
    with open(_ACTIVE_TAGS, 'r') as active_tags_stream:
        for tag in active_tags_stream.readlines():
            label, state, _, _ = tag.split(',')
            for directive in directives:
                if label == directive.label:
                    directive.set_state(state)


def getdirectives(**kwargs):
    with open(_ERA_MODEL_PATH, 'r') as model_stream:
        matrix = _json.load(model_stream)
    directives = [d for zone in matrix for ds in matrix[zone].values() for d in ds]
    directives = [_OptionalRule(d['label'], d['state'], d['count'], d['interface']) for d in directives]
    _update_directive_states(directives)
    return [{'label': d.label, 'enabled': d.is_active(), 'hidden': d.is_hidden(), 'interface': d.interface} for d in directives]
