# -*- coding: utf-8 -*-
"""
Support for era matrix format
"""

from datetime import datetime
import iptc as _iptc
import re as _re
from ipaddress import ip_interface as _ip_interface
from subprocess import Popen as _Popen, STDOUT as _STDOUT, PIPE as _PIPE


COMMENT = 'iptablesvolatiles'


def deldirectives(rule_id, **kwargs):
    table = _iptc.Table(_iptc.Table.FILTER)
    id_rule = COMMENT + ':' + rule_id
    delete = False
    for chain in table.chains:
        for rule in chain.rules:
            for match in rule.matches:
                if match.name == 'comment' and match.parameters['comment'] == id_rule:
                    chain.delete_rule(rule)
                    delete = True
                    break
    if delete:
        return {'message': {'text': 'La règle est supprimé',
                'type': 'info'}}
    return {'message': {'text': 'La règle introuvable',
            'type': 'error'}}


def setdirectives(source, destination, dport, protocol, **kwargs):
    try:
        ip = _ip_interface(source)
    except ValueError:
        return {'message': {'text': 'La source est une IP invalide',
                'type': 'error'}}
    try:
        ip = _ip_interface(destination)
    except ValueError:
        return {'message': {'text': 'La destination est une IP invalide',
                'type': 'error'}}
    if protocol not in ['UDP', 'TCP']:
        return {'message': {'text': 'Le protocole doit être soit "UDP" soit "TCP"',
                'type': 'error'}}
    multi = False
    if ':' in dport:
        dports = dport.split(':')
        if len(dports) != 2:
            return {'message': {'text': 'Le port est invalide (multiple :)',
                    'type': 'error'}}
    elif ',' in dport:
        multi = True
        dports = dport.split(',')
    else:
        dports = [dport]
    try:
        for dp in dports:
            int(dp)
    except ValueError:
        return {'message': {'text': 'Le port est invalide',
                'type': 'error'}}
    #

    cmd = ['iptables', '-I', 'FORWARD', '-s', source, '-d', destination]
    if protocol == 'UDP':
        cmd.extend(['-p', 'udp', '-m', 'state', '--state', 'NEW', '-m', 'udp'])
    elif protocol == 'TCP':
        cmd.extend(['-p', 'tcp', '-m', 'state', '--state', 'NEW', '-m', 'tcp', '--tcp-flags', 'SYN,RST,ACK', 'SYN'])
    id_rule = datetime.now().timestamp()
    if not multi:
        cmd.extend(['--dport', dport])
    else:
        cmd.extend(['-m', 'multiport', '--dports', dport])
    cmd.extend(['-m', 'comment', '--comment', COMMENT + ':' + str(id_rule), '-j', 'ACCEPT'])
    proc = _Popen(cmd, stderr=_STDOUT, stdout=_PIPE)
    stdout, stderr = proc.communicate()
    if proc.returncode:
        if stderr:
            msg = stderr
        else:
            msg = stdout
        return {'message': {'text': msg,
                'type': 'error'}}
    return {'message': {'text': 'La règle est appliquée',
            'type': 'info'}}


def getdirectives(**kwargs):
    directives = []
    table = _iptc.Table(_iptc.Table.FILTER)
    for chain in table.chains:
        for rule in chain.rules:
            id_rule = None
            for match in rule.matches:
                if match.name == 'comment' and match.parameters['comment'].startswith(COMMENT + ':'):
                    id_rule = match.parameters['comment'].split(':')[1]
                    break
            if id_rule:
                dport = ','.join([m.dport for m in rule.matches if m.dport])
                if not dport:
                    dport = ','.join([m.dports for m in rule.matches if m.dports])
                directives.append({'id': id_rule,
                                   'destination': rule.dst,
                                   'source': rule.src,
                                   'dport': dport,
                                   'protocol': rule.protocol})
    return directives
