# -*- coding: utf-8 -*-
"""
Support for era matrix format
"""

from subprocess import call as _call
import json as _json

_ACTIVE_TAGS = '/var/lib/eole/config/regles.csv'
_ERA_MODEL_PATH = '/var/lib/eole/config/era_matrix.json'

class _OptionalRule(object):
    OPTIONAL_MASK = 1
    ACTIVE_MASK = 2
    LOG_MASK = 4
    IPSET_MASK = 8
    HIDDEN_MASK = 16

    def __init__(self, label, default_state, count, interface):
        self.label = label
        self.default_state = int(default_state)
        self.state = int(default_state)
        self.count = count
        self.interface = interface

    def set_state(self, value):
        self.state = int(value)

    def is_optional(self):
        return self.OPTIONAL_MASK & self.state == self.OPTIONAL_MASK

    def is_hidden(self):
        return self.HIDDEN_MASK & self.state == self.HIDDEN_MASK

    def is_active(self):
        return self.ACTIVE_MASK & self.state == self.ACTIVE_MASK

    def activate(self):
        self.state = self.state | self.ACTIVE_MASK

    def deactivate(self):
        self.state = self.state ^ self.ACTIVE_MASK

    def output(self):
        return ','.join([self.label, str(self.state), str(self.count), self.interface])


def _update_directive_states(directives):
    with open(_ACTIVE_TAGS, 'r') as active_tags_stream:
        for tag in active_tags_stream.readlines():
            label, state, _, _ = tag.split(',')
            for directive in directives:
                if label == directive.label:
                    directive.set_state(state)


def getzonenames(**kwargs):
    with open(_ERA_MODEL_PATH, 'r') as model_stream:
        matrix = _json.load(model_stream)
    return list(matrix.keys())

def getdirectives(zone, matrix=None, **kwargs):
    with open(_ERA_MODEL_PATH, 'r') as model_stream:
        matrix = _json.load(model_stream)
    directives = [d for ds in matrix[zone].values() for d in ds]
    directives = [_OptionalRule(d['label'], d['state'], d['count'], d['interface']) for d in directives]
    _update_directive_states(directives)
    return [{'label': d.label, 'enabled': d.is_active()} for d in directives
            if d.is_optional() and not d.is_hidden()]

def setdirectives(zone, directives, **kwargs):
    # mettre à jour la liste des directives activées : pour chaque directive, l’inscrire ou l’enlever du fichier
    tags = []
    with open(_ACTIVE_TAGS, 'r') as active_tags_stream:
        active_tags = active_tags_stream.readlines()
        for active_tag in active_tags:
            print(active_tag)
            active_tag = active_tag.split(',')
            print(active_tag)
            tags.append(_OptionalRule(*active_tag))

    with open(_ACTIVE_TAGS, 'w') as active_tags_stream:
        for active_tag in tags:
            for index, label in enumerate([d['label'] for d in directives]):
                if label == active_tag.label:
                    if directives[index]['enabled']:
                        active_tag.activate()
                    else:
                        active_tag.deactivate()
            active_tags_stream.write(active_tag.output())
    # bastion regen
    bastion_regen_cmd = ['bastion', 'regen']
    regen_status = _call(bastion_regen_cmd)
    # 01-export-era
    export_era_cmd = ['/usr/share/eole/posttemplate/01-export-era']
    export_status = _call(export_era_cmd)
    return True
