# -*- coding: utf-8 -*-
"""
Support for era matrix format
"""

from datetime import datetime as _datetime
from json import loads as _loads
from ipaddress import ip_interface as _ip_interface, ip_network as _ip_network
from subprocess import Popen as _Popen, STDOUT as _STDOUT, PIPE as _PIPE
import argparse as _argparse
import shlex as _shlex
from pyeole.process import system_out as _system_out
from creole.eosfunc import calc_netmask as _calc_netmask


_COMMENT = 'iptablesvolatiles'
_TABLES = ['filter']


_parser = _argparse.ArgumentParser()
_parser.add_argument('-A', '--chain', type=str)
_parser.add_argument('-s', '--source', type=str, default='0.0.0.0/0')
_parser.add_argument('-d', '--destination', type=str, default='0.0.0.0/0')
_parser.add_argument('-i', '--in_', type=str)
_parser.add_argument('-o', '--out', type=str)
_parser.add_argument('-p', '--protocol', type=str, default='ip')
_parser.add_argument('-m', type=str)
_parser.add_argument('--state', type=str)
_parser.add_argument('--dport', type=str, default='')
_parser.add_argument('--tcp-flags', type=str, nargs=2)
_parser.add_argument('--match-set', type=str, nargs=2)
_parser.add_argument('--comment', type=str, default='')
_parser.add_argument('--limit', type=str)
_parser.add_argument('--log-prefix', type=str)
_parser.add_argument('-j', '--target', type=str)
_parser.add_argument('--icmp-type', type=str)
_parser.add_argument('--to-source', type=str)
_parser.add_argument('--to-ports', type=str)


def _cmdline(source, destination, dport, protocol, multi=False, id_rule=None, is_delete=False):
    cmd = ['iptables']
    if not is_delete:
        cmd.append('-I')
    else:
        cmd.append('-D')
    cmd.extend(['FORWARD', '-s', source, '-d', destination])
    if protocol == 'UDP':
        cmd.extend(['-p', 'udp', '-m', 'state', '--state', 'NEW', '-m', 'udp'])
    elif protocol == 'TCP':
        cmd.extend(['-p', 'tcp', '-m', 'state', '--state', 'NEW', '-m', 'tcp', '--tcp-flags', 'SYN,RST,ACK', 'SYN'])
    if id_rule is None:
        id_rule = _datetime.now().timestamp()
    if not multi:
        cmd.extend(['--dport', dport])
    else:
        cmd.extend(['-m', 'multiport', '--dports', dport])
    cmd.extend(['-m', 'comment', '--comment', _COMMENT + ':' + str(id_rule), '-j', 'ACCEPT'])
    proc = _Popen(cmd, stderr=_STDOUT, stdout=_PIPE)
    stdout, stderr = proc.communicate()
    if proc.returncode:
        if stderr:
            msg = stderr
        else:
            msg = stdout
        raise Exception(msg)


def deldirectives(rule_id, **kwargs):
    try:
        rc, out, err = _system_out(['iptables-save', '-t', _TABLES[0]])
        if rc != 0:
            raise Exception('cannot extract iptable rules')
        for line in out.strip().split('\n'):
            if f' --comment "{_COMMENT}:{rule_id}" ' in line:
                cmd = 'iptables -D' + line[2:]
                proc = _Popen(cmd, stderr=_STDOUT, stdout=_PIPE, shell=True)
                stdout, stderr = proc.communicate()
                if proc.returncode:
                    if stderr:
                        msg = stderr
                    else:
                        msg = stdout
                    raise Exception(msg)
                return {'message': {'text': 'La règle est supprimé',
                        'type': 'info'}}
        raise Exception(f'règle {rule_id} inconnue')
    except Exception as err:
        return {'message': {'text': f'Ne peut supprimer la règle : {err}',
                'type': 'error'}}


def setdirectives(source, destination, dport, protocol, **kwargs):
    try:
        ip = _ip_interface(source)
    except ValueError:
        return {'message': {'text': 'La source est une IP invalide',
                'type': 'error'}}
    try:
        ip = _ip_interface(destination)
    except ValueError:
        return {'message': {'text': 'La destination est une IP invalide',
                'type': 'error'}}
    if protocol not in ['UDP', 'TCP']:
        return {'message': {'text': 'Le protocole doit être soit "UDP" soit "TCP"',
                'type': 'error'}}
    multi = False
    if ':' in dport:
        dports = dport.split(':')
        if len(dports) != 2:
            return {'message': {'text': 'Le port est invalide (multiple :)',
                    'type': 'error'}}
    elif ',' in dport:
        multi = True
        dports = dport.split(',')
    else:
        dports = [dport]
    try:
        for dp in dports:
            int(dp)
    except ValueError:
        return {'message': {'text': 'Le port est invalide',
                'type': 'error'}}
    #

    try:
        _cmdline(source, destination, dport, protocol, multi)
    except Exception as err:
        return {'message': {'text': str(err),
                'type': 'error'}}
    return {'message': {'text': 'La règle est appliquée',
            'type': 'info'}}


def _address_to_ead(ip):
    if isinstance(ip, dict):
        addr = ip['prefix']['addr']
        cidr = ip['prefix']['len']
        ip = _ip_network(f'{addr}/{cidr}').with_netmask
    else:
        ip = f'{ip}/255.255.255.255'
    return ip


def getdirectives(**kwargs):
    directives = []
    for table in _TABLES:
        rc, out, err = _system_out(['iptables-save', '-t', table])
        if rc != 0:
            raise Exception('cannot extract iptable rules')
        for line in out.strip().split('\n'):
            if _COMMENT not in line:
                continue
            try:
                nline = _shlex.split(line)
                args = _parser.parse_args(nline)
                if not args.comment.startswith(f'{_COMMENT}:'):
                    continue
                id_rule = args.comment.split(':', 1)[-1]
                source = args.source
                if '/' in source:
                    ip, cidr = source.split('/')
                    if cidr == '0':
                        cidr = '0.0.0.0'
                    network = _calc_netmask(cidr)
                    source = f'{ip}/{network}'
                destination = args.destination
                if '/' in destination:
                    ip, cidr = destination.split('/')
                    if cidr == '0':
                        cidr = '0.0.0.0'
                    network = _calc_netmask(cidr)
                    destination = f'{ip}/{network}'
                directives.append({'id': id_rule,
                                   'source': source,
                                   'destination': destination,
                                   'dport': args.dport,
                                   'protocol': args.protocol,
                                   })
            except Exception as err:
                # cannot parse this line
                pass
    return directives
