from era.tool.pairtype import extendabletype
class FwObject(metaclass=extendabletype):
    """Classe de base des objets du Firewall (directives, extremites, etc.)
    Cette classe définit l'ensemble des caractères communs de tous les objets
    du Firewall. En particulier, chaque objet peut-être 'hérité' d'un modèle,
    et être ou non 'éditable'.

    """

    instanciable_attributes = ()

    def __init__(self, inherited = False, editable = True, mime_type = None):
        """
        inherited : définit si l'objet est hérité d'un modèle ou pas
        editable : définit si l'objet est modifiable ou pas (lié à l'import)
        mime_type : le mime_type de l'objet
        """
        self.inherited = inherited
        self.editable = editable
        self.mime_type = mime_type

    def is_inherited(self):
        """Pour savoir si cet objet a été hérité d'un modèle
        """
        return self.inherited

    def set_inherited(self, inherited):
        """Définit si l'objet est hérité d'un modèle ou pas
        """
        self.inherited = inherited

    def is_editable(self):
        """Renvoie True si l'objet est éditable, False sinon
        """
        return self.editable

    def set_editable(self, editable):
        """Fixe l'éditablilité de l'objet
        """
        self.editable = editable

    def get_type(self):
        """Renvoie le type de l'objet
        """
        return self.mime_type

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        return any([getattr(self, attrname, '').startswith('%%')
                    for attrname in self.instanciable_attributes])

    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        unspecified = set()
        for attrname in self.instanciable_attributes:
            value = getattr(self, attrname, '')
            if value.startswith('%%'):
                if value.endswith('%%'):
                    unspecified.add(value[2:-2])
                else:
                    unspecified.add(value[2:])
        return unspecified
