# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
Module chargé de gérer les fenêtres de services.
"""

import gtk, gtk.glade
import gobject

from era.noyau.fwobjects import Service, ServiceGroupe
from era.noyau.dpatterns import Singleton
from era.noyau.erreurs import InvalidPortListError, EmptyNameError
from gtk_models import create_service_treestore, create_group_treestore, \
     set_empty_combo_model, set_combo_model
from ihm_utils import (create_error_dialog, create_yes_no_dialog, 
                                                          make_template_string)
from era.noyau.pool import library_store

class ServiceListviewDialog(Singleton):

    def __init__(self, glade_file,groupe=None):

        self.glade_file = glade_file

        if 'dlg' not in self.__dict__:
            self.glade = gtk.glade.XML(self.glade_file,"service_listview_dialog", "editeur")
            self.dlg = self.glade.get_widget('service_listview_dialog')
            self.treeview =  self.glade.get_widget('service_listview')
            self.prop_label =  self.glade.get_widget('service_properties_label')
            self.group_listview =  self.glade.get_widget('list_group_service')
#            self.treeview.set_headers_visible(1)
#            self.group_listview.set_headers_visible(1)
            self.init_treeview_column()
            self.init_group_liststore()
            selection = self.treeview.get_selection()
            selection.set_mode(gtk.SELECTION_SINGLE)
            selection.connect('changed',self.item_selected)
            handlers = {'on_service_listview_dialog_response' : self.close,
                        'on_service_listview_dialog_delete_event' : self.stay_alive,
                        'on_edit_service_button_clicked' : self.edit_service,
                        'on_del_service_button_clicked' : self.del_service,
                        'on_button_add_service_clicked' : self.add_service,
                        'on_button_add_service_group_clicked' : self.add_service_group,
                        'on_service_listview_button_press_event' : self.edit_service_on_click,
                        }
            self.glade.signal_autoconnect(handlers)
#            self.refresh_treeview()
        self.init_model()

    def add_service(self, *args):
        """
            callback de creation du service
        """
        serv_dlg = ServiceDialog(self.glade_file, parent=self)
        serv_dlg.show_dialog()

    def add_service_group(self, *args):
        """
            affichage du dialogue de création de groupes de services
        """
        grp_dlg = ServiceGroupeDialog(self.glade_file, parent=self)
        grp_dlg.show_dialog()

    def init_model(self):
        """Initialise le modèle associé à l'arbre des services
        """
        self.treemodel = create_service_treestore()
        self.treemodel = create_group_treestore(self.treemodel)
        self.treeview.set_model(self.treemodel)


    def init_treeview_column(self):
        column = gtk.TreeViewColumn(_('services and groups list'),
                                    gtk.CellRendererText(),
                                    text = 0)
        self.treeview.append_column(column)
        #self.treeview.connect('button-press-event', self.item_selected)


    def item_selected(self,selection, *args):
        """Callback appelée lorsqu'on clique sur un service dans l'arbre
        """
        (store, itersel) = selection.get_selected()
        if itersel is None:
            self.selected_node = None
            return

        self.selected_node = store.get_value(itersel,1)
        # On met à jour les propriétés
        self.update_properties()


    def update_properties(self):
        """Mets à jour la partie 'propriétés' de la fenêtre de dialogue
        (partie droite)
        """
        # on vérifie si l'objet peut-être modifié
        if self.selected_node is not None:
            if self.selected_node.is_editable():
                self.glade.get_widget('del_service_button').set_sensitive(1)
                self.glade.get_widget('edit_service_button').set_sensitive(1)
            else:
                self.glade.get_widget('del_service_button').set_sensitive(0)
                self.glade.get_widget('edit_service_button').set_sensitive(0)
#            self.prop_label.set_text(str(self.selected_node))
            self.group_store.clear()
            if self.selected_node.get_type() == "firewall/group" or self.selected_node.get_type() == "firewall/service":
                self.group_listview.set_sensitive(1)
                self.fill_group_liststore()
            else:
                self.group_listview.set_sensitive(0)
        else:
            self.glade.get_widget('del_service_button').set_sensitive(0)
            self.glade.get_widget('edit_service_button').set_sensitive(0)
            self.group_listview.set_sensitive(0)
#            self.prop_label.set_text("")
            self.group_store.clear()


    def init_group_liststore(self):
        """Initialise le modèle pour la liste des services d'un groupe.
        """
        self.group_store = gtk.ListStore(gobject.TYPE_STRING)
        self.group_listview.set_model(self.group_store)
        column = gtk.TreeViewColumn(_('services in group'),
                                    gtk.CellRendererText(),
                                    text = 0)
        self.group_listview.append_column(column)


    def fill_group_liststore(self):
        """Remplit le modèle de la liste des services du groupe.
        """
        if not self.selected_node:
            return
        index = 0
        if hasattr(self.selected_node, "services"):
            for serv in self.selected_node.services:
                iter_sel = self.group_store.append()
                self.group_store.set(iter_sel,0,serv)
                index += 1
        else:
            iter_sel =  self.group_store.append()
            self.group_store.set(iter_sel,0,str(self.selected_node))
            index += 1

    def edit_service(self,*args):
        """Ouvre la fenetre d'edition de service
        """
        if self.selected_node:
            if self.selected_node.get_type() == "firewall/group":
                # édition d'un groupe
                editdlg=ServiceGroupeDialog(self.glade_file,self.selected_node)
                editdlg.show_dialog()
            else:
                # édition d'un service
                editdlg=ServiceDialog(self.glade_file,self.selected_node)
                editdlg.show_dialog()


    def edit_service_on_click(self, widget, event, *args):
        """
            edition d'un service sur le double-click
        """
        # bouton gauche
        if event.button == 1:
            # double-clic
            if event.type == gtk.gdk._2BUTTON_PRESS:
               self.edit_service(widget, event, args)

    def del_service(self,*args):
        """supprime un service ou un groupe de la bibliothèque
        """
        if self.selected_node:
            # on vérifie si l'objet est encore utilisé
            if self.selected_node.used > 0:
                dial_error = create_error_dialog(_("object still used"),self.dlg)
                dial_error.show()
            else:
                # dialogue de confirmation
                dial_conf = create_yes_no_dialog(_("remove service confirmation") % self.selected_node.name)
                # dial_conf=gtk.MessageDialog(parent=self.dlg,buttons=gtk.BUTTONS_YES_NO)
                # dial_conf.label.set_text(
                dial_conf.set_default_response(gtk.RESPONSE_NO)
                ret = dial_conf.run()
                dial_conf.destroy()
                # on teste le retour du dialogue
                if ret == gtk.RESPONSE_YES:
                    selection = self.treeview.get_selection()
                    model, iter_sel = selection.get_selected()

                    if self.selected_node.get_type() == "firewall/group":
                        # suppression d'un groupe
                        # mise à jour des services correspondants
                        for service in self.selected_node.services:
                            service.used -= 1
                        library_store.service_groups.remove(self.selected_node)
                    else:
                        # suppresion d'un service
                        library_store.services.remove(self.selected_node)

                    self.treemodel.remove(iter_sel)

    def show_dialog(self, *args):
        self.treeview.set_size_request(250,300)
        self.dlg.show_all()


    def stay_alive(self, *args):
        self.close()
        return True


    def close(self, *args):
        self.dlg.hide()
        self.group_store.clear()

class ServiceGroupeDialog(Singleton):
    """Gère l'édition des groupes de services
    """

    def __init__(self,glade_file,groupe=None, parent=None):
        """Glade : l'objet XML glade
        parent : la fenêtre d'affichage des services
        """
        self.parent = parent
        self.glade_file = glade_file

        if 'dlg' not in self.__dict__:
            self.services_groupe = []
            self.glade = gtk.glade.XML(self.glade_file,"service_group_dialog", "editeur")
            self.dlg = self.glade.get_widget('service_group_dialog')
            self.name_entry = self.glade.get_widget('entryGroupName')
            self.combo = self.glade.get_widget('comboServices')
            store=gtk.ListStore(str)
            self.combo.set_model(store)
            self.combo.set_active(0)
            cell = gtk.CellRendererText()
            self.combo.pack_start(cell, True)
            self.combo.add_attribute(cell, 'text', 0)

            self.libelle_groupe = self.glade.get_widget('service_group_libelle')
            self.groupe_tree = self.glade.get_widget("treeview_group_services")

            self.init_treeview()

            handlers = {
                'on_service_group_dialog_response' : self.validate_edit,
                'on_service_group_dialog_close' : self.hide_and_flush,
                'on_service_group_dialog_delete_event' : self.stay_alive,
                'on_button_group_add_service_clicked' : self.add_service,
                'on_button_group_del_service_clicked' : self.del_service,
                }

            self.glade.signal_autoconnect(handlers)

        self.flush()
        self.groupe = groupe
        if groupe != None:
            # on édite un groupe existant
            self.name_entry.set_text(groupe.id)
            self.name_entry.set_sensitive(0)
            self.libelle_groupe.set_text(groupe.libelle)
            for serv in groupe.services:
                serv_name=str(serv.name)
                self.services_groupe.append(serv_name)
                self.remove_combo_by_name(serv_name)
            self.fill_treeview()


    def init_treeview(self):
        # initialisation du TreeView
        self.groupe_store = gtk.ListStore(gobject.TYPE_STRING)
        self.groupe_tree.set_model(self.groupe_store)
        renderer = gtk.CellRendererText()
        self.column = gtk.TreeViewColumn(_('chosen services'),renderer,text=0)
        self.groupe_tree.append_column(self.column)
        self.selection=self.groupe_tree.get_selection()
        self.selection.set_mode(gtk.SELECTION_SINGLE)

    def fill_treeview(self):
        #remplissage du treeview
        for serv_name in self.services_groupe:
            add_iter = self.groupe_store.append()
            self.groupe_store.set_value(add_iter,0,serv_name)

    def validate_group_name(self,*args,**kwargs):
        name = self.name_entry.get_text()
        nom_valid = 1
        if name != "":
            for group in library_store.service_groups:
                if group.id == name:
                    valid = 0
                    break
        else:
            nom_valid = 0

        if nom_valid:
            alerte=create_error_dialog(_('existing group'),self.dlg)
            alerte.show()
#            self.name_entry.set_text("")
            self.name_entry.grab_focus()

    def remove_combo(self, model, path, iterator, name):
        """
            callback appelé lors d'une iteration sur le store
        """
        path = [str(i) for i in path]
        iterator = model.get_iter_from_string(":".join(path))
        if name == model.get_value(iterator, 0):
            model.remove(iterator)

    def remove_combo_by_name(self, name):
        """
            rempli le combo_box des app_group
        """
        self.combo.get_model().foreach(self.remove_combo, name)
        self.combo.set_active(0)

    def add_service(self,*args):
        # mise à jour des listes de services
        iterator = self.combo.get_active_iter()
        if iterator == None:
            return
        store = self.combo.get_model()
        serv_name = store.get_value(iterator, 0)
        if serv_name != "":
            self.services_groupe.append(serv_name)
            # on met à jour la combo_box
            self.remove_combo_by_name(serv_name)

            # mise à jour du treeview
            add_iter = self.groupe_store.append()
            self.groupe_store.set_value(add_iter,0,serv_name)

    def del_service(self,*args):
        # mise à jour des listes de services
        # on récupère le service sélectionné
        (store,iterdel) = self.selection.get_selected()
        serv_name=self.groupe_store.get_value(iterdel,0)
        if serv_name != "":
            self.services_groupe.remove(serv_name)
            # on met à jour la combo_box
            self.combo.append_text(serv_name)
            # mise à jour du treeview
            self.groupe_store.remove(iterdel)


    def validate_edit(self,dial,response_code,*args):
        if response_code == gtk.RESPONSE_OK:
            if self.groupe == None:
                # vérification de l'id du groupe
                name = self.name_entry.get_text()
                nom_valid=1
                if name != "":
                    for group in library_store.service_groups:
                        if group.id == name:
                            nom_valid=0
                            break
                else:
                    nom_valid=0

                if nom_valid == 0:
                    alerte=create_error_dialog(_('existing group'),self.dlg)
                    alerte.show()
                    self.name_entry.set_text("")
                    self.name_entry.grab_focus()
                    return

            if len(self.services_groupe) == 0:
                alerte=create_error_dialog(_('empty group'),self.dlg)
                alerte.show()
                return

            # on récupère le libellé
            libelle_texte=self.libelle_groupe.get_text()
            #création de la liste des groupes
            new_service_list=[]
            for serv in library_store.services:
                if serv.name in self.services_groupe:
                    new_service_list.append(serv)

            if self.groupe == None:
              # nouveau groupe
                new_groupe=ServiceGroupe(name,libelle_texte,new_service_list)
                library_store.add_group(new_groupe)
                for service in new_service_list:
                    service.used += 1
                self.flush()
            else:
                # modification du groupe existant
                self.groupe.libelle = libelle_texte
                # mise à jour du nombre d'utilisation des anciens services
                for service in self.groupe.services:
                    service.used -= 1
                # mise à jour du nombre d'utilisation des nouveaux services
                for service in new_service_list:
                    service.used += 1
                # mise à jour de la liste de services
                self.groupe.services = new_service_list
                self.hide_and_flush()
                # mise a jour de la boite de dialogue d'affichage des service
            if self.parent != None:
               self.parent.init_model()
        else:
            self.hide_and_flush()

    def hide_and_flush(self, *args):
        self.dlg.hide()
        self.flush()

    def stay_alive(self,*args):
        self.hide_and_flush()
        return True

    def flush(self):
        """Réinitialise l'ensemble des champs de la fenêtre d'édition de service
        """
        self.name_entry.set_text("")
        self.libelle_groupe.set_text("")
        self.combo.get_model().clear()
        self.services_groupe = []
        for serv in library_store.services:
            self.combo.append_text(serv.name)
        self.groupe_store.clear()
        self.groupe = None
        self.name_entry.set_sensitive(1)

    def show_dialog(self):
        self.dlg.show_all()


class ServiceDialog(Singleton):
    """Gère l'édition de services
    """

    def __init__(self, glade_file, serv_edit=None, parent=None):
        """
        glade : l'objet XML glade
        """
        self.glade_file = glade_file
        self.glade = gtk.glade.XML(self.glade_file,"service_dialog", "editeur")
        self.parent = parent
        if 'dlg' not in self.__dict__:
            self.dlg = self.glade.get_widget('service_dialog')
            self.name_entry = self.glade.get_widget('service_name_entry')
            self.port_from_entry = self.glade.get_widget('service_port_from_entry')
            self.port_to_entry = self.glade.get_widget('service_port_to_entry')
            self.variable_port_entry = self.glade.get_widget('service_variable_port_entry')
            self.variable_service_port_radiobutton = self.glade.get_widget('variable_service_port_radiobutton')
            self.protocol_combo_entry = self.glade.get_widget('service_protocol_combo')
            self.protocol_combo_entry_model = set_combo_model(self.protocol_combo_entry, ['tcp', 'udp', 'esp', 'icmp'])
            self.libelle_textview = self.glade.get_widget('service_libelle_textview')
            self.variable_protocol_entry = self.glade.get_widget('service_variable_protocol_entry')
            self.port_from_entry.connect('activate', self.define_start_port)
            self.port_to_entry.connect('activate', self.define_end_port)
            self.protocol_radiobutton = self.glade.get_widget('service_protocol_radiobutton')
            self.variable_protocol_radiobutton = self.glade.get_widget('variable_service_protocol_radiobutton')

            self.ip_var_radiobutton = self.glade.get_widget('ip_var_radiobutton')

            handlers = {
                'on_service_dialog_response' : self.validate_edit,
                'on_service_dialog_delete_event' : self.stay_alive,
                }

            self.glade.signal_autoconnect(handlers)

        self.start_port = self.end_port = 0

        # pas forcement utile : c'est renseigne dans le fichier glade
        self.protocol_radiobutton.set_active(True)
        self.variable_protocol_radiobutton.set_active(False)

        self.service=serv_edit
        if serv_edit != None:
            self.name_entry.set_text(serv_edit.name)
            self.name_entry.set_sensitive(0)
            if len(serv_edit.port_list) == 1 and str(serv_edit.port_list[0]).startswith('%'):
#                self.port_from_entry.set_text('')
#                self.port_to_entry.set_text('')
                self.variable_port_entry.set_text(serv_edit.port_list[0])
            else:
                self.port_from_entry.set_text(str(serv_edit.port_list[0]))
                self.port_to_entry.set_text(str(serv_edit.port_list[-1]))

            self.libelle_textview.set_text(serv_edit.libelle)

            if serv_edit.protocol.startswith('%'):
                self.variable_protocol_radiobutton.set_active(True)
                self.variable_protocol_entry.set_text(serv_edit.protocol)
            else:
#                iterator = self.protocol_combo_entry_model.get_iter_from_string(serv_edit.protocol.upper())
#                self.protocol_combo_entry.set_active_iter(iterator)
                iterator = self.protocol_combo_entry_model.get_iter_root()
                if iterator:
                    while iterator:
                        if self.protocol_combo_entry_model.get_value(iterator,0).lower() == serv_edit.protocol.lower():
                            self.protocol_combo_entry.set_active_iter(iterator)
                        iterator = self.protocol_combo_entry_model.iter_next(iterator)

                #self.protocol_combo_entry.set_text(serv_edit.protocol)

    def define_start_port(self, *args):
        """Callback appelée après que le signal 'activate' ait été émis
        pour le port d'entrée
        """
        try:
            self.start_port = int(self.port_from_entry.get_text())
            if not self.port_to_entry.get_text():
                self.port_to_entry.set_text('%s'%self.start_port)
        except ValueError:
            dlg = create_error_dialog(_('invalid beginning port'))
            dlg.show_all()

    def define_end_port(self, *args):
        """Callback appelée après que le signal 'activate' ait été émis
        pour le port d'arrivée
        """
        try:
            self.end_port = int(self.port_to_entry.get_text())
            if self.end_port < self.start_port:
                raise InvalidPortListError
        except ValueError:
            dlg = create_error_dialog(_('invalid end port'))
            dlg.show_all()

        except InvalidPortListError:
            dlg = create_error_dialog(_('unordered ports'))
            dlg.show_all()

    def stay_alive(self,*args):
        self.hide_and_flush(*args)
        return True

    def hide_and_flush(self, *args):
        self.dlg.hide()
        self.flush()

    def flush(self):
        """Réinitialise l'ensemble des champs de la fenêtre d'édition de service
        """
        self.name_entry.set_text("")
        self.libelle_textview.set_text("")
        self.port_from_entry.set_text("")
        self.port_to_entry.set_text("")
        self.protocol_combo_entry_model = set_combo_model(self.protocol_combo_entry, [''])
        iterator = self.protocol_combo_entry_model.get_iter_root()
        self.protocol_combo_entry.set_active_iter(iterator)
        self.protocol_combo_entry_model = set_combo_model(self.protocol_combo_entry, ['tcp', 'udp', 'esp', 'icmp'])
        self.service = None
        self.name_entry.set_sensitive(1)
        self.start_port = self.end_port = 0
        self.variable_protocol_entry.set_text("")
        self.variable_port_entry.set_text("")

    def validate_edit(self, dialog, response_code):
        """Callback appelée lorsqu'on valide l'édition d'un service.
        """
        if response_code != gtk.RESPONSE_OK:
            self.dlg.hide()
            self.flush()
        else:
            try:
                if self.variable_service_port_radiobutton.get_active():
                    port_list = [make_template_string(self.variable_port_entry.get_text())]
                else:
                    self.start_port = int(self.port_from_entry.get_text())
                    self.end_port = int(self.port_to_entry.get_text())

                    if self.start_port > self.end_port:
                        raise ValueError
                    elif self.start_port != self.end_port:
                        port_list = range(self.start_port, self.end_port+1)
                    else:
                        port_list = [self.start_port]

                if self.variable_protocol_radiobutton.get_active():
                    protocol = make_template_string(self.variable_protocol_entry.get_text())
                else:
                    protocol = self.protocol_combo_entry.get_active_text()

                libelle = self.libelle_textview.get_text()
                name = self.name_entry.get_text()

                if not name:
                    raise EmptyNameError

                if self.service == None:
                    #vérification de la non existence du nom (nouveau service)
                    for serv in library_store.services:
                        if serv.name == name:
                            dlg_er = create_error_dialog(_('existing service'),self.dlg)
                            dlg_er.show()
                            return

                    new_service = Service(name, protocol, port_list, len(library_store.services), libelle)
                    library_store.add_service(new_service)
                    self.flush()
                else:
                    self.service.protocol=protocol
                    self.service.port_list=port_list
                    self.service.libelle=libelle
                    self.hide_and_flush()

            except ValueError:
                dlg = create_error_dialog(_('invalid ports'))
                dlg.show_all()
            except EmptyNameError:
                dlg = create_error_dialog(_('empty service'))
                dlg.show_all()
            else:
                self.start_port = self.end_port = 0

            if self.parent != None:
               self.parent.init_model()

    def show_dialog(self):
        self.dlg.show_all()

