# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################


"""
Module gérant les objets de bases manipulés par l'éditeur
Ces objets sont :
    - les flux
    - les zones
    - les extrémités
    - les services / groupes de services
    - les directives
"""

from itertools import chain

from era.tool.pairtype import extendabletype
from era.noyau.constants import *
from era.noyau.dpatterns import Observer, Observable
from era.noyau.erreurs import BadPortError, TagError, InvalidZoneError
from era.noyau.pool import library_store

# Classe DirectiveStore ########################################################
class DirectiveStore(Observable):
    """ Modèle pour le classeur de directives
    c'est à partir de maintenant que les flux sont réellement orientés
    """

    def __init__(self, directive_list=[], default_policy=True, src_zone=None, dest_zone=None):

        Observable.__init__(self)
        self.classeur_directives = []
        self.src_zone = src_zone
        self.dest_zone = dest_zone

        for directive in directive_list:
            self.add_directive(directive)
        # flag d'inversion de la politique par defaut
        self.default_policy = default_policy
        # permet de savoir si la politique par defaut est inversée
        self.reverse = False

    def get_non_standard_directives(self):
        """
            renvoie les directives non standard du classeur

        """
        non_standard = []
        for directive in self.classeur_directives:
            if directive.action_is_default():
                non_standard.append(directive)
        return non_standard

    def reverse_default_policy(self, reverse):
        """
            - change la directive par défaut
            - actualise le flag reverse
        """
        self.default_policy = not self.default_policy
        self.reverse = reverse
        self.notify_observers()

    def set_default_policy_to_false(self):
        self.default_policy = False
        self.reverse = False
        self.notify_observers()

    def set_default_policy_to_true(self):
        self.default_policy = True
        self.reverse = False
        self.notify_observers()

    def get_zones(self):
        """
            renvoie les zones src et dest du directive_store
        """
        return (self.src_zone, self.dest_zone)

    def add_directive(self, directive, load=0):
        """ Ajoute une directive au classeur
        """
        if load == 0:
            directive.set_priority(len(self.classeur_directives)+1)
        self.classeur_directives.append(directive)

        self.notify_observers()

    def del_directive(self, index):
        """ Supprime une directive du classeur
        """
        for directive in self.classeur_directives[index+1:]:
            directive.set_priority(directive.priority - 1)
        del(self.classeur_directives[index])
        self.classeur_directives.sort()
        self.notify_observers()

    def get_directive_index(self, directive):
        """
            renvoie l'index de la directive dans le classeur
        """
        return self.classeur_directives.index(directive)

    def swap(self, index1, index2):
        """Echange l'ordre des deux directives pointées par index1 et index2
        """
        self.classeur_directives[index2].set_priority(index1+1)
        self.classeur_directives[index1].set_priority(index2+1)
        self.classeur_directives.sort()
        self.notify_observers()

    def directive_list(self):
        """ renvoie la liste des directives de ce modèle
        """
        return self.classeur_directives

class FwObject:
    """Classe de base des objets du Firewall (directives, extremites, etc.)
    Cette classe définit l'ensemble des caractères communs de tous les objets
    du Firewall. En particulier, chaque objet peut-être 'hérité' d'un modèle,
    et être ou non 'éditable'.

    """
    __metaclass__ = extendabletype

    instanciable_attributes = ()

    def __init__(self, inherited = False, editable = True, mime_type = None):
        """
        inherited : définit si l'objet est hérité d'un modèle ou pas
        editable : définit si l'objet est modifiable ou pas (lié à l'import)
        mime_type : le mime_type de l'objet
        """
        self.inherited = inherited
        self.editable = editable
        self.mime_type = mime_type

    def is_inherited(self):
        """Pour savoir si cet objet a été hérité d'un modèle
        """
        return self.inherited

    def set_inherited(self, inherited):
        """Définit si l'objet est hérité d'un modèle ou pas
        """
        self.inherited = inherited

    def is_editable(self):
        """Renvoie True si l'objet est éditable, False sinon
        """
        return self.editable

    def set_editable(self, editable):
        """Fixe l'éditablilité de l'objet
        """
        self.editable = editable

    def get_type(self):
        """Renvoie le type de l'objet
        """
        return self.mime_type

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        return any([getattr(self, attrname, '').startswith('%%')
                    for attrname in self.instanciable_attributes])

    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        unspecified = set()
        for attrname in self.instanciable_attributes:
            value = getattr(self, attrname, '')
            if value.startswith('%%'):
                if value.endswith('%%'):
                    unspecified.add(value[2:-2])
                else:
                    unspecified.add(value[2:])
        return unspecified


# EXTREMITE #############################################################
class Extremite(FwObject):
    """ Classe représentant une extrémité
    """
    instanciable_attributes = ('name', 'netmask')

    # XXX FIXME : C'est moche le coup du all_zone, trouver mieux !!
    def __init__(self, zone, name, libelle, ip_list, netmask,
                 all_zone = False, subnet=0):
        """
        zone : la zone dans laquelle est incluse l'extrémité
        name : le nom de l'extrémité.
        libelle : un descriptif de l'extrémité
        ip_list : la liste des IPs
        netmask : le netmask de l'extrémité
        subnet : indique que l'extremite est un sous-réseau (permet d'utiliser le bon dialogue)
        all_zone : un booléen qui indique l'extrémité est l'ensemble de la zone
        """
        FwObject.__init__(self, False, True, "firewall/extremite")
        self.all_zone = all_zone
#        # Si l'extrémité représente toute la zone, elle ne doit pas
#        # être éditable
#        if self.all_zone:
#            self.set_editable(False)
        self.zone = zone
        if all_zone:
            self.name = self.zone.name
        else:
            self.name = name
        self.libelle = libelle
        self.ip_list = ip_list
        self.netmask = netmask
        self.subnet = subnet
        self.used = 0

    def __str__(self):
        l = ['Nom\t: %s'%self.name]
        l.append(u'Zone\t: %s'%self.zone.name)
        l.append(u'Libelle\t: %s'%self.libelle)
        l.append(u'Netmask\t: %s'%self.netmask)
        return '\n'.join(l)

    def __repr__(self):
        return self.name

    def __iter__(self):
        """Pour faciliter l'itération sur les ips
        """
        return iter(self.ip_list)

    def __cmp__(self, other):
        """
        Permet d'ordonner les extrémitées dans l'ordre alphabétique
        """
        if other is None:
            return 1
        return cmp(self.name, other.name)

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        if FwObject.needs_instanciation(self):
            return True
        return any([ip for ip in self.ip_list if ip.startswith('%%')])

    # FIXME a mettre dans un visiteur ayant cette responsabilité seule
    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        unspecified = FwObject.unspecified_vars(self)
        for ip in self.ip_list:
            if ip.startswith('%%'):
                if ip.endswith('%%'):
                    unspecified.add(ip[2:-2])
                else:
                    unspecified.add(ip[2:])
        return unspecified

class Directive(FwObject):
    """
    Classe définissant un ensemble de règles
    """

    def __init__(self, src_list, dest_list, service, action, attrs,
                 nat_extr=None, nat_port=None, is_group=False,
                 src_inv=False, dest_inv=False, serv_inv=False,
                 libelle="pas de description", time=None,
                 user_group=None, mark=None, app_group=None):
        """
        src_list : la liste des extrémités sources
        dest_list : la liste des extrémités destinations
        service : le service (ou groupe) associé à la directive
        action : un masque définissant les actions
        attrs : une valeur définissant si la règle est optionnelle et /ou active
        src_inv : flag d'inversion de l'extrémité source
        dest_inv : flag d'inversion de l'extrémité destination
        serv_inv : flag d'inversion du service
        libelle : un descriptif de la directive (et donc du groupe de règles)
        user_group : groupe d'utilisateurs (peut être None)
        mark : marquage des paquets [opérateur, marque]
        """
        FwObject.__init__(self, False, True, "firewall/directive")
        assert service is not None, _("service undefined error")
        self.libelle = libelle
        self.time = time
        self.src_list = src_list
        self.src_inv = src_inv
        self.dest_list = dest_list
        self.dest_inv = dest_inv
        self.service = service
        self.serv_inv = serv_inv
        self.action = action
        self.attrs = int(attrs)
        self.priority = 0
        self.accept = 0
        self.nat_extr = nat_extr
        self.nat_port = nat_port
        self.mark = mark
        self.service_group = is_group
        self.user_group = user_group
        self.app_group = app_group
        self.tag = ""
        self.ipsec = 0
        # ipsets exceptions
        self.exceptions = []

    def action_is_default(self):
        """Retourne vrai si l'action est DENY ou ALLOW, false sinon
        """
        return self.action <= 2

    def is_active(self):
        """renvoie 'Vrai' si la règle est active
        """
        return self.attrs & DIRECTIVE_ACTIVE == DIRECTIVE_ACTIVE

    def set_active(self, attrs):
        """
        attrs: 1 ou 3
        """
        if attrs not in (1, 3):
            raise Exception('Attrs doit être 1 ou 3 dans set_active')

        #si active
        if self.attrs & DIRECTIVE_ACTIVE == DIRECTIVE_ACTIVE:
            #demande de passer en inactif
            if attrs == 1:
                self.attrs = self.attrs ^ DIRECTIVE_ACTIVE
        #si inactive
        else:
            #demande de passer en actif
            if attrs == 3:
                self.attrs = self.attrs ^ DIRECTIVE_ACTIVE

    def is_optional(self):
        """ renvoie 'Vrai' si la règle est optionnelle
        """
        return self.attrs & DIRECTIVE_OPTIONAL == DIRECTIVE_OPTIONAL

    def get_app_group(self):
        """
            :return:fwobjects/AppGroup
        """
        return library_store.get_app_group(self.app_group)

    def is_logged(self):
        """ renvoie 'Vrai' si on doit journaliser les évènements relatifs à
        cette directive
        """
        return self.attrs & DIRECTIVE_LOG == DIRECTIVE_LOG

    def is_authenticated(self):
        """
            renvoie True si la directive est authentifiée
            (càd possède un groupe d'utilisateur)
        """
        if self.user_group != None:
            return True
        return False

#    def is_ipsec(self):
#        """ renvoie 'Vrai' si on doit generer des parametres ipsec
#        """
#        return self.attrs & DIRECTIVE_IPSEC == DIRECTIVE_IPSEC

    def is_marked(self):
        """ renvoie 'Vrai' si on doit marquer cette règle
        """
        return self.mark is not None

    def is_hidden(self):
        """ renvoie 'Vrai' si on doit marquer cette règle
        """
        return self.attrs & DIRECTIVE_HIDDEN == DIRECTIVE_HIDDEN

    def get_source_zone(self):
        """ Renvoie la zone source de la directive
        """
        return self.src_list[0].zone

    def get_dest_zone(self):
        """ Renvoie la zone destination de la directive
        """
        return self.dest_list[0].zone

    def is_montante(self):
        """ renvoie 'Vrai' si la directive part d'une zone moins sécurisée vers
        une zone plus sécurisée
        """
        return self.get_source_zone() < self.get_dest_zone()

#    def get_directive_type(self):
#        """Renvoie le type de la directive (UP ou DOWN)
#        Méthode crée par convenance
#        """
#        src_zone = self.get_source_zone()
#        dest_zone = self.get_dest_zone()
#        if src_zone <= dest_zone:
#            return ACTION_DENY
#        else:
#            return ACTION_ALLOW

    def set_priority(self, priority):
        """Définit la priorité de la directive
        priority : un entier définissant la priorité
        """
        self.priority = priority

    def set_time(self, time):
        """Définit les options horaires de la directive
        priority : un dictionnaire avec les bonnes options
        """
        self.time = time

    def update_attrs(self, col_number):
        if col_number == LOG_COLUMN:
            self.attrs = self.attrs ^ DIRECTIVE_LOG
        elif col_number == MARK_COLUMN:
            self.attrs = self.attrs ^ DIRECTIVE_MARK
        elif col_number == ACTIVE_COLUMN:
            self.attrs = self.attrs ^ DIRECTIVE_ACTIVE
        elif col_number == OPTIONAL_COLUMN:
            self.attrs = self.attrs ^ DIRECTIVE_OPTIONAL
        else:
            print "Warning : type d'action inconnu"

    def action_to_string(self):
        d = {ACTION_DENY : _('deny'),
             ACTION_ALLOW : _('allow'),
             ACTION_REDIRECT : _('Redirect'),
             ACTION_DNAT : _('dnat'),
             ACTION_MASK : _('mask'),
             ACTION_FORWARD : _('Forward')
             }

        return d[self.action]

    def __cmp__(self, other):
        """ Permet de comparer la priorité de deux directives
        dir1 < dir2 si dir1.priority < dir2.priority
        """
        if other is None:
            return 1
        return cmp(self.priority, other.priority)

    def __str__(self):
        l = ['Directive : ']
        l.append('\t src = %s'%self.src_list)
        l.append('\t dest = %s'%self.dest_list)
        l.append('\t service = %s'%self.service)
        l.append('\t priority = %s'%self.priority)
        l.append('\t action = %s'%self.action_to_string())
        l.append('\t log = %s'%self.is_logged())
        l.append('\t mark = %s'%self.is_marked())
        l.append('\t active = %s'%self.is_active())
        l.append('\t optional = %s'%self.is_optional())

        return '\n'.join(l)

    def set_tag(self, tag):
        """Affectation d'un attribut de tag la directive
        si celle-ci est optionnelle.
        """
        self.tag = tag

    def get_tag(self):
        """Renvoie le tag de la directive s'il existe.
        """
        if hasattr(self, "tag"):
            return self.tag
        else :
            raise TagError

    def set_attrs(self, attrs):
        """active ou désactive une directive optionnelle
        """
        self.attrs = int(attrs)

    def get_attrs(self):
        """renvoie l'attrs
        """
        return self.attrs

    def get_libelle(self):
        """renvoie le libelle
        """
        return self.libelle

    def as_list(self):
        """ Renvoie une liste représentant la directive.
        L'élément N de la liste renvoyée est la valeur de l'attribut se trouvant
        à la colonne N du modèle de l'arbre GTK
        """
        # si les extremites sont inversées, on affiche un "!" devant
        if self.src_inv == 1:
            src_list = "!"+str(self.src_list)
        else:
            src_list = str(self.src_list)

        if self.dest_inv == 1:
            dest_list = "!"+str(self.dest_list)
        else:
            dest_list = str(self.dest_list)

        if self.serv_inv == 1:
            service = "!" + str(self.service)
        else:
            service = str(self.service)

        if hasattr(self.time, 'name'):
            timerange = self.time.name
        else:
            timerange = _("undefined")

        if hasattr(self.user_group, 'name'):
            user_group = self.user_group.name
        else:
            user_group = _("undefined")

        if self.app_group != None:
            app_group = self.app_group
        else:
            app_group = _("undefined")

        if self.mark != None and self.mark[0] is not None:
            # operation (set, and, or) et marque
            mark = "%s (%s)" % (self.mark[1],self.mark[0])
        else:
            mark = _("undefined")


        return [self.priority,src_list, dest_list, service, self.action_to_string(),timerange,user_group, app_group,
                mark, self.is_logged(), self.is_optional(), self]

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        if self.nat_extr:
            checks = (self.src_list, self.dest_list, self.service, [self.nat_extr])
        else:
            checks = (self.src_list, self.dest_list, self.service)
        return any([obj.needs_instanciation() for obj in chain(*checks)])

    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        if self.nat_extr:
            checks = (self.src_list, self.dest_list, self.service, [self.nat_extr])
        else:
            checks = (self.src_list, self.dest_list, self.service)
        return set(chain(*[obj.unspecified_vars() for obj in chain(*checks)]))

# ______________________________________________________________________

class Service(FwObject):
    instanciable_attributes = ('protocol',)

    def __init__(self, name, protocol, port_list, id, libelle) :
        """
        name : the service's name
        protocol : the concerned protocol
        port_list : a list of all concerned ports
        id : l'identifiant du service
        libelle : description du service
        """
        FwObject.__init__(self, False, True, "firewall/service")
        self.name = name
        self.protocol = protocol
        self.port_list = port_list
        self.id = id
        self.libelle = libelle
        self.used = 0

    def __eq__(self, other):
        """ On redéfinit __eq__ en plus de __cmp__ pour s'assurer
        que seules deux services ayant même nom sont égaux.
        """
        if other is None:
            return False
        return self.name == other.name

    def __cmp__(self, other):
        """ Permet de trier alphabétiquement deux services
        """
        if other is None:
            return 1
        return cmp(self.name, other.name)

    def __iter__(self):
        """XXX **Bof** : pour ne pas avoir à différencier un service d'un groupe
        """
        return iter([self])


    def __str__(self):
        return "service : %s, protocol = %s, port = %s"%(self.name, self.protocol,
                str(self.port_list).replace('u', ''))


    def __repr__(self):
        return "<%s %s>" % (self.__class__.__name__, self.name)

class ServiceGroupe(FwObject):
    """Groupe de service
    """
    def __init__(self, id, libelle, services=[]) :
        """
        id : le nom (qui sert aussi d'identifiant) du groupe
        services : liste des services du groupe
        libelle : le descriptif du groupe
        """
        FwObject.__init__(self, False, True, "firewall/group")
        self.id = id
        self.services = services
        self.libelle = libelle
        self.name = self.id
        self.used = 0

    def __eq__(self, other):
        """ On redéfinit __eq__ en plus de __cmp__ pour s'assurer
        que seules deux services ayant même nom sont égaux.
        """
        if other is None:
            return False
        return self.id == other.id

    def __cmp__(self, other):
        """ Permet de trier alphabétiquement deux services
        """
        if other is None:
            return 1
        return cmp(self.id, other.id)

    def add_service(self, service) :
        """Ajoute un service au groupe de services
        et vérifie que le service n'est pas déjà existant
        """
        if (service in self.services) :
            raise ExistingServiceError, _('existing service')
        else :
            self.services.append(service)

    def __str__(self):
        liste_serv = []
        longueur = len(self.services)
        for serv in self.services[0:3]:
            liste_serv.append(str(serv.name))
        if longueur > 4:
            points = ', ...'
        else:
            points = ''
        return "groupe %s : %s%s"%(self.id, ', '.join(liste_serv), points)

    def __iter__(self):
        """Pour simplifier l'itération sur les services du groupe de services"""
        return iter(self.services)

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        return any([s.needs_instanciation() for s in self.services])

    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        return set(chain(*[s.unspecified_vars() for s in self.services]))

class UserGroup(FwObject):
    """
        Utilisateur ou groupe utilisateur
        nufwobject groupe utilisateur pour le filtrage authentifie
    """
    def __init__(self, id, name):
        """
        id : entier (gid du groupe)
        name : le descriptif du groupe
        """
        FwObject.__init__(self, False, True, "firewall/user_group")
        self.id = id
        self.name = name
        # used : indique si l'objet est référencé dans un autre objet
        self.used = 0

    def __eq__(self, other):
        """ On redéfinit __eq__ en plus de __cmp__ pour s'assurer
        que seules deux services ayant même nom sont égaux.
        """
        if other is None:
            return False
        return self.name == other.name

    def __cmp__(self, other):
        """ Permet de trier alphabétiquement deux services
        """
        if other is None:
            return 1
        return cmp(self.name, other.name)

    def __str__(self):
        return "<user_group:%s>" % self.name

    def __repr__(self):
        return "<user_group:%s> %s" % (self.name,id(self))

# ______________________________________________________________________

class Zone(FwObject):
    """ Classe Zone
    """
    instanciable_attributes = ('ip', 'network', 'netmask', 'interface')

    def __init__(self, name, level, ip, network="0.0.0.0",
                 netmask="255.255.255.255", interface="lo"):
        """name : nom de la zone"""
        FwObject.__init__(self, False, True, "firewall/zone")
        self.name = name
        self.level = int(level)
        self.ip = ip
        self.network = network
        self.netmask = netmask
        self.interface = interface
        self._listeners = []

    def __eq__(self, other):
        """ On redéfinit __eq__ en plus de __cmp__ pour s'assurer
        que seules deux zones ayant même nom (et même niveau)
        sont égales. Sinon, avec __cmp__, deux zones de noms différents, mais
        de niveaux identiques seraient considérées comme égales.
        L{era.tests.test_zone.test_zone_equal}
        """
        if other is None:
            return False

        return self.name == other.name and self.level == other.level

    def strict_eq(self, other):
        if other is None:
            return False
        return self.level == other.level

    def strict_cmp(self, other):
        """
            comparaison stricte (pas d'ordonnancement en fonction du nom)
        """
        if other is None:
            return 1
        if self.level == other.level:
            return 0
        else:
            return cmp(self.level, other.level)

    def __cmp__(self, other):
        """ L{era.tests.test_zone.test_zone_compare}
        Permet de comparer le niveau de deux zones
        zone1 < zone2 si zone1.level < zone2.level
        """
        if other is None:
            return 1
        if self.level == other.level:
            return cmp(self.name,other.name)
        else:
            return cmp(self.level, other.level)

    def __str__(self):
        return "<zone:%s>" % self.name

    def __repr__(self):
        return "<zone:%s> %s" % (self.name,id(self))

    def add_listener(self, listener):
        """ Ajouter un auditeur de l'objet zone
        """
        self._listeners.append(listener)

    def notify_listeners(self):
        """ Méthode qui prévient tous les auditeurs d'un changement
        """
        for l in self._listeners:
            l.zone_changed()

class Flux(FwObject,Observer):

    def __init__(self, zoneA, zoneB):
        """
        Attention : le flux n'est pas orienté ! Il n'y a pas de différence entre :
        f = Flux(z1,z2) et f = Flux(z2,z1)
        zoneA et zoneB : les deux extrémités du flux
        """
        FwObject.__init__(self, False, True, "firewall/flux")

        self.zoneA = zoneA
        self.zoneB = zoneB

        # self.directives = []
        down_default_policy=True
        if zoneA.strict_eq(zoneB):
            down_default_policy=False
        self.up_directives_store = DirectiveStore(default_policy=False,
             src_zone=self.get_lowlevel_zone(), dest_zone=self.get_highlevel_zone())
        # down_default_policy est fixé lorsque l'attribut n'est pas présent pas dans le xml
        self.down_directives_store = DirectiveStore(default_policy=down_default_policy,
             src_zone=self.get_highlevel_zone(), dest_zone=self.get_lowlevel_zone())
        self.up_directives_store.register_observer(self)
        self.down_directives_store.register_observer(self)

        self._listeners = []

    def __eq__(self, other):
        """Retourne vrai si les deux flux s'appliquent aux mêmes zones
        """
        if other is None:
            return 0
        return sorted([other.zoneA.name, other.zoneB.name]) == \
               sorted([self.zoneA.name, self.zoneB.name])

    def append_directives_from_flux(self, other):
        """
            ajoute les directives du flux other au flux self
        """
        for directive in other.up_directives_store.classeur_directives:
            self.up_directives_store.add_directive(directive)
        for directive in other.down_directives_store.classeur_directives:
            self.down_directives_store.add_directive(directive)


    def set_up_default_policy(self, boolean):
        """
            permet de changer l'attribut de default_policy
            pour le store montant
            **IMPORTANT** : cette méthode ne doit être appelée
            qu'au chargement du fichier, en effet, le reverse ne sera jamais
            remis à False

        """
        if boolean != self.up_directives_store.default_policy:
            self.up_directives_store.default_policy = boolean
            self.up_directives_store.reverse = True

    def set_down_default_policy(self, boolean):
        """
            permet de changer l'attribut de default_policy
            pour le store descendant
            **IMPORTANT** : cette méthode ne doit être appelée
            qu'au chargement du fichier, en effet, le reverse ne sera jamais
            remis à False
        """
        if boolean != self.down_directives_store.default_policy:
            self.down_directives_store.default_policy = boolean
            self.down_directives_store.reverse = True

    def up_directives_model(self):
        """ Renvoie le modèle associé aux directives montantes
        """
        return self.up_directives_store
        # return [d for d in self.directives if d.is_montante()]


    def down_directives_model(self):
        """ Renvoie le modèle associé aux directives descendantes
        """
        return self.down_directives_store
        # return [d for d in self.directives if not d.is_montante()]


    def up_directives_list(self):
        """ Renvoie la liste des directives montantes de ce flux
        Cette méthode est un raccourci pour :
        f.up_directives_model().directive_list()
        """
        return self.up_directives_store.directive_list()


    def down_directives_list(self):
        """ Renvoie la liste des directives descendantes de ce flux
        Cette méthode est un raccourci pour :
        f.down_directives_model().directive_list()
        """
        return self.down_directives_store.directive_list()

    def directives_list(self):
        """
        renvoie les directives up **et** down
        """
        return self.up_directives_store.directive_list().extend(self.down_directives_store.directive_list())

    def update(self):
        """Appelé lorsque l'un des directive_store a changé
        """
        self.notify_listeners()

    def get_highlevel_zone(self):
        """
            récupère la zone de niveau de sécurité le plus élevé
            dans le flux
        """
        if self.zoneA > self.zoneB:
            return self.zoneA
        elif self.zoneB > self.zoneA:
            return self.zoneB
        else:
            # les deux zones ont mme niveau et mme nom
            raise InvalidZoneError, "les flux ne peuvent pas avoir la mme zone"

    def get_lowlevel_zone(self):
        """
            récupère la zone de niveau de sécurité le moins élevé
            dans le flux
        """
        if self.zoneA < self.zoneB:
            return self.zoneA
        elif self.zoneB < self.zoneA:
            return self.zoneB
        else:
            # les deux zones ont mme niveau et mme nom
            raise InvalidZoneError, "les flux ne peuvent pas avoir la mme zone"

    def get_up_reverse(self):
        """
            renvoie True s'il y a une inversion de la politique par defaut
            dans le flux montant
        """
        return  self.up_directives_store.reverse

    def get_down_reverse(self):
        """
            renvoie True s'il y a une inversion de la politique par defaut
            dans le flux descendant
        """
        return self.down_directives_store.reverse

    def add_listener(self, listener):
        """ Ajouter un auditeur de l'objet flux
        """
        # assert listener not in self._listeners
        self._listeners.append(listener)

    def notify_listeners(self):
        """ Méthode qui prévient tous les auditeurs d'un changement
        """
        # FIXME : supprimer l'ancien mvc
        for l in self._listeners:
            l.flux_changed()

    def __str__(self):
        return "Flux : [%s <===> %s]"%(self.zoneA.name, self.zoneB.name)

    def __repr__(self):
        return "Flux : [%s <===> %s]"%(self.zoneA.name, self.zoneB.name)

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        # py2.3
        return any([d.needs_instanciation()
                    for d in chain(self.up_directives_list(), self.down_directives_list())])

    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        return set(chain(*[d.unspecified_vars()
                           for d in chain(self.up_directives_list(), self.down_directives_list())
                           if d.needs_instanciation()]))

# ______________________________________________________________________

class Range(FwObject):

    def __init__(self, range_dict) :
        """
        name : the service's name
        range_dict:  parameters for time module
        """
        FwObject.__init__(self, False, True, "firewall/range")
        self.used = 0
        self.name = range_dict['name']
        for opt in ('timestart','timestop','datestart','datestop','weekdays'):
            setattr(self, opt, range_dict.get(opt, ''))

    def __eq__(self, other):
        """ On redéfinit __eq__ en plus de __cmp__ pour s'assurer
        que seules deux services ayant même nom sont égaux.
        """
        if other is None:
            return False
        return self.name == other.name

    def __cmp__(self, other):
        """ Permet de trier alphabétiquement deux services
        """
        if other is None:
            return 1
        return cmp(self.name, other.name)

    def __str__(self):
        txt= ["%s :" % self.name]
        if self.weekdays != '':
            txt.append(" (%s)" % self.weekdays)
        if self.timestart != '':
            txt.append(" %s - %s" % (self.timestart, self.timestop))
        if self.datestart != '':
            txt.append(" %s - %s" % (self.datestart, self.datestop))
        return ''.join(txt)
# ______________________________________________________________________

class QosClass(FwObject):
    """classe de qos """

    def __init__(self, id, bandwidth, libelle="pas de description", priority=1, zone=None) :
        """
        id : l'identifiant à partir duquel le bitmask pourra être récupéré
        bandwidth : un pourcentage
        zone : le nom de zone à laquelle la qos appartient
        """
        FwObject.__init__(self, False, True, "firewall/qosclass")
        self.id = id
        self.priority = priority
        self.bandwidth = float(bandwidth)
        self.libelle = libelle
        self.zone = zone

    def __eq__(self, other):
        """ On redéfinit __eq__ en plus de __cmp__ pour s'assurer
        que seules deux services ayant même nom sont égaux.
        """
        if other is None:
            return False
        return self.id == other.id

    def __cmp__(self, other):
        """ Permet de trier alphabétiquement deux services
        """
        if other is None:
            return 1
        return cmp(self.id, other.id)

    def set_bandwidth(self, bandwidth):
        """permet de remettre le bandwidth en float
        (la bande passante est en string quand elle vient de la fenêtre de répartition)
        """
        self.bandwidth = float(bandwidth)

    def save(self, stream, indent=0):
        """ Méthode qui sauvegarde une classe de qos au format interne
        stream : le flux dans lequel on écrit
        """
        stream.write('%s<qos id="%s" priority="%s" zone="%s" libelle="%s" bandwidth="%f"/>\n'% \
                     (" "*indent, self.id, self.priority, self.zone, self.libelle, self.bandwidth))

    def __str__(self):
        return "qosclass : %s, bandwidth = %f"%(self.id, self.bandwidth)

    def __repr__(self):
        return "<%s %s>" % (self.__class__.__name__, self.zone)
# ______________________________________________________________________

class ApplicationGroup(FwObject):
    """
        nufwobject groupe d'applications (ex: firefox)
        pour le filtrage applicatif
        une directive ne manipule **que** des groupes d'application
    """
    def __init__(self, name, description="pas de description", applications=[]):
        """
            :name: nom du groupe
            :applications: initialise les applications
            :description: libellé (pour l'interface)
            :used: flag pour savoir si une Directive utilise ce groupe d'applications
        """
        FwObject.__init__(self, False, True, "firewall/application_group")
        self.name = name
        self.description = description
        self.applications = applications
        self.used = 0

    def add_application(self, application):
        """
            ajout d'une application
            :application: fwobj Application
        """
        self.applications.append(application)

    def remove(self, application):
        """
            suppression d'une application
        """
        self.applications.remove(application)

    def get_applications(self):
        """
            :return: tous les chemins de toutes les applications du groupe
        """
        return self.applications

    def get_paths(self):
        """
            :return: tous les chemins des applications de ce groupe
        """
        return library_store.get_applications_paths(self.applications)

    def __str__(self):
        return """%s :
%s""" % (str(self.description), "\n".join(self.applications))

    def __repr__(self):
        return "%s %s" % (self.__class__.__name__, self.name)

class Application(FwObject):
    """
        nufwobject object application à partir duquel sont créés
        les groupes d'application pour le filtrage applicatif

        Une application peut :
        - être intégrée à un groupe d'application afin d'être
          associée à une directive
        - être présente indépendamment dans le xml afin d'être utilisée dans le frontend

    """

    def __init__(self, name, description="pas de description", paths=[]):
        """
            :name: nom
            :description: libellé de des applications (pour le frontend)
            :used: flag pour savoir si un AppGroup utilise cette application
        """
        FwObject.__init__(self, False, True, "firewall/application")
        self.name = name
        self.description = description
        self.paths = paths
        self.used = 0

    def __str__(self):
        return """%s:
%s""" % (str(self.description), "\n".join(self.paths))

    def __repr__(self):
        return "<%s %s>" % (self.__class__.__name__, self.name)

    def get_paths(self):
        return self.paths
# ______________________________________________________________________
# FIXME A IMPORTER DE eole-firewall (twobjects.py)
class TcpWrapper:
    def __init__(self, ip, name):
        self.ip = ip
        self.name = name

    def __str__(self):
        if self.ip.strNetmask() == '255.255.255.255':
            ip = str(self.ip)
        else:
            ip = str(self.ip.strNormal(2))
        return ('%s:%s' % (self.name, ip))
