# -*- coding: UTF-8 -*-

"""
Tests unitaires de cohérence de la matrice de flux
"""

import autopath
from era.noyau.erreurs import *
from era.noyau.fwobjects import *
from era.noyau.models import *
    
FLUX_ZONE_A = """[Flux : [zB <===> zA], Flux : [zC <===> zA]]"""

FLUX_ZONE_C_REMOVED = """[Flux : [zB <===> zA]]"""


def setup_module(module):
    _init_zones(module)
    _init_services(module)
    _init_extremites(module)
    _init_directives(module)
    _init_matrice(module)
    
def _init_zones(module):
    module.zoneA = Zone('zA', 30, '127.0.0.1')
    module.zoneB = Zone('zB', 31, '127.0.0.1')
    module.zoneC = Zone('zC', 32, '127.0.0.1')
    
    module.zones = {'zoneA': module.zoneA,
                    'zoneB': module.zoneB,
                    'zoneC': module.zoneC
                    }
    
def _init_services(module):
    module.s1 = Service("serv1", "TCP", [80], "serv1", "libelle_Internet")
    module.s2 = Service("serv2", "TCP", [480], "serv2", "libelle2")
    module.s3 = Service("Youpi", "UDP", [4513], "serv3", "libelle3")
    module.s4 = Service("echo-request", "ICMP", [0], "echo-request", "règle icmp echo-request")
    module.sg1 = ServiceGroupe("Groupe1", "test de groupe de services",[s1,s2])
    module.services = {"serv1" : module.s1, "serv2" : module.s2, "Youpi" : module.s3, "icmp" : module.s4}
    module.liste_services = [module.s1,module.s2,module.s3,module.s4]
    module.groupe_services = {"grp1" : module.sg1}

def _init_extremites(module):
    module.e1 = Extremite(zones['zoneA'],"extr1", "libelle1", ["10.0.0.1"], "255.255.255.255")
    module.e2 = Extremite(zones['zoneA'],"extr2", "libelle2", ["10.0.0.2"], "255.255.255.255")
    module.e3 = Extremite(zones['zoneB'],"extr3", "libelle3", ["10.0.0.3"], "255.255.255.255")
    module.e4 = Extremite(zones['zoneC'],"extr4", "libelle4", ["10.0.0.4"], "255.255.255.255")
    module.extremites = {"extr1" : module.e1,"extr2" : module.e2,"extr3" : module.e3,"extr4" : module.e4}
    module.groupe_extremites = {'grp1' : [module.e1,module.e2,module.e3], 'grp2' : [module.e2,module.e3,module.e4]}

def _init_directives(module):
    module.d0 = Directive([module.extremites['extr1']], [module.extremites['extr3']],
                        module.services['serv1'], ACTION_DENY, 0)
    
    module.d1 = Directive([module.extremites['extr3']],[module.extremites['extr1']], 
                        module.services['serv1'], ACTION_ALLOW, 0)

    directives = [module.d0, module.d1]


def _init_matrice(module) :
    matrice = MatrixModel()

    for zone in zones :
        matrice.add_zone(zones[zone])
    matrice.flux[0].up_directives_store.add_directive(module.d0)
    matrice.flux[0].up_directives_store.add_directive(module.d1)
    module.matrice = matrice


def test_flux() :
    """Test sur les flux reliés à la zone A
    """
    assert str(matrice.get_flux_list(zones["zoneA"])) ==  FLUX_ZONE_A

def test_del_zone() :
    """Test de réorganisation des flux sur la suppression d'une zone dans la matrice
    """
    matrice.remove_zone(zones["zoneC"])
    assert str(matrice.get_flux_list(zones["zoneA"])) == FLUX_ZONE_C_REMOVED


def test_directive_montante() :
    """Test de vérification du positionnement d'une directive dans la matrice
    """
    assert d0.is_montante()
    assert not d1.is_montante()

## need instanciation


def _modify_and_test_instanciation(obj):
    for attr in obj.instanciable_attributes:
        initial = getattr(obj, attr)
        setattr(obj, attr, '%%some_variable%%')
        assert matrice.needs_instanciation()
        setattr(obj, attr, initial)
        # on revérifie juste que tout est remis dans l'ordre
        assert not matrice.needs_instanciation()
    

def test_needs_instanciation():
    assert matrice.needs_instanciation() == False
    _modify_and_test_instanciation(zones['zoneA'])
#    _modify_and_test_instanciation(services['serv1'])

def test_unspecified_vars():
    assert matrice.unspecified_vars() == set()
    zoneA = zones['zoneA']
    zoneA.ip = '%%ip0%%'
    zoneA.network = '%%network0%%'
    assert matrice.unspecified_vars() == set(['ip0', 'network0'])
    matrice.flux[0].up_directives_store.add_directive(d0)
    serv1 = services['serv1']
    serv1.protocol = '%%protocol0%%'
    assert matrice.unspecified_vars() == set(['ip0', 'network0', 'protocol0'])


#def test_unspecified_vars_extremites():
#    assert matrice.unspecified_vars() == set()
#    extr = extremites['extr1']
#    extr.ip_list[0] = '%%ip1%%'
#    extr.netmask = '%%netmask1%%'
#    assert matrice.unspecified_vars() == set(['ip1', 'netmask1'])
#    
