# -*- coding: UTF-8 -*-
###########################################################################
# 
# Eole NG  
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr 
#  
###########################################################################

"""
Module gérant les fenêtres liées aux extrémités
"""

import gtk, gtk.glade
import gobject

from era.noyau.dpatterns import Singleton
from era.noyau.pool import library_store
from era.noyau.fwobjects import Extremite

from gtk_models import create_machine_treestore, create_all_machines_treestore
from ihm_utils import create_error_dialog, create_info_dialog, create_yes_no_dialog


class ExtremiteListviewDialog(Singleton):

    def __init__(self, glade_file, zone,liste_zones=None):
        self.glade_file = glade_file
        self.zone = zone

        if 'dlg' not in self.__dict__:
            self.glade = gtk.glade.XML(self.glade_file,"extremite_listview_dialog", "editeur")
            self.dlg = self.glade.get_widget('extremite_listview_dialog')
            self.treeview = self.glade.get_widget('extremite_listview')
            self.prop_label = self.glade.get_widget('extremite_prop_label')
            self.ip_listview = self.glade.get_widget('extremite_ip_listview')
            self.ip_listview.set_headers_visible(1)
            self.init_treeview_column()
            self.init_ip_liststore()
            selection = self.treeview.get_selection()
            selection.set_mode(gtk.SELECTION_SINGLE)
            selection.connect('changed',self.item_selected)
            handlers = {'on_extremite_listview_dialog_response' : self.close,
                        'on_extremite_listview_dialog_delete_event' : self.stay_alive,
                        'on_button_del_extremite_clicked' : self.del_extremite,
                        'on_button_edit_extremite_clicked' : self.edit_extremite
                        }
            self.glade.signal_autoconnect(handlers)
            
        self.init_model(liste_zones)
        

    def init_ip_liststore(self):
        """Initialise le modèle pour la liste des IPs.
        """
        self.ip_store = gtk.ListStore(gobject.TYPE_STRING)
        self.ip_listview.set_model(self.ip_store)
        column = gtk.TreeViewColumn(_('ip list'),
                                    gtk.CellRendererText(),
                                    text = 0)
        self.ip_listview.append_column(column)
        
        
    def fill_ip_liststore(self):
        """Remplit le modèle de la liste d'IPs.
        """
        if not self.selected_node:
            return
        index = 0
        for ip in self.selected_node.ip_list:
            iter_add = self.ip_store.append()
            self.ip_store.set(iter_add, 0, ip)
            #self.ip_store.emit('row-inserted',(index,),iter_add)
            index += 1


    def init_model(self,zones=None):
        """Initialise le modèle associé à l'arbre des extrémités
        """
        if self.zone is not None:
            self.treemodel = create_machine_treestore(zone = self.zone)
        else:
            self.treemodel= create_all_machines_treestore(zones=zones)

        self.treeview.set_model(self.treemodel)
        if self.zone is not None:
            # si on a une seule zone, on déroule le sous arbre correspondant par défaut
            self.treeview.expand_all()

    
    def init_treeview_column(self):
        column = gtk.TreeViewColumn(_('available extremite'),
                                    gtk.CellRendererText(),
                                    text = 0)
        self.treeview.append_column(column)

    
    def item_selected(self, selection, *args):
        """Callback appelée lorsqu'on clique sur une extrémité dans l'arbre
        """
        model, iter_sel = selection.get_selected()
        if iter_sel is not None:
            path = model.get_path(iter_sel)
            self.selected_node = model[path][1]
        else:
            self.selected_node = None
        # On met à jour les propriétés
        self.update_properties()
        

    def edit_extremite(self, *args):
        """edition de l'extrémité choisie (ouvre un dialogue)
        """
        if self.selected_node:
            # édition de l'extremite
            if self.selected_node.subnet == 1:
                # l'extrémité est un sous réseau
                editdlg=DialogReseauManager(self.glade_file,extremite=self.selected_node)
            else:
                editdlg=DialogMachinesManager(self.glade_file,extremite=self.selected_node)
            editdlg.show_dialog()


    def del_extremite(self, *args):
        """suppression de l'extremite choisie
        """
        if self.selected_node:
            # on vérifie si l'objet est encore utilisé
            if self.selected_node.used > 0:
                dial_error = create_error_dialog(_("object still used"),self.dlg)
                dial_error.show()
            else:
                # dialogue de confirmation
                dial_conf=create_yes_no_dialog(_("remove extremity confirmation"))
                #dial_conf=gtk.MessageDialog(parent=self.dlg,buttons=gtk.BUTTONS_YES_NO)
                #dial_conf.label.set_text(
                dial_conf.set_default_response(gtk.RESPONSE_NO)
                ret=dial_conf.run()
                dial_conf.destroy()
                # on teste le retour du dialogue
                if ret == gtk.RESPONSE_YES:
                    selection = self.treeview.get_selection()
                    model, iter_del = selection.get_selected()
                    # on vérifie que cette extremite n'est pas présente dans une directive
                    #suppression effective de l'extremite
                    del library_store.extremites[self.selected_node.name]
                    self.treemodel.remove(iter_del)


    def update_properties(self):
        """Mets à jour la partie 'propriétés' de la fenêtre de dialogue
        (partie droite)
        """
        # on vérifie si l'objet peut-être modifié
        if self.selected_node is not None:
            if self.selected_node.is_editable():
                self.glade.get_widget('button_del_extremite').set_sensitive(1)
                self.glade.get_widget('button_edit_extremite').set_sensitive(1)
            else:
                self.glade.get_widget('button_del_extremite').set_sensitive(0)
                self.glade.get_widget('button_edit_extremite').set_sensitive(0)
            self.prop_label.set_text(str(self.selected_node))
            self.ip_store.clear()
            self.fill_ip_liststore()
        else:
            self.glade.get_widget('button_del_extremite').set_sensitive(0)
            self.glade.get_widget('button_edit_extremite').set_sensitive(0)
            self.prop_label.set_text("")
            self.ip_store.clear()
        

    def show_dialog(self, *args):
        self.treeview.set_size_request(450,300)
        self.dlg.show_all()


    def stay_alive(self, *args):
        self.close()
        return True

    
    def close(self, *args):
        self.dlg.hide()
        


class DialogReseauManager(Singleton):
    """Classe du dialogue de création d'un sous réseau
    # FIXME changer les nom 'ip' en 'network'
    """
    
    def __init__(self, glade_file, zone=None, extremite=None):
        """
        glade : objet XML glade
        """
        self.zone=zone
        self.extremite=extremite
        self.glade_file = glade_file

        if 'dlg' not in self.__dict__:
            self.glade = gtk.glade.XML(self.glade_file,"extremite_reseau_dialog", "editeur")
            self.dlg = self.glade.get_widget('extremite_reseau_dialog')
            self.name_entry = self.glade.get_widget('extremite_reseau_name_entry')
            self.libelle = self.glade.get_widget('extremite_reseau_libelle')
            self.network_var_entry = self.glade.get_widget('ip_subnet_variable_entry')
            self.network_radiobutton = self.glade.get_widget('ip_subnet_radiobutton')
            self.network_var_radiobutton = self.glade.get_widget('ip_subnet_variable_radiobutton')
            self.network_box = self.glade.get_widget('extremite_reseau_ip_hbox')
            self.netmask_var_entry = self.glade.get_widget('netmask_subnet_variable_entry')
            self.netmask_radiobutton = self.glade.get_widget('netmask_subnet_radiobutton')
            self.netmask_var_radiobutton = self.glade.get_widget('netmask_subnet_variable_radiobutton')
            self.netmask_box = self.glade.get_widget('extremite_reseau_netmask_hbox')


            for i in range(1,5):
                ip_entry = self.glade.get_widget('extremite_reseau_ip_entry%d'%i)
                netmask_entry = self.glade.get_widget('extremite_reseau_netmask_entry%d'%i)
                ip_entry.connect('focus_out_event', self.set_mask,netmask_entry)

            handlers = {
                'on_extremite_reseau_dialog_response':self.validate_edit,
                'on_extremite_reseau_dialog_close':self.hide_and_flush,
                'on_extremite_reseau_dialog_delete_event':self.stay_alive,
                "on_ip_subnet_radiobutton_toggled" : self.network_type_changed,
                "on_ip_subnet_variable_radiobutton_toggled" : self.network_type_changed,
                "on_netmask_subnet_radiobutton_toggled" : self.netmask_type_changed,
                "on_netmask_subnet_variable_radiobutton_toggled" : self.netmask_type_changed
                }

            self.glade.signal_autoconnect(handlers)

        self.template_network = False
        self.template_netmask = False
        self.network_radiobutton.set_active(True)
        self.netmask_radiobutton.set_active(True)

        if self.zone != None:
            if self.zone.network.startswith('%'):
                self.set_ip('ip','...')
            else:
                self.set_ip('ip',self.zone.network)

            if self.zone.network.startswith('%'):
                self.set_ip('netmask','255.255.255.255')
            else:
                self.set_ip('netmask',self.zone.netmask)
            # blocage des entrées à ne pas modifier en fonction du netmask
            #nb=0
            #for mask in self.zone.netmask.split("."):
            #    nb+=1
            #    if mask=="255":
            #        # on empêche la modification de cette partie de l'ip
            #        self.glade.get_widget('extremite_reseau_ip_entry%d'%(nb)).set_sensitive(0)
            #        self.glade.get_widget('extremite_reseau_netmask_entry%d'%(nb)).set_sensitive(0)
            #    else:
            #        # on autorise
            #        self.glade.get_widget('extremite_reseau_ip_entry%d'%(nb)).set_sensitive(1)
            #        self.glade.get_widget('extremite_reseau_netmask_entry%d'%(nb)).set_sensitive(1)
                    
            
        # dans le cas d'une édition, on récupère les données de l'extrémité
        if self.extremite is not None:
            self.zone = self.extremite.zone
            self.name_entry.set_text(self.extremite.name)
            self.name_entry.set_sensitive(0)
            self.liste_ips = self.extremite.ip_list
            self.libelle.set_text(self.extremite.libelle)

            if self.extremite.ip_list[0].startswith('%') :   
                self.network_var_radiobutton.set_active(True)
                self.network_var_entry.set_text(self.extremite.ip_list[0])
            else :
                self.set_ip('ip',self.extremite.ip_list[0])

            if self.extremite.netmask.startswith('%') :   
                self.netmask_var_radiobutton.set_active(True)
                self.netmask_var_entry.set_text(self.extremite.netmask)
            else :
                self.set_ip('netmask',self.extremite.netmask)


    def set_mask(self,entry_ip,event,entry_netmask):
        if entry_ip.get_text()=="0":
            entry_netmask.set_text("0")

    def get_ip(self, type = 'ip'):
        """ Returns a string representing the IP contained in the text entries
        """
        assert type in ('ip', 'netmask')

        entry_var = self.glade.get_widget(type+"_subnet_variable_entry")

        if type == 'ip':
            template = self.template_network
        else :
            template = self.template_netmask

        if template:
            result = make_template_string(entry_var.get_text())
        else :


            widget_names = ['extremite_reseau_%s_entry%d'%(type, i) for i in range(1,5)]

            ip_parts = []
            for w_name in widget_names:
                entry = self.glade.get_widget(w_name)
                # vérifiaction de la validité des données
                ## FIXME (faire assert 0 <= int(entry.get_text()) <= 255):
                ip_parts.append(entry.get_text())

            result = '.'.join(ip_parts)

        return result
   
    def set_ip(self, type = 'ip', ip="255.255.255.255"):
        """ Remplit les champs de saisie d'une ip avec la valeur spécifiée
        """
        assert type in ('ip', 'network', 'netmask')
        widget_names = ['extremite_reseau_%s_entry%d'%(type, i) for i in range(1,5)]
        ip_parts=ip.split('.')
        
        part=0

        for w_name in widget_names:
            entry = self.glade.get_widget(w_name)
            entry.set_text(ip_parts[part])
            part=part + 1

    def network_type_changed(self, *args):
        """Callback appelée pour déterminer quel type de network est entré
        """
        template_network = not self.network_radiobutton.get_active()
        self.network_var_entry.set_sensitive(template_network)
        self.network_box.set_sensitive(not template_network)
        self.template_network = template_network

    def netmask_type_changed(self, *args):
        """Callback appelée pour déterminer quel type de netmask est entré
        """
        template_netmask = not self.netmask_radiobutton.get_active()
        self.netmask_var_entry.set_sensitive(template_netmask)
        self.netmask_box.set_sensitive(not template_netmask)
        self.template_netmask = template_netmask
        
    def validate_edit(self,dialog,response_code):
        """Validation du dialogue
           on vérifie les données saisies dans le cas ou la réponse est RESPONSE_OK
           sinon, on cache le dialogue et on réinitialise les champs
        """
        if response_code == gtk.RESPONSE_OK:
            #on vérifie la validité du sous réseau saisi
            zone=self.zone
            nom=self.name_entry.get_text()
            libelle=self.libelle.get_text()
            ip=self.get_ip()
            netmask=self.get_ip('netmask')
            
            # vérification de la présence du nom
            if nom == "":
                alerte=create_error_dialog(_('fill the name'),self.dlg)
                alerte.show()
                self.name_entry.grab_focus()
                return
                
            
            # vérification ip et netmask
            ip_valide = 1
            try:
                if self.template_network == False :
                    for nb in ip.split('.'):
                        if (int(nb) > 255) or (int(nb) < 0):
                            ip_valide = 0
                if self.template_netmask == False :
                    for nb in netmask.split('.'):
                        if (int(nb) > 255) or (int(nb) < 0):
                            ip_valide = 0
            except ValueError:
                ip_valide = 0
                        
            if ip_valide==0:
                alerte=create_error_dialog(_('not in range'),self.dlg)
                alerte.show()
                self.glade.get_widget('extremite_reseau_ip_entry1').grab_focus()
                return
        

            if self.extremite is None:
                # création d'une nouvelle extrémité
                
                # on vérifie qu'elle n'existe pas déjà
                for ext in library_store.extremites.values():
                    if ext.name == nom:
                        alerte=create_error_dialog(_('existing name'),self.dlg)
                        alerte.show()
                        self.name_entry.grab_focus()
                        return
                new_extr=Extremite(zone,nom,libelle,[ip],netmask,subnet=1)
                # enregistrement de l'extrémité dans la librairie
                library_store.add_extremite(new_extr)
                # boite de dialogue
                dial=create_info_dialog(_("added extremite"))
                dial.show_all()
                # 
                self.flush()
            
            else:
                #édition d'une extrémité existante
                self.extremite.libelle=libelle
                self.extremite.ip_list=[ip]
                self.extremite.netmask=netmask
                self.hide_and_flush()
        else:
            self.hide_and_flush()
            

    def hide_and_flush(self,*args):
        """remise à 0 des champs et disparition de la fenetre
        """
        self.dlg.hide()
        self.flush()
    
    def flush(self):
        """réinitialisation des zones de saisie
        """
        # on réinitialise tous les champs
        self.name_entry.set_text("")
        self.libelle.set_text("")
        if self.zone != None:
            if self.zone.network.startswith('%'):
                self.set_ip('ip','...')
            else:
                self.set_ip('ip',self.zone.network)

            if self.zone.network.startswith('%'):
                self.set_ip('netmask','255.255.255.255')
            else:
                self.set_ip('netmask',self.zone.netmask)
        self.extremite = None
        self.name_entry.set_sensitive(1)
        # on réinitialise les champs de templates
        self.network_var_entry.set_text('')
        self.network_radiobutton.set_active(True)
        self.netmask_var_entry.set_text('')
        self.netmask_radiobutton.set_active(True)
        
    
    def stay_alive(self,*args):
        """court circuit de la destruction de la fenetre
        """
        self.hide_and_flush()
        return True

    def show_dialog(self):
        self.dlg.show_all()



class DialogMachinesManager(Singleton):
    """Classe du dialogue de création d'une liste de machines
    """
    
    def __init__(self, glade_file, zone=None, extremite=None):
        """
        glade : objet XML glade
        """
        self.zone=zone
        self.extremite=extremite
        self.liste_ips = []
        self.glade_file = glade_file
        
        if 'dlg' not in self.__dict__:
            self.glade = gtk.glade.XML(self.glade_file,"extremite_machines_dialog", "editeur")
            self.dlg = self.glade.get_widget('extremite_machines_dialog')
            self.name_entry = self.glade.get_widget('extremite_machines_name_entry')
            self.libelle = self.glade.get_widget('extremite_machines_libelle')
            self.ip_entry1 = self.glade.get_widget('extremite_machines_ip_entry1')
            self.ip_entry2 = self.glade.get_widget('extremite_machines_ip_entry2')
            self.ip_entry3 = self.glade.get_widget('extremite_machines_ip_entry3')
            self.ip_entry4 = self.glade.get_widget('extremite_machines_ip_entry4')
            self.ip_list_tree = self.glade.get_widget('extremite_machines_ip_list')
            self.ip_list_tree.connect("row-activated", self.ip_list_button_pressed)
            self.ip_box = self.glade.get_widget('extremite_ip_box')
            
            self.ip_radiobutton = self.glade.get_widget('ip_radiobutton2')
            self.ip_var_entry = self.glade.get_widget('ip_variable_entry1')
            
            self.init_treeview()
            handlers = {
                'on_extremite_machines_dialog_response':self.validate_edit,
                'on_extremite_machines_dialog_close':self.hide_and_flush,
                'on_extremite_machines_dialog_delete_event':self.stay_alive,
                'on_extremite_machines_ip_add_clicked':self.add_ip,
                'on_extremite_machines_ip_entry4_activate':self.add_ip,
                'on_ip_variable_entry_activate' : self.add_ip,
                'on_ip_radiobutton_toggled' : self.ip_type_changed,
                'on_ipvar_toggled' : self.ip_type_changed,
                }
            
            self.glade.signal_autoconnect(handlers)


        # dans le cas d'une édition, on récupère les données de l'extrémité
        if self.extremite is not None:
            self.zone = self.extremite.zone
            self.name_entry.set_text(self.extremite.name)
            self.name_entry.set_sensitive(0)
            self.liste_ips = self.extremite.ip_list
            self.libelle.set_text(self.extremite.libelle)
        
        self.template_ip = False
        if self.zone != None:
            if self.zone.network.startswith('%'):
                self.set_ip('...')
            else:
                self.set_ip(self.zone.network)
            # blocage des entrées à ne pas modifier en fonction du netmask
            #nb=0
            #for mask in self.zone.netmask.split("."):
            #    nb+=1
            #    if mask=="255":
            #        # on empêche la modification de cette partie de l'ip
            #        self.glade.get_widget('extremite_machines_ip_entry%d'%(nb)).set_sensitive(0)
            #    else:
            #        # on autorise la modification
            #        self.glade.get_widget('extremite_machines_ip_entry%d'%(nb)).set_sensitive(1)
            
        # mise à jour du TreeView
        self.fill_treeview(self.liste_ips)


    def init_treeview(self):
        self.store = gtk.ListStore(gobject.TYPE_STRING)
        self.ip_list_tree.set_model(self.store)
        renderer = gtk.CellRendererText()
        column = gtk.TreeViewColumn(_('ip list'),renderer,text=0)
        self.ip_list_tree.append_column(column)


    def fill_treeview(self,liste):
        """fonction d'initialisation des ips dans le treeview
        (cas de l'édition d'une extrémité existante)
        """
        for ip in liste:
            iter_sel=self.store.append()
            self.store.set_value(iter_sel,0,ip)


    def get_ip(self):
        """ Returns a string representing the IP contained in the text entries
        """
        widget_names = ['extremite_machines_ip_entry%d'%(i) for i in range(1,5)]
        # FIXME : comment est renseigné template_ip ? 
        template = self.template_ip
        if template:
            entry_var = self.glade.get_widget("ip_variable_entry1")
            result = make_template_string(entry_var.get_text())
        else :
            ip_parts = []
            for w_name in widget_names:
                entry = self.glade.get_widget(w_name)
                ip_parts.append(entry.get_text())
            result = '.'.join(ip_parts)
        
        return result
   
    def set_ip(self, ip="255.255.255.255"):
        """ Remplit les champs de saisie d'une ip avec la valeur spécifiée
            '255.255.255.255' par défaut
        """
        widget_names = ['extremite_machines_ip_entry%d'%(i) for i in range(1,5)]
        ip_parts=ip.split('.')
        if len(ip_parts) == 4:
            part=0
            for w_name in widget_names:
                entry = self.glade.get_widget(w_name)
                entry.set_text(ip_parts[part])
                part=part + 1
        else:
            dialog = create_error_dialog(_("error in the xml file"))
            dialog.run()
            dialog.destroy()
            
    def add_ip(self,*args):
        """ ajout d'une ip à la liste des machines de l'extrémité
        """
        #vérification de l'ip, ajout à la liste, et mise à jour du treeview

        ip_valide = 1
        # vérification des ip
        # FIXME : d'où vient template_ip ?
        if self.template_ip:
            ip = self.ip_var_entry.get_text()
            ip = make_template_string(ip)
        else:
            ip = self.get_ip()
	    ip_parts = []
            for nb in ip.split('.'):
	        try:
		  ip_num_tmp=int(nb.strip())
		  ip_parts.append(str(ip_num_tmp))
		except:
		  ip_num_tmp=0
		  ip_valide=0
		nb=ip_num_tmp
                if (nb > 255) or (nb < 0) or (ip_valide == 0):
                    ip_valide=0

            if ip_valide == 0:
                alerte=create_error_dialog(_('not in range'),self.dlg)
                alerte.show()
                self.glade.get_widget('extremite_machines_ip_entry1').grab_focus()
                return
            else:
                ip='.'.join(ip_parts)
            
        # on vérifie que l'ip n'est pas déjà présente dans cette extrémité
        for previous_ip in self.liste_ips:
            if previous_ip == ip:
                alerte=create_error_dialog(_('ip in list'),self.dlg)
                alerte.show()
                self.glade.get_widget('extremite_machines_ip_entry1').grab_focus()
                return  
            
        # ajout de l'ip à la liste
        self.liste_ips.append(ip)
        # ajout au treeview
        iter_add=self.store.append()
        self.store.set_value(iter_add,0,ip)
        if not self.template_ip:
            self.glade.get_widget('extremite_machines_ip_entry1').set_text("")
            self.glade.get_widget('extremite_machines_ip_entry2').set_text("")
            self.glade.get_widget('extremite_machines_ip_entry3').set_text("")
            self.glade.get_widget('extremite_machines_ip_entry4').set_text("")
            self.glade.get_widget('extremite_machines_ip_entry1').grab_focus()
        else:
            self.glade.get_widget('ip_variable_entry1').set_text("")
            self.glade.get_widget('ip_variable_entry1').grab_focus()

    def ip_list_button_pressed(self, treeview, event, column):
        """Callback appelée lors d'un clic de souris sur le treeview.
        On l'utilise pour capturer le double-clic.
        """
        self.del_ip(column) 

    def del_ip(self, column):
        selection=self.ip_list_tree.get_selection()
        (store,iterdel) = selection.get_selected()
        #selected_ip = store.get_value(iterdel,0)
        selected_ip = self.store.get_value(iterdel,0)
        self.store.remove(iterdel)
        self.liste_ips.remove(selected_ip)

    def ip_type_changed(self, *args):
        """Callback appelée pour déterminer quel type d'ip est entré
        """
        template_ip = not self.ip_radiobutton.get_active()
        self.ip_var_entry.set_sensitive(template_ip)
        self.ip_box.set_sensitive(not template_ip)
        self.template_ip = template_ip
                

    def validate_edit(self,dialog,response_code):
        """Validation du dialogue
           on vérifie les données saisies dans le cas ou la réponse est RESPONSE_OK
           sinon, on cache le dialogue et on réinitialise les champs
        """
        if response_code == gtk.RESPONSE_OK:
            # on vérifie la validité du sous réseau saisi
            zone=self.zone
            nom=self.name_entry.get_text()
            libelle=self.libelle.get_text()
            ip=self.get_ip()
            
            # vérification de la présence du nom
            if nom == "":
                alerte=create_error_dialog(_('fill the name'),self.dlg)
                alerte.show()
                self.name_entry.grab_focus()
                return
            
            if self.extremite == None:
                # vérification de la non-existence de l'extrémité
                if nom in library_store.extremites:
                    alerte=create_error_dialog(_('existing name'),self.dlg)
                    alerte.show()
                    self.name_entry.grab_focus()
                    return

                #création d'une nouvelle extrémité
                new_extr=Extremite(zone,nom,libelle,self.liste_ips,"255.255.255.255")
                library_store.add_extremite(new_extr)
                dial=create_info_dialog(_("added extremite"))
                dial.show_all()
                self.flush() 

            else:
                #édition d'une extrémité existante
                self.extremite.libelle=libelle
                # FIXME : liste_ips ça vient d'où ?
                self.extremite.ip_list=self.liste_ips
                self.hide_and_flush()
        else:
            self.hide_and_flush()


    def stay_alive(self,*args):
        """court circuit de la destruction de la fenetre
        """
        self.hide_and_flush()
        return True


    def hide_and_flush(self,*args):
        """remise à 0 des champs et disparition de la fenetre
        """
        self.dlg.hide()
        self.flush()
    

    def flush(self):
        """ réinitialisation des champs de saisie
        """
        # on réinitialise tous les champs
        self.name_entry.set_text("")
        self.libelle.set_text("")
        self.liste_ips = []
        self.store.clear()
        self.fill_treeview(self.liste_ips)
        if self.zone.network.startswith('%'):
            self.set_ip('...')
        else:
            self.set_ip(self.zone.network)
        self.extremite = None
        self.name_entry.set_sensitive(1)
        self.ip_var_entry.set_text('')
        self.ip_radiobutton.set_active(True)
    
    def show_dialog(self):
        self.dlg.show_all()


# XXX FIXME : mettre cette fonction ailleurs !
def make_template_string(value):
    """Rajoute les %% avant et / ou après value
    """
    return '%%' + value.replace('%','')
    
