# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
Module chargé de gérer les fenêtres liées aux directives.

"""
import gtk, gtk.glade

from era.noyau.constants import bad_chars
from era.noyau.dpatterns import Observer, Singleton
from era.noyau import fwobjects
from gtk_models import create_objects_treestore, create_extremite_treemodel, \
     create_directive_treemodel, set_empty_combo_model, set_combo_model
from ihm_utils import *
from ipsets import DirectiveIpsets
from era.noyau.constants import *
from era.noyau.pool import library_store
from era.noyau.erreurs import BadPortError, TagError
from era.noyau.path import GID_GUEST

class DirectiveSummaryView(Observer):
    """ Vue pour l'ensemble des règles
    """

    def __init__(self, glade_file, ctrl, directive_store):
        """
        glade : l'objet XML glade
        ctrl : le controller pour cette vue
        """
        self.glade_file = glade_file
        self.glade = gtk.glade.XML(self.glade_file,"directive_summary_dialog", "editeur")
        self.ctrl = ctrl
        # Création de la vue et du modèle
        if directive_store is None:
#            self.set_model(fwobjects.DirectiveStore())
            raise Exception, "le classeur de directive est None"
        else:
            self.set_model(directive_store)
        # XXX FIXME : BEURK !
        self.ctrl.model = self.model
        self.gtk_model = create_directive_treemodel()
        # la fenêtre principale
        self.main_window = self.glade.get_widget('directive_summary_dialog')
        # affichage des zones du directive store
        self.label = self.glade.get_widget('directive_summary_label')
        src_zone, dest_zone = directive_store.get_zones()
        self.main_window.set_title(_('directive summary') + ' : %s -> %s' % (src_zone.name, dest_zone.name))
        # la liste des règles
        self.treeview = self.glade.get_widget('directive_summary_treeview')

        self.default_policy = self.glade.get_widget('reverse_defaut_policy')
        self.default_policy.set_active(directive_store.reverse)
        if src_zone.inherited or dest_zone.inherited:
            self.default_policy.set_sensitive(False)
        # initialisation des données du treeview
        self.init_treeview()
        self.ctrl.gtk_model = self.gtk_model
        self.fill_treeview()

        self.treeview.set_headers_visible(1)
        handlers = {
            "on_directive_summary_dialog_response" : self.hide_window,
            "on_directive_new_item_activate" : (self.ctrl.add_directive, directive_store),
            "on_directive_summary_del_button_clicked" : (self.ctrl.del_directive, self.treeview, self, self.main_window),
            "on_close_item_activate" : self.hide_window,
            "on_directive_summary_dialog_close" : self.hide_window,
            "on_directive_summary_delete_event" : self.hide_window,
            "on_directive_summary_down_button_clicked" : (self.ctrl.directive_down, self.treeview),
            "on_directive_summary_up_button_clicked" : (self.ctrl.directive_up, self.treeview),
            "on_directive_summary_treeview_button_press_event" : self.ctrl.button_pressed,
            "on_directive_summary_add_button_clicked" : (self.ctrl.add_directive, directive_store),
            "on_edit_button_clicked": (self.ctrl.directive_edit, self.treeview),
            "on_reverse_defaut_policy_toggled": (self.reverse_default_policy, self.treeview),
            "on_directive_summary_treeview_drag_end": (self.ctrl.reorder_treeview),
            }

        self.glade.signal_autoconnect(handlers)

    def init_treeview(self):
        # création du treeview des directives
        col_names = [_("PRIORITY"), _("SRC"), _("DEST"), _("SERVICE"), _("ACTION"), _("TIMERANGE"),
                     _("USER_GROUP"), _("APPLICATION"), _("MARK"), _("LOG"), _("OPTIONAL")]
        col_index = 0
        for col_name in col_names:
            if col_index < 9:
                renderer = gtk.CellRendererText()
                column = gtk.TreeViewColumn(col_name, renderer, text = col_index)
            else:
                renderer = gtk.CellRendererToggle()
                column = gtk.TreeViewColumn(col_name, renderer, active = col_index)
            self.treeview.append_column(column)
            col_index += 1

        self.treeview.set_model(self.gtk_model)

    def fill_treeview(self):
        for directive in self.model.directive_list():
            self.gtk_model.append(directive.as_list())

    def set_model(self, model):
        """
        """
        self.model = model
        self.model.register_observer(self)

    def show_window(self):
        """ Affiche la fenêtre de dialogue
        """
        self.main_window.show_all()

    def hide_window(self, *args):
        """ Cache la fenêtre de dialogue
        """
        self.main_window.hide()
        return False
    def reverse_default_policy(self, default_policy, treeview, *args):
        return_code =self.ctrl.reverse_default_policy(default_policy, treeview)
        if not return_code:
            self.hide_window()


    def update(self):
        """ Mets à jour la liste des directives affichées
        """
        pass

class DirectiveManager:
    """ Classe qui joue le rôle du Controller pour la gestion des directives.
    """
    def __init__(self, glade_file):
        """
        glade : l'objet XML glade
        """
        self.glade_file = glade_file
        # self.model = fwobjects.DirectiveStore()
        self.model = None
        self.gtk_model = None

    def reverse_default_policy(self, default_policy, treeview, *args):
        """

            callback appelée lors du clic sur le checkbutton d'inversion de
            la politique par defaut
        """
        dialog = create_yes_no_dialog(_("All the standard directives are going to be removed. Are you sure ?"))
        dialog.show_all()
        dialog.set_default_response(gtk.RESPONSE_NO)
        ret = dialog.run()
        dialog.destroy()
        # on teste le retour du dialogue
        if ret == gtk.RESPONSE_YES:
            reverse_status = default_policy.get_active()
            self.model.reverse_default_policy(reverse_status)
            self.del_standard_directives(treeview)
            self.reorder_treeview(treeview)
            return True
        else:
            return False


    def reorder_treeview_and_model(self, liststore, path, iterator, *args):
        """
            réorganise le treeview après un déplacement par glisser-déposer
        """
        # l'iterateur va de 0 a n ; la priorite de 1 a n+1
        priority = liststore.get_path(iterator)[0]+1
        liststore.set_value(iterator, 0, priority)
        directive = liststore.get_value(iterator, 11)
        directive.set_priority(priority)

    def reorder_treeview(self, treeview, *args):
        """
            callback pour réorganiser le treeview par glisser-déposer
        """
        model = treeview.get_model()
        model.foreach(self.reorder_treeview_and_model)

    def del_standard_directives(self, treeview):
        """
            supprime toutes les directives standard du classeur
            une directive standard est tout ce qui est en ALLOW, DENY
        """
        for directive in self.model.get_non_standard_directives():
            self.model.del_directive(self.model.get_directive_index(directive))
            treeview.get_model().foreach(self.remove_in_treeview, directive)

    def remove_in_treeview(self, model, path, iterator, directive):
        """
            supprime une directive dans le treeview
        """
        path = [str(i) for i in path]
        iterator = model.get_iter_from_string(":".join(path))
        if directive == model.get_value(iterator, 11):
            model.remove(iterator)

    def add_directive(self, item,  directive_store):
        """ Lance la fenêtre d'édition de règle
        """
        zone1, zone2 = directive_store.get_zones()
        directive_dlg = DirectiveDialog(self.glade_file, self.model,
                                        zone1, zone2, self)
        directive_dlg.show_dialog()

    def del_directive(self,item,treeview,view,dialog,*args):
        """Supprime la directive selectionnée
        """
        selection = treeview.get_selection()
        store, iter_sel = selection.get_selected()
        if iter_sel is not None:
            # on vérifie si la directive est éditable ou non (directives importées)
            index = store.get_path(iter_sel)[0]
            selected_dir = store[index][-1]
            if selected_dir.is_editable() == 0:
                dial_alerte=create_error_dialog(_('non editable selection'))
                dial_alerte.show()
            else:
                # demande de confirmation et suppression de la directive
                dial_conf = create_yes_no_dialog(_('remove confirmation'))
                # dial_conf=gtk.MessageDialog(dialog,gtk.DIALOG_MODAL,buttons=gtk.BUTTONS_YES_NO)
                # dial_conf.label.set_text(
                dial_conf.set_default_response(gtk.RESPONSE_NO)
                ret = dial_conf.run()
                dial_conf.destroy()
                # on teste le retour du dialogue
                if ret == gtk.RESPONSE_YES:
                    # suppression dans le classeur de directives
                    # libération des services et extremites
                    selected_dir.service.used -= 1
                    for extr in selected_dir.src_list:
                        extr.used -= 1
                    for extr in selected_dir.dest_list:
                        extr.used -= 1
                    if selected_dir.time != None:
                        selected_dir.time.used -= 1
                    if selected_dir.user_group != None:
                        selected_dir.user_group.used -= 1
                    self.model.del_directive(index)
                    # mise à jour de l'affichage
                    store.remove(iter_sel)
                    self.gtk_model.clear()
                    view.fill_treeview()

    def swap(self, index1, index2):
        """Echange l'ordre des deux directives pointées par index1 et index2
        """
        row1 = self.gtk_model[index1]
        row2 = self.gtk_model[index2]
        col_index = 1
        # On inverse l'ensemble des contenus des colonnes, sauf pour la première
        # qui représente la priorité
        # XXX on ne peut pas faire row[1:] sur un TreeRowModel
        for col_index in range(1,len(row1)):
            val = row1[col_index]
            row1[col_index] = row2[col_index]
            row2[col_index] = val

        self.model.swap(index1, index2)

    # XXX : ListStore.swap() n'est pas défini
    def directive_up(self, button, treeview):
        """Callback appelée pour remonter une directive
        """
        selection = treeview.get_selection()
        model, iter_up = selection.get_selected()
        line_index = model.get_path(iter_up)[0]
        if line_index > 0:
            self.swap(line_index-1, line_index)
            # on sélectionne la ligne à sa nouvelle position
            path_prev=model.get_path(iter_up)
            path_new=(path_prev[0]-1,)
            selection.select_path(path_new)

    def directive_down(self, button, treeview):
        """Callback appelée pour redescendre une directive
        """
        selection = treeview.get_selection()
        model, iter_down = selection.get_selected()
        line_index = model.get_path(iter_down)[0]

        if line_index < (len(model)-1):
            self.swap(line_index, line_index+1)
            # on sélectionne la ligne à sa nouvelle position
            path_prev=model.get_path(iter_down)
            path_new=(path_prev[0]+1,)
            selection.select_path(path_new)

    def button_pressed(self, treeview, event):
        """Callback appelée lors d'un clic de souris sur le treeview.
        On l'utilise pour capturer le double-clic.
        """
        # bouton gauche
        if event.button == 1:
            # double-clic
            if event.type == gtk.gdk._2BUTTON_PRESS:
                self.directive_edit(treeview)

    def directive_edit(self, treeview):
        """
          callback sur le bouton simplement pour editer la directive
        """
        selection = treeview.get_selection()
        store, iter_sel = selection.get_selected()
        if iter_sel is not None:
            path = store.get_path(iter_sel)
            selected_dir = store[path][-1]
#            dir_type = fwobjects.ACTION_ALLOW

            dir_dlg = DirectiveDialog(self.glade_file,
                                        self.model,
                                        selected_dir.get_source_zone(),
                                        selected_dir.get_dest_zone(),
                                        self,
                                        selected_dir, iter_sel)

            dir_dlg.show_dialog()

class DirectiveDialog(Singleton):
    """ Classe qui gère l'affichage de l'édition de directives
    """
    def __init__(self, glade_file, directive_store,
                 src_zone, dest_zone, ctrl,
                 edited_dir = None, dir_iter = None):
        """
        glade : l'objet XML glade
        directive_store : le classeur de directives
        """
        self.store = directive_store
        self.glade_file = glade_file
        self.directive_attrs = 0
        self.default_policy = directive_store.default_policy
        self.service_group = False
        self.ctrl = ctrl
        self.src_zone = src_zone
        self.dest_zone = dest_zone
        self.src_name = src_zone.name
        self.dest_name = dest_zone.name

        if 'dialog' not in self.__dict__ :
            self.src_handler_id  = None
            self.dest_handler_id  = None
            self.glade = gtk.glade.XML(self.glade_file,"directive_editor", "editeur")
            self.dialog = self.glade.get_widget('directive_editor')
            # self.dialog.set_size_request(670,390)
            self.available_view = self.glade.get_widget('rule_edit_available_tree')
            # self.directive_view = self.glade.get_widget('rule_edit_rule_view')
            self.src_listview = self.glade.get_widget('rule_edit_src_listview')
            self.dest_listview = self.glade.get_widget('rule_edit_dest_listview')
            self.service_button = self.glade.get_widget('drop_service_button')
            self.service_button.set_use_underline(False)
            self.action_menu = self.glade.get_widget('rule_edit_action_combobox')
            store=gtk.ListStore(str, bool)
            self.action_menu.set_model(store)
            cell = gtk.CellRendererText()
            self.action_menu.pack_start(cell, True)
            self.action_menu.add_attribute(cell, 'text', 0)
            self.action_menu.add_attribute(cell, 'sensitive', 1)

            self.extr_button = self.glade.get_widget('drop_extremite_button')
            self.extr_button.set_use_underline(False)
            self.port_nat_entry = self.glade.get_widget('port_nat_entry')
            self.range_button = self.glade.get_widget('drop_timerange_button')
            self.range_button.set_use_underline(False)
            self.user_group_button = self.glade.get_widget('drop_user_group_button')
            self.user_group_button.set_use_underline(False)
            self.user_group_frame = self.glade.get_widget('user_group_frame')

            self.app_group_button = self.glade.get_widget('drop_application_group_button')
            self.app_group_button.set_use_underline(False)
            self.app_group_frame = self.glade.get_widget('application_group_frame')
#            self.app_group_frame.set_sensitive(False)

            self.accept_button = self.glade.get_widget('accept_button')
            self.ip_nat_label = self.glade.get_widget('ip_nat_label')
#            self.ip_nat_label.set_use_underline(False)
            self.port_nat_label = self.glade.get_widget('port_nat_label')
            self.src_zone_label = self.glade.get_widget('src_zone_label')
            self.dest_zone_label = self.glade.get_widget('dest_zone_label')
            self.log_toggle = self.glade.get_widget('rule_edit_log_button1')
            self.ipsec_toggle = self.glade.get_widget('rule_edit_ipsec_button')
#            self.optional_toggle = self.glade.get_widget('rule_edit_optional_button')
            self.mark_combo = self.glade.get_widget('mark_combo')
            self.mark_value = self.glade.get_widget('mark_value')
#            self.mark_value.set_sensitive(False)
            self._init_extremites_columns()
            self._init_available_columns()
            self.tag_combo = self.glade.get_widget('tag_combo')

            self.tag_combo_model = set_empty_combo_model(self.tag_combo)
#            self.tag_combo_model = gtk.ListStore(str)
#            cell = gtk.CellRendererText()
#            self.tag_combo.pack_start(cell, True)
#            self.tag_combo.set_model(self.tag_combo_model)
#            self.tag_combo_model.clear()

            #entry.set_text("")
            #self.tag_combo.add_attribute(cell, 'text', 0)
            # self.tag_combo.set_use_arrows_always(True)

            # inversion des extremites
            self.src_inv_button = self.glade.get_widget('src_exclude_checkbutton')
            self.dest_inv_button  = self.glade.get_widget('dest_exclude_checkbutton')
            self.serv_inv_button = self.glade.get_widget('serv_exclude_checkbutton')
            # libelle
            self.libelle = self.glade.get_widget('libelle_entry')
            handlers = {'on_directive_editor_response' : self.validate_edit,
#                        'on_rule_edit_log_button_toggled' : (self.toggled, fwobjects.DIRECTIVE_LOG ),
#                        'on_rule_edit_ipsec_button_toggled' : (self.toggled, fwobjects.DIRECTIVE_IPSEC ),
#                        'on_rule_edit_optional_button_toggled' : (self.optional_toggled, fwobjects.DIRECTIVE_OPTIONAL),
                        'on_rule_edit_action_combobox_changed' : self.set_sensitive,
                        'on_rule_edit_accept_button_changed' : self.set_sensitive,
                        'on_rule_edit_src_listview_button_press_event' : self.remove_extremite,
                        'on_rule_edit_src_listview_key_press_event' : self.remove_extremite2,
                        'on_rule_edit_dest_listview_button_press_event' : self.remove_extremite,
                        'on_rule_edit_dest_listview_key_press_event' : self.remove_extremite2,
                        # inversion des extrémités
                        'on_src_exclude_checkbutton_toggled' : self.set_sensitive,
                        'on_dest_exclude_checkbutton_toggled' : self.set_sensitive,
                        'on_serv_exclude_checkbutton_toggled' : self.set_sensitive,
                        # combo pour les directive optionnelles
                        # 'on_rule_edit_select_tag_combo_changed' : self.tag_list
                        'on_mark_combo_changed':self.set_sensitive,
                        'on_drop_timerange_button_button_press_event':self.range_clicked,
                        'on_drop_user_group_button_button_press_event':self.user_group_clicked,
                        'on_drop_application_group_button_button_press_event':self.app_group_clicked,
                        'on_rule_edit_ipsets_button_press_event': self.ipsets_button,
                        }
            self.glade.signal_autoconnect(handlers)
            self.dialog.connect('delete_event', self.hide_and_flush)

            selection = self.available_view.get_selection()
            selection.connect('changed',self.store_selected)
            self._fill_tag_combo()

        else:
            self.flush()

#        self._guess_zones(zoneA,zoneB)
        self.src_zone_label.set_label('<span foreground="#a0569a" weight="bold">%s</span>'% \
                                      self.src_name)
        self.dest_zone_label.set_label('<span foreground="#a0569a" weight="bold">%s</span>'% \
                                       self.dest_name)

        self.edited_dir = edited_dir
        self.dir_iter = dir_iter

        self._init_buttons_lf()
        self._init_drag_destinations()
        self.add_default_policy_item()
        self._init_directive_view()
        self._init_available_view()

        # exceptions dialog box initialisation
        # this dialog is always there, then showed if necessary
        self.ipsets_dlg = DirectiveIpsets(self.glade_file)

        if self.edited_dir is not None:
            self._fill_dialog()
            # si la directive n'est pas éditable, on grise la vbox principale
            if not self.edited_dir.is_editable():
                self.glade.get_widget('hpaned3').set_sensitive(0)
            else:
                self.glade.get_widget('hpaned3').set_sensitive(1)
        else:
            self.glade.get_widget('hpaned3').set_sensitive(1)
            self.selected_service = None
            self.selected_range = None
            self.selected_user_group = None
            self.selected_app_group = None
            self.nat_extr = None
            self.set_sensitive()

    def _fill_dialog(self):
        """Remplit la boîte de dialogue avec les données de la directive
        à éditer. (self.edited_dir)
        """
        # On remplit la liste des sources
        for extr in self.edited_dir.src_list:
            self.src_model.append([extr.name,extr.libelle, extr.zone.name])
        # On remplit la liste des destinations
        for extr in self.edited_dir.dest_list:
            self.dest_model.append([extr.name,extr.libelle, extr.zone.name])

        self.service_button.set_label(str(self.edited_dir.service))
        self.selected_service = self.edited_dir.service
        self.selected_range = self.edited_dir.time
        # récupération des groupes d'utilisateurs
        self.selected_user_group = self.edited_dir.user_group
        self.selected_app_group = self.edited_dir.app_group
        if self.edited_dir.user_group != None:
            self.user_group_button.set_label(str(self.edited_dir.user_group.name))
            if self.edited_dir.app_group != None:
                self.app_group_button.set_label(str(self.selected_app_group))
            else:
                self.app_group_button.set_label(_("drop application_group"))
        # tag ead
        self.libelle.set_text(self.edited_dir.libelle)
        if self.edited_dir.get_tag() != "":
            active_iter = get_iter_from_value(self.tag_combo, self.edited_dir.get_tag())
            #self.tag_combo_model.set(cur_iter, 0, self.edited_dir.get_tag())
            self.tag_combo.set_active_iter(active_iter)
            attrs_optional = fwobjects.DIRECTIVE_OPTIONAL
        else:
            #self.tag_combo_model.clear()
            attrs_optional = 0
        # récupération de la plage horaire
        if self.edited_dir.time != None:
            self.range_button.set_label(self.edited_dir.time.name)
        # récupération des options de marquage
        model = self.mark_combo.get_model()
        try:
            cur_iter = model.get_iter_root()
            while cur_iter:
                if model.get_value(cur_iter,0) == self.edited_dir.mark[0]:
                    self.mark_combo.set_active_iter(cur_iter)
                    self.mark_value.set_text(self.edited_dir.mark[1])
                    break
                cur_iter=model.iter_next(cur_iter)
        except:
            self.mark_combo.set_active_iter(model.get_iter_root())
        # récupere l'etat des logs
        if self.edited_dir.is_logged():
            self.log_toggle.set_active(True)

        # Action ACCEPT pour SNAT, DNAT et FORWARD
        if self.edited_dir.accept == 1:
            self.accept_button.set_active(True)
        else:
            self.accept_button.set_active(False)

        # ipsec
        if self.edited_dir.ipsec == 1:
            self.ipsec_toggle.set_active(True)
        else:
            self.ipsec_toggle.set_active(False)

        # récupération de l'inversion des extrémité et du service
        self.src_inv_button.set_active(self.edited_dir.src_inv)
        self.dest_inv_button.set_active(self.edited_dir.dest_inv)
        self.serv_inv_button.set_active(self.edited_dir.serv_inv)

        if self.edited_dir.nat_extr is not None:
            self.extr_button.set_label(self.edited_dir.nat_extr.name)
            self.nat_extr = self.edited_dir.nat_extr

        else:
            self.nat_extr = None

        if self.edited_dir.nat_port is not None:
            self.port_nat_entry.set_text(str(self.edited_dir.nat_port))
        item_index = {fwobjects.ACTION_DENY : 0,
                      fwobjects.ACTION_ALLOW : 0,
                      fwobjects.ACTION_REDIRECT : 1,
                      fwobjects.ACTION_DNAT : 2,
                      fwobjects.ACTION_MASK : 3,
                      fwobjects.ACTION_FORWARD: 4}

        self.action_menu.set_active(item_index[self.edited_dir.action])
        self.action_type = self.edited_dir.action

        # il ne faut pas reprendre directive_attrs dans la directive éditée,
        # sinon, sa valeur est inversée lorsque les cases sont cochées dans _directive_toggle
        self.directive_attrs = attrs_optional
        self.set_sensitive()
        self.ipsets_dlg.set_exceptions(self.edited_dir.exceptions)

    def _fill_tag_combo(self):
        self.tag_combo.get_model().clear()
        if len(library_store.tags.keys()) > 0:
            for key in library_store.tags.keys():
#                iter = self.tag_combo_model.append()
                self.tag_combo_model.append([key]) #set_value(iter, 0, key)

    def _set_nufw_sensitive(self):
        # si nufw activé
        inv_button = False
        app_group = True
        if self.user_group_button.get_label() == _("drop user_group"):
            app_group = False
            inv_button = True
        else:
            if self.selected_user_group != None:
                if int(self.selected_user_group.id) == int(GID_GUEST):
                    app_group = False

        return inv_button, app_group

    def set_combo_sensitive(self, model, path, iterator, value):
        """
            callback appelé lors d'une iteration sur le store
        """
        path = [str(i) for i in path]
        iterator = model.get_iter_from_string(":".join(path))
        if model.get_value(iterator, 0) == _('DNAT') or model.get_value(iterator, 0) == _('SNAT'):
            model.set_value(iterator, 1, value)
            #model.remove(iterator)

    def set_sensitive(self, widget=None, *args):
        """
            grise/dégrise un ensemble de widget
        """
        if not hasattr(self, 'old_action_menu'):
            return
        action_menu = self.action_menu.get_active()
        if action_menu == -1:
            return
        # si la case user_group est activé
        user_group = True
        # si app_groupe est possible (user_group + pas non-identifié)
        app_group = True
        # si forward, (s/d)nat
        accept_button_sensitive = True
        accept = self.accept_button.get_active()
        ip_nat = False
        # si (s/d)nat
        port_nat = False
        # si la user_group est vide
        inv_button = True
        # le type d'action
        action_type = False
        if action_menu == 0:
#            action_type = self.default_policy
            if self.default_policy == False:
                action_type = fwobjects.ACTION_ALLOW
            else:
                action_type = fwobjects.ACTION_DENY
        elif action_menu == 1:
            action_type = fwobjects.ACTION_REDIRECT
        elif action_menu == 2:
            action_type = fwobjects.ACTION_DNAT
        elif action_menu == 3:
            action_type = fwobjects.ACTION_MASK
        elif action_menu == 4:
            action_type = fwobjects.ACTION_FORWARD

        if not action_type:
            return
        self.action_type = action_type
        if self.src_inv_button.get_active() == True or self.dest_inv_button.get_active() == True or self.serv_inv_button.get_active() == True:
            user_group = False
        if self.mark_combo.get_active_text() != "" and self.mark_combo.get_active_text() != None:
            inv_button = True
            user_group = False
        if (self.action_type == fwobjects.ACTION_DENY
                or self.action_type == fwobjects.ACTION_ALLOW ):
            inv_button, app_group = self._set_nufw_sensitive()
            ip_nat = False
            accept_button_sensitive = False

        if (self.action_type ==  fwobjects.ACTION_DNAT or self.action_type == fwobjects.ACTION_MASK):
            # il faut dégriser éventuellement le marquage
            if accept:
                ip_nat = False
                port_nat = False
            else:
                ip_nat = True
                port_nat = True
            user_group = False
            app_group = False

        if (self.action_type == fwobjects.ACTION_FORWARD):
            # il faut dégriser éventuellement le marquage
            accept_button_sensitive = False
            self.accept_button.set_active(True)
            ip_nat = False
            port_nat = False
            user_group = False
            app_group = False

        # Cas particulier du redirect : ça va sur Localhost
        if self.action_type == fwobjects.ACTION_REDIRECT:
            accept_button_sensitive = False
            port_nat = True
            inv_button, app_group = self._set_nufw_sensitive()

        self.port_nat_label.set_sensitive(port_nat)
        self.port_nat_entry.set_sensitive(port_nat)
        if not port_nat:
            self.port_nat_entry.set_text('')

        self.accept_button.set_sensitive(accept_button_sensitive)
        self.ip_nat_label.set_sensitive(ip_nat)
        self.extr_button.set_sensitive(ip_nat)
        if not ip_nat:
            self.extr_button.set_label(_('drop extremite'))

        self.mark_combo.set_sensitive(inv_button)
        #n'active que si pas nul
        if inv_button:
            if self.mark_combo.get_active_text() == "" or self.mark_combo.get_active_text() == None:
                self.mark_value.set_sensitive(False)
                self.mark_value.set_text('')
            else:
                self.mark_value.set_sensitive(True)
        else:
            self.mark_combo.set_active(0)
            self.mark_value.set_text('')
#        self.src_inv_button.set_sensitive(inv_button)
#        self.dest_inv_button.set_sensitive(inv_button)
#        self.serv_inv_button.set_sensitive(inv_button)
#        if not inv_button:
#            self.src_inv_button.set_active(False)
#            self.dest_inv_button.set_active(False)
#            self.serv_inv_button.set_active(False)

        self.user_group_frame.set_sensitive(user_group)
        if not user_group:
            self.user_group_button.set_label(_("drop user_group"))
            self.selected_user_group = None
            app_group = False

        self.app_group_frame.set_sensitive(app_group)
        if not app_group:
            self.app_group_button.set_label(_("drop application_group"))
            self.selected_app_group = None

        # en cas d'interdiction + application : change le label
        if self.default_policy == True:
            model=self.action_menu.get_model()
            iterator = model.get_iter_first()
            if self.app_group_button.get_label() != _("drop application_group"):
                model.set_value(iterator, 0, _("deny any other application"))
            else:
                model.set_value(iterator, 0, _("deny"))

        # active/desactive le menu action
        if not inv_button and self.old_action_menu == True:
            self.old_action_menu=False
            store=self.action_menu.get_model()
            store.foreach(self.set_combo_sensitive, False)
        elif inv_button and self.old_action_menu == False:
            self.old_action_menu=True
            store=self.action_menu.get_model()
            store.foreach(self.set_combo_sensitive, True)

        self.test_check_invert(self.src_listview.get_model(), True)
        self.test_check_invert(self.dest_listview.get_model(), False)


    def _init_buttons_lf(self):
        """Initialise le L&F (Look and feel) des boutons 'drop' ...
        """
        # initialisation du bouton : 'drop a service'
        self.service_button.remove(self.service_button.get_children()[0])
        l = gtk.Label(_("drop a service"))
        l.set_use_markup(True)
        self.service_button.add(l)
        self.service_button.show_all()
        # initialisation du bouton : 'drop extremite'
        self.extr_button.remove(self.extr_button.get_children()[0])
        l = gtk.Label(_("drop extremite"))
        l.set_use_markup(True)
        self.extr_button.add(l)
        self.extr_button.show_all()
        # initialisation du bouton : 'drop user_group'
        self.user_group_button.remove(self.user_group_button.get_children()[0])
        l = gtk.Label(_("drop user_group"))
        l.set_use_markup(True)
        self.user_group_button.add(l)
        self.user_group_button.show_all()

    def _init_drag_destinations(self):
        """Initialise l'ensemble des destinations de DnD
        """
        self.service_button.connect("drag_data_received", self.receive_service)
        self.service_button.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                          gtk.DEST_DEFAULT_HIGHLIGHT |
                                          gtk.DEST_DEFAULT_DROP,
                                          [( "library/object", 0, 0 )],
                                          gtk.gdk.ACTION_COPY)

        self.extr_button.connect("drag_data_received", self.nat_button_receive_extremite)
        self.extr_button.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                       gtk.DEST_DEFAULT_HIGHLIGHT |
                                       gtk.DEST_DEFAULT_DROP,
                                       [( "library/object", 0, 0 )],
                                       gtk.gdk.ACTION_COPY)

        if self.src_handler_id is not None:
            self.src_listview.disconnect(self.src_handler_id)
        self.src_handler_id = self.src_listview.connect("drag_data_received", self.receive_extremite, self.src_name)
        self.src_listview.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                        gtk.DEST_DEFAULT_HIGHLIGHT |
                                        gtk.DEST_DEFAULT_DROP,
                                        [( "library/object", 0, 0 )],
                                        gtk.gdk.ACTION_COPY)

        if self.dest_handler_id is not None:
            self.dest_listview.disconnect(self.dest_handler_id)
        self.dest_handler_id = self.dest_listview.connect("drag_data_received", self.receive_extremite, self.dest_name)
        self.dest_listview.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                         gtk.DEST_DEFAULT_HIGHLIGHT |
                                         gtk.DEST_DEFAULT_DROP,
                                         [( "library/object", 0, 0 )],
                                         gtk.gdk.ACTION_COPY)


        self.range_button.connect("drag_data_received", self.receive_range)
        self.range_button.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                          gtk.DEST_DEFAULT_HIGHLIGHT |
                                          gtk.DEST_DEFAULT_DROP,
                                          [( "library/object", 0, 0 )],
                                          gtk.gdk.ACTION_COPY)

        self.user_group_button.connect("drag_data_received", self.receive_user_group)
        self.user_group_button.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                          gtk.DEST_DEFAULT_HIGHLIGHT |
                                          gtk.DEST_DEFAULT_DROP,
                                          [( "library/object", 0, 0 )],
                                          gtk.gdk.ACTION_COPY)

        self.app_group_button.connect("drag_data_received", self.receive_app_group)
        self.app_group_button.drag_dest_set(gtk.DEST_DEFAULT_MOTION |
                                          gtk.DEST_DEFAULT_HIGHLIGHT |
                                          gtk.DEST_DEFAULT_DROP,
                                          [( "library/object", 0, 0 )],
                                          gtk.gdk.ACTION_COPY)


    def add_default_policy_item(self):
        """Ajoute l'action 'Autoriser' ou 'Interdire' suivant le type de règle
        """
        liststore=self.action_menu.get_model()
        liststore.clear()
        #if self.default_policy == fwobjects.ACTION_ALLOW:

        if self.default_policy == False:
            liststore.append([_('allow'), True])
            self.action_type = fwobjects.ACTION_ALLOW
        else:
            liststore.append([_('deny'), True])
            self.action_type = fwobjects.ACTION_DENY

        liststore.append([_('Redirect'), True])
        liststore.append([_('DNAT'), True])
        liststore.append([_('SNAT'), True])
        liststore.append([_('Forward'), True])
        self.action_menu.set_active(0)
        self.old_action_menu=True

    def _init_directive_view(self):
        """ Initialise la vue et le modèle associés à la directive
        """
        self.src_listview.set_headers_visible(1)
        self.dest_listview.set_headers_visible(1)

        self.src_model = create_extremite_treemodel()
        self.dest_model = create_extremite_treemodel()

        self.src_listview.set_model(self.src_model)
        self.dest_listview.set_model(self.dest_model)


    def _init_extremites_columns(self):
        column_names = [_('name'), _('libelle'), _('zone')]

        col_index = 0
        for col_name in column_names:
            src_column = gtk.TreeViewColumn(col_name,
                                            gtk.CellRendererText(),
                                            text = col_index)

            dest_column = gtk.TreeViewColumn(col_name,
                                             gtk.CellRendererText(),
                                             text = col_index)
            self.src_listview.append_column(src_column)
            self.dest_listview.append_column(dest_column)

            col_index += 1


    def _init_available_view(self):
        """ Initialise la vue et le modèle associés à l'arbre des disponibilités
        """
        self.available_model = create_objects_treestore(self.src_zone,
                                                        self.dest_zone)
        self.available_view.set_model(self.available_model)
        self.available_view.connect_after('drag_data_get', self.send_callback)
        self.available_view.drag_source_set(gtk.gdk.BUTTON1_MASK,
                                            [( "library/object", 0, 0 )],
                                            gtk.gdk.ACTION_COPY)
        # init du D&D (Drag and drop) sur le bouton des timeranges pour les supprimer
        self.range_button.drag_source_set(gtk.gdk.BUTTON1_MASK,
                                            [( "library/object", 0, 0 )],
                                            gtk.gdk.ACTION_MOVE)
        self.range_button.connect("drag_end", self.drop_range)
        # init du D&D sur le bouton des user_groups pour les supprimer
        self.user_group_button.drag_source_set(gtk.gdk.BUTTON1_MASK,
                                            [( "library/object", 0, 0 )],
                                            gtk.gdk.ACTION_MOVE)
        self.user_group_button.connect("drag_end", self.drop_user_group)

        self.app_group_button.connect("drag_end", self.drop_app_group)

    def _init_available_columns(self):
        """Définit les colonnes de l'arbre des objets disponible
        """
        column_names = [_("available objects")]
        col_index = 0
        for col_name in column_names:
            renderer = gtk.CellRendererText()
            column = gtk.TreeViewColumn(col_name, renderer,
                                        text = col_index)
            self.available_view.append_column(column)
            col_index += 1
        # self.available_view.connect('button-press-event', self.store_selected)

    def store_selected(self, selection, *args):
        model, iter_sel = selection.get_selected()
        if iter_sel is not None:
            # iter_sel est nul si on replie un arbre
            path = model.get_path(iter_sel)
            self.selected_object = model[path][1]

    def receive_extremite(self, widget, context, x, y, selection, targetType, time, zone_name):
        """Appelée lors d'un Drop d'une extrémité
        """
        widget.emit_stop_by_name('drag_data_received')

        try:
            if self.selected_object.get_type() == "firewall/extremite":
                store = widget.get_model()
                added_extr = [row[0] for row in store]
                extr = self.selected_object

                if extr.zone.name != zone_name:
                    dlg = create_error_dialog(_("bad zone drop"))
                    dlg.show_all()
                    return True
                if extr.name not in added_extr :
                    store.append((extr.name, extr.libelle, extr.zone.name))
                else:
                    dlg = create_error_dialog(_("already added extremite"))
                    dlg.show_all()
                    return True

                self.set_sensitive()

        except TypeError:
            pass

    def test_check_invert(self, store, is_src):
        first_iter =  store.get_iter_first()
        if first_iter == None:
            if is_src == False:
                self.dest_inv_button.set_sensitive(1)
            else:
                self.src_inv_button.set_sensitive(1)
        else:
            # on récupère l'extrémité correspondante
            nb_ip = len(library_store.extremites[store.get_value(first_iter,0)].ip_list)
            if nb_ip > 1 or store.iter_next(store.get_iter_first()) is not None:
                # on grise la case à cocher
                if is_src == False:
                    self.dest_inv_button.set_sensitive(0)
                    self.dest_inv_button.set_active(0)
                else:
                    self.src_inv_button.set_sensitive(0)
                    self.src_inv_button.set_active(0)
            else:
                if is_src == False:
                    self.dest_inv_button.set_sensitive(1)
                else:
                    self.src_inv_button.set_sensitive(1)

    def remove_extremite(self, treeview, event, *args):
        """ Appelée lors d'un click dans les listes de sources et destinations
        """
        if event.button == 1:
            # double-clic
            if event.type == gtk.gdk._2BUTTON_PRESS:
                selection = treeview.get_selection()
                store, iter_del = selection.get_selected()
                if iter_del is not None:
                    path = store.get_path(iter_del)
                    store.remove(iter_del)
                    self.set_sensitive()

    def remove_extremite2(self, treeview, event, *args):
        """ Appelée lors d'un click dans les listes de sources et destinations
        """
        if event.keyval == gtk.gdk.keyval_from_name("Delete"):
            selection = treeview.get_selection()
            store, iter_del = selection.get_selected()
            if iter_del is not None:
                path = store.get_path(iter_del)
                store.remove(iter_del)
                self.set_sensitive()


    def nat_button_receive_extremite(self, button, context, x,y, selection, targetType, time):
        """Callback appelée lorsque l'on fait reçoit l'évènement 'drag_data_received'
        sur le bouton 'drop_extremite'
        """
        # XXX FIXME : tester si l'extrémité vient de la source ou de la dest !
        if self.selected_object.get_type() == "firewall/extremite":
            button.set_label(self.selected_object.name)
            self.nat_extr = self.selected_object

    def receive_service(self, button, context, x, y, selection, targetType,time):
        """Appelée lors d'un Drop dans le bouton 'drop service'
        """
        if self.selected_object.get_type() in ["firewall/service","firewall/group"]:
            if self.selected_object.get_type() == "firewall/group":
                self.glade.get_widget('serv_exclude_checkbutton').set_sensitive(0)
                self.glade.get_widget('serv_exclude_checkbutton').set_active(0)
            else:
                self.glade.get_widget('serv_exclude_checkbutton').set_sensitive(1)

            if self.selected_object.get_type() == "firewall/service":
                self.service_group = False
            else:
                self.service_group = True
            button.set_label(str(self.selected_object))
            self.selected_service = self.selected_object

    def drop_range(self, button, *args):
        """Appelé lors de la fin d'un d&d
        """
        if self.selected_range != None:
            button.set_label(_("drop timerange"))
            self.selected_range = None

    def range_clicked(self, button, event, *args):
        """Appelé lors de la fin d'un d&d
        """
        if event.type == gtk.gdk._2BUTTON_PRESS:
            if self.selected_range != None:
                button.set_label(_("drop timerange"))
                self.selected_range = None

    def drop_user_group(self, button, *args):
        """Appelé lors de la fin d'un d&d
        """
        if self.selected_user_group != None:
            button.set_label(_("drop user_group"))
            self.selected_user_goup = None
        self.set_sensitive()

    def user_group_clicked(self, button, event, *args):
        """Appelé lors de la fin d'un d&d
        """
    	if event.type == gtk.gdk._2BUTTON_PRESS:
            if self.selected_user_group != None:
                button.set_label(_("drop user_group"))
                self.selected_user_group = None
            self.set_sensitive()

    def drop_app_group(self, button, *args):
        """Appelé lors de la fin d'un d&d
        """
        if self.selected_app_group != None:
            button.set_label(_("drop application_group"))
            self.selected_app_goup = None
        self.set_sensitive()

    def app_group_clicked(self, button, event, *args):
        """Appelé lors de la fin d'un d&d
        """
    	if event.type == gtk.gdk._2BUTTON_PRESS:
            if self.selected_app_group != None:
                button.set_label(_("drop application_group"))
                self.selected_app_group = None
            self.set_sensitive()

    def receive_range(self, button, context, x, y, selection, targetType,time):
        """Appelée lors d'un Drop dans le bouton 'drop range'
        """
        if self.selected_object.get_type() == "firewall/range":
            button.set_label(self.selected_object.name)
            self.selected_range = self.selected_object

    def receive_user_group(self, button, context, x, y, selection, targetType,time):
        """Appelée lors d'un Drop dans le bouton 'drop user_group'
        """
        if self.selected_object.get_type() == "firewall/user_group":
            button.set_label(self.selected_object.name)
            self.selected_user_group = self.selected_object
        self.set_sensitive()

    def receive_app_group(self, button, context, x, y, selection, targetType,time):
        """Appelée lors d'un Drop dans le bouton 'drop app_group'
        """
        if self.selected_object.get_type() == "firewall/application_group":
            button.set_label(self.selected_object.name)
            self.selected_app_group = self.selected_object.name
        self.set_sensitive()

    def send_callback(self, widget, context, selection, targetType, eventTime):
        if self.selected_object is not None:
            selection.set(selection.target, 8, self.selected_object.name)

#    def toggled(self, button, flag):
#        """ Callback appelée lorsque l'on (dé)coche une case de la fenêtre
#        """
#        print self.directive_attrs
#        self.directive_attrs = self.directive_attrs ^ flag
#        print self.directive_attrs

    def flush(self):
        """ Vide le dialogue d'édition
        """
        # On vide la liste de sources / dests
        self.src_listview.get_model().clear()
        self.src_inv_button.set_active(False)
        self.dest_listview.get_model().clear()
        self.dest_inv_button.set_active(False)
        self.serv_inv_button.set_active(False)
        # On réinitialise les boutons de DnD
        self._init_buttons_lf()
        self.port_nat_entry.set_text("")
        self.add_default_policy_item()
        self.range_button.set_label(_("drop timerange"))
        self.selected_service = None
        self.selected_range = None
        self.selected_user_group = None
        self.accept_button.set_active(False)
        self.nat_extr = None
        self.nat_port = None
        # On décoche tous les toggles
        self.log_toggle.set_active(False)
        self.ipsec_toggle.set_active(False)
        # self.mark_toggle.set_active(False)
#        self.optional_toggle.set_active(False)
        #self.tag_combo.set_popdown_strings(library_store.tags.keys())
#        self.tag_combo_model.clear()
        self.tag_combo_model = set_combo_model(self.tag_combo, [''])
        iterator = self.tag_combo_model.get_iter_root()
        self.tag_combo.set_active_iter(iterator)
        self._fill_tag_combo()

        # options de marquage
        model = self.mark_combo.get_model()
        self.mark_combo.set_active_iter(model.get_iter_root())
        self.mark_value.set_text("")
        self.directive_attrs = 0
        self.libelle.set_text("")
        self.dialog.window.raise_()
        self.user_group_button.set_label(_("drop user_group"))
        self.set_sensitive()
        self.ipsets_dlg.set_exceptions(None)

    def hide_and_flush(self, *args):
        """ Cache la fenêtre et la réinitialise
        """
        self.dialog.hide()
        self.flush()
        return True

    def get_extr_list(self, store):
        """ Récupère la liste des noms d'extrémités d'un modèle
        store : le modèle à traiter
        """
        return [row[0] for row in store]

    def validate_edit(self, dialog, response_code, *args):
        """ Callback appelée lorsque l'on a fini d'éditer une directive
        """
        # On ne tient compte que si l'on clique sur OK
        if response_code != gtk.RESPONSE_OK:
            return self.hide_and_flush()

        src_names = self.get_extr_list(self.src_listview.get_model())
        dest_names = self.get_extr_list(self.dest_listview.get_model())
        src_list = [library_store.extremites[name] for name in src_names]
        dest_list = [library_store.extremites[name] for name in dest_names]
        self.nat_port = self.port_nat_entry.get_text()
        if self.nat_port != '':
            try:
                self.nat_port = int(self.nat_port)
            except:
                self.nat_port = make_template_string(self.nat_port)
        else:
            self.nat_port = 0
        active_iter = self.tag_combo.get_active_iter()
        if active_iter:
            self.opt_tag = self.tag_combo_model.get_value(active_iter,0)
        else:
            self.opt_tag = self.tag_combo.get_active_text()
        self.log_value = self.log_toggle.get_active()
        if self.libelle.get_text() :
            libelle = self.libelle.get_text()
        else :
            libelle = _("no description")

        if not len(src_list):
            src_list.append(library_store.extremites[self.src_zone.name])
        if not len(dest_list):
              dest_list.append(library_store.extremites[self.dest_zone.name])

        mark_operator = self.mark_combo.get_active_text()
        if mark_operator != "":
            mark = (mark_operator, self.mark_value.get_text())
        else:
            mark = None

        try:
            for char in bad_chars:
                if char in self.opt_tag:
                    raise AssertionError(_("bad character in the tag: {0}").format(self.opt_tag))
            try:
                self.opt_tag.decode('ascii')
            except UnicodeDecodeError:
                raise AssertionError(_("bad character in the tag: {0}").format(self.opt_tag))

#            if self.selected_service is not None:
#                # tcpwrapper validation : la destination doit etre le bastion
#                if self.selected_service.is_tcpwrapper_type() and dest_list[0].zone.name != "bastion":
#                        # il suffit de regarder la premiere extremite pour voir dans quelle zone on est
#                        raise AssertionError(_("forbidden destination extremity with a tcpwrapper service"))
            # exceptions validation
            if self.ipsets_dlg.get_exceptions() is not None:
                if len(self.ipsets_dlg.get_exceptions()) != 0 and self.action_type not in (fwobjects.ACTION_ALLOW, fwobjects.ACTION_DENY,
                    fwobjects.ACTION_REDIRECT, fwobjects.ACTION_FORWARD):
                    raise AssertionError(_("it is forbidden to use ipset exceptions with DNAT or SNAT action type"))
                # si il y a des ipsets et qu'il y a des "tout sauf", ça doit être interdit
                if len(self.ipsets_dlg.get_exceptions()) != 0 and (self.src_inv_button.get_active() == True or
                    self.dest_inv_button.get_active() == True):
                    raise AssertionError(_("it is forbidden to use ipset exception with inverted extremities"))
            # si le service est "tous", la redirection doit etre interdite
            if self.selected_service is not None:
                if self.selected_service.name == 'tous' and self.action_type == ACTION_REDIRECT: #and self.nat_port == 0:
                    raise AssertionError(_("it is forbidden to redirect when the selected service is 'all'"))
            #si on demande de log
            if self.log_value:
                #et que ca ne log pas
                if not self.directive_attrs & fwobjects.DIRECTIVE_LOG == fwobjects.DIRECTIVE_LOG:
                    #on log
                    self.directive_attrs = self.directive_attrs ^ fwobjects.DIRECTIVE_LOG
            #si on demande de ne pas log
            else:
                #et que ca log
                if self.directive_attrs & fwobjects.DIRECTIVE_LOG == fwobjects.DIRECTIVE_LOG:
                    #delog
                    self.directive_attrs = self.directive_attrs ^ fwobjects.DIRECTIVE_LOG

            if self.edited_dir:
                if self.opt_tag.strip() != "":
                    if not self.edited_dir.is_optional():
                        self.directive_attrs = self.directive_attrs ^ fwobjects.DIRECTIVE_OPTIONAL
                else:
                    if self.edited_dir.is_optional():
                        self.directive_attrs = self.directive_attrs ^ fwobjects.DIRECTIVE_OPTIONAL
            else:
                if self.opt_tag.strip() != "":
                    self.directive_attrs = self.directive_attrs ^ fwobjects.DIRECTIVE_OPTIONAL
            if self.opt_tag.strip() == "":
                library_store.tags[str(self.opt_tag)] = (False,False)
            else:
                if self.opt_tag not in library_store.tags.keys():
                    library_store.tags.update({str(self.opt_tag):(False,False)})
            d = fwobjects.Directive(src_list, dest_list, self.selected_service, self.action_type,self.directive_attrs,
                                    nat_port = self.nat_port, nat_extr = self.nat_extr,
                                    is_group = self.service_group,
                                    src_inv = self.src_inv_button.get_active(),
                                    dest_inv = self.dest_inv_button.get_active(),
                                    serv_inv = self.serv_inv_button.get_active(),
                                    libelle = libelle,
                                    time = self.selected_range,
                                    mark = mark,
                                    user_group = self.selected_user_group,
                                    app_group = self.selected_app_group
                                    )
            # ipsets exceptions
            exceptions = self.ipsets_dlg.get_exceptions()
            d.exceptions = exceptions

#            if self.glade.get_widget('rule_edit_optional_button').get_active() == 1:
            d.set_tag(self.opt_tag)
            # accept button
            if self.accept_button.get_active():
                d.accept = 1
            else:
                d.accept = 0
            # ipsec
            if self.ipsec_toggle.get_active():
                d.ipsec = 1
            else:
                d.ipsec = 0
            # on met à jour les services ,extremites et plages horaires pour leur signaler qu'ils sont utilisés
            self.selected_service.used += 1
            for extr in src_list:
                extr.used += 1
            for extr in dest_list:
                extr.used += 1
            if self.selected_range != None:
                self.selected_range.used += 1
            if self.selected_user_group != None:
                self.selected_user_group.used += 1

            if self.edited_dir is not None:
                # c'est une directive éditée (pas une nouvelle directive)
                self.edited_dir.exceptions = exceptions
                # on libère les anciens services, extremités et plages horaires
                self.edited_dir.service.used -= 1
                for extr in self.edited_dir.src_list:
                    extr.used -= 1
                for extr in self.edited_dir.dest_list:
                    extr.used -= 1
                if self.edited_dir.time != None:
                    self.edited_dir.time.used -= 1
                if self.selected_user_group != None:
                    self.selected_user_group.used -= 1

                # on met à jour la directive
                self.edited_dir.src_list = d.src_list
                self.edited_dir.dest_list = d.dest_list
                self.edited_dir.service = d.service
                self.edited_dir.action = d.action
                self.edited_dir.attrs = d.attrs
                self.edited_dir.tag = d.tag
                self.edited_dir.ipsec = d.ipsec
                self.edited_dir.accept = d.accept
                #self.edited_dir.priority = d.priority
                self.edited_dir.nat_extr = d.nat_extr
                self.edited_dir.nat_port = d.nat_port
                self.edited_dir.service_group = d.service_group
                self.edited_dir.libelle = d.libelle
                self.edited_dir.time = d.time
                self.edited_dir.user_group = d.user_group
                self.edited_dir.app_group = d.app_group
                self.edited_dir.mark = d.mark
                # inversion
                self.edited_dir.src_inv = d.src_inv
                self.edited_dir.dest_inv = d.dest_inv
                self.edited_dir.serv_inv = d.serv_inv

                if not self.edited_dir.action_is_default():
                    if ( self.edited_dir.action == ACTION_DNAT or self.edited_dir.action == ACTION_MASK ) and self.edited_dir.accept == 0:
                        assert self.edited_dir.nat_extr is not None, _("nat extr undefined error")
                        # vérifions la validitié du port
                        # IMPORTANT : le port 0 signifie TOUT (tous les ports)
                        if isinstance(self.edited_dir.nat_port, int):
                            if not 0 <= self.edited_dir.nat_port <= 65535:
                                raise BadPortError, _("bad port error")

                col_index = 0
                for val in self.edited_dir.as_list():
                    # self.store.notify_observers()
                    self.ctrl.gtk_model.set(self.dir_iter, col_index, val)
                    col_index += 1

                return self.hide_and_flush()
            else:
                # nouvelle directive
                self.store.add_directive(d)
                self.ctrl.gtk_model.append(d.as_list())
                dlg = create_info_dialog(_('directive created'),self.dialog)
                dlg.show_all()

        except (BadPortError, AssertionError, TagError), e:
            dlg = create_error_dialog(str(e))
            dlg.show_all()
            return

        self.flush()

    def show_dialog(self):
        """ Affiche la fenêtre de dialogue d'édition de directive
        """
        self.dialog.show_all()

    def assert_directive_valid(self, directive):
        """ Teste si la directive est valide
        """
        pass

    def ipsets_button(self, *args):
        self.ipsets_dlg.show_dialog()

