# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
    la fenetre principale
"""
# ____________________________________________________________
# generic imports
import os, tempfile, base64
import gtk, gtk.glade
# ____________________________________________________________
# compiler
from era import version
from era.tool.standalone import ask_for_unspecified_vars
from era.backend import compiler
# si creole n'est pas present sur la machine
try:
    from era.backend.nufw_acl_plaintext import gen_acls as gen_nufw_acl_plaintext
except:
    pass
# import xmlrpclib depuis zephir_client si disponible (corrige des pbs avec
# caratères spéciaux dans les mots de passe).
try:
    from zephir.eolerpclib import EoleProxy as ServerProxy
except:
    from xmlrpclib import ServerProxy
# ____________________________________________________________
# fwobjects
from era.noyau.erreurs import TagError
from era.noyau.path import setup_path, GLADE_FILE, ERA_DIR
from era.noyau.pool import library_store
from era.noyau.initialize import initialize_app
from era.noyau.fwobjects import ACTION_DENY, ACTION_ALLOW
# ____________________________________________________________
# ihm widgets
from era.ihm.ihm_utils import build_image_button, create_error_dialog, \
     create_about_dialog, create_info_dialog, create_ok_cancel_dialog, \
     destroy_parent, create_yes_no_dialog, create_warning_dialog
from era.ihm.zone import ZoneDialogManager
from era.ihm.directive import DirectiveManager, DirectiveSummaryView
from era.ihm.extremite import ExtremiteListviewDialog, DialogReseauManager, \
     DialogMachinesManager
from era.ihm.service import ServiceListviewDialog, ServiceGroupeDialog, \
     ServiceDialog
from era.ihm.user_group import UserGroupDialog
from era.ihm.optionnel import LibelleOptionnelDialog
from era.ihm.ranges import TimeRangeDialog
from era.ihm.qosshare import QosShareWindow
from era.ihm.applications import ApplicationListviewDialog
# ____________________________________________________________
from era.ihm.options import LibelleOptionsDialog

class MatrixView:
    """ Vue pour la matrice de flux
    """
    def __init__(self, glade_file, ctrl, model=None, rc_file=None, mvcview= None, mvcctrl=None):
        """
        glade : l'objet XML glade
        ctrl : le controller pour cette vue
        """
        # FIXME transitoire : permet de petit à petit passer à pygtkmvc
        self.mvcview = mvcview
        self.mvcctrl = mvcctrl
        # FIXME : inutile de passer le ficher glade à l'init (on le connait)
        # initialisation de la fenêtre (libglade)
        self.glade_file = glade_file
        self.glade = gtk.glade.XML(self.glade_file, "matrix_window", "editeur")
        self.zeph_user=self.zeph_pwd=self.zeph_addr=self.zeph_port=self.connected=None
        self.ctrl = ctrl
        self.model = model
        self.ctrl.saved = 1
        # flag pour indiquer si on doit convertir les variables creole
        self.creole_conv = False
        # la fenêtre principale
        self.main_window = self.glade.get_widget('matrix_window')
        # la table contenant la matrice de flux
        self.zones_table = self.glade.get_widget('matrix_table')
        # la status bar de la fenêtre
        self.status_bar = self.glade.get_widget('matrix_stat_bar')
        # La boîte verticale contenant tous les éléments de la fenêtre
        # On en a besoin pour enlever et remettre les nouvelles tables
        self.main_vbox = self.glade.get_widget('matrix_main_vbox')
        # widget de partage de qos
        # self.bwshare = False

        handlers = { 'on_matrix_toolbar_add_button_clicked' : self.ctrl.add_zone,
                     'on_matrix_toolbar_dump_button_clicked' : self.export,
                     'on_matrix_export_zephir_item_activate' : (self._connect_zephir,'send'),
                     'on_matrix_import_zephir_item_activate' : (self._connect_zephir,'get'),
                     'on_matrix_window_delete_event' : self.quit,
                     'on_matrix_quit_item_activate' : self.quit,
                     'on_matrix_open_item_activate' : (self.ctrl.import_firewall,self),
                     'on_import_model_activate' : (self.ctrl.import_firewall_model,self),
                     'on_matrix_save_item_activate' : (self.ctrl.save, self),
                     'on_matrix_saveas_item_activate' : (self.ctrl.save_as, self),
                     'on_matrix_new_item_activate' : (self.ctrl.new_firewall, self),
                     'on_matrix_add_service_activate' : self.ctrl.add_service,
                     'on_matrix_add_service_group_activate' : self.ctrl.add_service_group,
                     'on_matrix_services_view_activate' : self.ctrl.show_services,
                     'on_matrix_user_groups_view_activate' : self.ctrl.show_user_groups,
                     'on_matrix_extremite_view_activate' : self.ctrl.show_all_extremites,
                     'on_matrix_about_item_activate' : self.about,
                     'on_matrix_dump_item_activate' : self.export,
                     'on_inclusion_statique1_activate' : self.open_include,
                     'on_libelles_optionnels_activate' : self.libelle_optionnels,
                     'on_libelle_ranges_activate' : self.timerange,
                     'on_libelle_options_activate' : self.libelle_options,
                     'on_qos_activate': self.show_qosshare,
                     'on_properties_activate': self.view_properties,
                     'on_application_filter_activate': self.application_filter,

                     }

        self.glade.signal_autoconnect(handlers)
        if model is not None:
            self.set_model(model)
        else:
            self.model = model
            self.display_message(_('new document'))

    def view_properties(self, *args):
        "Affiche les proprietes du modele (version, modele herite, ...)"
        if (library_store.model != "" and library_store.model is not None):
            model = library_store.model
        else:
            model = "pas de modèle"
        version = library_store.version
        properties = """
        version du modèle : %s
        modele père : %s
        """ % (version, model)
        dialog = create_info_dialog(properties)
        dialog.show()
        dialog.grab_focus()

    def set_model(self, model):
        self.model = model
        self.model.register_observer(self)
        self.update()

    def show_window(self):
        """ Affiche la fenêtre de dialogue
        """
        self.main_window.show_all()

    def libelle_options(self, *args):
        """
        Affiche la fenêtre des options
        """
        dlg = LibelleOptionsDialog(self.glade_file)
        dlg.show_dialog()

    def libelle_optionnels(self, *args):
        """
        Affiche la fenêtre des libellés optionnels
        """
        dlg = LibelleOptionnelDialog(self.glade_file)
        dlg.show_dialog()

    def timerange(self, *args):
        """
        Affiche la fenêtre des plages horaires
        """
        dlg = TimeRangeDialog(self.glade_file)
        dlg.show_dialog()

    def application_filter(self, *args):
        """
          Lance la fenêtre pour le filtrage applicatif
        """
        app = ApplicationListviewDialog(self.glade_file)

    def open_include(self, *args):
        """Affiche La fenetre d'édition de l'inclusion statique
        """
        if not hasattr(self,'include_glade'):
            self.include_glade = gtk.glade.XML(self.glade_file, "dialog_include", "editeur")
            if library_store.rules:
                inc_buff = self.include_glade.get_widget('text_include').get_buffer()
                inc_buff.set_text(library_store.rules)
            self.include_glade.get_widget('label_include').set_text(_('label_include'))
            dic_signal = {'on_valid_include_clicked' : (self.ctrl.valid_include, self),
                          'on_close_include_clicked' : self.close_include,
                          'on_dialog_include_delete_event' : self.close_include
                         }
            self.include_glade.signal_autoconnect(dic_signal)
        else:
            if library_store.rules:
                inc_buff = self.include_glade.get_widget('text_include').get_buffer()
                inc_buff.set_text(library_store.rules)
            else:
                inc_buff = self.include_glade.get_widget('text_include').get_buffer()
                inc_buff.set_text("")
            self.include_glade.get_widget('dialog_include').show_all()

    def close_include(self, button, *args):
        """cache la fenêtre des includes
        """
        dialog = create_ok_cancel_dialog(_("Close witout saving ?"))
        dialog.show()
        reponse = dialog.run()
        if reponse != gtk.RESPONSE_OK:
            dialog.destroy()
        else:
            dialog.destroy()
            self.include_glade.get_widget('dialog_include').hide()

        return True

    # ____________________________________________________________
    # status bar
    def set_status_bar_value(self, msg):
        """
            updates the status bar
            :param msg: new status bar value
        """
        self.display_message(msg)


    def display_message(self, msg):
        """ Affiche un message d'information
        """
        context_id = self.status_bar.get_context_id('Message info')
        self.status_bar.push(context_id, msg)
    # ____________________________________________________________

    def update(self):
        """Méthode de mise a jour du tableau des flux
        """
        table = self.create_table()
        self.update_table(table)

    def show_qosshare(self, arg):
        """affichage des bandes de qos
        liste des zones pour la bande passante de la matrice de qos
        en fait la qos ne peut avoir lieu que sur l'extérieur.
        """
        zones=[zone for zone in self.model.real_zones() ] # a priori ext est dedans if zone.name != 'exterieur']
        self.bwshare = QosShareWindow(self.glade_file)
        self.bwshare.gen_bandwidths(zones)
        self.bwshare.show()
    # ____________________________________________________________
    def build_table_button(self, table, directive_store, directives_list, min_index, max_index):
        """
            Constructeur de bouton du tableau des flux en fonction des couleurs
        """
        red = "#de8781"
        green = "#85c990"
        if directive_store.default_policy:

            color = green
        else:
            color = red
        lendir=len(directives_list)
        if lendir > 1:
            dirstr = "directives"
        else:
            dirstr = "directive"
        button = build_image_button("%d %s"%(lendir, dirstr),
                                   ERA_DIR+'/images/search25.png', color="%s"%color)
        button.connect('clicked', self.ctrl.show_directives, directive_store)
        table.attach(button, min_index+1, min_index+2,
                     max_index+1, max_index+2)

    def create_table(self):
        """ Crée une nouvelle table en fonction des données du modèle
        """
        nb_zones = len(self.model.zones)
        table = gtk.Table(nb_zones+1, nb_zones+1)
        table.set_homogeneous(True)

        index = 1
        # Pour chaque zone, on crée un bouton, et on l'attache
        for zone in self.model.zones:
            button = gtk.Button(zone.name)
            button.set_relief(gtk.RELIEF_NONE)
            button.connect("button_press_event", self.ctrl.show_zone_ctx_menu, zone)
            # affichage de la qos
            if zone.name == 'exterieur':
                button.connect("clicked", self.show_qosshare)

            # On attache en titre de colonne ET titre de ligne
            table.attach(button, 0, 1, index, index+1)

            button = gtk.Button(zone.name)
            # affichage de la qos
            # button.connect("clicked", self.show_qosshare, zone)
            if zone.name == 'exterieur':
                button.connect("clicked", self.show_qosshare)

            button.set_relief(gtk.RELIEF_NONE)
            button.connect("button_press_event",self.ctrl.show_zone_ctx_menu, zone)
            table.attach(button, index, index+1, 0, 1)
            index += 1

        # La diagonale = pas de labels
        for i in range(nb_zones+1):
            image = gtk.Image()
            if i==0:
                image.set_from_file(ERA_DIR+"/images/matrix_arrow.xpm")
            elif i == nb_zones:
                image.set_from_file(ERA_DIR+"/images/eole.xpm")
            #button = gtk.Button('')
            #button.set_relief(gtk.RELIEF_NONE)
            #button.set_sensitive(1)

            table.attach(image, i, i+1, i, i+1)
        for flux in self.model.flux:
            down_zone = flux.get_lowlevel_zone()
            up_zone = flux.get_highlevel_zone()
            max_index = self.model.get_zone_index(up_zone)
            min_index = self.model.get_zone_index(down_zone)

            # cas directive DOWN (descendante)
            self.build_table_button(table, flux.down_directives_model(), flux.down_directives_list(), min_index, max_index)

            # cas directive UP (MONTANTE)
            self.build_table_button(table, flux.up_directives_model(), flux.up_directives_list(), max_index, min_index)

        return table


    def update_table(self, table):
        """ Mets à jour la table contenant la matrice de flux
        """
        # cacher l'ancienne table pour éviter les scintillements
        self.zones_table.hide()
        # Enlever l'ancienne table de la boîte
        self.main_vbox.remove(self.zones_table)
        self.zones_table.destroy()

        self.zones_table = table
        # On empile la nouvelle table, et on la remet à la bonne position
        self.main_vbox.pack_start(self.zones_table)
        self.main_vbox.reorder_child(self.zones_table,2)

        # On affiche cette table
        # self.zones_table.show()
        self.main_vbox.show_all()
        # self.main_window.show_all()
    # ____________________________________________________________

    def export(self, *args) :
        """
        glade : l'objet XML glade
        """
        self.export_warn()

    def export_cb(self, *args):
        """callback d'avertissement pour l'exportation
        """
        if self.ctrl.filename is not None:
            default_savename = os.path.splitext(os.path.basename(self.ctrl.filename))[0] + ".sh"
            chooser = gtk.FileChooserDialog(title="firewall export",action=gtk.FILE_CHOOSER_ACTION_SAVE,
                                      buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_SAVE,gtk.RESPONSE_OK))
            chooser.set_default_response(gtk.RESPONSE_OK)
            chooser.set_filename(default_savename)
            filtre = gtk.FileFilter()
            filtre.set_name("sh files")
            filtre.add_mime_type("text")
            filtre.add_pattern("*.sh")
            chooser.add_filter(filtre)
            folder = os.path.join(ERA_DIR,'modeles')
            chooser.set_current_folder(folder)
            chooser.add_shortcut_folder(folder)
            response = chooser.run()
            if response == gtk.RESPONSE_OK:
                self.save_file(chooser.get_filename())

            chooser.destroy()
        else:
            self.export_warn()

    def save_file(self, filename):
        """exporte le .sh
        """
        try:
            matrix_model = initialize_app(self.ctrl.filename)
            try:
                import creole
                variables = None
            except:
                unspecified_vars = matrix_model.unspecified_vars()
                if unspecified_vars:
                    variables = ask_for_unspecified_vars(unspecified_vars, self.main_window)
                else:
                    variables = None
            # si authentification alors
            # generation du fichier nufw_acl_plaintext
            if list(matrix_model.visit_user_group()) != []:
                gen_nufw_acl_plaintext(os.path.splitext(filename)[0]+"_acl_plaintext.sh", self.ctrl.filename)
            # ecriture du script iptables dans le fichier de sortie
            compiler.compile_matrix(matrix_model, filename, variables, no_creole=True)

        except IOError:
            dlg = create_error_dialog(_('export error'))
            dlg.show_all()

    def export_warn(self):
        """avertissement : enregistrer le xml avant
        """
        dialog = create_ok_cancel_dialog(_("save before export"))
        dialog.show()
        reponse = dialog.run()
        if reponse == gtk.RESPONSE_OK:
            dialog.destroy()
            self.export_cb()

        dialog.destroy()
# ____________________________________________________________
# exportation zephir

    def _connect_zephir(self, widget, command):
        """
            connection au zephir
            :param command:
        """

        if self.ctrl.filename is None and command == 'send':
            # fichier non sauvegardé
            dial_err = create_error_dialog(_("filename not set"),self.main_window)
            dial_err.show_all()
            return

        if not self.connected:
            glade = gtk.glade.XML(self.glade_file,"connect_zephir", "editeur")
            # ouverture de la boite de dialogue connect_zephir
            dialog = glade.get_widget('connect_zephir')
            login = glade.get_widget('entry_login_zeph')
            passwd = glade.get_widget('entry_pwd_zeph')
            adresse = glade.get_widget('entry_addr_zeph')
            port = glade.get_widget('entry_port_zeph')
            # récupération des anciennes données
            if self.zeph_user is not None:
                login.set_text(self.zeph_user)
            if self.zeph_pwd is not None:
                passwd.set_text(self.zeph_pwd)
            if self.zeph_addr is not None:
                adresse.set_text(self.zeph_addr)
            if self.zeph_port is not None:
                port.set_text(self.zeph_port)
            else:
                try:
                    from zephir.zephir_conf import zephir_conf
                except:
                    pass
                else:
                    glade.get_widget('entry_addr_zeph').set_text(str(zephir_conf.adresse_zephir))
            log_ok = 0
            while log_ok == 0:
                ret=dialog.run()
                if ret == gtk.RESPONSE_OK:
                    login.grab_focus()
                    self.zeph_user = login.get_text()
                    self.zeph_pwd = passwd.get_text()
                    self.zeph_addr = adresse.get_text()
                    self.zeph_port = port.get_text()
                    try:
                        self.zephir = ServerProxy('https://%s:%s@%s:%s' % (self.zeph_user,self.zeph_pwd,self.zeph_addr,self.zeph_port))
                        self.zephir.get_permissions(str(self.zeph_user))
                    except Exception, e:
                        dial_err = create_error_dialog(_("error_connect_zephir"),dialog)
                        dial_err.show_all()
                    else:
                        # authentification réussie sur zephir
                        self.connected = 1
                        log_ok = 1
                        dialog.destroy()
                        if command=='get':
                            self.get_zephir()
                        if command=='send':
                            self.send_zephir()
                else:
                    dialog.destroy()
                    # log_ok = 1
                    return
        else:
            # déjà authentifié
            if command=='get':
                self.get_zephir()
            if command=='send':
                self.send_zephir()

    def get_zephir(self, *args):
        # on demande les informations de connexion et le n° de serveur pour zephir
        glade = gtk.glade.XML(self.glade_file,"firewall_send_zephir", "editeur")
        self.combo_variante = glade.get_widget('combo_variante')
        self.combo_groupe = glade.get_widget("combo_groupe")
        self.combo_variante.set_sensitive(False)
        self.combo_groupe.set_sensitive(False)
        # ouverture de la boite de dialogue connect_zephir
        dial_zeph = glade.get_widget('firewall_send_zephir')
        self.variantes={}
        try:
            # recherche de l'id du module amon
            liste=self.zephir.modules.get_module()[1]
            self.id_amon={}
            for mod in liste:
                if mod['libelle'].startswith("amon"):
                    self.id_amon[mod['id']] = mod['libelle']
            # recherche des variantes d'amon
            liste=self.zephir.modules.get_variante()[1]
            for var in liste:
                if var['module'] in self.id_amon.keys() and var['libelle'] != "standard":
                    libelle = "%s (%s)" % (var['libelle'],self.id_amon[var['module']])
                    self.variantes[libelle]=(var['id'],var['module'])
            self.combo_variante.get_model().clear()
            if self.variantes.keys() != []:
                for key in self.variantes.keys():
                    self.combo_variante.append_text(key)
#                        self.combo_variante.set_popdown_strings(self.variantes.keys())
        except Exception, e:
            pass
        self.combo_groupe.hide()
        glade.get_widget("radiobutton2").hide()
        glade.get_widget("radiobutton3").show()
        glade.get_widget("entry_id_server").grab_focus()
        handlers = {
            "on_firewall_send_zephir_response" : (self._get_zephir, glade)
            ,"on_firewall_send_zephir_close" : self.close
            ,"on_radiobutton3_toggled" : (self._toggle_zserv, glade)
            }
        glade.signal_autoconnect(handlers)

    def _get_zephir(self, dial_zeph, response_code, glade):
        if (response_code == gtk.RESPONSE_OK) :
            # réponse ok
            if glade.get_widget('radiobutton3').get_active() == True:
                # mode variante, on récupère la liste des modèles présents
                mode = 'variante'
                id_dest=glade.get_widget('combo-entry-variante').get_text()
                id_dest,id_mod=self.variantes[str(id_dest)]
                try:
                    code_ret, var_files = self.zephir.modules.get_var_file(int(id_dest), int(id_mod), 'fichiers_zephir')
                    assert code_ret == 1
                    # affichage d'un dialogue de sélection du modèle
                    modeles = []
                    for fic in var_files:
                        if fic.endswith('.xml'):
                            modeles.append(fic[:fic.index('.xml')])
                    if modeles == []:
                        dial=create_error_dialog(_('no_model_found'))
                        dial.show_all()
                        dial_zeph.hide()
                        return
                    else:
                        dial_sel = gtk.MessageDialog(None, gtk.DIALOG_MODAL, gtk.MESSAGE_QUESTION, gtk.BUTTONS_OK_CANCEL, _('select a model'))
                        dial_sel.set_title(_('model_selection'))
                        vbox = dial_sel.vbox.get_children()[0].get_children()[1]
                        combo = gtk.combo_box_entry_new_text()
                        for fic in modeles:
                            combo.append_text(fic)
                        combo.set_active(0)
                        vbox.pack_start(combo,False,False)
                        dial_sel.show_all()
                        ret_code = dial_sel.run()
                        if ret_code == gtk.RESPONSE_OK:
                            # récupération du nom de modèle dans la combobox
                            modele_var = combo.child.get_text()
                            # lecture du modèle sur zephir
                            code_ret, data = self.zephir.modules.get_var_file(int(id_dest), int(id_mod), 'fichiers_zephir/%s.xml' % modele_var)
                            res = code_ret, data, modele_var
                        dial_sel.destroy()
                    dial_zeph.hide()
                except Exception,e:
                    errmsg = _('error_connect_zephir') + str(e)
                    self.display_message(errmsg)
                    dlg = create_error_dialog(errmsg)
                    dlg.show_all()
                    dial_zeph.hide()
                    return
            else:
                # mode serveur : on récupère le modèle actif
                try:
                    id_serveur=glade.get_widget('entry_id_server').get_text()
                    res=self.zephir.serveurs.get_bastion(int(id_serveur))
                except Exception,e:
                    errmsg = _('error_connect_zephir') + str(e)
                    self.display_message(errmsg)
                    dlg = create_error_dialog(errmsg)
                    dlg.show_all()
                    dial_zeph.hide()
                    return
            if res[0] == 0:
                errmsg = _('error_get_zephir') + ' ' + str(res[1])
                self.display_message(errmsg)
                dlg = create_error_dialog(errmsg)
                dlg.show_all()
            else:
                # on a bien récupéré le fichier, on charge le modèle en mémoire
                try:
                    temp_path = tempfile.gettempdir() + os.sep + res[2] + ".xml"
                    f=open(temp_path,'w')
                    f.write(base64.decodestring(res[1]))
                    f.close()
                    self.ctrl.load_firewall(None, self, inherited = False, zephir = temp_path)
                except:
                    dial=create_error_dialog(_('error_get_zephir'))
                    dial.show_all()
            dial_zeph.hide()
        else :
            # réponse cancel
            dial_zeph.hide()

    def send_zephir(self, *args):
        """
            envoi vers le zephir
            on demande les informations de connexion et le n° de serveur pour zephir
        """

        # FIXME : donner le choix entre sauvegarde pour un serveur ou une variante ?
        glade = gtk.glade.XML(self.glade_file,"firewall_send_zephir", "editeur")
        # ouverture de la boite de dialogue send_zephir
        dial_zeph = glade.get_widget('firewall_send_zephir')
        # on récupère la liste des groupes et les variantes d'amon
        self.variantes={}
        self.combo_variante = glade.get_widget('combo_variante')
        self.combo_groupe = glade.get_widget("combo_groupe")
        self.combo_variante.set_sensitive(False)
        self.combo_groupe.set_sensitive(False)
        groupes={}
        glade.get_widget("radiobutton2").show()
        glade.get_widget("radiobutton3").show()
        glade.get_widget('entry_id_server').grab_focus()
        try:
            # recherche de l'id du module amon
            liste=self.zephir.modules.get_module()[1]
            self.id_amon={}
            for mod in liste:
                if mod['libelle'].startswith("amon"):
                    self.id_amon[mod['id']] = mod['libelle']
            # recherche des variantes d'amon
            liste=self.zephir.modules.get_variante()[1]
            for var in liste:
                if var['module'] in self.id_amon and var['libelle'] != "standard":
                    libelle = "%s (%s)" % (var['libelle'],self.id_amon[var['module']])
                    self.variantes[libelle]=(var['id'],var['module'])
            # recherche des groupes de serveur
            liste=self.zephir.serveurs.get_groups()[1]
            for groupe in liste:
                groupes[groupe[1]]=groupe[0]

            self.combo_groupe.get_model().clear()
            if groupes.keys() != []:
                for key in groupes.keys():
                    self.combo_groupe.append_text(key)
            self.combo_variante.get_model().clear()
            if self.variantes.keys() != []:
                for key in self.variantes.keys():
                    self.combo_variante.append_text(key)
        except Exception, e:
            dial=create_error_dialog(str(e))
            dial.show_all()
        else:
            try:
                from zephir.zephir_conf import zephir_conf
            except:
                pass
            else:
                glade.get_widget('entry_id_server').set_text(str(zephir_conf.id_serveur))

        handlers = {
            "on_firewall_send_zephir_response" : (self._send_zephir, glade, groupes)
            ,"on_firewall_send_zephir_close" : self.close
            ,"on_radiobutton2_toggled" : (self._toggle_zserv,glade)
            ,"on_radiobutton3_toggled" : (self._toggle_zserv,glade)
            }
        glade.signal_autoconnect(handlers)

    def _send_zephir(self, dial_zeph, response_code, glade, groupes):
        if (response_code == gtk.RESPONSE_OK) :
            # réponse ok
            # on récupère les données de la boite de dialogue
            mode = 'serveur'
            id_dest=None
            if glade.get_widget('radiobutton1').get_active() == True:
                id_dest=glade.get_widget('entry_id_server').get_text()
                id_dest=int(id_dest)
            elif glade.get_widget('radiobutton2').get_active() == True:
                mode = 'groupe'
                id_dest=glade.get_widget('combo-entry-groupe').get_text()
                id_dest=groupes[str(id_dest)]
            elif glade.get_widget('radiobutton3').get_active() == True:
                mode = 'variante'
                id_dest=glade.get_widget('combo-entry-variante').get_text()
                id_dest = self.variantes[str(id_dest)][0]

            # on sauvegarde le fichier dans un emplacement temporaire
            old_filename = self.ctrl.filename
            temp_path = tempfile.mktemp()
            self.ctrl.filename = temp_path
            self.ctrl._do_save(self)
            # on remet le chemin d'origine pour la sauvegarde du fichier
            self.ctrl.filename = old_filename
            try:
                temp_file = open(temp_path,'r')
                buff_zephir = temp_file.read()
                temp_file.close()
                os.unlink(temp_path)
            except :
                # erreur de lecture du fichier xml temporaire
                self.display_message(_('error_temp_zephir'))
            else:

                if self.ctrl.filename is None:
                    dial = create_error_dialog(_('filename not set'))
                    dial.show_all()
                    dial_zeph.hide()
                    return

                # envoi du buffer à zephir
                try:
                    if id_dest != None:
                        modele_bastion = os.path.splitext(os.path.basename(self.ctrl.filename))[0]
                        if mode == 'serveur':
                            # envoi sur un serveur
                            res=self.zephir.serveurs.save_bastion(id_dest, base64.encodestring(buff_zephir), modele_bastion)
                        elif mode == 'groupe':
                            # envoi sur un groupe
                            res=self.zephir.serveurs.save_bastion_groupe(id_dest, base64.encodestring(buff_zephir), modele_bastion)
                        else:
                            # envoi sur une variante
                            res=self.zephir.modules.add_files(id_dest,{"dicos_var":[],"persos_var":[],"patchs_var":[],"fichiers_var":[['/usr/share/era/modeles/'+modele_bastion+'.xml', base64.encodestring(buff_zephir)]],"rpms_var":[]})

                except Exception, e:
                    self.display_message(_('error_connect_zephir') + str(e))
                else:
                    if res[0] == 0:
                        self.display_message(_('error_send_zephir') + res[1])
                    else:
                        self.display_message(_('send_zephir_ok') % (mode,str(id_dest)))
            dial_zeph.hide()
        else :
            # réponse cancel
            dial_zeph.hide()

    def _toggle_zserv(self,widget,glade,*args):
        """active la bonne zone de saisie"""
        b_serv = glade.get_widget("entry_id_server")
        b_serv.set_sensitive(glade.get_widget("radiobutton1").get_active())
        self.combo_groupe.set_sensitive(glade.get_widget("radiobutton2").get_active())
        self.combo_variante.set_sensitive(glade.get_widget("radiobutton3").get_active())

# end zephir
# ____________________________________________________________

    def close(self, fs, *args):
        fs.hide()

    def about(self, *args):
        """Callback appelée lorsque l'on clique sur 'About'
        """
        about = create_about_dialog()
        about.run()
        about.destroy()

    def quit(self, *args):
        """ Callback 'quit'
        """
        if self.ctrl.saved == 0:
            dialog = create_ok_cancel_dialog(_("alert quit"),self.main_window)
            dialog.show()
            reponse = dialog.run()
            if reponse == gtk.RESPONSE_OK:
                gtk.main_quit()
            dialog.destroy()
            return True
        else :
            gtk.main_quit()

class MatrixManager:
    """ Classe gérant la matrice de flux
    """
    def __init__(self, glade_file, model = None):
        """
        glade_file : le fichier glade XML.
        """
        self.glade_file = glade_file
        self.model = model
        self.inherited = None
        self.name = _('new document')
        # le nom du fichier firewall sur lequel on travaille
        self.filename = None
        # load_defaults(model, library_store.services)
        # conversion du dictionnaire creole (si on n'est pas sur un amon)
        self.creole_conv = False
    # ____________________________________________________________

    # FIXME je ne suis pas sûr que ça soit utilisé
    def set_model(self, model):
        self.model = model
        self.update()
    # ____________________________________________________________

    def show_zone_ctx_menu(self, zone_button, event, zone):
        """ Callback appelée lorsque l'on clique sur un bouton de zone
        """
        if event.button == 3:
            glade = gtk.glade.XML(self.glade_file, "zone_ctx_menu", "editeur")
            menu = glade.get_widget('zone_ctx_menu')
            handlers = {'on_edit_zone_item_activate' : self.edit_zone,
                        'on_delete_zone_item_activate' : self.delete_zone,
                        'on_new_subnet_item_activate' : self.add_subnet,
                        'on_new_machinelist_item_activate' : self.add_machines,
                        'on_show_extremites_item_activate' : self.show_extremites,
                        }
            if not zone.is_editable():
                # On désactive lest items edit et delete
                edit_item = glade.get_widget('edit_zone_item')
                edit_item.set_sensitive(False)
                delete_item = glade.get_widget('delete_zone_item')
                delete_item.set_sensitive(False)
            glade.signal_autoconnect(handlers)
            self.selected_zone = zone
            menu.popup(None, None, None, event.button, event.time)
            return True
        elif (event.type == 5) and (zone.is_editable()):
            self.selected_zone = zone
            self.edit_zone(zone_button)
            return True
        return False

    def show_directives(self, button, directive_store):
        """
        """
        ctrl = DirectiveManager(self.glade_file)

        dir_view = DirectiveSummaryView(self.glade_file, ctrl,
                                        directive_store)
        dir_view.show_window()
# ____________________________________________________________
    def valid_include(self, button, view):
        """valide la saisie des includes
        """
        text = view.include_glade.get_widget('text_include').get_buffer()
        start = text.get_start_iter()
        end = text.get_end_iter()
        library_store.rules = text.get_text(start,end)
        view.include_glade.get_widget('dialog_include').hide()
# ____________________________________________________________

    def add_zone(self, *args):
        zone_dlg = ZoneDialogManager(self.glade_file, self.model)
        zone_dlg.show_dialog()

    def edit_zone(self, widget):
        zone_dlg = ZoneDialogManager(self.glade_file, self.model,
                                     self.selected_zone)
        zone_dlg.show_dialog()

    def delete_zone(self, widget):
        library_store.remove_extremites(self.selected_zone)
        self.model.remove_zone(self.selected_zone)

    def add_subnet(self, widget):
        if self.selected_zone.name != 'bastion':
            subnet_dlg = DialogReseauManager(self.glade_file,self.selected_zone)
            subnet_dlg.show_dialog()
        else:
            dial_error = create_error_dialog(_("add subnet not allowed"))
            dial_error.show()

    def show_extremites(self, widget):
        """affichage de la liste des extrémités rattachées à une zone
        """
        ext_dlg = ExtremiteListviewDialog(self.glade_file, self.selected_zone)
        ext_dlg.show_dialog()

    def show_all_extremites(self,widget):
        """affichage de la liste de toutes les extrémités définies
        """
        ext_dlg = ExtremiteListviewDialog(self.glade_file,zone=None,liste_zones=self.model.zones)
        ext_dlg.show_dialog()

    def show_services(self, widget):
        """ affichage de la fenêtre de visualisation des services
        """
        serv_dlg = ServiceListviewDialog(self.glade_file)
        serv_dlg.show_dialog()

    def add_service_group(self, widget):
        """ affichage du dialogue de création de groupes de services
        """
        grp_dlg = ServiceGroupeDialog(self.glade_file)
        grp_dlg.show_dialog()

    def show_user_groups(self, widget):
        """ affichage de la fenêtre de visualisation des services
        """
        user_group_dlg = UserGroupDialog(self.glade_file)
        user_group_dlg.show_dialog()

    def add_service(self, widget):
        """ affichage du dialogue de création de service
        """
        serv_dlg = ServiceDialog(self.glade_file)
        serv_dlg.show_dialog()

    def add_machines(self, widget):
        #if self.selected_zone.name != 'bastion':
        machines_dlg = DialogMachinesManager(self.glade_file,self.selected_zone)
        machines_dlg.show_dialog()
        #else:
        #    dial_error = create_error_dialog(_("add extremite not allowed"))
        #    dial_error.show()

    def _do_save(self, view):
        """Méthode de sauvegarde 'réelle'
        """
        # affecte les identifiants
        # self._parse_directives(view)
        try:
            view.model.save(self.filename, inherited=self.inherited, creole_conv=self.creole_conv)
            self.saved = 1;
        except Exception, e:
            import traceback
            traceback.print_exc()
            create_error_dialog('Erreur inconnue : ' + str(e)).show_all()


    def cb_fs_save(self, ok_button, view):
        """Callback appelée lorsqu'on a choisi le fichier dans le
        sélecteur
        """
        fs = ok_button.get_toplevel()
        filename = fs.get_filename()
        # ajoute l'extension si elle n'est pas renseignee par l'utilisateur
        if not filename.endswith(".xml"):
            filename = filename + ".xml"
        fs.destroy()
        try:
            os.stat(filename)
        except OSError, e:
            self.filename = filename
            self._do_save(view)
            view.display_message(filename)
        else:
            dial_conf = create_ok_cancel_dialog(_("file exists"), view.main_window)
            dial_conf.set_default_response(gtk.RESPONSE_NO)
            ret = dial_conf.run()
            dial_conf.destroy()
            if ret == gtk.RESPONSE_OK:
                self.filename = filename
                view.display_message(self.filename)
                self.name = os.path.basename(self.filename)
                self._do_save(view)
                # si le modèle est converti de creole 1 vers creole 2, on le recharge
                # pour avoir les variables correctes
                if self.creole_conv == True:
                    inherited_model = self.model.inherited
                    inherited = False
                    if inherited_model != None:
                        inherited = True
                    model = initialize_app(filename, inherited)
                    view.set_model(model)
                    # si le modèle chargé a un modèle, on le récupère
                    # (pas dans le cas où on importe : c'est le nom du fichier importé)
                    self.inherited = inherited_model
                # si des variables ne sont pas retrouvées en cas de conversion
                # on les affiche
                if self.model.unknown_vars != []:
                    dlg = create_warning_dialog(_('unknown_vars') + ' :\n\n%s' % '\n'.join(self.model.unknown_vars))
                    dlg.show_all()

    def save(self, item, view):
        """
          Callback appelé pour sauvegarder un firewall au format interne.
          (fichier > enregister)
        """
        # on réinitialise le flag de conversion avant test de la version
        self.creole_conv = False
        self._save_update_version(item, view)
        if self.filename is None:
            fs = gtk.FileSelection(_("save file"))
            fs.ok_button.connect("clicked",self.cb_fs_save, view)
            fs.cancel_button.connect('clicked', destroy_parent)
            filename = os.path.join(ERA_DIR,'modeles')+os.sep
            fs.set_filename(filename)
            fs.show()
        else:
            self._do_save(view)

    def save_as(self, item, view):
        """Callback appelée pour sauvegarder un firewall au format interne.
        """
        # on réinitialise le flag de conversion avant test de la version
        self.creole_conv = False
        self._save_update_version(item, view)
        chooser = gtk.FileChooserDialog(title=_("save file"),action=gtk.FILE_CHOOSER_ACTION_SAVE,
                                  buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_SAVE,gtk.RESPONSE_OK))
        chooser.set_default_response(gtk.RESPONSE_OK)
        chooser.set_filename(os.path.join(ERA_DIR,'modeles')+os.sep)
        filtre = gtk.FileFilter()
        filtre.set_name("xml files")
        filtre.add_mime_type("text/xml")
        filtre.add_pattern("*.xml")
        chooser.add_filter(filtre)
        folder = os.path.join(ERA_DIR,'modeles')
        chooser.set_current_folder(folder)
        chooser.add_shortcut_folder(folder)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            self.cb_fs_save(chooser, view)
        chooser.destroy()

    def _save_update_version(self, item, view):
        """fonction de conversion des anciens formats lors de la sauvegarde du modèle
        """
        self.model.unknown_vars = []
        if library_store.get_version() != version:
            versions = ['1.2','2.0', '2.3']
            if library_store.get_version() == 1.0:
                versions.insert(0,'1.0')
            dlg = create_yes_no_dialog(_('maj'))
            hbox = dlg.vbox.get_children()[0]
            c_version = gtk.combo_box_new_text()
            for vers in versions:
                iter = c_version.append_text(vers)
            c_version.set_active(len(versions)-1)
            hbox.pack_start(c_version, False, False, 0)
            hbox.show_all()
            response = dlg.run()
            dlg.set_default_response(gtk.RESPONSE_NO)
            if response == gtk.RESPONSE_YES:
                # on regarde la version demandée
                vers = c_version.get_active_text()
                if library_store.get_version() < 2.0:
                    if vers == '2.0' or vers == '2.3':
                        # version utilisant creole 1, on convertit
                        self.creole_conv = True
                library_store.set_version(vers)
            dlg.destroy()
            return True

    def new_firewall(self, item, view):
        """Callback appelée lorsque l'on clique sur 'New'
        """
        # affichons un avertissement lorsque le modèle en cours va être supprimé
        if len(self.model.flux) != 1 and len(self.model.zones) != 2 :
            dialog = create_yes_no_dialog(_("are you sure you want to create a new firewall ? (all modifications will be lost)"))
            dialog.set_default_response(gtk.RESPONSE_NO)
            response = dialog.run()
            if response == gtk.RESPONSE_YES:
                model = initialize_app()
                view.set_model(model)
                view.display_message(_('new document'))
                self.name = _('new document')
                self.model = model
            dialog.destroy()

    def load_firewall(self, filename, view, inherited = False, zephir = None):
        """Callback appelée après le sélecteur de fichiers pour ouvrir
        un firewall
        """
        if zephir is None:
            # le filename est celui passe en parametre
            pass
        else:
            filename=zephir
        if inherited:
            self.inherited = filename
        try:
            model = initialize_app(filename, inherited)
            # si le modèle chargé a un modèle, on le récupère
            # (pas dans le cas où on importe : c'est le nom du fichier importé)
            if not inherited:
                self.inherited = model.inherited

        except Exception, e:
            dlg = create_error_dialog(str(e))
            dlg.show_all()
            return
        # FIXME : a mettre dans un controle plus general du modele
        # vérifie que toutes les directives optionnelles ont des tags
        try:
            model.check_optional_list()
        except:
            dlg = create_error_dialog(_('empty filename'),view.main_window)
            dlg.show_all()

        view.set_model(model)
        view.display_message(filename)
        self.name = os.path.basename(filename)
        self.model = model
        if inherited:
            self.inherited = filename
        if not inherited:
            self.filename = filename
            #self.view.display_message(self.filename)
        self.saved = 0;

    def import_firewall(self, item, view):
        """Callback appelée lorsque l'on clique sour 'Ouvrir'
        """
        self._load_xml_file_dialog(_("import firewall"), False, item, view)

    def import_firewall_model(self, item, view):
        """Callback appelée lorsque l'on choisit 'importer un modele'
        """
        self._load_xml_file_dialog(_('import firewall model'), True, item, view)

    def _load_xml_file_dialog(self, libelle, ok_boolean, item, view):
        chooser = gtk.FileChooserDialog(title=None,action=gtk.FILE_CHOOSER_ACTION_OPEN,
                                  buttons=(gtk.STOCK_CANCEL,gtk.RESPONSE_CANCEL,gtk.STOCK_OPEN,gtk.RESPONSE_OK))
        chooser.set_default_response(gtk.RESPONSE_OK)
        filtre = gtk.FileFilter()
        filtre.set_name("xml files")
        filtre.add_mime_type("text/xml")
        filtre.add_pattern("*.xml")
        chooser.add_filter(filtre)
        folder = os.path.join(ERA_DIR,'modeles')
        chooser.set_current_folder(folder)
        chooser.add_shortcut_folder(folder)
        response = chooser.run()
        if response == gtk.RESPONSE_OK:
            self.load_firewall(chooser.get_filename(), view, ok_boolean)
        chooser.destroy()

# ____________________________________________________________
def main(model, model_file):
    """
        point d'entrée de l'interface
        :param model: model = noyau.initialize.initialize_app()
    """
    ctrl = MatrixManager(GLADE_FILE, model)
    view = MatrixView(GLADE_FILE, ctrl, model)
    view.show_window()
    if model_file:
        ctrl.load_firewall(model_file, view)
    gtk.main()

