# -*- coding: UTF-8 -*-
"""Module de tests unitaires pour l'API de génération
de règles iptables
"""

from py.test import raises

import autopath
import sys

from era.backend import iptrule
from cStringIO import StringIO

def test_parameter():
    """
        test de la classe Parametre
    """
    pass

class LStringIO(list):
    """petite classe qui émule write et getvalue
    en utilisant des listes
    """
    write = list.append
    def getvalue(self, join_char = ''):
        return join_char.join()

def test_match_parameter():
    """Teste MatchParameter (-m)
    """
    param = iptrule.MatchParameter('policy')
    assert str(param) == '-m policy'

def test_matchset_parameter():
    param = iptrule.MatchSetParameter('setname')
    assert str(param) == '--match-set setname src'
    param = iptrule.MatchSetParameter('setname', src=False)
    assert str(param) == '--match-set setname dst'

def test_matchset():
    param = iptrule.MatchSet()
    assert str(param) == '-m set'

## ipsec
def test_proto_parameter():
    """
        ProtoParameter (--proto, pour ipsec)
    """
    param = iptrule.ProtoParameter('est')
    assert str(param) == '--proto est'

def test_policy_parameter():
    """
        ProtoParameter (--proto, pour ipsec)
    """
    param = iptrule.PolicyParam('in')
    assert str(param) == '--pol in'

def test_dir_parameter():
    """
        DirParameter
    """
    param = iptrule.DirParameter('in')
    assert str(param) == '--dir in'
## end ipsec

def test_protocol_parameter():
    """
        ProtocolParameter (-p)
    """
    # protocole court (-p)
    param = iptrule.ProtocolParameter('est')
    assert str(param) == '-p est'


def test_ip_options():
    """Teste IPOptions (-s / -d)
    """
    param = iptrule.SrcIPParameter('0.0.0.0', '0')
    assert str(param) == '-s 0.0.0.0/0'

    param = iptrule.DestIPParameter('0.0.0.0')
    assert str(param) == '-d 0.0.0.0'

def test_interface_options():
    """Teste InterfaceParameter (-i / -o)
    """
    param = iptrule.InterfaceParameter('eth2', out = False)
    assert str(param) == '-i eth2'
    param = iptrule.InterfaceParameter('eth2', out = True)
    assert str(param) == '-o eth2'
    param = iptrule.InterfaceParameter('eth2', out = False, inverted = True)
    assert str(param) == '-i ! eth2'


def test_icmp_options():
    """Teste ICMPParameter
    """
    param = iptrule.ICMPProtocol('source-quench')
    assert '-p icmp' in str(param)
    assert  '--icmp-type source-quench' in str(param)

    param = iptrule.ICMPProtocol('source-quench', inverted = True)
    assert '-p icmp' in str(param)
    assert  '!' in str(param)

def test_tcp_options():
    """Teste TCPOptions
    """
    param = iptrule.TCPProtocol(dport = '1', sport = '2')
    assert str(param) == '-p tcp --dport 1 --sport 2'
    param = iptrule.TCPProtocol(dport = [1,10], sport = '1')
    assert str(param) == '-p tcp --dport 1:10 --sport 1'
    param = iptrule.TCPProtocol(dport = 1, sport = 2, flags = ['SYN', 'ACK'])
    assert str(param) == '-p tcp --dport 1 --sport 2 --tcp-flags SYN,ACK SYN'
    raises(AssertionError, iptrule.TCPProtocol, 1, 2, ['SYNY'])


def test_udp_options():
    """Teste UDPOptions
    """
    param = iptrule.UDPProtocol(dport = 1, sport = [2, 10])
    assert str(param) == '-p udp --dport 1 --sport 2:10'


def test_basic_target():
    """Teste les cibles basiques (DROP, ACCEPT, etc.)
    """
    target = iptrule.Target('DROP')
    assert str(target) == '-j DROP'


def test_redirect_target():
    """Teste cible redirect
    """
    target = iptrule.Redirect(2)
    assert str(target) == '-j REDIRECT --to-ports 2'
    target = iptrule.Redirect([2,10])
    assert str(target) == '-j REDIRECT --to-ports 2-10'


def test_redirect_target():
    """Teste cible redirect
    """
    target = iptrule.Masquerade(2)
    assert str(target) == '-j MASQUERADE --to-ports 2'
    target = iptrule.Masquerade([2,10])
    assert str(target) == '-j MASQUERADE --to-ports 2-10'


def test_log_target():
    """Teste cible LOG
    """
    target = iptrule.Log('warn', 'intrusion de :')
    assert str(target) == '-j LOG --log-level warn --log-prefix intrusion de :'


def test_ulog_target():
    """Teste la cible LOG
    """
    target = iptrule.Log(1, 'prefixe', 4, 2)
    assert '-j LOG ' in str(target)
    target = iptrule.Log(1, 'prefixe')
    assert '-j LOG ' in str(target)

def test_mark_target():
    """Teste la cible MARK
    """
    target = iptrule.Mark('marque')
    assert str(target) == '-j MARK --set-mark marque'


def test_reject_target():
    """Teste la cible REJECT
    """
    target = iptrule.Reject('icmp-net-unreachable')
    assert str(target) == '-j REJECT --reject-with icmp-net-unreachable'

    raises(AssertionError, iptrule.Reject, 'blabla')


def test_tos_target():
    """Teste la cible TOS
    """
    target = iptrule.Tos(16)
    assert str(target) == '-j TOS --set-tos 16'
    target = iptrule.Tos('Minimize-Delay')
    assert str(target) == '-j TOS --set-tos Minimize-Delay'

    raises(AssertionError, iptrule.Tos, 'blabla')


def test_nat_target():
    """Teste les cibles DNAT / SNAT
    """
    target = iptrule.Dnat([['0.0.0.0', '0.0.0.1'],
                                '1.1.1.1',
                                '2.2.2.2'],
                                [12, '13-15', 0])
    assert str(target) == '-j DNAT --to-destination 0.0.0.0-0.0.0.1:12 --to-destination 1.1.1.1:13-15 --to-destination 2.2.2.2'

    target = iptrule.Dnat([['0.0.0.0', '0.0.0.1'],
                                '1.1.1.1'],
                                ['12-15', 0])
    assert str(target) == '-j DNAT --to-destination 0.0.0.0-0.0.0.1:12-15 --to-destination 1.1.1.1'

    raises(AssertionError, iptrule.Nat, [], [], 'toto')

