# -*- coding: UTF-8 -*-
"""
visite les directives de la matrice de flux 
récupère les directives optionnelles
"""
import autopath

from era.noyau.erreurs import *
from era.noyau.constants import *

from era.noyau.fwobjects import *
from era.noyau.models import *


class InitMatrice :
    def __init__(self):
        """
        - instancie la matrice
        - peuple un flux avec deux directives
          (une montante, une descendante)
        """
        self._init_zones()
        self._init_services()
        self._init_extremites()
        self._init_directives()
        self._init_matrice()
        # affectation des directives
        self._bind_directive_to_matrix()

    def _init_zones(self):
        zoneA = Zone('zA', 30, '127.0.0.1')
        zoneB = Zone('zB', 31, '127.0.0.1')
        zoneC = Zone('zC', 32, '127.0.0.1')
        
        self.zones = {'zoneA': zoneA,
                 'zoneB': zoneB,
                 'zoneC': zoneC
                      }

    def _init_services(self):
        s1 = Service("serv1", "TCP", [80], "serv1", "libelle_Internet")
        s2 = Service("serv2", "TCP", [480], "serv2", "libelle2")
        s3 = Service("Youpi", "UDP", [4513], "serv3", "libelle3")
        s4 = Service("echo-request", "ICMP", [0], "echo-request", "règle icmp echo-request")
        sg1 = ServiceGroupe("Groupe1", "test de groupe de services",[s1,s2])
        self.services = {"serv1" : s1, "serv2" : s2, "Youpi" : s3, "icmp" : s4}
        self.liste_services = [s1,s2,s3,s4]
        self.groupe_services = {"grp1" : sg1}

    def _init_extremites(self):
        e1 = Extremite(self.zones['zoneA'],"extr1", "libelle1", ["10.0.0.1"], "255.255.255.255")
        e2 = Extremite(self.zones['zoneA'],"extr2", "libelle2", ["10.0.0.2"], "255.255.255.255")
        e3 = Extremite(self.zones['zoneB'],"extr3", "libelle3", ["10.0.0.3"], "255.255.255.255")
        e4 = Extremite(self.zones['zoneC'],"extr4", "libelle4", ["10.0.0.4"], "255.255.255.255")
        self.extremites = {"extr1" : e1,"extr2" : e2,"extr3" : e3,"extr4" : e4}
        self.groupe_extremites = {'grp1' : [e1,e2,e3], 'grp2' : [e2,e3,e4]}

    def _init_directives(self):
        """
        [zA : 30, zB : 31]
        zA -> zB : montant
        zB -> zA : descendant

        extr1 in zoneA
        extr3 in zoneB
        d0 : zA -> zB in flux.up_directives_list()
        d1 : zB -> zA in flux.down_directives_list()

        """
        d0 = Directive([self.extremites['extr1']], [self.extremites['extr3']],
                       self.services['serv1'], ACTION_DENY, 0)

        d1 = Directive([self.extremites['extr3']],[self.extremites['extr1']], 
                       self.services['serv1'], ACTION_ALLOW, 0)

        self.directives = [d0,d1]

    def _init_active_directives(self):
        """Quelques directives optionnelles actives ou optionnelles inactives 
        """

        d0 = Directive([self.extremites['extr1']], [self.extremites['extr3']],
                       self.services['serv1'], ACTION_DENY, 1)

        d1 = Directive([self.extremites['extr3']],[self.extremites['extr1']], 
                       self.services['serv1'], ACTION_ALLOW, 3)

        return [d0,d1]


    def bind_active_directives(self):
        """Dans le flux : [zB <===> zA]
        """
        d0,d1 = self._init_active_directives()
        d0.set_tag("d_zero")
        d1.set_tag("d_un")
        self.matrice.flux[1].up_directives_store.add_directive(d0)
        self.matrice.flux[1].down_directives_store.add_directive(d1)


    def _init_matrice(self) :
        self.matrice = MatrixModel()

        for zone in self.zones :
            self.matrice.add_zone(self.zones[zone])

    def _bind_directive_to_matrix(self):
        """Dans le flux : [zB <===> zA]
        """
        self.matrice.flux[1].up_directives_store.add_directive(self.directives[0])
        self.matrice.flux[1].down_directives_store.add_directive(self.directives[1])

    def get_matrice(self):
        return self.matrice


def test_get_flux():
    m = InitMatrice().get_matrice()
    flux1 = m.get_flux_list(m.zones[0])
    # Flux : [zB <===> zA], Flux : [zC <===> zA]]
    assert len(flux1) == 2

def test_get_directive_list():
    m = InitMatrice().get_matrice()
    flux_list = m.flux
    assert len(m.flux) == 3
    flux = m.flux[1]
    up = flux.up_directives_list()[0]
    assert len(up.src_list) == 1
    assert len(up.dest_list) == 1

    down = flux.down_directives_list()[0]
    assert len(down.src_list) == 1
    assert len(down.dest_list) == 1

def test_parse_directives():
    m = InitMatrice().get_matrice()
    flux_list = m.flux
    directives = []
    # seule maniere de parser des directives
    for f in flux_list:
      directives.extend(f.up_directives_list())
      directives.extend(f.down_directives_list())

    for d in directives:
      assert d.attrs == 0

    directives[0].attrs = 1
    for f in flux_list:
      for d in f.up_directives_list():
        assert d.attrs == 1
      for d in f.down_directives_list():
        assert d.attrs == 0

def test_active_list():
    init = InitMatrice()
    init.bind_active_directives()
    m = init.get_matrice()
    assert m.get_active_list() == [('d_zero', 1, 1, 'lo'), ('d_un', 3, 1, 'lo')]

def test_visit_directives():
    init = InitMatrice()
    init.bind_active_directives()
    m = init.get_matrice()
    assert m.get_active_list() == [('d_zero', 1, 1, 'lo'), ('d_un', 3, 1, 'lo')]

    update_list = [['d_zero',3]]
    m.visit_directives(update_list)
    assert m.get_active_list() == [('d_zero', 3, 1, 'lo'), ('d_un', 3, 1, 'lo')]

def test_tag_list():
    init = InitMatrice()
    init.bind_active_directives()
    m = init.get_matrice()
    assert m.get_tag_list() ==  ['d_zero', 'd_un']
