# -*- coding: UTF-8 -*-
###########################################################################
# 
# Eole NG  
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr 
#  
###########################################################################

"""
Module chargé de gérer les fenêtres pour le filtrage applicatif
"""

import gtk, gtk.glade
import gobject

from era.noyau.fwobjects import ApplicationGroup, Application
from era.noyau.dpatterns import Singleton
from era.noyau.erreurs import InvalidPortListError, EmptyNameError
from gtk_models import create_application_treestore, create_application_group_treestore, \
                       create_application_path_treestore, create_app_treestore
from ihm_utils import create_error_dialog, create_yes_no_dialog
from era.noyau.pool import library_store

class ApplicationListviewDialog(Singleton):

    def __init__(self, glade_file,groupe=None):

        self.glade_file = glade_file
        
        if 'dlg' not in self.__dict__:
            self.glade = gtk.glade.XML(self.glade_file,"application_listview_dialog", "editeur")
            self.dlg = self.glade.get_widget('application_listview_dialog')
            self.treeview =  self.glade.get_widget('application_listview')
            self.application_label =  self.glade.get_widget('applications_label')
            self.init_treeview_column()
#            self.init_group_liststore()
            selection = self.treeview.get_selection()
            selection.set_mode(gtk.SELECTION_SINGLE)
            selection.connect('changed',self.item_selected)
            handlers = {'on_application_listview_dialog_response' : self.close,
                        'on_application_listview_dialog_delete_event' : self.stay_alive,
                        'on_edit_application_button_clicked' : self.edit_application,
                        'on_del_application_button_clicked' : self.del_application,
                        'on_button_add_application_clicked' : self.add_application,
                        'on_button_add_application_group_clicked' : self.add_application_group, 
                        'on_application_listview_button_press_event' : self.edit_application_on_click,
                        }
            self.glade.signal_autoconnect(handlers)
#            self.refresh_treeview()
        else:
            self.flush()
            self.show_dialog()
        self.init_model()

    def flush(self):
        self.application_label.set_text("")

    def add_application(self, *args):
        """
            callback de creation du application
        """
        serv_dlg = ApplicationDialog(self.glade_file, parent=self)
        serv_dlg.show_dialog()
        
    def add_application_group(self, *args):
        """ 
            affichage du dialogue de création de groupes de applications
        """
        grp_dlg = ApplicationGroupDialog(self.glade_file, parent=self)
        grp_dlg.show_dialog()

    def init_model(self):
        """Initialise le modèle associé à l'arbre des applications
        """
        self.treemodel = create_application_treestore()
        self.treemodel = create_application_group_treestore(self.treemodel)
        self.treeview.set_model(self.treemodel)


    def init_treeview_column(self):
        column = gtk.TreeViewColumn('applications and groups list',
                                    gtk.CellRendererText(),
                                    text = 0)
        self.treeview.append_column(column)
        #self.treeview.connect('button-press-event', self.item_selected)


    def item_selected(self, selection, *args):
        """Callback appelée lorsqu'on clique sur un application dans l'arbre
        """
        (store, itersel) = selection.get_selected()
        if itersel is None:
            self.selected_node = None
            return
        
        self.selected_node = store.get_value(itersel,1)
        # On met à jour les propriétés
        self.update_properties()

    def refresh(self):
        """
            recharge le treeview et le update_properties
        """
        self.init_model()
        self.update_properties()

    def update_properties(self):
        """Mets à jour la partie 'propriétés' de la fenêtre de dialogue
        (partie droite)
        """
        # on vérifie si l'objet peut-être modifié
        if self.selected_node is not None:
            if self.selected_node.is_editable():
                self.glade.get_widget('del_application_button').set_sensitive(1)
                self.glade.get_widget('edit_application_button').set_sensitive(1)
            else:
                self.glade.get_widget('del_application_button').set_sensitive(0)
                self.glade.get_widget('edit_application_button').set_sensitive(0)
#            self.prop_label.set_text(str(self.selected_node))
#            self.group_store.clear()
            if self.selected_node.get_type() == "firewall/application_group" or self.selected_node.get_type() == "firewall/application":
                self.application_label.set_text(str(self.selected_node))
        else:
            self.glade.get_widget('del_application_button').set_sensitive(0)
            self.glade.get_widget('edit_application_button').set_sensitive(0)
            self.application_label.set_text("")
#            self.group_store.clear()
            
    def edit_application(self,*args):
        """Ouvre la fenetre d'edition de application
        """
        if self.selected_node:
            if self.selected_node.get_type() == "firewall/application_group":
                # édition d'un groupe
                # FIXME : 
                editdlg=ApplicationGroupDialog(self.glade_file,self.selected_node, parent=self)
                editdlg.show_dialog()
            else:
                # édition d'un application
                editdlg=ApplicationDialog(self.glade_file, app_edit=self.selected_node, parent=self)
                editdlg.show_dialog()

    def edit_application_on_click(self, widget, event, *args):
        """
            edition d'un application sur le double-click
        """
        # bouton gauche
        if event.button == 1:
            # double-clic
            if event.type == gtk.gdk._2BUTTON_PRESS:
                self.edit_application(widget, event, args)

    def del_application(self,*args):
        """supprime un application ou un groupe de la bibliothèque
        """
        if self.selected_node:
            # on vérifie si l'objet est encore utilisé
            if self.selected_node.used > 0:
                dial_error = create_error_dialog(_("object still used"),self.dlg)
                dial_error.show()
            else:
                # dialogue de confirmation
                dial_conf = create_yes_no_dialog(_("remove application confirmation : ") + self.selected_node.name)
                dial_conf.set_default_response(gtk.RESPONSE_NO)
                ret = dial_conf.run()
                dial_conf.destroy()
                # on teste le retour du dialogue
                if ret == gtk.RESPONSE_YES:
                    selection = self.treeview.get_selection()
                    model, iter_sel = selection.get_selected()
                    if self.selected_node.get_type() == "firewall/application_group":
                        # suppression d'un groupe (firewall/application_group)
                        # mise à jour des applications correspondants (TODO)
#                         for application in self.selected_node.applications:
#                            application.used -= 1
                        del(library_store.app_groups[self.selected_node.name])
                    else:
                        # suppresion d'un application (firewall/application)
                        del(library_store.applications[self.selected_node.name])
                    self.treemodel.remove(iter_sel)
                
    def show_dialog(self, *args):
#        self.treeview.set_size_request(250,300)
        self.dlg.show_all()

    def stay_alive(self, *args):
        self.close()
        return True
    
    def close(self, *args):
        self.dlg.hide()
#        self.group_store.clear()

def validate_path(path):
    """
        validation de chemins pour le filtrage applicatif
        :path: nom long du chemin
    """
    if path.strip() == '':
        raise Exception, _("empty string")
    return path


class ApplicationDialog(Singleton):
    """Gère l'édition de applications
    """

    def __init__(self, glade_file, app_edit=None, parent=None):
        """
        glade : l'objet XML glade
        """
        self.app_edit = app_edit
        self.glade_file = glade_file
        self.glade = gtk.glade.XML(self.glade_file,"application_dialog", "editeur")
        self.parent = parent
        if 'dlg' not in self.__dict__:
            self.dlg = self.glade.get_widget('application_dialog')
            self.name_entry = self.glade.get_widget('application_name_entry')
            self.application_name_entry = self.glade.get_widget('application_name_entry')
            self.application_libelle_entry = self.glade.get_widget('application_libelle_entry')
            self.path_list_treeview = self.glade.get_widget("application_path_treeview")
            self.application_path_entry = self.glade.get_widget("application_path_entry")

            selection = self.path_list_treeview.get_selection()
            selection.set_mode(gtk.SELECTION_SINGLE)
            selection.connect('changed',self.item_selected)
            handlers = {
                'on_application_dialog_response' : self.validate_edit,
                'on_application_path_entry_activate' : self.application_path_add,
                'on_application_dialog_delete_event' : self.stay_alive,
                'on_add_path_application_button_clicked': self.application_path_add,
                'on_remove_application_button_clicked':self.application_path_del,
                }
        
            self.glade.signal_autoconnect(handlers)
        else:
            self.flush()
            self.show_dialog()

        if self.app_edit != None:
            self.fill_application()
        else:
            self.init_model()
            self.init_treeview()

    def fill_application(self):
        """
            remplit les données de l'application en cours dans le dialog
        """
        self.name_entry.set_text(self.app_edit.name)
        self.application_libelle_entry.set_text(self.app_edit.description)
        self.name_entry.set_sensitive(0)
        self.init_treeview()
        
    ## gestion du treestore des paths         
    def init_treeview(self):
        """
            remplissage du treeview des paths de l'application en cours
        """
        self.init_model()
        self.init_treeview_column()

    def add_line_treeview(self, path):
        """
            ajoute une ligne au treeview des paths
        """
        self.treestore.append(None, [path])
        
    def init_model(self):
        """Initialise le modèle associé à l'arbre des applications
        """
        self.treestore = create_application_path_treestore(self.app_edit)
        self.path_list_treeview.set_model(self.treestore)

    def init_treeview_column(self):
        column = gtk.TreeViewColumn(_('paths list'),
                                    gtk.CellRendererText(),
                                    text = 0)
        self.path_list_treeview.append_column(column)
        #self.treeview.connect('button-press-event', self.item_selected)

    def application_path_add(self, *args):
        """
            callback lorsqu'on clique sur ajout de path
        """
        path = self.application_path_entry.get_text()
        # etape de validation du path      
        try:
            path = validate_path(path)
            self.add_line_treeview(path)
            self.application_path_entry.set_text("")
        except Exception, msg:
            dialog = create_error_dialog(str(msg))
            dialog.run()
            dialog.destroy()

    def item_selected(self, *args):
        """Callback appelée lorsqu'on clique sur un application dans l'arbre
        """
        selection = self.path_list_treeview.get_selection()
        (store, itersel) = selection.get_selected()

        if itersel is None:
            self.selected_node = None
            return
        self.selected_node = store.get_value(itersel,0)
        return self.selected_node

    def application_path_del(self, *args):
        """
            suppression d'un path pour une application
        """
        selected_node = self.item_selected()
        if selected_node:
            # dialogue de confirmation
            dial_conf = create_yes_no_dialog(_("remove path confirmation : ") + selected_node)
            dial_conf.set_default_response(gtk.RESPONSE_NO)
            ret = dial_conf.run()
            dial_conf.destroy()
            # on teste le retour du dialogue
            if ret == gtk.RESPONSE_YES:
                selection = self.path_list_treeview.get_selection()
                model, iter_sel = selection.get_selected()
                # suppression effective du path dans l'application                
                #self.app_edit.paths.remove(selected_node)
                # supression dans le treeview
                model.remove(iter_sel)
        else:
            dial_conf = create_error_dialog(_("please select a path"))
            dial_conf.set_default_response(gtk.RESPONSE_NO)
            ret = dial_conf.run()
            dial_conf.destroy()
            
    def stay_alive(self,*args):
        self.hide_and_flush(*args)
        return True
    
    def hide_and_flush(self, *args):
        self.dlg.hide()
        self.flush()

    def flush(self):
        """Réinitialise l'ensemble des champs de la fenêtre d'édition de application
        """
        self.name_entry.set_text("")
        self.application_libelle_entry.set_text("")
        self.application = None
        self.name_entry.set_sensitive(1)
        self.application_path_entry.set_text('')
        # vide le treeview des paths
        treeviewcolumn = self.path_list_treeview.get_column(0)
        if treeviewcolumn != None:
            self.path_list_treeview.remove_column(treeviewcolumn)
        self.treestore.clear()        

    def validate_edit(self, dialog, response_code):
        """Callback appelée lorsqu'on valide l'édition d'un application.
        """
        if response_code != gtk.RESPONSE_OK:
            self.hide_and_flush()
        else:
            try:
                name = self.name_entry.get_text()
                libelle = self.application_libelle_entry.get_text()
                treeiter = self.treestore.get_iter_first()
                paths = []
                paths.append(self.treestore.get_value(treeiter, 0))
                while treeiter != None:
                    treeiter = self.treestore.iter_next(treeiter)
                    if treeiter != None:
                        paths.append(self.treestore.get_value(treeiter, 0))
                if not name:
                    raise EmptyNameError

                if self.app_edit == None:
                    # nom déjà pris par une appli ou un groupe d'appli
                    if name in library_store.applications.keys() or \
                            name in library_store.app_groups.keys():
                        dlg_er = create_error_dialog(_('existing application'))
                        dlg_er.run()
                        return
                application = Application(name, libelle, paths)
                library_store.add_application(application)
                self.hide_and_flush()
                # mise à jour de la fenêtre parents
                self.parent.refresh()
            except Exception, e:
                dlg = create_error_dialog(_('empty application'))
                dlg.show_all()

    def show_dialog(self):
        self.dlg.show_all()
        
        
### groupe d'applications
class ApplicationGroupDialog(Singleton):
    """Gère l'édition des groupes d'applications
    """

    def __init__(self, glade_file, app_group_edit=None, parent=None):
        """
        glade : l'objet XML glade
        """
        self.app_group_edit = app_group_edit
        self.glade_file = glade_file

        self.glade = gtk.glade.XML(self.glade_file,"application_group_dialog")
        self.parent = parent
        if 'dlg' not in self.__dict__:
            self.dlg = self.glade.get_widget('application_group_dialog')
            self.name_entry = self.glade.get_widget('application_group_name_entry')
            self.application_group_libelle_entry = self.glade.get_widget('application_group_libelle_entry')
            self.application_treeview = self.glade.get_widget("application_treeview")
            #self.application_entry = self.glade.get_widget("application_entry")
#            container = self.glade.get_widget("hbox151")
            self.application_combobox = self.glade.get_widget("application_combobox")
#            self.application_combobox = gtk.combo_box_new_text()
#            container.pack_end(self.application_combobox)
            self.store=gtk.ListStore(str)
            self.application_combobox.set_model(self.store)
            self.application_combobox.set_active(0)
            cell = gtk.CellRendererText()
            self.application_combobox.pack_start(cell, True)
            self.application_combobox.add_attribute(cell, 'text', 0)

            selection = self.application_treeview.get_selection()
            selection.set_mode(gtk.SELECTION_SINGLE)
            selection.connect('changed',self.item_selected)
            handlers = {
                'on_application_group_dialog_response' : self.validate_edit,
                'on_application_dialog_delete_event' : self.stay_alive,
                'on_application_add_button_clicked': self.application_add,
                'on_application_del_button_clicked':self.application_del,
                }
            self.glade.signal_autoconnect(handlers)
        else:
            self.flush()
            self.show_dialog()

        if self.app_group_edit != None:
            self.fill_application()
        else:
            self._fill_combobox()
            self.init_model()
            self.init_treeview()

    def _fill_combobox(self):
        """
            remplit la liste deroulante 
        """
        for app in library_store.applications.keys():
            self.application_combobox.append_text(app)
        self.application_combobox.set_active(0)

    def fill_application(self):
        """
            remplit les données de l'application en cours dans le dialog
        """
        self.name_entry.set_text(self.app_group_edit.name)
        self.application_group_libelle_entry.set_text(self.app_group_edit.description)
        self.name_entry.set_sensitive(0)
        self._fill_combobox()
        self.init_treeview()
                
    ## gestion du treestore des applications
    def init_treeview(self):
        """
            remplissage du treeview des applications du groupe en cours
        """
        self.init_model()
        self.init_treeview_column()

    def add_line_treeview(self, path):
        """
            ajoute une ligne au treeview des applications
        """
        self.treestore.append(None, [path])
        
    def init_model(self):
        """Initialise le modèle associé à l'arbre des applications
        """
        self.treestore = create_app_treestore()
        self.application_treeview.set_model(self.treestore)
        if self.app_group_edit != None:
            for name in self.app_group_edit.applications:
                self.remove_combo_add_listview(name)

    def remove_combo(self, model, path, iterator, name):
        """
            callback appelé lors d'une iteration sur le store
        """
        path = [str(i) for i in path]
        iterator = self.store.get_iter_from_string(":".join(path))
        if name == model.get_value(iterator, 0):
            self.store.remove(iterator)

    def remove_combo_add_listview(self, name):
        """
            rempli le combo_box des app_group
        """
        self.add_line_treeview(name)
        self.store.foreach(self.remove_combo, name)
        self.application_combobox.set_active(0)

    def init_treeview_column(self):
        column = gtk.TreeViewColumn(_('paths list'),
                                    gtk.CellRendererText(),
                                    text = 0)
        self.application_treeview.append_column(column)
        #self.treeview.connect('button-press-event', self.item_selected)

    def application_add(self, *args):
        """
            callback lorsqu'on clique sur ajout de path
        """
        
        iterator = self.application_combobox.get_active_iter()
        if iterator != None: 
            name = self.store.get_value(iterator, 0)
            self.remove_combo_add_listview(name)
        
        
    def item_selected(self, *args):
        """Callback appelée lorsqu'on clique sur un application dans l'arbre
        """
        selection = self.application_treeview.get_selection()
        (store, itersel) = selection.get_selected()

        if itersel is None:
            self.selected_node = None
            return
        self.selected_node = store.get_value(itersel,0)
        return self.selected_node

    def application_del(self, *args):
        """
            suppression d'une application pour un groupe d'applications
        """
        selected_node = self.item_selected()
        if selected_node:
            # dialogue de confirmation
            dial_conf = create_yes_no_dialog(_("remove application confirmation : ") + selected_node)
            dial_conf.set_default_response(gtk.RESPONSE_NO)
            ret = dial_conf.run()
            dial_conf.destroy()
            # on teste le retour du dialogue
            if ret == gtk.RESPONSE_YES:
                selection = self.application_treeview.get_selection()
                model, iter_sel = selection.get_selected()
                #reajoute dans le combo
                self.application_combobox.append_text(model.get_value(iter_sel, 0))
                self.application_combobox.set_active(0)
                # suppression effective du path dans l'application                
                #self.app_group_edit.paths.remove(selected_node)
                # supression dans le treeview
                model.remove(iter_sel)
        else:
            dial_conf = create_error_dialog(_("please select an application"))
            dial_conf.set_default_response(gtk.RESPONSE_NO)
            ret = dial_conf.run()
            dial_conf.destroy()
            
    def stay_alive(self,*args):
        self.hide_and_flush(*args)
        return True
    
    def hide_and_flush(self, *args):
        self.dlg.hide()
        self.flush()

    def flush(self):
        """Réinitialise l'ensemble des champs de la fenêtre d'édition de application
        """
        self.name_entry.set_text("")
        self.application_group_libelle_entry.set_text("")
        self.application = None
        self.name_entry.set_sensitive(1)
        #FIXME flush le combo
        #self.application_entry.set_text('')
        # vide le treeview des paths
        treeviewcolumn = self.application_treeview.get_column(0)
        if treeviewcolumn != None:
            self.application_treeview.remove_column(treeviewcolumn)
        self.treestore.clear()        
        # flush du store du combo
        self.store.clear()

    def validate_edit(self, dialog, response_code):
        """
        Callback appelée lorsqu'on valide l'édition 
        d'un groupe applications
        """
        if response_code != gtk.RESPONSE_OK:
            self.hide_and_flush()
        else:
            try:
                name = self.name_entry.get_text()
                libelle = self.application_group_libelle_entry.get_text()
                treeiter = self.treestore.get_iter_first()
                applications = []
                applications.append(self.treestore.get_value(treeiter, 0))
                while treeiter != None:
                    treeiter = self.treestore.iter_next(treeiter)
                    if treeiter != None:
                        applications.append(self.treestore.get_value(treeiter, 0))
                if not name:
                    raise EmptyNameError

                if self.app_group_edit == None:
                    # nom déjà pris par une appli ou un groupe d'appli
                    if name in library_store.applications.keys() or \
                            name in library_store.app_groups.keys():
                        dlg_er = create_error_dialog(_('existing application'))
                        dlg_er.run()
                        return

                    application = ApplicationGroup(name, libelle, applications)
                    library_store.add_application_group(application)                    
                    
                else:
                    self.app_group_edit.name = name
                    self.app_group_edit.description = libelle
                    self.app_group_edit.applications = applications
                    
                self.hide_and_flush()
                # mise à jour de la fenêtre parents
                self.parent.refresh()
            except Exception, e:
                dlg = create_error_dialog(_('empty application'))
                dlg.show_all()

    def show_dialog(self):
        self.dlg.show_all()
                
