# -*- coding: UTF-8 -*-
"""Teste les attributs optionnels des directives
les directives sont testees directement a partir du xml

``era.noyau.domparsers.instantiate_directive_from_dom``

actions possibles pour l'attribut attrs :

era/doc/html/DirectivesOptionnelles.html

"""
import autopath

from xml.dom.minidom import parseString
from era.noyau import domparsers

from era.noyau.fwobjects import Zone,  Service, ServiceGroupe, Extremite, Directive, Flux

DIRECTIVE_NODE = """<directive id='essai' service='serv1' priority='1' action='1' attrs='0' src_inv='0' dest_inv='0' serv_inv='0' libelle='pas de description'>
<source name='extr1' />
<source name='extr2' />
<destination name='extr3' />
</directive>
"""

DIRECTIVE_OPTIONAL = """<directive id='essai' service='serv1' priority='1' action='1' attrs='1' src_inv='0' dest_inv='0' serv_inv='0' libelle='pas de description'>
<source name='extr1' />
<source name='extr2' />
<destination name='extr3' />
</directive>
"""

DIRECTIVE_OPTIONAL_ACTIVE = """<directive id='essai' service='serv1' priority='1' action='1' attrs='3' src_inv='0' dest_inv='0' serv_inv='0' libelle='pas de description'>
<source name='extr1' />
<source name='extr2' />
<destination name='extr3' />
</directive>
"""

DIRECTIVE_OPTIONAL_ACTIVE_SEVEN = """<directive id='essai' service='serv1' priority='1' action='1' attrs='5' src_inv='0' dest_inv='0' serv_inv='0' libelle='pas de description'>
<source name='extr1' />
<source name='extr2' />
<destination name='extr3' />
</directive>
"""

class InitDirective:
    """generateur de directives
    d = InitDirective().get_directive()
    L{era.noyau.domparsers.instantiate_directive_from_dom}
    """

    def __init__(self):
        self._init_zones()
        self._init_extremites()
        self._init_services()
        self._init_directive()

    def _init_zones(self):

        zoneA = Zone('zoneA', 30, '127.0.0.1')
        zoneB = Zone('zoneB', 31, '127.0.0.1')
        zoneC = Zone('zoneC', 32, '127.0.0.1')
        self.zones = {'zoneA': zoneA,
                 'zoneB': zoneB,
                 'zoneC': zoneC
                 }

    def _init_extremites(self):
        e1 = Extremite(self.zones['zoneA'],"extr1", "libelle1", ["10.0.0.1"], "255.255.255.255")
        e2 = Extremite(self.zones['zoneA'],"extr2", "libelle2", ["10.0.0.2"], "255.255.255.255")
        e3 = Extremite(self.zones['zoneB'],"extr3", "libelle3", ["10.0.0.3"], "255.255.255.255")
        e4 = Extremite(self.zones['zoneC'],"extr4", "libelle4", ["10.0.0.4"], "255.255.255.255")
        self.extremites = {"extr1" : e1,"extr2" : e2,"extr3" : e3,"extr4" : e4}

    def _init_services(self):
        s1 = Service("serv1", "TCP", [80], "serv1", "libelle_Internet")
        s2 = Service("serv1", "TCP", [480], "serv2", "libelle2")
        s3 = Service("Youpi", "UDP", [4513], "serv3", "libelle3")
        s4 = ServiceGroupe("Groupe1", "test de groupe de srevices",[s1,s2])
        self.services = {"serv1" : s1, "serv2" : s2, "Youpi" : s3}
        self.groups = {"Group1" : s4}

    def _init_directive(self):
        doc = parseString(DIRECTIVE_NODE)
        self.directive = domparsers.instantiate_directive_from_dom(doc.documentElement,
                                              self.extremites, self.services, self.groups)
        doc = parseString(DIRECTIVE_OPTIONAL)
        self.directive_optional = domparsers.instantiate_directive_from_dom(doc.documentElement,
                                              self.extremites, self.services, self.groups)
        doc = parseString(DIRECTIVE_OPTIONAL_ACTIVE)
        self.directive_active = domparsers.instantiate_directive_from_dom(doc.documentElement,
                                              self.extremites, self.services, self.groups)

        doc = parseString(DIRECTIVE_OPTIONAL_ACTIVE_SEVEN)
        self.directive_active_seven = domparsers.instantiate_directive_from_dom(doc.documentElement,
                                              self.extremites, self.services, self.groups)

    def get_directive(self):
        return self.directive

    def get_directive_optional(self):
        return self.directive_optional

    def get_directive_active(self):
        return self.directive_active

    def get_directive_active_seven(self):
        return self.directive_active_seven

def test_directive():
    """Crée une directive a partir du xml
    attrs=0
    """
    d = InitDirective().get_directive()
    assert d.libelle == u'pas de description'
    assert d.attrs == 0

def test_optional():
    """créer une directive a partir du xml
    vérifie qu'elle est optionnelle
    L{era.noyau.domparsers.instantiate_directive_from_dom}

    différentes valeurs pour attrs :
    0 : pas optionnelle
    1 : optionnelle non active
    3 : optionnelle et active
    """
    d = InitDirective().get_directive()
    assert d.libelle ==  u'pas de description'
    assert d.attrs == 0
    assert d.is_optional() == 0
    assert d.is_active() == 0
    d = InitDirective().get_directive_optional()
    assert d.libelle ==  u'pas de description'
    assert d.attrs == 1
    assert d.is_optional() == 1
    assert d.is_active() == 0

def test_directive_active():
    """ crée une directive à partir du xml
        L{era.noyau.domparsers.instantiate_directive_from_dom}
    """
    d = InitDirective().get_directive_active()
    assert d.libelle ==  u'pas de description'
    assert d.attrs == 3
    assert d.is_active() ==    1

def test_directive_set_active():
    """
    """
    d = InitDirective().get_directive_active()
    d.set_active(1)
    assert d.is_active() == 0
    d.set_active(3)
    assert d.is_active() == 1

def test_directive_set_active_seven():
    """
    """
    d = InitDirective().get_directive_active_seven()
    d.set_active(1)
    assert d.is_active() == 0
    d.set_active(3)
    assert d.is_active() == 1
    d.set_active(3)
    assert d.is_active() == 1
