# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""Contient l'ensemble des structures utilisées dans les règles iptables

Une règle Iptable peut être composée de :
  - paramètres (options)
  - une cible

Un paramètre est une option iptables lui-même composé d'options éventuellement liées
"""
import types
from era.backend.utils import dump_port_range, gen_ip_netmask, gen_ip_couples

from era.noyau.constants import ACTION_DENY, ACTION_ALLOW, ACTION_REDIRECT, \
     ACTION_DNAT, ACTION_MASK, ACTION_FORWARD

from era.noyau.pool import library_store

## Règles ######################################################################
class IPTRule(object):
    """Classe abstraite d'une règle iptable
    """
    def __init__(self, chain_name = None, table_name = 'filter'):
        """
         - chain_name : le nom de la chaine itpables
        """
        self.table_name = table_name
        self.chain_name = chain_name
        self.params = []

    def add_parameter(self, param):
        """Ajoute un paramètre à la règle
        """
        if param is not None:
            self.params.append(param)
        # else : logger warning ?

    def add_parameters(self, param_seq):
        """Ajoute une liste de parmètres à la règle"""
        for param in param_seq:
            self.add_parameter(param)

    def __str__(self):
        """Représentation iptables de cette règle
        """
        return ''

    def set_chain_name(self, chain_name):
        """Pour spéficier une chaine particulière
        """
        self.chain_name = chain_name

    def set_table_name(self, table_name):
        """Pour spécifier la table à modifier
        """
        self.table_name = table_name

    def check(self):
        """Vérifie la cohérence de la règle
        Lève une exception InvadidIPTRule en cas d'erreur
        """
        pass

class TargettedRule(IPTRule):
    """Classe représentant une règle iptable *ciblée* (-j target)
    """
    def __init__(self, target = None, chain_name = None, table_name = 'filter'):
        """
         - chain_name : le nom de la chaine itpables
         - target : la cible (instance de Target ou string)
        """
        super(TargettedRule, self).__init__(chain_name, table_name)
        self.set_target(target)

    def set_target(self, target):
        """Repositionne la valeur de la cible
         - target : soit une string représentant la cible, soit un objet Target
        """
        actions = {ACTION_DENY : 'DROP',
                   ACTION_ALLOW : 'ACCEPT',
                   ACTION_REDIRECT : 'REDIRECT',
                   ACTION_FORWARD : 'SNAT',
                   }
        if type(target) in (types.StringType, types.UnicodeType):
            self.target = Target(target)
        elif type(target) == types.IntType:
            try:
                self.target = Target(actions[target])
            except KeyError:
                raise ValueError('Target value "%s" not in %s' % (target,
                                                                  actions))
        else:
            self.target = target


    def get_target(self):
        """Renvoie la cible
        """
        return self.target


    def check(self):
        """Vérifie la cohérence de la règle
        """
        for param in self.params:
            param.check() # .check(self.params)


class CreationRule(IPTRule):
    """Classe pour gérer les créations de chaines, etc.
    """
    pass

## Paramètres ##################################################################
class IParameter(object):
    """Représente un paramètre d'une règle iptables
    """
    def __str__(self):
        raise NotImplementedError()

    def set_value(self, value):
        """Pour fixer la valeur du paramètre
        """
        raise NotImplementedError()

    def _parameters_options(self):
        """Renvoie les options spécifiques aux paramètres
        """
        raise NotImplementedError()

class Parameter(IParameter):
    """Classe de base pour les paramètres
    """
    def __init__(self, p_name, value=None, long_name=True, inverted=False):
        """
         - p_name : le nom du paramètre (option, ex : protocol, match, etc.)
         - value : la valeur de cette option
         - long_name : si le nom du paramètre est une option longue ou courte
        """
        self.inverted = inverted
        self.p_name = p_name
        self.p_value = value
        self.long_name = long_name

    def set_value(self, value):
        """Pour positionner la valeur du paramètre
        """
        self.p_value = value

    def _parameters_options(self):
        """Comportement par défaut : pas d'option spécifique
        """
        return ''

    def __str__(self):
        """Comportement de base pour str()
        """
        options = self._parameters_options()
        invert = ""
        if self.inverted:
            invert = "! "
        if options:
            if self.p_value == 'esp':
                return invert + '%s%s %s %s' % ((self.long_name and '--') or '-',
                                       self.p_name,
                                       self.p_value, options)
            else:
                return invert + '%s%s %s %s' % ((self.long_name and '--') or '-',
                                       self.p_name,
                                       self.p_value, options)
        else:
            if self.p_value == 'esp':
                return invert + '%s%s %s' % ((self.long_name and '--') or '-',
                                    self.p_name,
                                    self.p_value)
            else:
                return invert + '%s%s %s' % ((self.long_name and '--') or '-',
                                    self.p_name,
                                    self.p_value)

# ____________________________________________________________
# effective FORWARD IMPLEMENTATION
class InsertParameter(Parameter):
    def __init__(self, p_name):
        """Gestion des paramètres relatifs aux protocoles
         - t_name : type d'insertion (FORWARD...)

        """
        super(InsertParameter, self).__init__('I', value=p_name, long_name=False)


class ToParameter(Parameter):
    def __init__(self, p_name):
        """Gestion des paramètres relatifs aux protocoles
         - t_name : type d'insertion (FORWARD...)

        """
        super(ToParameter, self).__init__('to', value=p_name, long_name=True)

class MarkParameter(Parameter):
    """
        option de marquage
    """
    def __init__(self, p_name):
        """Gestion des paramètres relatifs aux protocoles
         - p_name : le nom du protocle
        """
        super(MarkParameter, self).__init__('mark', p_name)

## Match options ###############################################################
class MatchParameter(Parameter):
    """Pour les paramètres match
    (Par exemple -m limit --limit=...)
    """
    def __init__(self, match_name):
        """Gestion des paramètres relatifs aux protocoles
         - match : nom de l'option à matcher
        """
        super(MatchParameter, self).__init__('m', match_name, False)

# _____________________ipsec parameters__________________________
class MatchPolicy(MatchParameter):
    """Pour gérer -m policy """
    def __init__(self):
        super(MatchPolicy, self).__init__('policy')

class MatchSet(MatchParameter):
    """Pour gérer -m set """
    def __init__(self):
        super(MatchSet, self).__init__('set')

class ProtoParameter(Parameter):
    """Définit un paramètre de proto (ipsec)
    """
    def __init__(self, p_name):
        """Gestion des paramètres relatifs aux protocoles
         - p_name : le nom du protocle
        """
        super(ProtoParameter, self).__init__('proto', p_name, long_name=True)

class DirParameter(Parameter):
    """Définit un paramètre --dir in|out (ipsec)
    """
    def __init__(self, p_name):
        """Gestion des paramètres relatifs aux protocoles
         - p_name : le nom du protocle
        """
        super(DirParameter, self).__init__('dir', p_name, long_name=True)

## --pol ##
class PolicyParam(Parameter):
    """Pour gérer --pol
    """
    def __init__(self, p_name='ipsec'):
        super(PolicyParam, self).__init__('pol', value=p_name, long_name=True)

## --match-set <name>
class MatchSetParameter(Parameter):
    def __init__(self, p_name, inverted=False, src=True):
        self.src = src
        super(MatchSetParameter, self).__init__('match-set', value=p_name,
            long_name=True, inverted=inverted)

    def _parameters_options(self):
        if self.src:
            return 'src'
        else:
            return 'dst'
#        return 'src,dst'
# _______________________________________________________________________________

## Paramètres protocole ########################################################
class ProtocolParameter(Parameter):
    """Définit un paramètre de protocole
    """
    def __init__(self, p_name, inverted=False):
        """Gestion des paramètres relatifs aux protocoles
         - p_name : le nom du protocle
        """
        super(ProtocolParameter, self).__init__('p', p_name, False, inverted=inverted)

class ESPProtocol(ProtocolParameter):
    def __init__(self, inverted=False):
        super(ESPProtocol, self).__init__('esp', inverted=inverted)

# TCP ##################################
class TCPProtocol(ProtocolParameter):
    """Options spécifiques TCP
    """
    def __init__(self, dport = None, sport = None,
                 flags = None, inverted = False):
        """
         - sport (source port) : soit un port, soit une liste
           La liste ne devrait, dans ce cas, contenir que les deux
           extrêmes de la rangée de ports
         - dport (destination port) : soit un port, soit une liste
           La liste ne devrait, dans ce cas, contenir que les deux
           extrêmes de la rangée de ports
         - flags : (SYN, ACK, FIN, RST, URG, PSH, ALL, NONE)
         - syn : true ou false si on ne veut matcher que les paquets
           dont le bit syn est placé, et avec ACK et FIN vides
        /!\ les options 'tcp-option', 'tcp-flags', et 'mss' ne sont pas gérées
        """
        super(TCPProtocol, self).__init__('tcp')
        self.dport = dport
        self.sport = sport
        flags = flags or []
        for flag in flags:
            assert flag in ('SYN', 'ACK', 'FIN', 'RST',
                            'URG', 'PSH', 'ALL', 'NONE')
        self.flags = flags
        self.inverted = inverted

    def _parameters_options(self):
        """Génère ces options en syntaxe *iptables*
        """
        out = []
        invert = ""
        if self.inverted:
            invert = "! "

        if self.dport:
            out.append(invert + '--dport ' + dump_port_range(self.dport,
                                                    inverted = self.inverted))
        if self.sport:
            out.append(invert + '--sport ' + dump_port_range(self.sport,
                                                    inverted = self.inverted))

        if self.flags:
            """il y a aussi un raccourci : --syn"""
            out.append('--tcp-flags %s SYN' % (','.join(self.flags)))

        return ' '.join(out)

# UDP ##################################
# XXX FIXME UDP et TCP ?! : quasiment les mêmes classes
class UDPProtocol(ProtocolParameter):
    """Options spécifiques UDP
    """

    def __init__(self, dport = None, sport = None, inverted = False):
        """
         - sport (source port) : soit un port, soit une liste
           La liste ne devrait, dans ce cas, contenir que les deux
           extrêmes de la rangée de ports
         - dport (destination port) : soit un port, soit une liste
           La liste ne devrait, dans ce cas, contenir que les deux
           extrêmes de la rangée de ports
        """
        super(UDPProtocol, self).__init__('udp')
        self.dport = dport
        self.sport = sport
        self.inverted = inverted

    def _parameters_options(self):
        """Génère ces options en syntaxe *iptables*
        """
        out = []
        invert = ""
        if self.inverted:
            invert = "! "

        if self.dport:
            out.append(invert + '--dport ' + dump_port_range(self.dport,
                                                    inverted = self.inverted))
        if self.sport:
            out.append(invert + '--sport ' + dump_port_range(self.sport,
                                                    inverted = self.inverted))

        return ' '.join(out)

# ICMP #################################
class ICMPProtocol(ProtocolParameter):
    """Classe spécifique pour gérer les options ICMP
    """
    def __init__(self, icmp_type=None, inverted=False):
        super(ICMPProtocol, self).__init__('icmp')
        # XXX FIXME : faire un assert icmp_type in (...)
        self.inverted = inverted
        self.icmp_type = icmp_type

    def _parameters_options(self):
        if self.icmp_type:
            return '%s --icmp-type %s' % ((self.inverted and '!') or '',
                                                            self.icmp_type)
        return ''

# LOG #################################
class LogParameter(Parameter):
    """Classe spécifique pour gérer les options ICMP
    """

    def __init__(self, libelle, level='', prefix='', tcp_sequence=False,
                 tcp_options=False, ip_options=False):
        """Gestion des paramètres relatifs aux protocoles
         - p_name : le nom du protocle
        """
        super(LogParameter, self).__init__(p_name='j', value='LOG', long_name=False)
        self.libelle = libelle
        if level not in ['debug','info','notice','warning','err','crit','alert','emerg']:
            level = ''
        self.level = level
        self.prefix = prefix
        self.tcp_sequence = tcp_sequence
        self.tcp_options = tcp_options
        self.ip_options = ip_options

    def _parameters_options(self):
        """rajoute les options --log-level ...
        """
        out = []
        out.append('--log-prefix "iptables: %s"'  % self.libelle)
        if self.level != '':
            out.append('--log-level %s' % self.level)
        if self.tcp_sequence:
            out.append('--log-tcp-sequence')
        if self.tcp_options:
            out.append('--log-tcp-options')
        if self.ip_options:
            out.append('--log-ip-options')
        return ' '.join(out)

# TIME #################################
class TimeParameter(Parameter):
    """Classe spécifique pour gérer les options time
    http://www.netfilter.org/patch-o-matic/pom-base.html#pom-base-time
    """
    allowed = ('timestart', 'timestop', 'datestart', 'datestop', 'weekdays')
    def __init__(self, time):
        """connection tracking state
        """
        super(TimeParameter, self).__init__('m', 'time', False)
        self.time = time
        # On vérifie que tous les paramètres de dates sont reconnus
        not_found = True
        for option in self.allowed:
            if hasattr(time, option):
                not_found = False
                break
        if not_found == True:
            raise ValueError("not recognized option")

    def _parameters_options(self):
        out = []
        for option in self.allowed:
            if getattr(self.time, option) != '':
                out.append('--%s %s' % (option, getattr(self.time, option)))

        return ' '.join(out)




# -m state #############################
class MatchState(MatchParameter):
    """Pour gérer -m state
    """
    allowed = ('INVALID', 'ESTABLISHED', 'NEW', 'RELATED')
    def __init__(self, states):
        """connection tracking state
        """
        super(MatchState, self).__init__('state')
        self.states = [state.upper() for state in states]
        # On vérifie que tous les états sont valides
        for state in self.states:
            if state not in self.allowed:
                raise ValueError("L'état %r n'est pas dans %s" % (state,
                                                                  self.allowed))

    def _parameters_options(self):
        """--state state
        """
        return '--state %s' % ','.join(self.states)

# -m limit #############################
class MatchLimit(MatchParameter):
    """Pour gérer -m limit
    """
    unit_dict = {'s'       : "second",
                 'm'       : "minute",
                 'h'       : "hour",
                 'd'       : "day",
                 'j'       : "jour",
                 'second'  : "second",
                 'seconde' : "second",
                 'minute'  : "minute",
                 'hour'    : "hour",
                 'heure'   : "hour",
                 'day'     : "day",
                 'jour'    : "day",
                 }

    def __init__(self, rate = 3, unit = 'hour', burst = 5):
        """
         - rate : nombre de paquet limite
         - unit : unité pour mesurer le nombre de paquets :
           * s, seconde
           * m, minute
           * d, day
           default = 3/h
         - burst : nombre initial maxium de paquets à matcher
        """
        super(MatchLimit, self).__init__('limit')
        self.rate = rate
        try:
            self.unit = self.unit_dict[unit]
        except KeyError:
            raise ValueError("Impossible de reconnaitre l'unité %s" \
                             ", devrait être s, m, h ou d")
        self.burst = burst

    def _parameters_options(self):
        """--state state
        """
        out = []
        # On n'affiche l'option que si elle est différente des valeurs
        # par défaut (3/h)
        if not (self.rate == 3 and self.unit == 'hour'):
            out.append('--limit %s/%s' % (self.rate, self.unit))

        # On n'affiche l'option que si elle est différente des valeurs
        # par défaut (5)
        if self.burst != 5:
            out.append('--limit-burst %s' % (self.burst))

        return ' '.join(out)



# XXX FIXME : manque conntrack, mark, etc.
## Interface Options ###########################################################
class InterfaceParameter(Parameter):

    def __init__(self, interface_name, out = False, inverted = False):
        """
         - out définit si c'est c'est pour l'entrée ou la sortie du paquet
        """
        if out:
            super(InterfaceParameter, self).__init__('o', interface_name, False)
        else:
            super(InterfaceParameter, self).__init__('i', interface_name, False)

        self.out = out
        self.inverted = inverted

    def __str__(self):
        if self.p_value:
            if self.out:
                return '-o %s%s' % ((self.inverted and '! ') or '',
                                    self.p_value)
            else:
                return '-i %s%s' % ((self.inverted and '! ') or '',
                                    self.p_value)

        return ''


## IPOptions ###################################################################
class IPParameter(Parameter):
    """utilisée pour gérer les options '-s' et '-d'
    """
    def __init__(self, src, address, mask = None, inverted = False):
        """
         - src devrait valoir True ou False suivant que vont matcher la source
           ou la destination
        """
        if src:
            super(IPParameter, self).__init__('s', long_name=False)
        else:
            super(IPParameter, self).__init__('d', long_name=False)

        assert address, "Address must be specified"
        self.src = src
        self.address = address
        self.mask = mask
        self.inverted = inverted
        self._update_value()

    def _update_value(self):
        """Mets à jour self.p_value avec les valeurs passées au constructeur
        """
        out = []
        out.append(self.address)
        # On ajoute la définition du masque si précisée
        if self.mask:
            self.p_value = ' '.join(out) + '/%s' % self.mask
        else:
            self.p_value = ' '.join(out)



class SrcIPParameter(IPParameter):
    """Classe définie pour faire un raccourci
    vers : IPParameter(1, ...)
    """
    def __init__(self, address, mask = None, inverted = False):
        super(SrcIPParameter, self).__init__(True, address, mask, inverted)


class DestIPParameter(IPParameter):
    """Classe définie pour faire un raccourci
    vers : IPParameter(0, ...)
    """
    def __init__(self, address, mask = None, inverted = False):
        super(DestIPParameter, self).__init__(False, address, mask, inverted)


## Cibles ######################################################################
class Target(Parameter):
    """Représente la cible d'une règle iptables
    """

    def __init__(self, t_name):
        """
         - t_name : le nom de la cible
        """
        # XXX FIXME : ugly hack for netbios-ext and co
        if '-' not in t_name and '_' not in t_name:
            t_name = t_name.upper()

        super(Target, self).__init__('j', t_name, False)


# Redirect #############################
class Redirect(Target):
    """target redirect
    """

    def __init__(self, port):
        super(Redirect, self).__init__('REDIRECT')
        self.dport = port

    def _parameters_options(self):
        """rajoute l'option --to-ports
        """
        return '--to-ports ' + dump_port_range(self.dport, sep = '-')

# Masquerade ###########################
class Masquerade(Target):
    """masquerade target
    """

    def __init__(self, port):
        super(Masquerade, self).__init__('MASQUERADE')
        self.dport = port

    def _parameters_options(self):
        """rajoute l'option --to-ports
        """
        return '--to-ports ' + dump_port_range(self.dport, sep = '-')


# Log ##################################
class Log(Target):
    """log target
    """
    # XXX FIXME : faire un assert sur level (warn, error, etc.)
    def __init__(self, level, prefix='iptables: ', tcp_sequence=False,
                 tcp_options=False, ip_options=False):
        super(Log, self).__init__('LOG')
        self.level = level
        self.prefix = prefix
        self.tcp_sequence = tcp_sequence
        self.tcp_options = tcp_options
        self.ip_options = ip_options

    def _parameters_options(self):
        """rajoute les options --log-level ...
        """
        out = []
        out.append('--log-level %s' % self.level)
        out.append('--log-prefix %s'  % self.prefix)
        if self.tcp_sequence:
            out.append('--log-tcp-sequence')
        if self.tcp_options:
            out.append('--log-tcp-options')
        if self.ip_options:
            out.append('--log-ip-options')
        return ' '.join(out)


## ULog #################################
#class ULog(Target):
#    """cible ULOG
#    """
#    # XXX bof ...
#    def __init__(self, nlgroup = 0, prefix = '',
#                 cprange = -1, treshold = -1):
#        super(ULog, self).__init__('ULOG')
#        assert nlgroup in range(33)
#        self.nlgroup = nlgroup
#        self.prefix = prefix
#        self.cprange = cprange
#        self.treshold = treshold

#    def _parameters_options(self):
#        """rajoute les options --ulog-nlgroup ...
#        """
#        out = []
#        if self.nlgroup:
#            out.append('--ulog-nlgroup %s' % self.nlgroup)
#        if self.prefix:
#            out.append('--ulog-prefix %s' % self.prefix)
#        if self.cprange >= 0:
#            out.append('--ulog-cprange %s' % self.cprange)
#        if self.treshold >= 0:
#            out.append('--ulog-qtreshold %s' % self.treshold)
#        return ' '.join(out)


# Mark #################################
class Mark(Target):
    """cible MARK
    """
    def __init__(self, mark, operation='set'):
        super(Mark, self).__init__('MARK')
        self.mark = mark
        self.operation = operation

    def _parameters_options(self):
        """rajoute l'option --set-mark
        """
        if self.operation == 'or':
            return '--or-mark %s' % self.mark
        elif self.operation == 'and':
            return '--and-mark %s' % self.mark
        else:
            return '--set-mark %s' % self.mark


# Reject ###############################
class Reject(Target):
    """cible REJECT
    """
    def __init__(self, rej_type):
        super(Reject, self).__init__('REJECT')
        assert rej_type in ('icmp-net-unreachable', 'icmp-host-unreach-able',
                            'icmp-port-unreachable',  'icmp-proto-unreachable',
                            'cmp-net-prohibited', 'icmp-host-prohibited')
        self.rej_type = rej_type

    def _parameters_options(self):
        """option --reject-with
        """
        return '--reject-with %s' % self.rej_type


# Tos ##################################
class Tos(Target):
    """cible TOS
    """
    def __init__(self, tos):
        if type(tos) != types.IntType:
            assert tos in ('Minimize-Delay', 'Maximize-Throughput',
                           'Maximize-Reliability', 'Minimize-Cost',
                           'Normal-Service')

        super(Tos, self).__init__('TOS')
        self.tos = tos

    def _parameters_options(self):
        """option --set-tos
        """
        return '--set-tos %s' % self.tos


# Nat ##################################
class Nat(Target):
    """cible SNAT ou DNAT
    """
    # XXX ip_list et port_list devrait avoir la même longueur
    # => XXX passer un dictionnaire plutôt ?
    def __init__(self, ip_list, port_list, nat_type):
        """
         - Les élément dans port_list peuvent soit être un port simple, soit
         une rangée de ports.
           (Chaque élément générera une option 'to-source')
         - nat_type est soit 'SNAT', soit 'DNAT'
        """
        nat_type = nat_type.upper()
        assert nat_type in ('SNAT', 'DNAT')
        super(Nat, self).__init__(nat_type)
        # Si port_list est vide (ou None) => on considère que ça veut
        # dire : "aucun port à spécifier pour *toutes* les IPs"
        if not port_list:
            self.port_list = [0] * len(ip_list)
        else:
            self.port_list = port_list
        self.ip_list = ip_list
        if nat_type == 'SNAT':
            self.rule_type = 'source'
        else:
            self.rule_type = 'destination'

    def _parameters_options(self):
        """rajoute le(s) option(s) --to-source / --to-destination
        """
        sources = []
        for source_ip, port in zip(self.ip_list, self.port_list):
            l = ['--to-%s ' % self.rule_type]
            # Si c'est une liste => on doit créer une rangée d'IPs
            if type(source_ip) == types.ListType:
                start_ip = source_ip[0]
                end_ip = source_ip[-1]
                l.append('%s-%s' % (start_ip, end_ip))
            # On suppose que c'est soit liste (étendre à tuple ?), soit ip seule
            else:
                l.append('%s' % source_ip)

            # Est-ce qu'un port ou une rangée de ports a été définie ?
            # il ne faut pas que le port soit zero non plus
            if port:
                try:
                    if not int(str(port)) == 0:
                        l.append(':' + dump_port_range(port, sep = '-'))
                except:
                    l.append(':' + port)
            sources.append(''.join(l))

        return ' '.join(sources)


# Dnat #################################
class Dnat(Nat):
    """Classe définie uniquement pour faire un raccourci
    """
    def __init__(self, ip_list, port_list):
        super(Dnat, self).__init__(ip_list, port_list, 'DNAT')


# Snat #################################
class Snat(Nat):
    """Classe définie uniquement pour faire un raccourci
    """
    def __init__(self, ip_list, port_list):
        super(Snat, self).__init__(ip_list, port_list, 'SNAT')



# Nfqueue ###############################
class Nfqueue(Target):
    """cible NFQUEUE
    """
    def __init__(self):
        super(Nfqueue, self).__init__('NFQUEUE')


class ParameterFactory(object):
    """Fabrique pour paramètres de règles iptables
    """

    def target_from_directive(directive):
        """Renvoie la cible relative à cette directive
        """
        # DNAT
        if directive.action == ACTION_DNAT and directive.accept == 0:
            return Dnat([directive.nat_extr.ip_list[0]], [directive.nat_port])
        # SNAT or Forward (first SNAT part)
        elif ( directive.action == ACTION_MASK or directive.action == ACTION_FORWARD ) and directive.accept == 0:
            return Snat([directive.nat_extr.ip_list[0]], [directive.nat_port])
        elif directive.action == ACTION_REDIRECT:
            return Redirect([directive.nat_port])

        actions = {ACTION_DENY : 'DROP',
                   ACTION_ALLOW : 'ACCEPT',
                   }
        try:
            if directive.user_group:
                return Nfqueue()
            else:
                return Target(actions[directive.action])
        except KeyError:
            raise ValueError('Target value "%s" not in %s' % (directive.action,
                                                              actions))


    def interface(i_name, out = False, inverted = False):
        """Construit une option pour l'interface
         - i_name : le nom de l'interface
        """
        return InterfaceParameter(i_name, out, inverted)


    def protocol(protocol_name):
        """Construit une option de protocole"""
        return ProtocolParameter(protocol_name)


    def time_from_directive(directive):
        """construit un objet TimeParameter pour les options du module time
        """
        return TimeParameter(directive.time)


    def service_from_directive(directive):
        """Renvoie un objet *Parameter* représentant le service
        """
        for service in directive.service:
            yield ParameterFactory.service2(service, directive.serv_inv)


    def service2(service, serv_inv):
        protocol = service.protocol.lower()
        if protocol not in ('udp', 'tcp', 'icmp', 'esp', 'tout'):
            raise ValueError('protocol should be in (udp, tcp, esp, tout or icmp)' \
                             ', found %s' % protocol)
        if protocol == 'udp':
            return UDPProtocol(dport=service.port_list, inverted=serv_inv)
        elif protocol == 'tcp':
            if library_store.get_version() > 1.0:
                """ajout de la fonctionnalité de SYN pour le tcp"""
                return TCPProtocol(dport=service.port_list, inverted=serv_inv, flags=['SYN', 'RST', 'ACK'])
            else:
                return TCPProtocol(dport=service.port_list, inverted=serv_inv)
        elif protocol == 'icmp':
            return ICMPProtocol(service.name, inverted=serv_inv)
        elif protocol == 'esp':
            return ESPProtocol(inverted=serv_inv)
        else:
            # Protocole 'TOUT' <=> pas besoin de spécifier de protocole
            return None


    def service(serv, *args, **kwargs):
        """Construit une option pour le service serv
        """
        serv = serv.lower()
        if serv == 'udp':
            return UDPProtocol(*args, **kwargs)
        if serv == 'tcp':
            # FIXME : Ici il faudrait ajouter un TCPProtocol
            if library_store.get_version() > 1.0:
                dct_param = kwargs
                dct_param['flags'] = ['SYN', 'RST', 'ACK']
                return TCPProtocol(*args, **dct_param)
            else:
                return TCPProtocol(*args, **kwargs)

        if serv == 'icmp':
            return ICMPProtocol(*args, **kwargs)
        if serv == 'esp':
            return ESPProtocol()
        if serv == 'tout':
            return None
        raise ValueError('service should be one of ("udp", "tcp", "esp", "icmp")' \
                         ', found %s' % (serv))



    def from_ip(ip = 0, netmask = 0, src = True):
        """Construit les bons paramètres pour options IP
        """
        if src:
            return SrcIPParameter(ip, netmask)
        else:
            return DestIPParameter(ip, netmask)


    def match_state(states):
        """Renvoie les bons paramètres pour options -m state
        """
        return MatchState(states)


    def target(target_name, *args, **options):
        """Renvoie un objet cible à partir du nom et des options
        """
        t_name = target_name.upper()
        class_dict = {'LOG' : Log,
                      #'ULOG' : ULog,
                      'MASQUERADE' : Masquerade,
                      'REJECT' : Reject,
                      'REDIRECT' : Redirect,
                      'MARK' : Mark,
                      'TOS' : Tos,
                      'DNAT' : Dnat,
                      'SNAT' : Snat,
                      'NFQUEUE' : Nfqueue,
                      }

        try:
            return class_dict[t_name](*args, **options)
        except KeyError:
            # cible basique ou chaine
            return Target(t_name)

    def parameters_for_zones(from_zone, dest_zone):
        """Renvoie la liste des paramètres pour un flux entre ces deux zones
         - from_zone : la zone de départ des paquets
         - dest_zone : la zone de destination des paquets
        la liste des paramètres est :
         (1) interface 'in' pour la zone de départ
         (2) interface 'out' pour la zone d'arrivée
         (3) ip/netmask pour la zone de départ
         (4) ip/netmask pour la zone d'arrivée
        """
        params = []
        # Pour les interfaces
        if from_zone.name.lower() != 'bastion':
            params.append(InterfaceParameter(from_zone.interface))
        if dest_zone.name.lower() != 'bastion':
            params.append(InterfaceParameter(dest_zone.interface, out = True))
        # Toute une zone <=> 0/0
        # Pour les IPs / masques
        params.append(SrcIPParameter('0', '0'))
        params.append(DestIPParameter('0', '0'))
        # params.append(SrcIPParameter(from_zone.ip, from_zone.netmask))
        # params.append(DestIPParameter(dest_zone.ip, dest_zone.netmask))
        return params

    def parameters_for_extremites(from_extrem, dest_extrem, src_inv=False, dest_inv=False, local=False):
        """Génère une liste de liste de paramètres pour ces deux extrémités
        """
        fename = from_extrem.name.lower()
        dename = dest_extrem.name.lower()
        fzname = from_extrem.zone.name.lower()
        dzname = dest_extrem.zone.name.lower()
        # Spécificités du bastion au niveau de la définition des interfaces
        in_int = InterfaceParameter(from_extrem.zone.interface)
        out_int = InterfaceParameter(dest_extrem.zone.interface , True)
        dest_params = None
        couples = []
        #si la regle est dans la zone bastion pas de d'inferface de destation ou de source
        if dzname == 'bastion' and local:
            # Le netmask devrait être 255.x4 vu que c'est une IP machine
            # => pas besoin de le spécifier
            out_int = None
        # XXX A-t-on besoin de gérer le cas 'bastion en entrée' ?
        elif fzname == 'bastion' and local:
            int_int = None

        #ecrase la source ou la destination que si extremite bastion dans la zone bastion
        if dzname == 'bastion' and local and dename == 'bastion':
            dest_param = DestIPParameter(from_extrem.zone.ip)
            for ips, netmask in gen_ip_netmask(from_extrem):
                src_param = SrcIPParameter(ips, netmask, src_inv)
                couples.append((src_param, dest_param))
        elif fzname == 'bastion' and local and fename == 'bastion':
            src_param = SrcIPParameter(dest_extrem.zone.ip)
            for ipd, netmask in gen_ip_netmask(dest_extrem):
                dest_param = DestIPParameter(ipd, netmask, dest_inv)
                couples.append((src_param, dest_param))
        else:
            for (f_ip, f_netmask, d_ip, d_netmask) in \
                    gen_ip_couples(from_extrem, dest_extrem):
                src_param = SrcIPParameter(f_ip, f_netmask, src_inv)
                dest_param = DestIPParameter(d_ip, d_netmask, dest_inv)
                couples.append((src_param, dest_param))
        for src_param, dest_param in couples:
            yield (in_int, out_int, src_param, dest_param)

    def ipsec_from_directive(directive):
        """Add -m policy --pol ipsec --proto esp --dir in/out
        """
        parameters = []
        if directive.ipsec == 1:
            parameters.append(MatchPolicy())
            parameters.append(PolicyParam('ipsec'))
            parameters.append(ProtoParameter('esp'))
            if directive.get_source_zone().interface == "%%interface_gw":
                parameters.append(DirParameter('in'))
            elif directive.get_dest_zone().interface == "%%interface_gw":
                parameters.append(DirParameter('out'))
            else:
                # Neither -i eth0 nor -o eth0
                pass
        return parameters


    def ipset_from_directive(directive, ipset_name, src=True):
        "Add -m set --match-set [ipsets_table_name]"
        exceptions = directive.exceptions
        parameters = []
        parameters.append(MatchParameter('set')) # -m set
        parameters.append(MatchSetParameter(ipset_name, inverted=False, src=src))
        return parameters

    target_from_directive = staticmethod(target_from_directive)
    interface = staticmethod(interface)
    protocol = staticmethod(protocol)
    service_from_directive = staticmethod(service_from_directive)
    service = staticmethod(service)
    from_ip = staticmethod(from_ip)
    match_state = staticmethod(match_state)
    target = staticmethod(target)
    parameters_for_extremites = staticmethod(parameters_for_extremites)
    parameters_for_zones = staticmethod(parameters_for_zones)
    time_from_directive = staticmethod(time_from_directive)
    ipsec_from_directive = staticmethod(ipsec_from_directive)
    ipset_from_directive = staticmethod(ipset_from_directive)
    service2 = staticmethod(service2)
