# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
Module qui définit les modèles principaux de l'application, c'est à dire
le modèle de matrice de flux (classe MatrixModel) et le modèle qui contient
l'ensemble des objets de la bibliothèque comme les services ou les extrémités
(classe LibraryStore)

"""

# ____________________________________________________________
import re
from os.path import basename, join, isfile
from itertools import chain
from StringIO import StringIO
# ____________________________________________________________
from era.noyau.dpatterns import Observable, FluxListener, ZoneListener
from era.noyau.fwobjects import Zone, Flux, Extremite
from era.noyau.erreurs import EmptyNameError, ExistingZoneError, TrigrammeError, InvalidZoneError
from era.noyau.pool import library_store
from era.noyau.constants import *

#import du dictionnarie de conversion (passage en eole2 des fichiers xml)
try:
    from creole import dico
except:
    pass
try:
    _
except:
    _ = str

# encoding du format xml de sortie
CHARSET = 'UTF-8'

def update_hidden_directive(matrix_model, active_filename):
    if isfile(active_filename):
        fichier = file(active_filename)
        tgs = fichier.read().split('\n')
        tags = [tg for tg in tgs if not tg.startswith('#')]
        fichier.close()
        update_list = []
        for tag in tags:
            if tag.strip():
                update_list.append(tag)
        matrix_model.visit_hidden(update_list)

# FIXME supprimer l'ancien MVC
class MatrixModel(Observable, FluxListener, ZoneListener):
    """ Modèle pour la matrice de flux
    """

    def __init__(self, observers = []):
        """
        observers : la liste des observers par défaut
        """
        Observable.__init__(self)
        for obs in observers:
            self.register_observer(obs)
        self.flux = []
        self.zones = []

    def add_zone(self, zone, notify = 1):
        """ Ajoute une zone à la liste
        zone : la zone à ajouter
        """
        self.assert_zone_valid(zone)
        for z in self.zones :
            f = Flux(z, zone)
            f.add_listener(self)
            self.flux.append(f)

        self.zones.append(zone)
        # les zones sont ordonnées par le __cmp__ de Zone(Fwobject)
        self.zones.sort()
        zone.add_listener(self)

        # On crée une extrémité du même nom qui représente toute la zone
        # càd 0/0 depuis la carte ethernet de la zone

        e = Extremite(zone, zone.name,
                      _('entire zone'), ['0.0.0.0'],
                      '0.0.0.0', True, 1)
        library_store.add_extremite(e)

        if notify:
            self.notify_observers()


    def get_zone_index(self, zone):
        """
            :return: index de la zone
        """
        if zone not in self.zones:
            raise InvalidZoneError, "la zone %s n'est pas dans la matrice"% zone.name
        return self.zones.index(zone)

    def real_zones(self):
        """liste les zones correspondant à des cartes éthernet"""
        return [z for z in self.zones if not z.name == 'bastion']

    def remove_zone(self, zone):
        """Enlève la zone du modèle
        """
        self.zones.remove(zone)
        for f in self.get_flux_list(zone):
            self.flux.remove(f)
        self.notify_observers()

    def get_flux_list(self, zone):
        """Renvoie la liste des flux qui concernent une zone donnée
        zone : la zone dont on veut voir les flux associés
        """
        return [f for f in self.flux if zone.name in (f.zoneA.name, f.zoneB.name)]


    def directive_collector(self):
        "recupere **toutes** les directives de la matrice de flux"
        for f in self.flux:
            for directive in f.up_directives_list():
                yield directive
            for directive in f.down_directives_list():
                yield directive

    def visit_directives(self, update_list):
        """met à jour les attrs (actif, inactif) des directives de tous les flux de la matrice
        @param upate_list: [ [tag,attrs] ]
        """
        # formate update_liste en un dictionaire de type {tag:attrs}
        dico = dict(update_list)
        # collecte la liste des objets directives du modèle
        # maj les attrs des directives
        for d in self.directive_collector():
            if not d.is_hidden():
                try:
                    tag = d.get_tag()
                    if not d.is_hidden():
                        if tag in dico.keys():
                            d.set_active(dico[tag])
                except:
                    # TagError
                    pass

    def visit_hidden(self, update_list):
        """active les directives optionelles cachées en fonction des tags
        """
        # maj les attrs des directives
        for d in self.directive_collector():
            if d.is_hidden():
                try:
                    tag = d.get_tag()
                    if tag in update_list:
                        d.set_attrs(d.get_attrs() ^ DIRECTIVE_ACTIVE)
                except:
                    # TagError
                    pass

    def visit_user_group(self):
        """récupère la liste des directives ayant un groupe d'utilisateur
        sous forme de générateur
        """
        filtered = []
        for d in self.directive_collector():
            if (d.user_group or d.app_group):
                filtered.append(d)
        return filtered

    def get_active_list(self, hidden=True):
        """collecte les informations sur les directives optionnelles
        (en vue de les enregistrer dans un fichier csv ou de gérer l'interface)

        @return: [ [tag, attrs, nb] ]
        """
        # filtre les directives pour ne garder que les directives optionnelles
        result = {}
        for d in self.directive_collector():
            interface = d.src_list[0].zone.interface
            if hidden or not d.is_hidden():
                if d.is_optional():
                    # en principe, une directive optionnelle a un tag
                    try:
                        tag = d.get_tag()
                    except:
                        raise Exception, "une directive optionnelle n'a pas de tag"
                    # calcul de nb : nombre de directives ayant ce mme tag
                    if result.has_key(tag) :
                        result[tag][1] += 1
                    else :
                        isactive = {True: 3, False: 1}.get(d.is_active(), 1)
                        if d.is_hidden():
                            isactive = isactive ^ DIRECTIVE_HIDDEN
                        result[tag] = [isactive, 1, interface]

        # formattage sous forme de liste
        return [(tag, isactive, count, interface) for tag, (isactive, count, interface) in result.items()]

    def get_tag_list(self):
        """collecte de la liste des tags des directives optionnelles
        @return liste de tags
        """
        return [i[0] for i in self.get_active_list()]

    def check_optional_list(self):
        """vérifie la liste des tags des directives optionnelles
        """
        # collecte la liste des objets directives du modèle
        directives = []
        for f in self.flux:
            directives.extend(f.up_directives_list())
            directives.extend(f.down_directives_list())
        # filtre les directives pour ne garder que les directives optionnelles
        for d in directives:
            if d.is_optional():
                # en principe, une directive optionnelle a un tag
                d.get_tag()

    def zone_changed(self):
        """ Méthode de l'interface 'ZoneListener'
        """
        # On est obligé de retrier la liste puisqu'il est possible
        # de modifier le niveau d'une zone !
        self.zones.sort()
        self.notify_observers()


    def flux_changed(self):
        """ Méthode de l'interface 'FluxListener'
        """
        self.notify_observers()


    def assert_zone_valid(self, zone):
        """ Teste si la zone entrée est valide
        """

        if not len(zone.name):
            raise EmptyNameError, _('empty string')

        if not zone.name.startswith("%") :
            for z in self.zones:
                if z.name[0:3] == zone.name[0:3]:
                    raise TrigrammeError,  _( 'trigramme zone error' )

        for z in self.zones:
            if z.name == zone.name:
                raise ExistingZoneError, _('existing zone')


    def save_zones(self, stream, indent = 0):
        """Méthode qui sauvegarde les zones au format interne
        stream : le flux dans lequel on écrit
        """
        stream.write('%s<zones>\n'%(" "*indent))
        for zone in self.zones:
            # permet d'eviter de sauver en double lors d'héritage
            if not zone.is_inherited():
                zone.save(stream, indent+4)
        stream.write('%s</zones>\n'%(" "*indent))


    def save_flux(self, stream, indent = 0):
        """Méthode qui sauvegarde un flux au format interne
        stream : le flux dans lequel on écrit
        """
        stream.write('%s<flux-list>\n'%(" "*indent))
        for flux in self.flux:
            flux.save(stream, indent+4)
        stream.write('%s</flux-list>\n'%(" "*indent))


    def save(self, filename, inherited=False, creole_conv=False):
        """Méthode qui sauvegarde la matrice au format interne
        filename : le nom du fichier courant
        inherited : le modele parent (ou bien False ou "")
        creole_conv : conversion des variables creole1 en creole2 si True
        """
        stream = StringIO()
        # vérification des libellés de dir. optionnelles
        self.check_optional_list()
        stream.write(u'<?xml version="1.0" encoding="%s" ?>\n\n' % CHARSET)
        stream.write(u'<firewall name="%s" ' % filename)
        if inherited:
            # Cas sauvegarde d'un nouveau fichier à partir d'un import
            # ou d'un modéle ayant déjà un parent
            stream.write(u'model="%s" ' % inherited)
            if library_store.inherited_option_netbios is not None:
                # Le modèle parent a une valeur d'option renseignée
                if library_store.inherited_option_netbios != library_store.options['netbios']:
                    # La valeur du modèle fils est différente de la valeur du modèle héritée
                    stream.write(u'netbios="%s" '% library_store.options['netbios'])
            else:
                # Le modèle parent n'a pas d'option renseignée
                # Arrive lors d'un nouveau fichier créé à partir d'un import
                if library_store.initial_option_netbios != library_store.options['netbios']:
                    # La valeur de l'option a changé donc on l'écrit
                    stream.write(u'netbios="%s" '% library_store.options['netbios'])
            if library_store.inherited_option_qos is not None:
                # Le modèle parent a une valeur d'option renseignée
                if library_store.inherited_option_qos != library_store.options['qos']:
                    # La valeur du modèle fils est différente de la valeur du modèle héritée
                    stream.write(u'qos="%s" '% library_store.options['qos'])
            else:
                # Le modèle parent n'a pas d'option renseignée
                # Arrive lors d'un nouveau fichier créé à partir d'un import
                if library_store.initial_option_qos != library_store.options['qos']:
                    # La valeur de l'option a changé donc on l'écrit
                    stream.write(u'qos="%s" '% library_store.options['qos'])
        else:
            # Cas de sauvegarde d'un fichier sans héritage
            stream.write(u'netbios="%s" '% library_store.options['netbios'])
            stream.write(u'qos="%s" '% library_store.options['qos'])

        stream.write(u'version="%s">\n'% library_store.get_version())

        self.save_zones(stream,4)
        library_store.save(stream,4)
        self.save_flux(stream,4)
        stream.write(u'</firewall>')

        fh = open(filename, "w")
        data = stream.getvalue()
        # le StringIO permet de convertir l'encoding en une seule fois
        data = data.encode(CHARSET)
        if creole_conv == True:
            # conversion des variables creole si nécessaire
            data = self.creole_convert(data)
        # sauvegarde des données
        fh.write(data)
        fh.close()
        # pour tests, on retourne le stream
        return stream

    def creole_convert(self, data):
        """Convertit les variables creole1 en variables creole2
        """
        # supposons que creole est disponible sur la machine
        from creole.config import datadir as creole_datadir
        self.unknown_vars = []
        dict_convert = dico.read_convert(join(creole_datadir, 'convert', 'amon.csv'))
        #VARIABLE_RGX = re.compile('%%(\w+)%%')
        VARIABLE_RGX = re.compile('%%(.*?)%%')
        variables = VARIABLE_RGX.findall(data)
        dico_inv = {}
        for newvar, oldvars in dict_convert.items():
            if len(oldvars) == 1:
                dico_inv[oldvars[0]] = newvar
        unknown_vars = []
        for var_ori in variables:
            # recherche de la variable équivalente si dispo
            if var_ori in dico_inv.keys():
                data = data.replace('%%'+var_ori+'%%', '%%'+dico_inv[var_ori])
            else:
                data = data.replace('%%'+var_ori+'%%', '%%'+var_ori)
                self.unknown_vars.append(var_ori)
        return data

    def needs_instanciation(self):
        """renvoie True si l'objet contient des variables à instancier,
        sinon False
        """
        # py2.3
        return any([obj.needs_instanciation()
                    for obj in chain(self.zones, self.flux)])

    def unspecified_vars(self):
        """renvoie la liste des variables non spécifiés (non instanciées)
        trouvées sur l'objet
        """
        # py2.3
        return set(chain(*[obj.unspecified_vars() for obj in chain(self.zones, self.flux)
                           if obj.needs_instanciation()]))

