# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
Module définissant les parsers XML pour charger les fichiers
presets au démarrage de l'application ou les fichiers de la
librairie par défaut.
(Parser DOM)
"""

from xml.dom.minidom import parse, Node
from fwobjects import Zone, Directive, Extremite, Flux, Service, \
                      ServiceGroupe, Range, UserGroup, QosClass, \
                      ApplicationGroup, Application

def instantiate_include_from_dom(dom_element):
    """Construit une inclusion statique à partir d'un élément DOM
    """
    return ''.join([line.data for line in dom_element.childNodes]) or None

def instantiate_zone_from_dom(dom_element, inherited = False):
    """Construit une zone à partir d'un élément DOM
    L{era.tests.test_zone.test_parse_zone}
    """

    z = Zone(dom_element.getAttribute('name'),
             int(dom_element.getAttribute('level')),
             dom_element.getAttribute('ip'),
             dom_element.getAttribute('network'),
             dom_element.getAttribute('netmask'),
             dom_element.getAttribute('interface'))

    # Si l'objet est hérité, alors il n'est pas éditable
    z.set_inherited(inherited)
    if inherited:
        z.set_editable(False)

    # XXX FIXME BEURK! Quoiqu'il arrive, le bastion est inéditable
    if z.name == 'bastion':
        z.set_editable(False)
    return z

def instantiate_service_from_dom(dom_element, inherited = False):
    """Construit un service à partir d'un élément DOM
    L{era.tests.test_domparser.test_parse_service}
    """

    port_str = dom_element.getAttribute('ports')
    # nous regardons s'il s'agit d'un port ou d'une plage de ports
    if '-' in port_str :
        start, end = port_str.split('-')
        port_list = range(int(start), int(end)+1)
    else:
        # on ne caste plus parce que ça peut être une variable
        # port_list = [int(port_str)]
        port_list = [port_str]


    # print "création du service ",dom_element.getAttribute('name')
    s = Service(dom_element.getAttribute('name'),
                dom_element.getAttribute('protocol'),
                port_list,
                dom_element.getAttribute('id'),
                dom_element.getAttribute('libelle'))

    # Si l'objet est hérité, alors il n'est pas éditable
    s.set_inherited(inherited)
    tcpwrapper = dom_element.getAttribute('tcpwrapper')
    if not tcpwrapper == "":
        s.set_tcpwrapper(tcpwrapper)
    if inherited:
        s.set_editable(False)
    return s

def instantiate_qos_from_dom(root_node):
    """Construction des classes de qos"""
    qos_nodes = root_node.getElementsByTagName('qos')
    qos = []
    for dom_element in qos_nodes:
        id = dom_element.getAttribute('id')
        priority = dom_element.getAttribute('priority')
        zone = dom_element.getAttribute('zone')
        libelle = dom_element.getAttribute('libelle')
        bandwidth = dom_element.getAttribute('bandwidth')

        qos.append(QosClass(id, bandwidth, libelle=libelle, priority=priority, zone=zone))
    return qos

def instantiate_group_from_dom(dom_element, inherited = False, available_services = None):
    """Construit un groupe de service à partir d'un élément DOM
    L{era.tests.test_domparser.test_parse_group}

    """
    id = dom_element.getAttribute('id')
    libelle = dom_element.getAttribute('libelle')
    tcpwrapper = dom_element.getAttribute('tcpwrapper')
    service_list = []

    elements = [child for child in dom_element.childNodes if child.nodeType == Node.ELEMENT_NODE]
    for service_element in elements:
        for serv in available_services:
            if serv.name == service_element.getAttribute('name'):
                service_list.append(serv)
                serv.used += 1

    g = ServiceGroupe(id,libelle,service_list)
    # Si l'objet est hérité, alors il n'est pas éditable
    g.set_inherited(inherited)
    if inherited:
        g.set_editable(False)
    return g

def instantiate_flux_from_dom(dom_element, zones, flux_list, extremites, services, groups, ranges={}, user_groups={}, inherited=False):
    """Construit un flux à partir d'un élément DOM
    """
    zoneA_name = dom_element.getAttribute('zoneA')
    zoneB_name = dom_element.getAttribute('zoneB')

    if zoneA_name not in zones:
        raise Exception, "%s n'est pas une zone connue"%zoneA_name

    if zoneB_name not in zones:
        raise Exception, "%s n'est pas une zone connue"%zoneB_name

    flux = Flux(zones[zoneA_name], zones[zoneB_name])
    # XXX Trick : on essaye de récupérer le flux s'il existe venant d'un héritage
    try:
        flux = flux_list.pop(flux_list.index(flux))
    except ValueError:
        pass

    montantes_node = dom_element.getElementsByTagName('montantes')[0]
    up_nodes = montantes_node.getElementsByTagName('directive')
    for directive_node in up_nodes:
        d = instantiate_directive_from_dom(directive_node, extremites, services, groups, ranges, user_groups,
                                       inherited = inherited)
        flux.up_directives_store.add_directive(d,load = 1)

    if str(montantes_node.getAttribute('default_policy')).strip() != "":
        montantes_default_policy = montantes_node.getAttribute('default_policy')
        #flux.up_directives_store.default_policy = bool(int(montantes_default_policy))
        flux.set_up_default_policy(bool(int(montantes_default_policy)))

    descendantes_node = dom_element.getElementsByTagName('descendantes')[0]
    down_nodes = descendantes_node.getElementsByTagName('directive')

    for directive_node in down_nodes:
        d = instantiate_directive_from_dom(directive_node, extremites, services, groups, ranges, user_groups,
                                           inherited = inherited)
        flux.down_directives_store.add_directive(d)

    if str(descendantes_node.getAttribute('default_policy')).strip() != "":
        descendantes_default_policy = descendantes_node.getAttribute('default_policy')
        #flux.down_directives_store.default_policy = bool(int(descendantes_default_policy))
        flux.set_down_default_policy(bool(int(descendantes_default_policy)))
    # flux.set_editable(editable)
    return flux

def instantiate_directive_from_dom(dom_element, extremites, services, groups, ranges={}, user_groups={}, inherited = False):
    """Construit une directive à partir d'un élément DOM
    L{era.tests.test_directive}
    L{era.tests.test_domparser.test_parse_directive}
    """
    src_names = []
    dest_names = []
    service_group = False

    sources = dom_element.getElementsByTagName('source')
    for source_node in sources:
        extr_name = source_node.getAttribute('name')
        if extr_name not in extremites:
            raise Exception, "%s n'est pas une extrémité connue !"%extr_name
        else:
            src_names.append(extr_name)

    destinations = dom_element.getElementsByTagName('destination')
    for dest_node in destinations:
        extr_name = dest_node.getAttribute('name')
        if extr_name not in extremites:
            raise Exception, "%s n'est pas une extrémité connue !"%extr_name
        else:
            dest_names.append(extr_name)
    exceptions = []
    dom_exceptions = dom_element.getElementsByTagName('exception')
    for exp_node in dom_exceptions:
        extr_name = dest_node.getAttribute('name')
        exceptions.append(dict(
          name = str(exp_node.getAttribute('name')),
          ip = str(exp_node.getAttribute('ip')),
          eolvar = str(exp_node.getAttribute('eolvar')),
          src = bool(int(exp_node.getAttribute('src'))),
          dest = bool(int(exp_node.getAttribute('dest')))
        ))
    service_name = dom_element.getAttribute('service')
    # filtrage applicatif
    app_group = dom_element.getAttribute('app_group')
    if app_group == '':
        app_group = None

    service = None
    if service_name not in services:
        if service_name not in groups:
            raise Exception, "%s n'est pas un service connu !"%service_name
        else:
            service = groups[service_name]
            service_group = True
    else:
        service = services[service_name]

    src_inversion = int(dom_element.getAttribute('src_inv'))
    dest_inversion = int(dom_element.getAttribute('dest_inv'))
    serv_inversion = int(dom_element.getAttribute('serv_inv'))

    user_group_id = dom_element.getAttribute('user_group')
    user_group = None
    if user_group_id != '':
        if user_group_id not in user_groups:
            raise Exception, "%s n'est pas un utilisateur/groupe reconnu" % user_group_id
        else:
            user_group = user_groups[user_group_id]

    mark_operator = dom_element.getAttribute('mark_operator')
    mark_value = dom_element.getAttribute('mark_value')
    mark = None
    if mark_value != '':
        mark = (mark_operator, mark_value)

    action = int(dom_element.getAttribute('action'))
    attrs = int(dom_element.getAttribute('attrs'))
    ipsec = dom_element.getAttribute('ipsec')
    if ipsec != '':
        ipsec = int(ipsec)
    else:
        ipsec = 0

    accept = dom_element.getAttribute('accept')
    if accept != '':
        accept = int(accept)
    else:
        accept = 0
    range_name = dom_element.getAttribute('time')
    timerange = None
    if range_name != '':
        if not ranges.has_key(range_name):
            raise Exception, "%s n'est pas une plage horaire connue !" % range_name
        else:
            timerange = ranges[range_name]
            ranges[range_name].used += 1

    libelle = dom_element.getAttribute('libelle')

    nat_extr, nat_port = None, None
    if dom_element.hasAttribute('nat_extr'):
        nat_extr_name = dom_element.getAttribute('nat_extr')
        if nat_extr_name not in extremites:
            raise Exception, "%s n'est pas une extrémité connue !"%extr_name
        else:
            nat_extr = extremites[nat_extr_name]

    if dom_element.hasAttribute('nat_port') :
        nat_port = dom_element.getAttribute('nat_port')

    src_list = [extremites[name] for name in src_names]
    dest_list = [extremites[name] for name in dest_names]

    d = Directive(src_list, dest_list, service, action, attrs,
        nat_extr, nat_port, is_group = service_group,
        src_inv=src_inversion, dest_inv=dest_inversion, serv_inv=serv_inversion,
        libelle=libelle, time=timerange, user_group=user_group,
        mark=mark, app_group=app_group)

    d.set_priority(int(dom_element.getAttribute('priority')))
    d.ipsec = ipsec
    d.accept = accept
    d.exceptions = exceptions
    # récupération du tag
    if dom_element.hasAttribute('tag'):
        d.set_tag(dom_element.getAttribute('tag'))

    # on met à jour les services et extremites utilisés (+1 au compteur d'utilisation)
    service.used +=1
    for extr in src_list:
        extr.used += 1
    for extr in dest_list:
        extr.used += 1

    # Si l'objet est hérité, alors il n'est pas éditable
    d.set_inherited(inherited)
    if inherited:
        d.set_editable(False)

    return d

def instantiate_range_from_dom(dom_element, inherited = False):
    ranges = {}
    for time_node in dom_element.getElementsByTagName('ranges'):
        for range_node in dom_element.getElementsByTagName('range'):
            options = ['timestart','timestop','datestart','datestop','weekdays','name']
            timerange = {}
            for option in options:
                opt_val = range_node.getAttribute(option)
                if opt_val != None:
                    timerange[option] = opt_val
            # instanciation objet range
            obj = Range(timerange)
            obj.set_inherited(inherited)
            if inherited:
                obj.set_editable(False)
            ranges[timerange['name']] = obj
    return ranges

def instantiate_extremite_from_dom(dom_element, zones, inherited = False):
    """Construit une extrémité à partir d'un élément DOM
    L{era.tests.test_domparser.test_parse_extremite}
    """
    ip_list = []
    ip_nodes = dom_element.getElementsByTagName('ip')
    for ip_node in ip_nodes:
        ip_list.append(ip_node.getAttribute('address'))

    zone_name = dom_element.getAttribute('zone')
    if zone_name not in zones:
        raise Exception, _("unknown zone %s")%zone_name

    e_name = dom_element.getAttribute('name')
    all_zone = (zone_name == e_name)
    subnet = 0
    if dom_element.hasAttribute('subnet'):
        subnet = int(dom_element.getAttribute('subnet'))

    extr = Extremite(zones[zone_name],
                     e_name,
                     dom_element.getAttribute('libelle'),
                     ip_list,
                     dom_element.getAttribute('netmask'),
                     all_zone = all_zone,
                     subnet = subnet)

    extr_type = dom_element.getAttribute("type")
    extr.set_type(extr_type)
    interface = dom_element.getAttribute("interface")
    container_name = dom_element.getAttribute("container")
    extr.set_interface(interface)
    extr.container_name = container_name
    # Si l'objet est hérité, alors il n'est pas éditable
    extr.set_inherited(inherited)
    if inherited:
        extr.set_editable(False)
    return extr

def instantiate_user_group_from_dom(dom_element, inherited=False):
    """
    Groupe d'utilisateur authentifie
    L{era.tests.test_user_xml.test_parse_user_group}
    """
    id = dom_element.getAttribute("id")
    name = dom_element.getAttribute("name")
    ugrp = UserGroup(id, name)
    ugrp.set_inherited(inherited)
    return ugrp

def get_services_from_dom(dom_element, inherited = False):
    """Récupère l'ensemble des services et des groupes de service
    à partir d'un élément 'services'
    """
    service_list = []
    group_list = []
    elements = [child for child in dom_element.childNodes if child.nodeType == Node.ELEMENT_NODE]
    for child in elements:
        if child.tagName == "service":
            service_list.append(instantiate_service_from_dom(child, inherited = inherited))
        elif child.tagName == "groupe":
            group_list.append(instantiate_group_from_dom(child, inherited = inherited, available_services = service_list))

    return service_list, group_list

def instanciate_bandwidth_from_dom(root_node):
    """renvoie les attributs bandwidth de la balise racine classe de qos"""
    # il n'y a en principe qu'une conteneur de qos
    qos_classes = root_node.getElementsByTagName('qosclasses')
    try:
        qos_bandwidth = qos_classes[0]
        upload = qos_bandwidth.getAttribute('upload')
        download = qos_bandwidth.getAttribute('download')
        return upload, download
    except:
        return "", ""


def instanciate_appgroup_from_dom(dom_element, inherited=False):
    """
      :dom_element: du type::

        <app_group name="browser" description="navigateur web">
            <app name="firefox"/>
            <app name="opera"/>
        </app_group>
        <app_group name="mail" description="client de messagerie">
            <app name="outlook"/>
            <app name="thunderbird"/>
        </app_group>
      :return:
        { 'browser':("navigateur web, ['firefox', 'opera']),
          'mail': ("client de messagerie", ['oulook', 'thunderbird']),
        }
   """
    app_groups = {}
    for group in dom_element.getElementsByTagName('app_group'):
        app_group_name = group.getAttribute('name')
        app_group_description = group.getAttribute('description')
        paths = []
        for app in group.getElementsByTagName('app'):
            paths.append(app.getAttribute('name'))
        app_groups[app_group_name] = ApplicationGroup(app_group_name, description=app_group_description,
                                   applications=paths)
    return app_groups

def instanciate_application_from_dom(dom_element, inherited=False):
    """
        :dom_element: du type::

            <application name="opera" description="navigateur web opera">
              <path name="/usr/bin/opera"/>
              <path name="/usr/share/opera"/>
            </application>
        :return:
         {
          'opera':("navigateur web opera",
                    ['/usr/bin/opera', '/usr/share/opera'])
         }
    """
    applications = {}
    for application in dom_element.getElementsByTagName('application'):
        app_name = application.getAttribute('name')
        app_description = application.getAttribute('description')
        paths = []
        for path in application.getElementsByTagName('path'):
            paths.append(path.getAttribute('name'))
        applications[app_name] = Application(app_name, description=app_description,
                                  paths=paths)# (paths, app_description)
    return applications

## fonctions utilisees pour l'initialisation

def parse_basic_fwobjects(root_node, inherited, zones):
    """recupere les fwobjects de base (extremites, services, ...)
    """
    services, service_groups = {}, {}
    extremites = {}
    ranges = {}
    user_groups = {}

    services_node = root_node.getElementsByTagName('services')[0]
    sl, gl = get_services_from_dom(services_node, inherited=inherited)
    services = dict([(s.name, s) for s in sl])
    service_groups = dict([(g.id, g) for g in gl])

    user_group_nodes = root_node.getElementsByTagName('user_group')
    for child in user_group_nodes:
        user_group = instantiate_user_group_from_dom(child, inherited=inherited)
        user_groups[user_group.id] = user_group

    # On crée les objets plage horaires (range)
    try:
        ranges = instantiate_range_from_dom(root_node, inherited=inherited)
    except:
        ranges = {}

    # On crée les extrémités
    for extremite_elt in root_node.getElementsByTagName('extremite'):
        extr = instantiate_extremite_from_dom(extremite_elt, zones, inherited=inherited)
        extremites[extr.name] = extr

    return services, service_groups, user_groups, ranges, extremites

def parse_zone_file(filepath):
    """Fonction qui parse un fichier librairie de zones
    filepath : le path du fichier à parser.
    L{era.tests.test_default_lib.test_parse_zone_file}
    """
    document = parse(filepath).firstChild
    zone_nodes = document.getElementsByTagName('zone')
    zone_list = []
    for node in zone_nodes:
        zone_list.append(instantiate_zone_from_dom(node))

    return zone_list

#def parse_user_group_file(filepath):
#    """Fonction qui parse un fichier librairie de services
#    filepath : le path du fichier à parser.
#    """
#    document = parse(filepath).firstChild
#    user_groups = {}
#    user_group_nodes = document.getElementsByTagName('user_group')
#    for child in user_group_nodes:
#        user_group = instantiate_user_group_from_dom(child, inherited=False)
#        user_groups[user_group.id] = user_group
#    return user_groups

def parse_service_file(filepath):
    """Fonction qui parse un fichier librairie de services
    filepath : le path du fichier à parser.
    L{era.tests.test_default_lib.test_parse_service_file}
    L{era.tests.test_default_lib.test_parse_service_groupes}
    """
    document = parse(filepath).firstChild
    # services_node = document.getElementsByTagName('services')[0]
    return get_services_from_dom(document)

def parse_root_attributes(root_node):
    """
        parse les attribut de la racine xml (<firewall ...>)
    """
    root_attrs = {}
    ## model (heritage)
    # afin de pouvoir recuperer tous les modeles a l'enregistrement
    model = root_node.getAttribute('model')
    root_attrs['model'] = model
    ## options du modèle (netbios, qos en mode utilisateur ou normal)
    options = {}
    if root_node.hasAttribute('netbios'):
        # l'attribut netbios doit etre 0 ou 1
        netbios = int(root_node.getAttribute('netbios'))
    else:
        # si l'attribut netbios n'est pas renseigné
        netbios = None
    options['netbios'] = netbios
    root_attrs['options'] = options
    if root_node.hasAttribute('qos'):
        # l'attribut qos doit etre 0 ou 1
        qos = int(root_node.getAttribute('qos'))
    else:
        # si l'attribut qos n'est pas renseigné la qos est desenclenchee
        qos = None
    options['qos'] = qos
    if root_node.hasAttribute('version'):
        version = float(root_node.getAttribute('version'))
    else:
        # si pas de version --> ancien modèle, on considère qu'on est en version 1.0
        version=1.0
    root_attrs['version'] = version
    return root_attrs

def init_zones(root_node, inherited=False):
    """
        initialise les zones
    """
    zones = {}
    # On crée les zones
    for zone_elt in root_node.getElementsByTagName('zone'):
        zone = instantiate_zone_from_dom(zone_elt, inherited = inherited)
        zones[zone.name] = zone
    return zones

def init_flux(root_node, zones, flux_list, extremites, services, service_groups, ranges, user_groups, inherited=False):
    """
        creation des flux
    """
    flux_list = []
    # On crée les flux
    for flux_elt in root_node.getElementsByTagName('flux'):
        flux = instantiate_flux_from_dom(flux_elt,zones, flux_list, extremites, services, service_groups, ranges, user_groups, inherited = inherited)
        flux_list.append(flux)
    return flux_list

def init_static_rules(root_node):
    """
        On crée éventuellement l'inclusion statique
    """
    try:
        rule = root_node.getElementsByTagName('include')[0]
        rules = instantiate_include_from_dom(rule)
    except:
        rules = None
    return rules

def init_applications(root_node, inherited=False):
    """
        creation des applications et des groupes d'applications
        pour le filtrage applicatif
    """
    if len(root_node.getElementsByTagName('applications')) == 1:
        apps_node = root_node.getElementsByTagName('applications')[0]
        app_groups = instanciate_appgroup_from_dom(apps_node, inherited = inherited)
        apps = instanciate_application_from_dom(apps_node, inherited=inherited)
        applications = (app_groups, apps)
    else:
        applications = ({},{})
    return applications

