# -*- coding: UTF-8 -*-
"""Module de tests unitaires pour l'écriture des règles iptables"""
import time
from datetime import datetime
from os.path import join

import autopath

from era.backend.iptwriter import IPTWriter
from era.backend.compiler import Compiler
from era.noyau.initialize import initialize_app
from era.noyau.fwobjects import Directive, Zone, Extremite, Service, Range, \
     ACTION_ALLOW, ACTION_DNAT, ACTION_DENY



from utils import get_first_diff, purge_list, LStringIO

ZONES_CHAINS = ['/sbin/iptables -t filter -N bas-int',
                '/sbin/iptables -t filter -N int-bas',
                '/sbin/iptables -t filter -N int-ext',
                '/sbin/iptables -t filter -N ext-int',
                '/sbin/iptables -t filter -N bas-ext',
                '/sbin/iptables -t filter -N ext-bas',
                '/sbin/iptables -t filter -N ext-ext',
                '/sbin/iptables -t filter -N bas-bas',
                '/sbin/iptables -t filter -N int-int']


ZONES_DEFAULT_POLICY = [
    '/sbin/iptables -t filter -A ext-int -i eth0 -o eth1 -s 0/0 -d 0/0 -j DROP',
    '/sbin/iptables -t filter -A ext-bas -i eth0 -s 0/0 -d 0/0 -j DROP',
    '/sbin/iptables -t filter -A int-ext -i eth1 -o eth0 -s 0/0 -d 0/0 -j ACCEPT',
    '/sbin/iptables -t filter -A int-bas -i eth1 -s 0/0 -d 0/0 -j DROP',
    # Pas besoin de faire les règles au départ du bastion
    # '/sbin/iptables -t filter -A bas-ext -o eth0 -s 0/0 -d 0/0 -j ACCEPT',
    # '/sbin/iptables -t filter -A bas-int -o eth1 -s 0/0 -d 0/0 -j ACCEPT'
    ]


DEFAULT_INPUT_POLICY = ['/sbin/iptables -t filter -A INPUT -i eth0 -j ext-bas',
 '/sbin/iptables -t filter -A INPUT -i eth1 -j int-bas',
 '/sbin/iptables -t filter -A ext-bas -i eth0 -s 0/0 -d 0/0 -j DROP',
 '/sbin/iptables -t filter -A ext-int -i eth0 -o eth1 -s 0/0 -d 0/0 -j DROP',
 '/sbin/iptables -t filter -A int-bas -i eth1 -s 0/0 -d 0/0 -j DROP',
 '/sbin/iptables -t filter -A int-ext -i eth1 -o eth0 -s 0/0 -d 0/0 -j ACCEPT',
 '/sbin/iptables -t filter -N bas-bas',
 '/sbin/iptables -t filter -N bas-ext',
 '/sbin/iptables -t filter -N bas-int',
 '/sbin/iptables -t filter -N ext-bas',
 '/sbin/iptables -t filter -N ext-ext',
 '/sbin/iptables -t filter -N ext-int',
 '/sbin/iptables -t filter -N int-bas',
 '/sbin/iptables -t filter -N int-ext',
 '/sbin/iptables -t filter -N int-int']

DEFAULT_FORWARD_POLICY = ['/sbin/iptables -t filter -A FORWARD -i eth0 -o eth0 -j ext-ext',
 '/sbin/iptables -t filter -A FORWARD -i eth0 -o eth1 -j ext-int',
 '/sbin/iptables -t filter -A FORWARD -i eth1 -o eth0 -j int-ext',
 '/sbin/iptables -t filter -A FORWARD -i eth1 -o eth1 -j int-int',
 '/sbin/iptables -t filter -A INPUT -i eth0 -j ext-bas',
 '/sbin/iptables -t filter -A INPUT -i eth1 -j int-bas',
 '/sbin/iptables -t filter -A ext-bas -i eth0 -s 0/0 -d 0/0 -j DROP',
 '/sbin/iptables -t filter -A ext-int -i eth0 -o eth1 -s 0/0 -d 0/0 -j DROP',
 '/sbin/iptables -t filter -A int-bas -i eth1 -s 0/0 -d 0/0 -j DROP',
 '/sbin/iptables -t filter -A int-ext -i eth1 -o eth0 -s 0/0 -d 0/0 -j ACCEPT',
 '/sbin/iptables -t filter -N bas-bas',
 '/sbin/iptables -t filter -N bas-ext',
 '/sbin/iptables -t filter -N bas-int',
 '/sbin/iptables -t filter -N ext-bas',
 '/sbin/iptables -t filter -N ext-ext',
 '/sbin/iptables -t filter -N ext-int',
 '/sbin/iptables -t filter -N int-bas',
 '/sbin/iptables -t filter -N int-ext',
 '/sbin/iptables -t filter -N int-int']

DEFAULT_NETBIOS_POLICY = [
    '/sbin/iptables -t filter -A FORWARD -i eth0 -o eth1 -j netbios-ext',
    '/sbin/iptables -t filter -A FORWARD -i eth1 -o eth0 -j netbios-ext'
    ]

CUSTOM_NETBIOS_POLICY = [
    '/sbin/iptables -t filter -A FORWARD -i eth1 -o eth0 -j netbios-ext'
    ]

NO_NETBIOS_POLICY = [
    ]

INPUT = ['/sbin/iptables -t filter -A int-bas -p tcp --dport 80 -i eth1 -s 10.20.30.1/255.255.255.255 -d 10.20.30.0 -j ACCEPT']
#['/sbin/iptables -t filter -A int-bas -m state --state NEW -p tcp --dport 80 --tcp-flags SYN,RST,ACK SYN -i eth1 -s 10.20.30.1/255.255.255.255 -d 10.20.30.0 -j ACCEPT']

DEST_EXT = [
    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth0 -s 10.20.30.1/255.255.255.255 -d 0/0 -j ACCEPT'
    ]

DEST_FROM = [
    '/sbin/iptables -t filter -A ext-int -p tcp --dport 80 -i eth0 -o eth1 -s 0/0 -d 10.20.30.1/255.255.255.255 -j ACCEPT'
    ]

FROM_ALL_ZONE = [
    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth0 -s 0/0 -d 0/0 -j DROP',
    ]

DIRECTIVE_RESULT = ['/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.1/1.1.1.1 -d 40.50.60.1/1.1.1.1 -j DROP',
                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.2/1.1.1.1 -d 40.50.60.1/1.1.1.1 -j DROP',
                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.3/1.1.1.1 -d 40.50.60.1/1.1.1.1 -j DROP',
                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.1/1.1.1.1 -d 40.50.60.2/1.1.1.1 -j DROP',
                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.2/1.1.1.1 -d 40.50.60.2/1.1.1.1 -j DROP',
                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.3/1.1.1.1 -d 40.50.60.2/1.1.1.1 -j DROP']

# gestion du décalage UTC #19299
ts = time.time()
offset = int((datetime.fromtimestamp(ts) -
          datetime.utcfromtimestamp(ts)).total_seconds() / 3600)
hstart = 4-offset
hstop = 19-offset
if len(str(hstart)) == 1:
    hstart = "0{}".format(hstart)
if len(str(hstop)) == 1:
    hstop = "0{}".format(hstop)

DIRECTIVE_TIME_RESULT = ['/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.1/1.1.1.1 -d 40.50.60.1/1.1.1.1 -m time --timestart {}:00 --timestop {}:00 --datestart 2005:02:01 --datestop 2005:02:31 -j DROP'.format(hstart, hstop),
 '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.1/1.1.1.1 -d 40.50.60.2/1.1.1.1 -m time --timestart {}:00 --timestop {}:00 --datestart 2005:02:01 --datestop 2005:02:31 -j DROP'.format(hstart, hstop),
 '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.2/1.1.1.1 -d 40.50.60.1/1.1.1.1 -m time --timestart {}:00 --timestop {}:00 --datestart 2005:02:01 --datestop 2005:02:31 -j DROP'.format(hstart, hstop),
 '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.2/1.1.1.1 -d 40.50.60.2/1.1.1.1 -m time --timestart {}:00 --timestop {}:00 --datestart 2005:02:01 --datestop 2005:02:31 -j DROP'.format(hstart, hstop),
 '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.3/1.1.1.1 -d 40.50.60.1/1.1.1.1 -m time --timestart {}:00 --timestop {}:00 --datestart 2005:02:01 --datestop 2005:02:31 -j DROP'.format(hstart, hstop),
 '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.3/1.1.1.1 -d 40.50.60.2/1.1.1.1 -m time --timestart {}:00 --timestop {}:00 --datestart 2005:02:01 --datestop 2005:02:31 -j DROP'.format(hstart, hstop),
]


#['/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.1/1.1.1.1 -d 40.50.60.1/1.1.1.1 -m time --timestart 02:00 --timestop 19:00 --datestart 2005:02:01 --datestop 2005:02:31 --days Mon,Tue,Wed -j DROP',
#                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.2/1.1.1.1 -d 40.50.60.1/1.1.1.1 -m time --timestart 02:00 --timestop 19:00 --datestart 2005:02:01 --datestop 2005:02:31 --days Mon,Tue,Wed -j DROP',
#                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.3/1.1.1.1 -d 40.50.60.1/1.1.1.1 -m time --timestart 02:00 --timestop 19:00 --datestart 2005:02:01 --datestop 2005:02:31 --days Mon,Tue,Wed -j DROP',
#                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.1/1.1.1.1 -d 40.50.60.2/1.1.1.1 -m time --timestart 02:00 --timestop 19:00 --datestart 2005:02:01 --datestop 2005:02:31 --days Mon,Tue,Wed -j DROP',
#                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.2/1.1.1.1 -d 40.50.60.2/1.1.1.1 -m time --timestart 02:00 --timestop 19:00 --datestart 2005:02:01 --datestop 2005:02:31 --days Mon,Tue,Wed -j DROP',
#                    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth2 -s 10.20.30.3/1.1.1.1 -d 40.50.60.2/1.1.1.1 -m time --timestart 02:00 --timestop 19:00 --datestart 2005:02:01 --datestop 2005:02:31 --days Mon,Tue,Wed -j DROP']

DNAT_DIRECTIVE = [
    # '/sbin/iptables -t nat -A PREROUTING -s 0/0 -i $INT_EXT -p tcp --dport 8600 -j DNAT --to %%adresse_ip_web%%:22',
    # '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 8600 -i eth0 -s 0/0 -j DNAT --to-destination ip_web',
    '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 8600 -i eth0 -s 0/0 -d 40.50.60.10 -j DNAT --to-destination ip_web:22',
    '/sbin/iptables -t filter -A ext-int -p tcp --dport 22 -i eth0 -o eth1 -s 0/0 -d ip_web -j ACCEPT']


IN_OUT_XML = [
    '/sbin/iptables -t filter -A int-bas -p tcp --dport 22 -i eth1 -s 192.168.230.0/255.255.255.0 -d 192.168.230.100 -j ACCEPT',
    '/sbin/iptables -t filter -A int-bas -p tcp --dport 22 -i eth1 -s 192.168.230.50/255.255.255.255 -d 192.168.230.100 -j ACCEPT',
    # '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth0 -s 192.168.230.0/255.255.255.0 -d 0/0 -j DROP',
    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth0 -s 0/0 -d 0/0 -j DROP',
    '/sbin/iptables -t filter -A int-ext -p tcp --dport 80 -i eth1 -o eth0 -s 192.168.230.50/255.255.255.255 -d 0/0 -j DROP'
    ]

DNAT_XML = [
    '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 80 -i eth0 -s 0/0 -d %%adresse_ip_eth0%% -j DNAT --to-destination %%adresse_ip_web%%:80',
    '/sbin/iptables -t filter -A ext-int -p tcp --dport 80 -i eth0 -o eth1 -s 0/0 -d %%adresse_ip_web%% -j ACCEPT',
    '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 8600 -i eth0 -s 0/0 -d %%adresse_ip_eth0%% -j DNAT --to-destination %%adresse_ip_web%%:22',
    '/sbin/iptables -t filter -A ext-int -p tcp --dport 22 -i eth0 -o eth1 -s 0/0 -d %%adresse_ip_web%% -j ACCEPT'
    ]

SNAT_XML_PORT_VARIABLE = ['/sbin/iptables -t filter -A int-ext -i eth1 -o eth0 -s 0/0 -d 0/0 -j ACCEPT',
 '/sbin/iptables -t nat -A POSTROUTING -o eth0 -s 0/0 -d 0/0 -j SNAT --to-source %%adresse_ip_eth0%%:%%natport']

SNAT_XML = [
    '/sbin/iptables -t nat -A POSTROUTING -o eth0 -s 0/0 -d 0/0 -j SNAT --to-source %%adresse_ip_eth0%%',
    # Règle implicite
    '/sbin/iptables -t filter -A int-ext -i eth1 -o eth0 -s 0/0 -d 0/0 -j ACCEPT',
    # '/sbin/iptables -t filter -A int-ext -i eth1 -o eth0 -s ?? -d 0/0 -j ACCEPT',
    ]

PORT_VARIABLE_XML = ['/sbin/iptables -t filter -A ext-int -p tcp --dport %%natport -i eth0 -o eth1 -s 0/0 -d %%adresse_ip_web -j ACCEPT',
         '/sbin/iptables -t filter -A ext-int -p tcp --dport 22 -i eth0 -o eth1 -s 0/0 -d %%adresse_ip_web -j ACCEPT',
          '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 80 -i eth0 -s 0/0 -d %%adresse_ip_eth0 -j DNAT --to-destination %%adresse_ip_web:%%natport',
           '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 8600 -i eth0 -s 0/0 -d %%adresse_ip_eth0 -j DNAT --to-destination %%adresse_ip_web:22']

PORT_VARIABLE_TEMPLATE_XML = ['/sbin/iptables -t filter -A ext-int -p tcp --dport 22 -i eth0 -o eth1 -s 0/0 -d %%adresse_ip_web%% -j ACCEPT',
 '/sbin/iptables -t filter -A ext-int -p tcp --dport 80 -i eth0 -o eth1 -s 0/0 -d %%adresse_ip_web%% -j ACCEPT',
 '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 80 -i eth0 -s 0/0 -d %%adresse_ip_eth0%% -j DNAT --to-destination %%adresse_ip_web%%:80',
 '/sbin/iptables -t nat -A PREROUTING -p tcp --dport 8600 -i eth0 -s 0/0 -d %%adresse_ip_eth0%% -j DNAT --to-destination %%adresse_ip_web%%:22']

output = LStringIO()
def _check_output(expected):
    """Routine de vérification commune à presque tous les tests
        - Enlève les commentaires et trie la sortie du compilateur
        - Trie le résultat attendu
        - Compare les deux :
        - si identiques : OK
        - sinon, lève une AssertionError avec la première différence trouvée
    """
    uncommented = purge_list(output)
    uncommented.sort()
    expected.sort()
    # first_diff est un tuple

    first_diff = get_first_diff(uncommented, expected)
    if first_diff:
        assert uncommented == expected


## Tests pour comportement par défaut ######################################


def test_zones_to_chains():
    """L{era.backend.compiler.Compiler._zones_to_chains}
    Teste la création de chaines nommees
    '/sbin/iptables -t filter -N bas-int
    """
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    fw_file = join(autopath.HERE, "data",'input-output.xml')
    mmodel = initialize_app(fw_file)
    compiler._mmodel = mmodel
    compiler._zones_to_chains()
    global output
    output = compiler._output
    _check_output(ZONES_CHAINS)


def test_zones_default_policy():
    """L{era.backend.compiler.Compiler._zones_default_policy}
    règles par défaut inter chaines
    """
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    fw_file = join(autopath.HERE, "data",'input-output2.xml')
    mmodel = initialize_app(fw_file)
    # XXX
    compiler._mmodel = mmodel
    compiler._zones_default_policy()
    global output
    output = compiler._output
    _check_output(ZONES_DEFAULT_POLICY)


def test_default_input_policy():
    """L{era.backend.compiler.Compiler._default_input_policy}
    règles en INPUT
    """
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    fw_file = join(autopath.HERE, "data",'input-output.xml')
    mmodel = initialize_app(fw_file)
    # XXX
    compiler._mmodel = mmodel
    compiler._default_input_policy()
    global output
    output = compiler._output
    _check_output(DEFAULT_INPUT_POLICY)


def test_default_forward_policy():
    """L{era.backend.compiler.Compiler._default_forward_policy}
    règles en FORWARD
    """
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    fw_file = join(autopath.HERE, "data",'input-output.xml')
    mmodel = initialize_app(fw_file)
    # XXX
    compiler._mmodel = mmodel
    compiler._default_forward_policy()
    global output
    output = compiler._output
    _check_output(DEFAULT_FORWARD_POLICY)


#     def test_default_netbios():
#         """Teste l'écriture de règles netbios"""
#         fw_file = 'data/input-output.xml'
#         mmodel = initialize_app(fw_file)
#         # XXX
#         compiler._mmodel = mmodel
#         compiler._netbios_policy()
#         _check_output(DEFAULT_NETBIOS_POLICY)


#     def test_several_chains_netbios():
#         """Teste l'écriture de règles netbios avec plusieurs chaines
#         précisées"""
#         fw_file = 'data/input-output.xml'
#         mmodel = initialize_app(fw_file)
#         # On teste en passant une liste de chaines, ici, comme on donne
#         # toutes les chaines possibles du modèle, ça *doit* revenir
#         # au même qu'en mettant compiler.options['netbios'] = 1
#         compiler._mmodel = mmodel
#         compiler._netbios_policy(['int-ext', 'ext-int'])
#         _check_output(DEFAULT_NETBIOS_POLICY)


#     def test_unique_chain_netbios():
#         """Teste l'écriture de règles netbios avec une seule chaine
#         précisée"""
#         fw_file = 'data/input-output.xml'
#         mmodel = initialize_app(fw_file)
#         # On teste avec une seule chaine
#         compiler._mmodel = mmodel
#         compiler._netbios_policy(['int-ext'])
#         _check_output(CUSTOM_NETBIOS_POLICY)


## À partir d'objets construits à la main  #################################
def test_input():
    """L{era.backend.compiler.Compiler._process_directives}
    Test la presence de input a partir d'une directive
    Teste le compilateur quand 'bastion' est la zone d'arrivée : (INPUT)
    À tester :
        - pas d'option '-o' en sortie
        - On doit avoir : '-d ip_zone_depart' (sans netmask)
    """
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    ######################################
    # -*- construction de la directive -*-
    zoneA = Zone('interieur', 10, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('bastion', 10, '127.0.0.1')
    service = Service('un service', 'tcp', [80], 's1', "-*-")
    extr1 = Extremite(zoneA, 'extrA', 'azerty', ['10.20.30.1'],
                        '255.255.255.255')
    extr2 = Extremite(zoneB, 'bastion', 'Tout le bastion', ['127.0.0.1'],
                        '255.255.255.255')
    directive = Directive([extr1], [extr2], service, ACTION_ALLOW, 0)
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(INPUT)

def test_dest_exterieur():
    """L{era.backend.compiler.Compiler._process_directives}
    Teste le compilateur quand 'exterieur' est la zone d'arrivée :
    À tester :
        - On doit avoir : '-d 0/0' (tout, pas seulement jusqu'au routeur)
    """
    ######################################
    # -*- construction de la directive -*-
    zoneA = Zone('interieur', 10, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('exterieur', 10, '40.50.60.0', interface = 'eth0')
    service = Service('un service', 'tcp', [80], 's1', "-*-")
    extr1 = Extremite(zoneA, 'extrA', 'azerty', ['10.20.30.1'],
                        '255.255.255.255')
    extr2 = Extremite(zoneB, 'exterieur', 'dans exterieur', ['40.50.60.1'],
                        '255.255.255.255')
    directive = Directive([extr1], [extr2], service, ACTION_ALLOW, 0)
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(DEST_EXT)


def test_from_exterieur():
    """L{era.backend.compiler.Compiler._process_directives}
    Teste le compilateur quand 'exterieur' est la zone de départ :
    À tester :
        - On doit avoir : '-s 0/0' (tout, pas seulement jusqu'au routeur)
    """
    ######################################
    # -*- construction de la directive -*-
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    zoneA = Zone('interieur', 10, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('exterieur', 10, '40.50.60.0', interface = 'eth0')
    service = Service('un service', 'tcp', [80], 's1', "-*-")
    extr1 = Extremite(zoneA, 'extrA', 'azerty', ['10.20.30.1'],
                        '255.255.255.255')
    extr2 = Extremite(zoneB, 'exterieur', 'dans exterieur', ['40.50.60.1'],
                        '255.255.255.255')
    directive = Directive([extr2], [extr1], service, ACTION_ALLOW, 0)
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(DEST_FROM)


def test_from_all_zone():
    """L{era.backend.compiler.Compiler._process_directives}
    Teste le compilateur si l'extrémité de départ est 'toute la zone'
    À tester :
        - On doit avoir : '-s 0/0'
    """
    ######################################
    # -*- construction de la directive -*-
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    zoneA = Zone('interieur', 10, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('exterieur', 10, '40.50.60.0', interface = 'eth0')
    service = Service('un service', 'tcp', [80], 's1', "-*-")
    extr1 = Extremite(zoneA, 'extrA', 'azerty', ['10.20.30.1'],
                        '255.255.255.255', all_zone = True)
    extr2 = Extremite(zoneB, 'exterieur', 'dans exterieur', ['40.50.60.1'],
                        '255.255.255.255')
    directive = Directive([extr1], [extr2], service, ACTION_DENY, 0)
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(FROM_ALL_ZONE)


def test_process_directive():
    """L{era.backend.compiler.Compiler._process_directives}
    Teste la compilation d'une directive basique"""
    ######################################
    # -*- construction de la directive -*-
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    zoneA = Zone('interieur', 10, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('exterieur', 10, '40.50.60.0', interface = 'eth2')
    service = Service('internet restreint', 'tcp', [80], 's1', "Yahoo !")
    extr1 = Extremite(zoneA, 'extrA', 'azerty', ['10.20.30.1', '10.20.30.2'],
                        '1.1.1.1')
    extr2 = Extremite(zoneA, 'extrB', 'uiop', ['10.20.30.3'],
                        '1.1.1.1')
    extr3 = Extremite(zoneB, 'extrC', 'qsdfgh', ['40.50.60.1', '40.50.60.2'],
                        '1.1.1.1')

    directive = Directive([extr1, extr2], [extr3], service, ACTION_DENY, 0)
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(DIRECTIVE_RESULT)

def test_process_time_directive():
    """L{era.backend.compiler.Compiler._process_directives}
    compilation d'une directive basique avec des paramètres de temps
    """
    ######################################
    # -*- construction de la directive -*-
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    zoneA = Zone('interieur', 10, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('exterieur', 10, '40.50.60.0', interface = 'eth2')
    service = Service('internet restreint', 'tcp', [80], 's1', "Yahoo !")
    extr1 = Extremite(zoneA, 'extrA', 'azerty', ['10.20.30.1', '10.20.30.2'],
                        '1.1.1.1')
    extr2 = Extremite(zoneA, 'extrB', 'uiop', ['10.20.30.3'],
                        '1.1.1.1')
    extr3 = Extremite(zoneB, 'extrC', 'qsdfgh', ['40.50.60.1', '40.50.60.2'],
                        '1.1.1.1')

    directive = Directive([extr1, extr2], [extr3], service, ACTION_DENY, 0)
    directive.time = Range({'name':'range1', 'timestart':'04:00', 'timestop':'19:00', 'datestart':'2005:02:01', 'datestop':'2005:02:31', 'days':'Mon,Tue,Wed'})
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(DIRECTIVE_TIME_RESULT)


def test_process_dnat_directive():
    """Teste la compilation d'une directive DNAT

    """
    ######################################
    # -*- construction de la directive -*-
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    zoneA = Zone('interieur', 50, '10.20.30.0', interface = 'eth1')
    zoneB = Zone('exterieur', 0, '40.50.60.10', '40.50.60.0',
                    '255.255.255.0', interface = 'eth0')
    zoneC = Zone('bastion', 100, '127.0.0.1')
    http = Service('http', 'tcp', [80], 's1', "serveur web")
    ssh_haut = Service('ssh-haut', 'tcp', [8600], 's2', 'un autre ssh')
    exterieur = Extremite(zoneB, 'exterieur', 'azerty', ['10.20.30.1'],
                            '255.255.255.255')
    bastion = Extremite(zoneC, 'bastion', 'uiop', ['127.0.0.1'],
                        '255.255.255.255')
    serveur_web = Extremite(zoneA, 'serveur-web', 'qsdfgh', ['ip_web'],
                            '255.255.255.255')

    directive = Directive([exterieur], [bastion], ssh_haut, ACTION_DNAT, 0,
                            nat_extr = serveur_web, nat_port = 22)
    # -*- fin de la construction de la directive -*-
    ################################################
    compiler._process_directives([directive])
    global output
    output = compiler._output
    _check_output(DNAT_DIRECTIVE)


## À partir de fichiers XML ################################################
def test_in_out_xml():
    """L{era.backend.compiler.Compiler._process_flux}
    génération de règles simples à partir du XML d'entrée

    """
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    matrix_model = initialize_app(join(autopath.HERE, "data",'input-output.xml'))
    for flux in matrix_model.flux:
        compiler._process_flux(flux)
    global output
    output = compiler._output
    _check_output(IN_OUT_XML)

def test_dnat_xml():
    """L{era.backend.compiler.Compiler._process_flux}
    Teste la génération de règles DNAT à partir du XML d'entrée"""
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    matrix_model = initialize_app(join(autopath.HERE, "data",'dnat.xml'))
    for flux in matrix_model.flux:
        compiler._process_flux(flux)
    global output
    output = compiler._output
    _check_output(DNAT_XML)

def test_snat_xml():
    """L{era.backend.compiler.Compiler._process_flux}

    génération de règles SNAT à partir du XML d'entrée"""
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    matrix_model = initialize_app(join(autopath.HERE, "data",'snat.xml'))
    for flux in matrix_model.flux:
        compiler._process_flux(flux)
    global output
    output = compiler._output
    _check_output(SNAT_XML)


def test_dnat_xml_port_variable():
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    matrix_model = initialize_app(join(autopath.HERE, "data",'portvariable.xml'))
    for flux in matrix_model.flux:
        compiler._process_flux(flux)
    global output
    output = compiler._output
    _check_output(PORT_VARIABLE_XML)

def test_snat_xml():
    """L{era.backend.compiler.Compiler._process_flux}

    génération de règles SNAT à partir du XML d'entrée"""
    compiler = Compiler(IPTWriter, output, join(autopath.HERE, "data",'config.ini'))
    matrix_model = initialize_app(join(autopath.HERE, "data",'snatportvariable.xml'))
    for flux in matrix_model.flux:
        compiler._process_flux(flux)
    global output
    output = compiler._output
    _check_output(SNAT_XML_PORT_VARIABLE)


