# -*- coding: UTF-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""
Permet d'initialiser l'application
point d'entree : initialize_app

En particulier, ce module contient les fonctions de parsages "haut-niveau"
(<=> le DOM se fait plus bas), et la fonction initialize_app qui est utilisée
pour (ré)initiliaser l'application (au démarrage, ou dans "Open", ou dans
"Load")

"""

try:
    _
except:
    _ = str

from era.noyau.domparsers import *
from era.noyau.path import *
from era.noyau.constants import *
from era.noyau.models import MatrixModel
from era.noyau.pool import library_store

from xml.dom.minidom import parse
import os

LIB_PATH = os.path.join(ERA_DIR, "lib")
DEFAULT_LIB_PATH = os.path.join(LIB_PATH, "defaults")
ZONES_FILENAME = "zones.xml"
SERVICES_FILENAME = "services.xml"
USER_GROUPS_FILENAME = "user_groups.xml"

def load_defaults(matrix_model):
    """Charge les zones et services par défaut de l'application
    matrix_model : le modèle de matrice de flux dans lequel on ajoute
    les zones
    """
    zone_list = parse_zone_file(os.path.join(DEFAULT_LIB_PATH,ZONES_FILENAME))
    for z in zone_list:
        matrix_model.add_zone(z)
    load_default_services()
#    load_default_user_group()

def load_default_services():
    """ Charge la liste des services par défaut et l'ajoute à la liste passée
    en paramètre.
    """
    service_list, group_list = parse_service_file(os.path.join(DEFAULT_LIB_PATH,SERVICES_FILENAME))
    library_store.services = service_list
    library_store.service_groups = group_list

def add_default_services():
    """Charge la liste des services par défaut et l'ajoute éventuellement
    à la liste des services du library_store si ces services
    ne sont pas déjà présents
    """
    service_list, group_list = parse_service_file(os.path.join(DEFAULT_LIB_PATH, SERVICES_FILENAME))
    for serv in service_list:
        if serv not in library_store.services:
            library_store.add_service(serv)
    for grp in group_list:
        if grp not in library_store.service_groups:
            library_store.add_group(grp)

#def load_default_user_group():
#    """ Charge la liste des user_group par défaut et l'ajoute à la liste passée
#    en paramètre.
#    """
#    library_store.user_groups = parse_user_group_file(os.path.join(DEFAULT_LIB_PATH,USER_GROUPS_FILENAME))

def merge_flux(flux_list, new_flux_list):
    """merge des flux
    """
    for new_flx in new_flux_list:
        if new_flx not in flux_list:
            # si flux inexistant, on l'ajoute
            flux_list.append(new_flx)
        else:
            # si le nouveau flux est dans la liste des flux
            # on concatène les directives des deux
            for flx in flux_list:
                if new_flx == flx:
                    flx.append_directives_from_flux(new_flx)
                    break

def parse_firewall_file(filepath, inherited = False):
    """Parse un fichier firewall.
    filepath : le path du fichier à parser
    inherited : True si on parse le fichier en tant que modèle, False sinon
    L{era.tests.test_user_xml.test_parse_model_user_group}
    """
    #try:
    # On récupère le DOM document, puis le noeud racine
    document = parse(filepath)
    root_node = document.firstChild
    return _parse_firewall_file(root_node, inherited)
    #except Exception, e:
    #    raise Exception(_('problem at loading time') + ":"+filepath + "\n" + str(e))

def _parse_firewall_file(root_node, inherited):
    models = root_node.getAttribute('model')
    # Si ce firewall hérite d'un modèle => on parse ce modèle
    services, service_groups = {}, {}
    zones = {}
    extremites = {}
    ranges = {}
    flux_list = []
    model = None
    user_groups = {}
    options = {}
    app_groups = {}
    applications = {}
    inherited_qos_bandwidths = []
    library_store.inherited_option_netbios = None
    library_store.inherited_option_qos = None
    if (root_node.hasAttribute('model') and str(models) != ""):
        for model in [m.strip() for m in models.split(',')]:
            fwobjs = parse_firewall_file(model, inherited = True)
            zones.update(fwobjs['zones'])
            merge_flux(flux_list, fwobjs['flux_list'])
            extremites.update(fwobjs['extremites'])
            services.update(fwobjs['services'])
            service_groups.update(fwobjs['service_groups'])
            ranges.update(fwobjs['ranges'])
            user_groups.update(fwobjs['user_groups'])
            if fwobjs['options'][0]['qos'] is not None:
                library_store.inherited_option_qos = fwobjs['options'][0]['qos']
            if fwobjs['options'][0]['netbios'] is not None:
                library_store.inherited_option_netbios = fwobjs['options'][0]['netbios']
            inherited_qos_bandwidths = fwobjs['qos']
            inherited_upload, inherited_download = fwobjs['options'][1],  fwobjs['options'][2]
            app_groups.update(fwobjs['app_groups'])
            applications.update(fwobjs['applications'])
    # les attributs de la balise racine
    root_attrs = parse_root_attributes(root_node)
    model = root_attrs['model']
    library_store.model = model
    options = root_attrs['options']
    netbios = options['netbios']
    library_store.initial_option_netbios = options['netbios']
    library_store.initial_option_qos = options['qos']
    qos = options['qos']
    if netbios is not None:
        netbios = options['netbios']
    else:
        if model:
            # on recupere l'option du modèle hérité
            netbios = library_store.inherited_option_netbios
        else:
            netbios = default_options['netbios']
    if qos is not None:
        qos = options['qos']
    else:
        if model:
            # on recupere l'option du modèle hérité
            qos = library_store.inherited_option_qos
        else:
            qos = default_options['qos']
    # si la qos est activee mais non renseignee dans le modele fils,
    # on prend la qos du modele pere ->> mais il faut que l
    # la qos du modele pere existe
    qos_bandwiths = instantiate_qos_from_dom(root_node)
    upload, download = instanciate_bandwidth_from_dom(root_node)
    if qos == 1:
        if len(qos_bandwiths) == 0 and len(inherited_qos_bandwidths) != 0:
            qos_bandwiths = inherited_qos_bandwidths
            upload, download = inherited_upload, inherited_download

    library_store.set_options(dict(netbios=netbios, qos=qos))
    version = root_attrs['version']
    library_store.set_version(version)
    # fwobjects de base
    zones.update(init_zones(root_node, inherited=inherited))
    base_services, base_service_groups, base_user_groups, base_ranges, base_extremites = parse_basic_fwobjects(root_node, inherited, zones)
    services.update(base_services)
    service_groups.update(base_service_groups)
    user_groups.update(base_user_groups)
    ranges.update(base_ranges)
    extremites.update(base_extremites)
    # preparation pour le merge de flux
    new_flux_list = init_flux(root_node, zones, flux_list, extremites, services, service_groups, ranges, user_groups, inherited=inherited)
    merge_flux(flux_list, new_flux_list)
    rules = init_static_rules(root_node)
    if rules is not None:
        # escaping xml characters in static rules
        from xml.sax.saxutils import unescape
        rules = unescape(rules)
    options = (options, upload, download)
    # filtrage applicatif
    app_groups, applications = init_applications(root_node, inherited=inherited)
    return dict(zones=zones, flux_list=flux_list, extremites=extremites,
                  services=services, service_groups=service_groups,
                  rules=rules, netbios=netbios, ranges=ranges,
                  model=model, user_groups=user_groups,
                  qos=qos_bandwiths, options=options,
                  app_groups=app_groups, applications=applications,
                  )

def create_matrix_model(zones, flux, inherited=None):
    """Crée et initialise un modèle de matrice avec le dictionnaire
    zones et la liste flux
    """
    matrix_model = MatrixModel()
    # On ne passe pas par la méthode add_zone pour ne pas
    # créer les flux automatiquement
    for zone in zones.values():
        matrix_model.add_zone(zone)
    matrix_model.zones.sort()

    # On réécrit par dessus les flux créés automatiquement
    # dans add_zone()
    matrix_model.flux = flux
    for f in matrix_model.flux:
        f.add_listener(matrix_model)

    matrix_model.inherited = inherited

    return matrix_model

def init_library_store(extremites, services, service_groups, rules, netbios, ranges, user_groups, qos, options, app_groups, applications):
    """réinitialise library_store avec les extremites, services et
    groupes passés en paramètre
    options = {'netbios':netbios}
    qos : qosclasses
    """
    library_store.extremites = extremites
    library_store.services = services.values()
    library_store.qos = qos
    library_store.service_groups = service_groups.values()
    library_store.rules = rules
    library_store.ranges = ranges
    library_store.user_groups = user_groups
    #library_store.set_options(options)
    library_store.add_application_groups(app_groups)
    library_store.add_applications(applications)

# FIXME : a utiliser lors du chargement des modeles herites
def update_library_store(extremites, services, service_groups, ranges, user_groups):
    """ajoute au library_store les extremites, services et
    groupes passés en paramètre
    """
    library_store.extremites.update(extremites)
    library_store.services.extend(services.values())
    library_store.service_groups.extend(service_groups.values())
    library_store.ranges.update(ranges)
    library_store.user_groups.update(user_groups)

def initialize_app(fw_file = None, inherited = False):
    """fonction d'initialisation de l'application
    fw_file : le filepath du fichier firewall. Si fw_file vaut None,
    on retourne des modèles vides
    Retourne le modèle de matrice de flux créé.
    """
    if fw_file is not None:
#        zones, flux_list, extremites, services, service_groups, rules, netbios, ranges, model, user_groups, qos, options = \
        fwobjs = parse_firewall_file(fw_file, inherited)
        zones = fwobjs['zones']
        flux_list = fwobjs['flux_list']
        extremites = fwobjs['extremites']
        services = fwobjs['services']
        service_groups = fwobjs['service_groups']
        rules = fwobjs['rules']
        netbios = fwobjs['netbios']
        ranges = fwobjs['ranges']
        model = fwobjs['model']
        user_groups = fwobjs['user_groups']
        qos = fwobjs['qos']
        options = fwobjs['options']
        app_groups = fwobjs['app_groups']
        applications = fwobjs['applications']

        matrix_model = create_matrix_model(zones, flux_list, model)
        # FIXME: les bandwiths sont pour l'instant rangees dans les options
        init_library_store(extremites, services, service_groups, rules, netbios, ranges, user_groups, qos, options[0], app_groups, applications)
        library_store.set_bandwidth(options[1], options[2])
        library_store.clear_tags()
        # recharge les libs par defaut si necessaire
        add_default_services()
        # recupere les tags des directives optionnelles
        try:
            tags = matrix_model.get_active_list()
        except:
            tags = []
        for i in tags:
            # convertissons en
            active = i[1] & DIRECTIVE_ACTIVE == DIRECTIVE_ACTIVE
            hidden = i[1] & DIRECTIVE_HIDDEN == DIRECTIVE_HIDDEN
            library_store.tags[str(i[0])]=(active,hidden)
        #print app_groups.values()[0].applications

    else:
        library_store.clear()
        matrix_model = MatrixModel()
        load_defaults(matrix_model)
    return matrix_model

