# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""Compilateur de directives ERA en règles iptables

Le compilateur accepte un fichier de configuration qui permet de définir
le chemin d'accès au fichier de règles statiques ainsi que les options
relatives aux règles netbios, etc.
"""

import sys
import re
from os.path import abspath, normpath, join, dirname, isdir
from os import makedirs
from collections import defaultdict

from era.backend.processors import get_processor
from era.backend.iptrule import ParameterFactory, TargettedRule, InterfaceParameter
from era.backend.utils import gen_chains, gen_chains_interface
from era.noyau.pool import library_store
from era.noyau.path import CONFIG_FILE, ERA_DIR, MARKGROUP_FILE
from era.noyau.erreurs import InvalidZoneError
from era.noyau.fwobjects import Service
from era.backend.cheetah import Multivar

class Output(list):
    """Filtre permettant d'eliminer les doublons sur la sortie
    """
    datas= {'nat': [], 'filter': []}
    iptables_restore = False
    exec_path = '/sbin/iptables'
    #def __init__(self, iptables_restore, exec_path):
    #    self.iptables_restore = iptables_restore
    #    self.exec_path = exec_path
    #    super(Output, self).__init__()

    def write(self, data):
        self.append(str(data))

    def append_table(self, table, data):
        if self.iptables_restore:
            self.datas[table].append(str(data))
        else:
            self.append('%s -t %s %s' % (self.exec_path, table, str(data)))

    def getvalue(self, join_char = ''):
        return join_char.join(self)


class Compiler(object):
    """Compilateur de directives ERA en règles iptables
    """

    def __init__(self, writer_cls, output=None, config_file=None,
                ipsetdir='/tmp/ipsets',
                containerdir='/tmp/containers/iptables/',
                output_inclusion=None,
                rules_file=None):
        """
         - writer : la classe chargée d'écrire les règles *iptables*
         - output : où le résultat doit être écrit, par défaut : sys.stdout
         - config_file : le fichier de configuration du compilateur
        """
        # Impossible de faire "output or sys.stdout" si output = [] par exemple
        exec_path = "/sbin/iptables"
        Output.iptables_restore = rules_file is not None
        Output.exec_path = exec_path
        self._output = Output()
        self._output_inclusion = Output()
        # second rules pile for multiple file output
        # used for output in container files
        self.VARIABLE_RGX = None
        if output is None:
            self.output_fh = sys.stdout
        else:
            self.output_fh = output
        if output_inclusion is None:
            self.output_inclusion = sys.stdout
        else:
            self.output_inclusion = output_inclusion
        # there are multiple files for container outputs
        # one for each container
        self.container_output = defaultdict(Output)
        self.container_output_fh = dict()
        self.container_output_writer = dict()
        self._writer = None
        self._policy_order = []
        self.ipsetdir = ipsetdir
        self.rules_file = rules_file
        self.containerdir = containerdir
        self.options = {'exec_path'          : exec_path,
                        # begin : inclusion en début de fichier
                        'begin_static_file'  : None,
                        'middle_static_file' : None,
                        'end_static_file'    : None,
                        'icmp'               : True,
                        'cheetah'            : True,
                        # qos : si True, les règles mangle sont générées
                        # FIXME : à détecter automatiquement
                        'lxc_rules'         : None,
                        'qos'                : False}
        self._load_conf(config_file, writer_cls)
        self._mmodel = None

    def _load_conf(self, config_file, writer_cls):
        """Charge le fichier de configuration
        """
        import ConfigParser
        parser = ConfigParser.ConfigParser()
        parser.readfp(file(config_file))
        for option in parser.options('POLICY'):
            self.options[option] = parser.getboolean('POLICY', option)
        for option in parser.options('OPTIONS'):
            self.options[option] = parser.getboolean('OPTIONS', option)
        for option in parser.options('PATHS'):
            self.options[option] = parser.get('PATHS', option)
        self.writer_cls = writer_cls
        self._writer = writer_cls(self.options['exec_path'],
                                  output = self._output)


    def compile(self, matrix_model, variables=None, force=False, no_creole=False):
        """Compile la matrice de flux en règles iptables
        """
        # Etape -ALPHA (premilimnaire) : nettoyage, on vide ou cree le fichier markgroup.conf
        if not isdir(dirname(MARKGROUP_FILE)):
            makedirs(dirname(MARKGROUP_FILE))
        fh = file(MARKGROUP_FILE, 'w')
        fh.write("")
        fh.close()
        # compatibilite
        if library_store.get_version() >= 2.0:
            self.VARIABLE_RGX = re.compile('%%(\w+)')
        else:
            self.VARIABLE_RGX = re.compile('%%(\w+)%%')
        unspecified_vars = matrix_model.unspecified_vars()
        variables = variables or {}
        missing = unspecified_vars - set(variables)
        self._mmodel = matrix_model
        # 0. inclusion du sh bang
        self._output.write('#!/bin/sh\n')
        self._output_inclusion.write('#!/bin/sh\n')
        # 1 - Règles statiques
        self._include_file('begin_static_file')
        self._default_lo_policy()
        # 2 - Chargement des modules du noyau (insmod / modprobe)
        self._load_modules()
        # 3 - Création des chaines relatives aux couples zone-zone
        self._zones_to_chains()
        # 4 - Création des règles semi-statiques
        self._write_semi_static_rules()
        # 5 - directives -> règles iptables
        for flux in matrix_model.flux:
            self._process_flux(flux)
        if self.rules_file:
            self._output.write('/sbin/iptables-restore < {}\n'.format(self.rules_file))
        # 6 - 2ème inclusion statique avant la politique par défaut
        # inclusion des options de log iptables
        self._log_unmatched()
        # 7 - Politique par défaut
        self._default_policy()
        # 8 - Appel des chaines
        self._write_chains_call()
        # 10 - Inclusion du fichier de fin (enregistrement des chaînes)
        self._include_file('lxc_rules')
        self._include_file('end_static_file')
        # 11 - deuxieme passe qui genere du format cheetah
        if self.rules_file:
            if self.options['cheetah'] and not no_creole:
                for filt in self._output.datas:
                    self._output.datas[filt] = self._multivars(self._output.datas[filt])
            with open(self.rules_file + '.tmpl', 'w') as fh:
                for filt in self._output.datas:
                    fh.write('*{}\n'.format(filt))
                    for data in self._output.datas[filt]:
                        fh.write(data)
                    fh.write('COMMIT\n\n')
        if self.options['cheetah'] and not no_creole:
            self._output = self._multivars(self._output)
        # 9 - Inclusion statique provenant du fichier xml
        self._include_string(library_store.rules)
        # au final, transformons la liste en string
        output = "".join(self._output) # .getvalue()
        # instanciation des variables au dernier moment
        if not force and variables:
            try:
                output = self.VARIABLE_RGX.sub(lambda m:variables[m.group(1)], output)
            except KeyError, err:
                print "Erreur: Impossible de trouver l'information sur la variable", err
                sys.exit(1)
        self.output_fh.write(output)
        self.output_inclusion.write("".join(self._output_inclusion))
        # self.output_fh.close()
        # let's build the different container outputs and write into them
        for key, container_rules in self.container_output.items():
            self.container_output_fh[key] = file(join(self.containerdir, key + '.sh'), 'w')
            flush_container_output = "".join(self._multivars(container_rules))
            self.container_output_fh[key].write(flush_container_output)
            self.container_output_fh[key].close()

    def _include_file(self, file_type):
        """Inclut les règles statiques
         - file_type : le nom de l'option correspondant au fichier à
           inclure.
           file_type peut valoir :
            * 'begin_static_file' pour le fichier à inclure en tête
            * 'end_static_file' pour le fichier à inclure avant la
              définition de la politique par défaut
        """
        try:
            static_file = self.options[file_type]
        except KeyError:
            static_file = None
        if static_file:
            in_str = file(abspath(normpath(join(ERA_DIR,static_file))))
            comment = 'inclusion statique a partir du fichier %r' % static_file
            self._writer.comment(comment, 3)
            self._output.write(in_str.read())
            self._writer.comment("Fin de l'" + comment, 3)
            self._output.write('\n')

    def _include_string(self, text):
        """Inclusion statique à partir d'une chaine
        """
        if library_store.rules:
            self._output_inclusion.append("# Inclusion statique provenant du fichier xml \n")
            self._output_inclusion.append(str(text))
            self._output_inclusion.append('\n  \n')

    def _load_modules(self):
        """ Génère les insmod appropriés en fonction des modules
        spécifiés dans le modèle ERA """
        pass

    def _zones_to_chains(self):
        """Crée une chaine pour chaque couple (orienté) zone-zone
        L{era.tests.test_compiler.test_zones_to_chains}
        """
        self._writer.comment(u"Creation des chaines zone-zone", 3)
        for chain_name in gen_chains(self._mmodel.zones):
            if self.options['qos']: # FIXME: [QOS] Only create mangle if we use QoS
                self._writer.new_chain(chain_name.replace('-','_'), "mangle")
            self._writer.new_chain(chain_name)

        self._output.write('\n')


    def _write_semi_static_rules(self):
        """Définit l'ensemble des règles semi-statiques, c'est à dire
        les règles ICMP / RETOUR CONNEXION / NETBIOS pour lesquelles on a
        besoin de connaître le nom des chaînes mais dont la syntaxe
        est toujours la même
        """
        zones = self._mmodel.zones
        # On trie par niveau
        zones.sort()
        # Zones sans bastion
        states = ['established', 'related']
        # 1 - On autorise les retours de connexions
        for chain_name in gen_chains(zones, mirror = False):
            if self.options['qos']: # FIXME: [QOS] Only do it if zone has QoS
                rule = TargettedRule('ACCEPT', chain_name.replace('-','_'), "mangle")
                rule.add_parameter(ParameterFactory.match_state(states))
                self._writer.append_rule(rule)
            rule = TargettedRule('ACCEPT', chain_name)
            rule.add_parameter(ParameterFactory.match_state(states))
            self._writer.append_rule(rule)

        # 2 - On écrit les règles ICMP
        self._icmp_default_policy()
        # 3 - On écrit les règles netbios
        if int(library_store.options['netbios']) == 1:
            # Comportement par défaut : toutes les chaines sont prises
            # en compte
            self._netbios_policy()
        else:
            return

    def _log_unmatched(self):
        """Ecrire la règle de log de tout le reste
        """
        zones = self._mmodel.zones
        # On trie par niveau
        zones.sort()
        zone_index = [zone.name for zone in zones].index("exterieur")
        ext_interface = zones[zone_index].interface
        self._output.append_table('filter', '-A ext-bas -m limit --limit 120/min -i %s -j LOG --log-prefix "iptables connection attempt: "\n' % ext_interface)

    def _default_lo_policy(self):
        self._writer.new_chain('lo')
        rule = TargettedRule('ACCEPT', 'INPUT')
        rule.add_parameter(ParameterFactory.interface('lo'))
        self._writer.append_rule(rule)
        rule = TargettedRule('ACCEPT', 'OUTPUT')
        rule.add_parameter(ParameterFactory.interface('lo', out=True))
        self._writer.append_rule(rule)

    def _default_policy(self):
        """Ecrit les règles relatives à la politique par défaut
        """
        # On parcourt les ensembles optionels (icmp, netbios, ...)
        for optional_set in self._policy_order:
            # Si on doit écrire des règles pour cet ensemble, on appelle la
            # méthode correspondante
            if self.options[optional_set]:
                self.__policy_methods[optional_set](self)

        self._zones_default_policy()


    def _netbios_policy(self, chains = None):
        """Politique par défaut pour les règles netbios
        Les règles générées font partie des règles classées 'semi-statiques'
         - *chains* est un argument optionel qui est la liste de l'ensemble des
           chaines (<=> flux orientés) pour lesquelles on veut traiter les
           paquets netbios.
        Si *chains* n'est pas précisé, alors toutes les chaines seront
        concernées.
        """

        # juste une règle netbios concernant l'exterieur
        self._writer.new_chain('netbios-ext')
        for z in self._mmodel.zones:
            if z.name == 'exterieur':
                ext = z

        rule = TargettedRule('netbios-ext', 'FORWARD')
        # chain_name = '%s-%s' % (zone1.name[:3], zone2.name[:3])
        rule.add_parameter(ParameterFactory.interface(ext.interface, out=True))
        self._writer.append_rule(rule)

        #self._output.write('\n')

        rule = TargettedRule('DROP', 'netbios-ext')
        rule.add_parameter(ParameterFactory.service2(Service('typ', 'tcp', ['135'], None, None), False))
        self._writer.append_rule(rule)
        rule = TargettedRule('DROP', 'netbios-ext')
        rule.add_parameter(ParameterFactory.service2(Service('typ', 'udp', ['135'], None, None), False))
        self._writer.append_rule(rule)
        #
        rule = TargettedRule('DROP', 'netbios-ext')
        rule.add_parameter(ParameterFactory.service2(Service('typ', 'tcp', ['137', '139'], None, None), False))
        self._writer.append_rule(rule)
        rule = TargettedRule('DROP', 'netbios-ext')
        rule.add_parameter(ParameterFactory.service2(Service('typ', 'udp', ['137', '139'], None, None), False))
        self._writer.append_rule(rule)
        #
        rule = TargettedRule('DROP', 'netbios-ext')
        rule.add_parameter(ParameterFactory.service2(Service('typ', 'tcp', ['445'], None, None), False))
        self._writer.append_rule(rule)
        rule = TargettedRule('DROP', 'netbios-ext')
        rule.add_parameter(ParameterFactory.service2(Service('typ', 'udp', ['445'], None, None), False))
        self._writer.append_rule(rule)

#         zones = self._mmodel.zones
#         # On trie par niveau et on enlève la zone bastion
#         zones.sort()
#         zones = zones[:-1]
#         for zone1, zone2 in gen_couples(zones, zones):
#             if zone1 == zone2:
#                 continue
#             chain_name = '%s-%s' % (zone1.name[:3], zone2.name[:3])
#             if chains is not None and chain_name not in chains:
#                 continue
#             rule = TargettedRule('netbios-ext', 'FORWARD')
#             # Interface en entrée
#             rule.add_parameter(ParameterFactory.interface(zone1.interface))
#             # Interface en sortie
#             rule.add_parameter(ParameterFactory.interface(zone2.interface,
#                                                           True))
#             self._writer.append_rule(rule)
#         self._output.write('\n')


    def _icmp_default_policy(self):
        """Politique par défaut pour les règles icmp
        Les règles générées font partie des règles classées 'semi-statiques'
        """
        zones = self._mmodel.zones
        # On trie par niveau et on enlève la zone bastion
        zones.sort()
        zones = zones[:-1]
        self._writer.new_chain('icmp-acc')
        for zone in zones:
##         for zone1, zone2 in gen_couples(zones, zones):
##             if zone1 == zone2:
##                 continue
            rule = TargettedRule('icmp-acc', 'INPUT')
            # Protocole ICMP
            rule.add_parameter(ParameterFactory.protocol('icmp'))
            # Interface en entrée
            rule.add_parameter(ParameterFactory.interface(zone.interface))
            # Interface en sortie
            # rule.add_parameter(ParameterFactory.interface(zone2.interface,
            #                                               True))
            self._writer.append_rule(rule)
        for typ in ['destination-unreachable', 'source-quench', 'time-exceeded', 'parameter-problem', 'echo-reply']:
            rule = TargettedRule('ACCEPT', 'icmp-acc')
            # Protocole ICMP
            service = Service(typ, 'icmp', None, None, None)
            rule.add_parameter(ParameterFactory.service2(service, False))
            self._writer.append_rule(rule)

        self._output.write('\n')


    def _zones_default_policy(self):
        """Politique par défaut pour les règles relatives aux zones
        L{era.tests.test_compiler.test_zones_default_policy}
        """
        mmodel_flux = self._mmodel.flux
        for flux in mmodel_flux:
            try:
                zone1 = flux.get_highlevel_zone()
                zone2 = flux.get_lowlevel_zone()
            except InvalidZoneError, e:
                print e

            up_store = flux.up_directives_model()
            down_store = flux.down_directives_model()

            self.create_default_policy_rule(zone1, zone2, down_store.default_policy)
            self.create_default_policy_rule(zone2, zone1, up_store.default_policy)
        self._output.write('\n')

    def create_default_policy_rule(self, zone1, zone2, default_policy):
        """
            génère une règle de politique par défaut, qu'elle soit en accept
            ou en drop, en fonction du paramètre *default_policy*
        """

        if default_policy:
            target_type = 'ACCEPT'
        else:
            target_type = 'DROP'

        # pas de regles par défaut nécessaires à partir du bastion
        if zone1.name == 'bastion':
            return

        self._writer.comment('Paquets contenant la zone %s' % zone1.name, 2)
        chain_name = '%s-%s' % (zone1.name[:3], zone2.name[:3])
        rule = TargettedRule(target_type, chain_name)
        # Création / Ajout des paramètres relatifs aux deux zones
        params = ParameterFactory.parameters_for_zones(zone1, zone2)
        for param in params:
            rule.add_parameter(param)
        # On ajoute la règle générée à la table filter
        self._writer.append_rule(rule)

    def _write_chains_call(self):
        """Génère les appels aux chaines créées par le compilateur"""
        zones = self._mmodel.zones
        # On trie par niveau
        zones.sort()
        # Dans toutes les chaines générées ici, on n'a pas besoin de la
        # zone bastion => on l'enlève de la liste
        zones = zones[:-1]

        self._default_input_policy(zones)
        self._output.write('\n')
        self._default_forward_policy(zones)
        self._output.write('\n')

    def _default_input_policy(self, zones = None):
        """L{era.tests.test_compiler.test_default_input_policy}

        Écrit les règles de renvoi pour INPUT
        Si zones est passé en paramètre, il sera utilisé, et on
        considérera que la liste ne comprend pas la zone bastion,
        sinon, on ira reprendre les zones du modèles, en enlevant
        le bastion.
        (Intérêt de passer cet argument : juste question
        performance / redondance, ne pas refaire plusieurs fois
        zones = self._mmodel.zones, zones = zones[:-1], etc.)
        """
        if zones is None:
            zones = self._mmodel.zones
            # On trie par niveau
            zones.sort()
            # Dans toutes les chaines générées ici, on n'a pas besoin de la
            # zone bastion => on l'enlève de la liste
            zones = zones[:-1]
        # Règles INPUT => on veut tout ce qui arrive au bastion
        self._writer.comment('Renvoi vers nos chaines pour le INPUT', 1)
        # On parcourt les zones, sauf le bastion
        for zone in zones:
            # La cible <=> la chaine qu'on a créée
            target_type = '%s-bas' % (zone.name[:3])
            # ajout de l'interface d'entrée à la chaine
            interface = InterfaceParameter(zone.interface)
            if self.options['qos']: # FIXME: [QOS] Only when we use QoS
                rule = TargettedRule(target_type.replace('-','_'), 'INPUT', 'mangle')
                rule.add_parameter(interface)
                self._writer.append_rule(rule)
            # création de la flèche d'entrée
            rule = TargettedRule(target_type, 'INPUT')
            rule.add_parameter(interface)
            self._writer.append_rule(rule)

    def _default_forward_policy(self, zones = None):
        """ L{era.tests.test_compiler.test_default_forward_policy}

        Écrit les règles de renvoi pour FORWARD
        Si zones est passé en paramètre, il sera utilisé, et on
        considérera que la liste ne comprend pas la zone bastion,
        sinon, on ira reprendre les zones du modèles, en enlevant
        le bastion.
        (Intérêt de passer cet argument : juste question
        performance / redondance, ne pas refaire plusieurs fois
        zones = self._mmodel.zones, zones = zones[:-1], etc.)
        """
        if zones is None:
            zones = self._mmodel.zones
            # On trie par niveau
            zones.sort()
            # Dans toutes les chaines générées ici, on n'a pas besoin de la
            # zone bastion => on l'enlève de la liste
            zones = zones[:-1]

        # Règles FORWARD
        self._writer.comment('Renvoi vers nos chaines pour le FORWARD', 1)
        # On génère toutes les chaines possibles pour le forward
        for chain_name, input, output in gen_chains_interface(zones):
            # ajout de l'interface input
            input_interface = InterfaceParameter(input)
            # ajout de l'interface output
            output_interface = InterfaceParameter(output, out=True)

            if self.options['qos']: # FIXME: [QOS] Only for QoS
                rule = TargettedRule(chain_name.replace('-','_'), 'FORWARD', 'mangle')
                rule.add_parameter(input_interface)
                rule.add_parameter(output_interface)
                self._writer.append_rule(rule)
            # construction de la chaine
            rule = TargettedRule(chain_name, 'FORWARD')
            rule.add_parameter(input_interface)
            rule.add_parameter(output_interface)
            self._writer.append_rule(rule)

    def _process_flux(self, flux):
        """L{era.tests.test_compiler.test_in_out_xml}

        L{era.tests.test_compiler.test_dnat_xml}

        L{era.tests.test_compiler.test_snat_xml}

        Transforme les directives du flux en règles iptables"""
        # Directives montantes
        self._writer.comment(u'Regles montantes entre %s et %s' % \
                            (flux.zoneA, flux.zoneB), 2)
        up_directives_store = flux.up_directives_model()
        self._process_directives(flux.up_directives_list(),
            default_policy=up_directives_store.default_policy)
        # Directives descendantes
        self._writer.comment(u'Regles descendantes entre %s et %s' % \
                            (flux.zoneA, flux.zoneB), 2)
        down_directives_store = flux.down_directives_model()
        self._process_directives(flux.down_directives_list(),
             default_policy=down_directives_store.default_policy)

    def _process_directives(self, directives, default_policy=False):
        """ L{era.tests.test_compiler.test_input}
        L{era.tests.test_compiler.test_dest_exterieur}
        L{era.tests.test_compiler.test_from_exterieur}
        L{era.tests.test_compiler.test_from_all_zone}
        L{era.tests.test_compiler.test_process_directive}

        X{processeur} unité de transformation de règles iptables
        """
        # pile de fin pour l'accumulation de tous les types de regles (implicites, etc)
        final_rules = []
        for directive in directives:
        # la directive est elle active ?
            comment = directive.libelle.encode('utf-8')
            if directive.is_active() or not directive.is_optional():
                self._writer.comment(comment, 3)
                proc = get_processor(directive, default_policy=default_policy)
                proc.ipsetdir = self.ipsetdir
                rules = proc.process()
                for rule in rules:
                    self._writer.append_rule(rule)
                # la directive a-t-elle des regles implicites ?
                final_rules.extend(proc.get_implicit_rules())
                    # la directive est-elle authentifiee ?
                if hasattr(proc, "get_authenticated_rules"):
                    final_rules.extend(proc.get_authenticated_rules())
                # retrieving container oriented rules
                if hasattr(proc, 'get_container_rules'):
                    for key, container_rules in proc.get_container_rules().items():
                        for container_rule in container_rules:
                            self.container_output_writer[key] = self.writer_cls(self.options['exec_path'],
                                                                                output=self.container_output[key])
                        #    self.container_output[key].write(container_rule)
                        ##self.container_output_fh[key].write("\n".join(container_rules))
                            self.container_output_writer[key].append_rule(container_rule) #getvalue(join_char="\n")
            else:
                self._writer.comment(comment + " : INACTIVE", 3)
        # ecriture des regles de fin de flux
        for rule in final_rules:
            self._writer.append_rule(rule)

        # ajout des regles implicite pour le snat

    __policy_methods  = {
        # 'icmp'    : _icmp_default_policy,
        # 'netbios' : _netbios_default_policy,
        }

    def _multivars(self, flushed_output):
        """prise en compte des variables multiples
        et generation d'un template cheetah
        bouclant sur les valeurs multi-valuees
        """
        try:
            m = Multivar(flushed_output)
            return list(m.process())
#            for l in list(m.process()):
#                self._output.write(l)
        except Exception, e:
            print e
            sys.exit(1)
        # transforme la liste en string
#        return self._output

# def run(model_file, output_file = None):
#     """Fonction principale du script de compilation
#     """
#     # XXX as an option ...
#     # FIXME -- pour un backend dans era (implique de lancer era dans son repertoire)
#     CONFIG_FILE = './backend/data/config.ini'
#     from era.noyau.initialize import initialize_app
#     from era.backend.iptwriter import IPTWriter
#     matrix_model = initialize_app(model_file)

#     w = LStringIO()
#     # w = LStringIO().StringIO()

#     compiler = Compiler(IPTWriter, w, CONFIG_FILE)
#     compiler.compile(matrix_model)

#     string = w.getvalue()

#     if output_file is not None:
#         output = open(output_file, 'w')
#         output.write(string)
#     else:
#         output = sys.stdout
#         output.write(string)


def run(model_file, output_file=None, variables=None, force=False):
    """Traite les options passées en ligne de commande, et lance la
    compilation avec les bons paramètres"""
    # internationalisation
    import locale, gettext
    from era.noyau.initialize import initialize_app
    APP = 'editeur'
    DIR = 'i18n'
    # locale.setlocale (locale.LC_ALL, '')
    gettext.bindtextdomain (APP, DIR)
    gettext.textdomain (APP)
    # installons la fonction _() dans les builtins
    gettext.install(APP, DIR, unicode=1)

    try:
        matrix_model = initialize_app(model_file)
    except Exception, e :
        sys.exit(e)
    compile_matrix(matrix_model, output_file, variables, force)


def compile_matrix(matrix_model, output_file=None, variables=None, force=False, no_creole=False):
    """compile le modèle passé en paramètre
    @param matrix_model : la matrice de flux
    @param output_file : le chemin du fichier de sortir (sys.stdout sinon)
    @param variables : un dictionnaire des variables à substituer à
                       la génération
    """
    # lancement de l'appliquation
    from era.backend.compiler import Compiler
    from era.backend.iptwriter import IPTWriter


    if output_file:
        output = file(output_file,'w')
    else:
        output = sys.stdout
    try:
        compiler = Compiler(IPTWriter, output, config_file=CONFIG_FILE)
        compiler.compile(matrix_model, variables, force, no_creole)
    except Exception, e:
        print str(e)

