# -*- coding: utf-8 -*-
###########################################################################
#
# Eole NG
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr
#
###########################################################################

"""Processeurs de règles iptrules à partir de directives ERA

un processeur est chargé de transformer un objet ERA (directives de types
autorisation, interdiction, redirection, dnat, snat
en règles iptables correspondantes
"""
from os.path import join
from copy import deepcopy
from collections import defaultdict

from era.backend.iptrule import Target, Mark, TargettedRule, MarkParameter, \
    ParameterFactory, DestIPParameter, LogParameter, TCPProtocol, \
    UDPProtocol, ICMPProtocol, \
    ESPProtocol, MatchParameter, IPParameter, InterfaceParameter, \
    InsertParameter, MatchState

from era.backend.utils import gen_couples
from era.backend.markgroup import append_mark, group_is_in_markfile, get_mark
from era.backend.cheetah import Multivar
from era.noyau.constants import ACTION_DENY, ACTION_ALLOW
from era.noyau.pool import library_store
from era.noyau.fwobjects import Service
from creole.eosfunc import is_ip

#variable globale permettant d'incrémenter les marques (directives authentifiées)
iterator = 1

# ipsets exceptions file names have to be kept in memory because of the flush
IPSETS_NAMES = []

try:
    from creole.client import CreoleClient
    creole_client = CreoleClient()
    mode_conteneur_actif = creole_client.get_creole('mode_conteneur_actif')
    VIRTDISABLED = {'non': True, 'oui': False}.get(mode_conteneur_actif)
except:
    VIRTDISABLED = True


class DirectiveProcessor(object):
    """Classe de base pour traiter les directives ERA
    """

    def __init__(self, directive):
        """
         - directive : la directive ERA
        """
        self.directive = directive
        # self.directive = self._is_list(directive)
        self.table_name = 'filter'
        self.implicit_rules = []
        self.ipsetdir = '/tmp/ipsetdir'

    def process_time(self, rule):
        """process des paramètres de temps
        """
        if self.directive.time is not None:
            time_params = ParameterFactory.time_from_directive(self.directive)
            rule.add_parameter(time_params)
        return rule

    def process_log(self, chain_name, service_param, param_list, match_params=None):
        """Transforme une directive LOG en règles iptables
        """
        # définition du libellé de log
        zone1 = self.directive.get_source_zone()
        zone2 = self.directive.get_dest_zone()
        libelle = "era %s-%s" % (zone1.name[:3], zone2.name[:3])
        param = LogParameter(libelle)
        target = param
        # On crée la règle avec la bonne chaine
        # on met toujours 'filter pour les règles de log' !!
        rule = TargettedRule(target, chain_name, 'filter')
        # mstate
        if library_store.get_version() > 1.0:
            if match_params is not None:
                """ajout de la fonctionnalité de -m state (c'est soumis au versionning)"""
                rule.add_parameter(match_params)
        # On ajoute tous les paramètres à la règle
        rule.add_parameter(service_param)
        for param in param_list:
            rule.add_parameter(param)
        # on process les paramètre du module time
        self.process_time(rule)
        return rule

    def process_ipsets(self):
        rules = []
        if len(self.directive.exceptions) != 0:
            zone1 = self.directive.get_source_zone()
            zone2 = self.directive.get_dest_zone()
            #chain_name = '%s-%s' % (zone1.name[:3], zone2.name[:3])
            current_ipsets_name = str(zone1.name[0:3]) + '-' + \
                    str(zone2.name[0:3]) + '-' + \
                str(self.directive.priority)
            self.ipsets_name_src = 'bastion-' + current_ipsets_name + '-src'
            self.ipsets_name_dst = 'bastion-' + current_ipsets_name + '-dst'
            IPSETS_NAMES.append(self.ipsets_name_src)
            IPSETS_NAMES.append(self.ipsets_name_dst)
            self.ipsets_filename_src = self.ipsets_name_src + '.sh'
            self.ipsets_filename_dst = self.ipsets_name_dst + '.sh'
            # reverse in comparison to the default politics
            if self.default_policy:
                target = 'accept'
            else:
                target = 'drop'
            # retrieve all params to perfectly match the rule
            basic_rules = BasicDirectiveProcessor(self.directive).process(ipset=False)  # default!
            for rule in basic_rules:
                rule.set_target(target)
                params = ParameterFactory.ipset_from_directive(self.directive,
                        self.ipsets_name_src, src=True)
                rule.add_parameters(params)
                rules.append(rule)
            basic_rules = BasicDirectiveProcessor(self.directive).process(ipset=False) #default!
            for rule in basic_rules:
                rule.set_target(target)
                params = ParameterFactory.ipset_from_directive(self.directive,
                        self.ipsets_name_dst, src=False)
                rule.add_parameters(params)
                rules.append(rule)

            #for extr1, extr2 in gen_couples(self.directive.src_list,
            #                                self.directive.dest_list):
                # On récupère l'ensemble des paramètres IPs / Interfaces
                #param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                #                                                       self.directive.src_inv,
                #                                                       self.directive.dest_inv)
                #std_params = list(param_set)
                ## src rule
                #rule = TargettedRule(target=target, chain_name=chain_name)
                #params = ParameterFactory.ipset_from_directive(self.directive,
                #        self.ipsets_name_src, src=True)
                #for param in std_params:
                #    rule.add_parameters(param)
                #for param in params:
                #    rule.add_parameter(param)
                #rule.add_parameter(ProtocolParameter(self.directive.service.port_list)
                #rules.append(rule)
                ## dst rule
                #rule = TargettedRule(target=target, chain_name=chain_name)
                #params = ParameterFactory.ipset_from_directive(self.directive,
                #        self.ipsets_name_dst, src=False)
                #for param in std_params:
                #    rule.add_parameters(param)
                #for param in params:
                #    rule.add_parameter(param)
                #rules.append(rule)
            # now creates the ipset table
            self.add_ipsets_rules()
        return rules

    def add_ipsets_rules(self):
        ipset_create = "ipset create "
        ipset_add = "ipset add "
        ipsets_src_rules = []
        ipsets_dst_rules = []
        ipset_create_src = ipset_create + self.ipsets_name_src + ' hash:net \n'
        ipset_create_dst = ipset_create + self.ipsets_name_dst + ' hash:net \n'
        ipsets_src_rules.append(ipset_create_src)
        ipsets_dst_rules.append(ipset_create_dst)
        # add ipsets rules
        """
        TEMPLATE_IPSETS, instruction for use :
        - {eolvar} is like %%ma_variable
        - {ipset_add} is "ipset -A "
        """
        TEMPLATE_IPSETS = """{ipset_add} {ipsets_name} {eolvar}"""
        #TEMPLATE_IPSETS2 = """{ipset_add} {ipsets_name} {eolvar_ip}/{eolvar_network}"""
        for exception in self.directive.exceptions:
            exc = self._filter_exception(exception)
            if exc['src']:
                if exc['is_active'] == 'eolvar':
                    eolvar = exc['ipname']
                    ipsets_add_src = TEMPLATE_IPSETS.format(eolvar=eolvar,
                            ipset_add=ipset_add, ipsets_name=self.ipsets_name_src)
                    multi = Multivar([ipsets_add_src])
                    ipsets_src_rules += list(multi.process())
                elif exc['is_active'] == 'name':
                    ipsets_add_src = ipset_add + self.ipsets_name_src + ' [' + exc['ipname'] + ']\n'
                    ipsets_src_rules.append(ipsets_add_src)
                else:
                    ipsets_add_src = ipset_add + self.ipsets_name_src + ' ' + exc['ipname'] + '\n'
                    ipsets_src_rules.append(ipsets_add_src)
            else:
                if exc['is_active'] == 'eolvar':
                    eolvar = exc['ipname']
                    ipsets_add_dst = TEMPLATE_IPSETS.format(eolvar=eolvar,
                            ipset_add=ipset_add, ipsets_name=self.ipsets_name_dst)
                    multi = Multivar([ipsets_add_dst])
                    ipsets_dst_rules += list(multi.process())
                elif exc['is_active'] == 'name':
                    ipsets_add_dst = ipset_add + self.ipsets_name_dst + ' [' + exc['ipname'] + ']\n'
                    ipsets_dst_rules.append(ipsets_add_dst)
                else:
                    ipsets_add_dst = ipset_add + self.ipsets_name_dst + ' ' + exc['ipname'] + '\n'
                    ipsets_dst_rules.append(ipsets_add_dst)
        # Ajout si nécessaire du test IP/Domain sur l'entrée ipset src dans le template
        tplist = []
        for ipset_rule in ipsets_src_rules:
            tpline = ipset_rule
            if "ipset add" in ipset_rule:
                ipset_entry = tpline.split(" ")[-1].split("\n")[0]
                if "/" not in ipset_entry and not is_ip(ipset_entry) and "[" not in ipset_entry:
                    tpline = """%if %%is_ip({ipset_entry})
{ip_line}
%else
{domain_line}
%end if""".format(ipset_entry=ipset_entry, ip_line=tpline, domain_line=tpline.replace(ipset_entry, "[" + ipset_entry + "]"))
            tplist.append(tpline)
        ipsets_src_rules = tplist
        # Ajout si nécessaire du test IP/Domain sur l'entrée ipset dst dans le template
        tplist = []
        for ipset_rule in ipsets_dst_rules:
            tpline = ipset_rule
            if "ipset add" in ipset_rule:
                ipset_entry = tpline.split(" ")[-1].split("\n")[0]
                if "/" not in ipset_entry and not is_ip(ipset_entry) and "[" not in ipset_entry:
                    tpline = ipset_rule
                    tpline = """%if %%is_ip({ipset_entry})
{ip_line}
%else
{domain_line}
%end if""".format(ipset_entry=ipset_entry, ip_line=tpline, domain_line=tpline.replace(ipset_entry, "[" + ipset_entry + "]"))
            tplist.append(tpline)
        ipsets_dst_rules = tplist
        # write ipsets rules
        self._write_ipsets_rules(self.ipsets_filename_src, ipsets_src_rules)
        self._write_ipsets_rules(self.ipsets_filename_dst, ipsets_dst_rules)

    def _write_ipsets_rules(self, filename, ipsets_rules):
        filepath = join(self.ipsetdir, filename)
        fh = open(filepath, 'a')
        fh.write("\n".join(ipsets_rules))
        fh.close()

    def _is_list(self, directive):
        """
        Repère si la directive est composée d'une liste d'extemités
        ou d'un groupe de services.
        """
        if directive.src_inv:
            # la source est inversée
            if len(directive.src_list) >= 1 :
                # mais la source est une liste
                return self.invert_src(directive)

        if directive.dest_inv:
            # la destination est à inverser
            if len(directive.dest_list) >= 1:
                # mais la destination est une liste
                return self.invert_dest(directive)

        if directive.serv_inv:
            # le service est à inverser
            if directive.is_group:
                # mais le service est un groupe de services
                return self.invert_serv(directive)
        else:
            # tout va bien
            return directive

    def create_rule(self, target, chain_name, table, match_params, service_param, param_list):
        # On crée la règle avec la bonne chaine
        rule = TargettedRule(target, chain_name, table)
        # mstate
        if library_store.get_version() > 1.0:
            """ajout de la fonctionnalité de -m state (c'est soumis au versionning)"""
            if match_params != None:
                rule.add_parameter(match_params)
        # On ajoute tous les paramètres à la règle
        rule.add_parameter(service_param)
        for param in param_list:
            rule.add_parameter(param)
        # on process les paramètre du module time
        self.process_time(rule)
        return rule

    def get_implicit_rules(self):
        return self.implicit_rules

    def _filter_exception(self, exc):
        ipname = None
        if  str(exc['ip']).strip() != '':
            ipname = exc['ip']
            exc['is_active'] = 'ip'
        elif  str(exc['name']).strip() != '':
            ipname = exc['name']
            exc['is_active'] = 'name'
        elif str(exc['eolvar']).strip() != '':
            ipname = exc['eolvar']
            exc['is_active'] = 'eolvar'
        else:
            raise Exception("ipset extremity has no valid name or ip")
        exc['ipname'] = ipname
        return exc

class BasicDirectiveProcessor(DirectiveProcessor):
    """Processeur d'objet 'Directive' (de type Basic)
    """
    def __init__(self, directive, default_policy=False):
        self.default_policy = default_policy
        self.container_rules = defaultdict(list)
        super(BasicDirectiveProcessor, self).__init__(directive)

    def _inverted(self):
        # traiter self.directive
        pass

    def _basic_inversion(self):
        """inversion de l'action
        Inversion une authorisation en interdiction et réciproquement
        """
        if action == ACTION_ALLOW:
            return ACTION_DENY
        if action == ACTION_DENY:
            return ACTION_ALLOW
        # normalement ça ne passe pas par là
        # puisque c'est une directive basique...
        else:
            return action

    def invert_src(self, directive):
        """Inversion de la source
        """
        # inversion de l'action
        # cela permet de traiter la liste des extremites
        directive.action = self._invert_action(directive.action)
        directive.src_inv = False
        return directive

    def invert_dest(self, directive):
        """Inversion de la destination
        """
        return directive

    def invert_serv(self, directive):
        """Inversion du service
        """
        return directive

    def process(self, ipset=True):
        """Transforme une directive basique en règles iptables
        """
        self.authenticated_rules = []
        # pile de regles standard
        rules = []
        if ipset:
            rules.extend(self.process_ipsets())
        # pile de regles authentifiees par nufw
        target = ParameterFactory.target_from_directive(self.directive)

        # On parcourt les couples d'extrémités utilisées dans la directive
        for extr1, extr2 in gen_couples(self.directive.src_list,
                                        self.directive.dest_list):
            # On récupère l'ensemble des paramètres IPs / Interfaces
            param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                                                                   self.directive.src_inv,
                                                                   self.directive.dest_inv, True)
            chain_name = '%s-%s' % (extr1.zone.name[:3], extr2.zone.name[:3])
            for param_list in param_set:
                # on initialise la liste des paramètres de service
                service_params = ParameterFactory.service_from_directive(self.directive)
                match_params = ParameterFactory.match_state(['NEW'])
                for service_param in service_params:
                    # ajout des règles de log
                    if self.directive.is_logged():
                        logged_rule = self.process_log(chain_name, service_param, param_list, match_params)
                        rules.append(logged_rule)
                    # ajout des règles de marquage
                    if self.directive.is_marked():
                        operator, mark = self.directive.mark
                        rule = self.create_rule(Mark(mark, operator), "marquage", "mangle", match_params, service_param, param_list)
                        rules.append(rule)
                    if extr2.extr_type != 'conteneur':
                        rule = self.create_rule(target, chain_name, self.table_name, match_params, service_param, param_list)
                        # si une directive est authentifiee (nfqueue), on la déplace dans authenticated_rules
                        if self.directive.is_authenticated() == True:
                            self.authenticated_rules.append(rule)
                        rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
                        rules.append(rule)
                    else:
                        # ajout d'une règle en ACCEPT spécifique au conteneur
                        # si extr2.interface == 'containers', on n'entre pas ici
                        # et extr2.interface est le nom du conteneur en principe
                        # construction d'une regle de type :
                        # -A eth0-root -s 192.168.1.0/255.255.255.0 -p tcp --syn --dport 3128 -j ACCEPT
                        container_chain_name = extr2.interface + '-root'
                        container_param_list = []
                        for param in param_list:
                            if isinstance(param, InterfaceParameter):
                                param = deepcopy(param)
                                param.p_value = extr2.interface
                            container_param_list.append(param)
                        container_rule = self.create_rule(Target('ACCEPT'), container_chain_name, self.table_name, None,
                                                        service_param, container_param_list)
                        self.container_rules[extr2.container_name].append(container_rule)
        return rules

    def get_implicit_rules(self):
        """
            renvoie la pile des règles implicites (règles automatiques)
        """
        # pas de regles implicites dans le cas d'une directive basique pour l'instant
        return []

    def get_authenticated_rules(self):
        """
            Renvoie les règles authentifiées, cette pile est mise en dernier dans le flux
        """
        return self.authenticated_rules

    def get_container_rules(self):
        "retrieves the container oriented rules"
        # container_rules est de la forme  dict(bdd=['rule', 'rule'])
        return self.container_rules


class DNATDirectiveProcessor(DirectiveProcessor):
    """Processeur d'objet 'Directive' (de type DNAT)
    """

    def __init__(self, directive, is_container=False):
        """is_container: ajoute une règle d'autorisation sur le forward
                         si c'est un conteneur local
        """
        super(DNATDirectiveProcessor, self).__init__(directive)
        self.is_container = is_container

    def build_container_rule(self, extr1, extr2, service):
        #class PseudoRule: pass
        #rule = PseudoRule()
        rule = TargettedRule(target='ACCEPT', chain_name='FORWARD')
        #rule.target = '-j ACCEPT'
        #rule.chain_name = 'FORWARD'
        #rule.table_name = 'filter'
        srcint = InterfaceParameter(extr1.zone.interface)
        destint = InterfaceParameter('br0', out=True)
        # XXX duplicate from backed/utils.gen_ip_netmask()
        if extr1.name == 'exterieur' or extr1.all_zone:
            srcip = IPParameter(src=True, address='0', mask='0')
        else:
            srcip = IPParameter(src=True, address=extr1.ip_list[0], mask=extr1.netmask)
        destip = IPParameter(src=False, address=extr2.ip_list[0], mask=extr2.netmask)
        #protocol = ProtocolParameter(service.protocol)
        def service_generator():
            protocol = service.protocol.lower()
            if protocol not in ('udp', 'tcp', 'icmp', 'esp', 'tout'):
                raise ValueError('protocol should be in (udp, tcp, esp, tout or icmp)' \
                                 ', found %s' % protocol)
            if protocol == 'udp':
                yield UDPProtocol(dport = service.port_list)
            elif protocol == 'tcp':
                yield TCPProtocol(dport = service.port_list)
            elif protocol == 'icmp':
                yield ICMPProtocol(service.name)
            elif protocol == 'esp':
                yield ESPProtocol()
            else:
                # Protocole 'TOUT' <=> pas besoin de spécifier de protocole
                yield None
        ports = list(service_generator())
        # FIXME
        #port = service.port_list[0]
        #flag = "--tcp-flags SYN,RST,ACK SYN"
        #port = '--dport ' + service.port_list[0]
        #rule.params = [srcint, destint, srcip, destip]
        #rule.params.extend(ports)
        rule.add_parameters([srcint, destint, srcip, destip])
        rule.add_parameters(ports)
        rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
        return rule

    def process(self):
        """Transforme une directive NAT en règles iptables
        """
        rules = []
        if self.directive.accept == 0:
            target = ParameterFactory.target_from_directive(self.directive)
        else:
            target = Target('ACCEPT')

        # On parcourt les couples d'extrémités utilisées dans la directive
        for extr1, extr2 in gen_couples(self.directive.src_list,
                                        self.directive.dest_list):
            # On récupère l'ensemble des paramètres IPs / Interfaces
            param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                                                                   self.directive.src_inv,
                                                                   self.directive.dest_inv, True)
            main_chain = "PREROUTING"
            for param_list in param_set:
                for service in self.directive.service:
                    service_param = ParameterFactory.service2(service, self.directive.serv_inv)
                    # On crée la règle avec la bonne chaine
                    rule = TargettedRule(target, main_chain, 'nat')
                    # On ajoute tous les paramètres à la règle
                    rule.add_parameter(service_param)
                    in_int, out_int, src_param, dest_param = param_list
                    # PREROUTING => On enlève les '-o'
                    rule.add_parameters((in_int, src_param, dest_param))
                    # paramètres horaires
                    self.process_time(rule)
                    # On ajoute la règle aux autres
                    rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
                    rules.append(rule)
                    if self.directive.accept == 0:
                        ## Règle implicite : accept de la 1ère vers la 3ème extrêmité
                        extr3 = self.directive.nat_extr
                        if self.is_container:
                            rules.append(self.build_container_rule(extr1, extr3, service))
                        chain_name = '%s-%s' % (extr1.zone.name[:3],
                                                extr3.zone.name[:3])
                        # dest_param n'est pas bon (pas celui du comportement par défaut)
                        out_int = ''
                        if extr3.zone.interface != 'lo':
                            out_int = ParameterFactory.interface(extr3.zone.interface, True)
                        dest_param = DestIPParameter(extr3.ip_list[0])
                        implicit_rule = TargettedRule('accept', chain_name)

                        if isinstance(service_param, ICMPProtocol):
                            service_param2 = ParameterFactory.service('icmp',
                                service.name)
                        else:
                            ports = [self.directive.nat_port]
                            if ports == [u'0']:
                                ports = service.port_list

                            service_param_nat = Service('nat_port',
                                                        service.protocol,
                                                        ports,
                                                        None, 'nat_port')
                            service_param2 = ParameterFactory.service2(service_param_nat, self.directive.serv_inv)

                        # ajout des règles de log
                        if self.directive.is_logged():
                            log_rule = self.process_log(chain_name, service_param2, (in_int, out_int, src_param, dest_param))
                            rules.append(log_rule)

                        implicit_rule.add_parameter(service_param2)
                        # service_param == None -> service == u'tous'
                        if service_param is not None:
                            # ne pas rediriger un service particulier
                            # sur tous les ports (ne veut rien dire...)
                            # le ICMPProtocol par exemple n'a pas de port
                            if hasattr(service_param, 'dport'):
                                if service_param.dport == '0' and \
                                    service.name != u'tous':
                                    service_param.dport = None
                        # dest_param = DestIPParameter(self.directive.nat_extr.ip_list[0])
                        for param in (in_int, out_int, src_param, dest_param):
                            implicit_rule.add_parameter(param)
                        # paramètres horaires
                        self.process_time(implicit_rule)
                        self.implicit_rules.append(implicit_rule)

                        # ajout des règles de marquage
                        if self.directive.is_marked():
                            operator, mark = self.directive.mark
                            rule = self.create_rule(Mark(mark, operator), "marquage", "mangle", None, service_param, (in_int, out_int, src_param, dest_param))
                            rules.append(rule)
                        rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
        return rules

    def get_implicit_rules(self):
        return self.implicit_rules


class SNATDirectiveProcessor(DirectiveProcessor):
    """Processeur d'objet 'Directive' (de type SNAT)
    """

    def __init__(self, directive):
       super(SNATDirectiveProcessor, self).__init__(directive)

    def process(self):
        """Transforme une directive NAT en règles iptables
        """
        rules = []
        if self.directive.accept == 0:
            target = ParameterFactory.target_from_directive(self.directive)
        else:
            target = Target('ACCEPT')
        # On parcourt les couples d'extrémités utilisées dans la directive
        for extr1, extr2 in gen_couples(self.directive.src_list,
                                        self.directive.dest_list):
            # On récupère l'ensemble des paramètres IPs / Interfaces
            param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                                                                   self.directive.src_inv,
                                                                   self.directive.dest_inv)
            main_chain = "POSTROUTING"
            for param_list in param_set:
                service_params = ParameterFactory.service_from_directive(self.directive)
                for service_param in service_params:
                    in_int, out_int, src_param, dest_param = param_list
                    # On crée la règle avec la bonne chaine
                    rule = TargettedRule(target, main_chain, 'nat')
                    # On ajoute tous les paramètres à la règle
                    rule.add_parameter(service_param)
                    # POSTROUTING => On enlève les '-i'
                    rule.add_parameters((out_int, src_param, dest_param))
                    # paramètres horaires
                    self.process_time(rule)
                    # On ajoute la règle aux autres
                    rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
                    rules.append(rule)
                    ## Règle implicite : accept de la 1ère vers la 3ème extrêmité
                    extr3 = self.directive.nat_extr
                    chain_name = '%s-%s' % (extr1.zone.name[:3],
                                            extr2.zone.name[:3])
                    # ajout des règles de log
                    if self.directive.is_logged():
                        logged_rule = self.process_log(chain_name, service_param, param_list)
                        rules.append(logged_rule)
                    implicit_rule = TargettedRule('accept', chain_name)
                    #params = ParameterFactory.parameters_for_zones(extr1.zone,
                    #                                               extr2.zone)
                    implicit_rule.add_parameter(service_param)
                    # implicit_rule.add_parameter((out_int, src_param, dest_param))
                    for param in param_list:
                        implicit_rule.add_parameter(param)
                    # paramètres horaires
                    self.process_time(implicit_rule)
                    # ce n'est pas nécessaire si la directive est montante...
                    # if self.directive.is_montante():
                    # les regles implicites seront ajoutee ulterieurement
                    # (a la fin du flux)
                    self.implicit_rules.append(implicit_rule)
                    # ajout des règles de marquage
                    if self.directive.is_marked():
                        operator, mark = self.directive.mark
                        rule = self.create_rule(Mark(mark, operator), "marquage", "mangle", None, service_param, param_list)
                        rules.append(rule)
        return rules

    def get_implicit_rules(self):
        return self.implicit_rules


class RedirectDirectiveProcessor(BasicDirectiveProcessor):
    """Processeur pour cible REDIRECT
    Le comportement est le-même que celui par défaut, mais la table
    est 'nat' et pas 'filter
    """
    def __init__(self, directive):
        """
         - directive : la directive ERA
        """
        super(RedirectDirectiveProcessor, self).__init__(directive)
        self.table_name = 'nat'
        self.implicit_rules = []
        self.authenticated_rules = []

    def process(self):
        rules = []
        target = ParameterFactory.target_from_directive(self.directive)
        # on récupère le port de redirection pour la règle implicite
        redirect_port = target.dport[0]
        # On parcourt les couples d'extrémités utilisées dans la directive
        for extr1, extr2 in gen_couples(self.directive.src_list,
                                        self.directive.dest_list):
            # On récupère l'ensemble des paramètres IPs / Interfaces
            param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                                                                   self.directive.src_inv,
                                                                   self.directive.dest_inv,
                                                                   True)
            chain_name = 'PREROUTING'
            trigram_zones = '%s-%s' % (extr1.zone.name[:3], extr2.zone.name[:3])
            for in_int, out_int, src_param, dest_param in param_set:
                service_params = ParameterFactory.service_from_directive(self.directive)
                for service_param in service_params :
                    # ajout des regles ipsets:
                    if len(self.directive.exceptions) != 0:
                        zone1 = self.directive.get_source_zone()
                        zone2 = self.directive.get_dest_zone()
                        current_ipsets_name = str(zone1.name[0:3]) + '-' + \
                                              str(zone2.name[0:3]) + '-' + \
                                              str(self.directive.priority)
                        self.ipsets_name_src = 'bastion-' + current_ipsets_name + '-src'
                        self.ipsets_name_dst = 'bastion-' + current_ipsets_name + '-dst'
                        IPSETS_NAMES.append(self.ipsets_name_src)
                        IPSETS_NAMES.append(self.ipsets_name_dst)
                        self.ipsets_filename_src = self.ipsets_name_src + '.sh'
                        self.ipsets_filename_dst = self.ipsets_name_dst + '.sh'
                        # src
                        ipsetsrule = TargettedRule('RETURN', chain_name, self.table_name)
                        ipsetsrule.add_parameter(service_param)
                        ipsetsrule.add_parameters([in_int, src_param, dest_param])
                        ipsets_params = ParameterFactory.ipset_from_directive(self.directive,
                                self.ipsets_name_src, src=True)
                        ipsetsrule.add_parameters(ipsets_params)
                        rules.append(ipsetsrule)
                        # dst
                        dstipsetsrule = TargettedRule('RETURN', chain_name, self.table_name)
                        dstipsetsrule.add_parameter(service_param)
                        dstipsetsrule.add_parameters([in_int, src_param, dest_param])
                        ipsets_params = ParameterFactory.ipset_from_directive(self.directive,
                                self.ipsets_name_dst, src=False)
                        for param in ipsets_params:
                            dstipsetsrule.add_parameter(param)
                        rules.append(dstipsetsrule)
                        # ajout des regles implicites de redirection pour ipsets
                        implicit_ipsetsrule = TargettedRule('ACCEPT',
                                                trigram_zones, 'filter')
                        implicit_ipsetsrule.add_parameters([in_int, out_int,
                                                src_param, dest_param])
                        ipsets_params = ParameterFactory.ipset_from_directive(
                                self.directive, self.ipsets_name_src, src=True)
                        implicit_ipsetsrule.add_parameters(ipsets_params)
                        rules.append(implicit_ipsetsrule)
                        implicit_ipsetsrule = TargettedRule('ACCEPT',
                                                trigram_zones, 'filter')
                        implicit_ipsetsrule.add_parameters([in_int, out_int,
                                                src_param, dest_param])
                        ipsets_params = ParameterFactory.ipset_from_directive(
                                self.directive, self.ipsets_name_dst, src=False)
                        implicit_ipsetsrule.add_parameters(ipsets_params)
                        rules.append(implicit_ipsetsrule)

                    # On crée la règle de REDIRECT avec la bonne chaine
                    rule = TargettedRule(target, chain_name, self.table_name)
                    # On ajoute tous les paramètres à la règle
                    rule.add_parameter(service_param)
                    rule.add_parameters([in_int, src_param, dest_param])

                    if library_store.get_version() > 1.0:
                        # la règle n'est pas nécessaire pour les flux descendant
                        # on ajoute la règle implicite uniquement pour des version updatées des modèles
                        # ajout des règles de log
                        implicit_chain = '%s-bas' % extr1.zone.name[:3]
                        dport_param = TCPProtocol(dport = redirect_port)
                        # on ne loggue pas si la directive est authentifiée (c'est NuFW qui s'en charge)
                        if self.directive.is_logged() and not self.directive.is_authenticated():
                            log_rule = self.process_log(implicit_chain, dport_param, param_set)
                            rules.append(log_rule)
                        # c'est une règle de type input qui est générée
                        implicit_rule = TargettedRule('accept', implicit_chain)
                        # implicit_rule.add_parameter((in_int, src_param, dest_param))
                        implicit_rule.add_parameter(in_int)
                        # service_param c'est le --dport de la règle normale
                        # implicit_rule.add_parameter(service_param)
                        # on ajoute le bon --dport
                        implicit_rule.add_parameter(dport_param)
                        for param in param_set:
                            implicit_rule.add_parameter(param)
                        # paramètres horaires
                        self.process_time(implicit_rule)
                        # paramètres horaires
                        self.process_time(rule)
                        rules.append(implicit_rule)

                    rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
                    rules.append(rule)
                    # ajout des règles de marquage
                    if self.directive.is_marked():
                        operator, mark = self.directive.mark
                        rule = self.create_rule(Mark(mark, operator), "marquage", "mangle", None, dport_param, param_set)
                        rules.append(rule)
                    # nfqueue authenticated pour la redirection
                    if self.directive.is_authenticated():
                        # FIXME construire la regle de maniere plus normalisee
                        queue_rule = deepcopy(rule)
                        queue_target = Target("NFQUEUE")
                        queue_rule.target = queue_target
                        queue_rule.table_name = "mangle"
                        self.authenticated_rules.append(queue_rule)
                        # ajout d'un parametre de marquage sur les regle de redirection
                        match = MatchParameter('mark')
                        rule.add_parameter(match)
                        # ecrit la marque dans le fichier markgroup.conf (pour nufw)
                        global iterator
                        if not group_is_in_markfile(self.directive.user_group.id):
                            append_mark(self.directive.user_group.id, iterator)
                            iterator +=1
                        mark_param = MarkParameter("0x%s"%get_mark(self.directive.user_group.id))
                        rule.add_parameter(mark_param)
            if len(self.directive.exceptions) != 0:
              self.add_ipsets_rules()
        return rules

    def get_implicit_rules(self):
        return self.implicit_rules

    def get_authenticated_rules(self):
        """
            Renvoie les règles authentifiées, cette pile est mise en dernier dans le flux
        """
        return self.authenticated_rules


class ForwardDirectiveProcessor(DirectiveProcessor):
    """FORWARD directive
    """
    def __init__(self, directive):
        super(ForwardDirectiveProcessor, self).__init__(directive)
        self.directive = directive
        self.implicit_rules = []
        self.authenticated_rules = []

    def process(self):
        # On parcourt les couples d'extrémités utilisées dans la directive
        rules = []
        # FORWARD generated rule 1
        # On parcourt les couples d'extrémités utilisées dans la directive
        for extr1, extr2 in gen_couples(self.directive.src_list,
                                        self.directive.dest_list):
            # On récupère l'ensemble des paramètres IPs / Interfaces
            param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                                                                   self.directive.src_inv,
                                                                   self.directive.dest_inv)
            chain_name = '%s-%s' % (extr1.zone.name[:3], extr2.zone.name[:3])
            for param_list in param_set:
                service_params = ParameterFactory.service_from_directive(self.directive)
                for service_param in service_params:
                    in_int, out_int, src_param, dest_param = param_list
                    # On crée la règle avec la bonne chaine
                    rule = TargettedRule(target='ACCEPT')
                    param = InsertParameter('FORWARD')
                    rule.add_parameter(param)
                    # On ajoute tous les paramètres à la règle
                    rule.add_parameter(service_param)
                    # POSTROUTING => On enlève les '-i'
                    #if self.directive.accept == 0:
                    #    interface_name = self.directive.nat_extr.zone.interface
                    #    interface_redirect = InterfaceParameter(interface_name, out=False)
                    rule.add_parameters((out_int, src_param, dest_param))
                    # paramètres horaires
                    self.process_time(rule)
                    # On ajoute la règle aux autres
                    rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
                    rules.append(rule)
                    if self.directive.is_logged():
                        logged_rule = self.process_log(chain_name, service_param, param_list)
                        rules.append(logged_rule)
        # FORWARD generated rules 2
        # On parcourt les couples d'extrémités utilisées dans la directive
        for extr1, extr2 in gen_couples(self.directive.src_list,
                                        self.directive.dest_list):
            # On récupère l'ensemble des paramètres IPs / Interfaces
            param_set = ParameterFactory.parameters_for_extremites(extr1, extr2,
                                                                   self.directive.src_inv,
                                                                   self.directive.dest_inv)
            for param_list in param_set:
                service_params = ParameterFactory.service_from_directive(self.directive)
                for service_param in service_params:
                    in_int, out_int, src_param, dest_param = param_list
                    # On crée la règle avec la bonne chaine
                    rule = TargettedRule(target='ACCEPT')
                    param = InsertParameter('FORWARD')
                    rule.add_parameter(param)
                    # On ajoute tous les paramètres à la règle
                    rule.add_parameter(service_param)
                    # POSTROUTING => On enlève les '-i'
                    #interface_name = self.directive.nat_extr.zone.interface
                    #interface_redirect = InterfaceParameter(interface_name, out=False)
                    #rule.add_parameters((interface_redirect, out_int, src_param, dest_param))
                    rule.add_parameters((out_int, src_param, dest_param))
                    match = MatchState(['RELATED', 'ESTABLISHED'])
                    rule.add_parameter(match)
                    # paramètres horaires
                    self.process_time(rule)
                    # On ajoute la règle aux autres
                    rule.add_parameters(ParameterFactory.ipsec_from_directive(self.directive))
                    rules.append(rule)
        return rules

    def get_implicit_rules(self):
        return self.implicit_rules

    def get_authenticated_rules(self):
        """
            Renvoie les règles authentifiées, cette pile est mise en dernier dans le flux
        """
        return self.authenticated_rules

def get_processor(directive, default_policy=True):
    """Fabrique de processeurs
    L{era.tests.test_directive_types.test_processor_factory}
    """
    # si mode conteneur : si dest du flux bastion :
    # si l'extrémité n'est pas bastion : directive.action == 8 + interface = br0
    is_container = False
    if directive.get_dest_zone().name == 'bastion':
        dest_type = None
        if VIRTDISABLED == False:
            for extr in directive.dest_list:
                if extr.extr_type == 'normal':
                    if dest_type not in [None, 'normal']:
                        raise Exception('Directive vers bastion et conteneur impossible')
                    dest_type = 'normal'
                elif extr.extr_type == 'conteneur':
                    if dest_type not in [None, 'for_to_dnat']:
                        raise Exception('Directive vers bastion et conteneur impossible')
                    if extr.interface == "containers":
                        dest_type = 'for_to_dnat'
                    else:
                        # le traitement est géré au niveau de la directive
                        dest_type = 'normal'
                else:
                    pass
                    # gestion des ip aliases ayant comme destination le bastion
                    # peut etre un flag pour differencier le mode conteneur et non conteneur
                    # dans la gestion des aliases. pour l'instant le comportement est identique
                    dest_type = 'alias'
            if dest_type == 'for_to_dnat':
                directive = deepcopy(directive)
                directive.action = 8
                if len(directive.dest_list) != 1:
                    raise ValueError("Seulement une extrémité pour une directive d'INPUT d'un conteneur")
                extr = directive.dest_list[0]
                #zone = deepcopy(extr.zone)
                directive.nat_extr = deepcopy(extr)
                if type(directive.service) != Service:
                    raise ValueError('Pas de groupe de service pour une  directive vers un conteneur pour {0}'.format(directive.service.name))
                if len (directive.service.port_list) != 1:
                    raise ValueError('Pas de liste de service pour une directive vers un conteneur')
                directive.nat_port = directive.service.port_list[0]
                extr.zone.interface = 'br0'
                zoneip = extr.zone.ip
                extr.subnet = extr.zone.network
                extr.netmask = extr.zone.netmask
                ret = []
                for ip in extr.ip_list:
                    ret.append(zoneip)
                extr.ip_list = ret
                extr.name = u'bastion'
                is_container = True
        elif VIRTDISABLED == True:
            if len(directive.dest_list) != 1:
                raise Exception('Une directive vers la zone bastion ne peut avoir qu\'une et une seule extrémité de destination')
            extr = directive.dest_list[0]
            if extr.name != 'bastion':
                if len(extr.ip_list) == 1:
                    if extr.extr_type == 'normal':
                        pass
                        #raise Exception('Une extrémité de type normal n\'est pas possible dans la zone bastion')
                    elif extr.extr_type == 'conteneur':
                        directive.dest_list[0] = library_store.extremites['bastion']
                    elif extr.extr_type == 'alias':
                        pass
                    else:
                        raise Exception('Extrémité de type {0} non géré'.format(extr.extr_type))
                else:
                    #une directive vers bastion ne peut pas etre redirigée
                    #dans plusieurs conteneurs
                    #par exemple on ne peut pas faire "tout ce qui arrive
                    #sur le port 80 est redirigé dans le conteneur web ET
                    #proxy"
                    raise Exception('Une directive vers la zone bastion ne peut avoir qu\'une et une seule extrémité de destination')
    # _______________________________________________________________________
    # directive factory
    if directive.action in (1, 2): # ACTION_DENY, ACTION_ALLOW
        return BasicDirectiveProcessor(directive, default_policy=default_policy)
    if directive.action == 4: # ACTION_REDIRECT
        return RedirectDirectiveProcessor(directive)
    if directive.action == 8: # ACTION_DNAT
        return DNATDirectiveProcessor(directive, is_container)
    if directive.action == 16: # ACTION_MASK
        return SNATDirectiveProcessor(directive)
    if directive.action == 32: # ACTION_FORWARD
        return ForwardDirectiveProcessor(directive)

