# -*- coding: UTF-8 -*-
###########################################################################
# 
# Eole NG  
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  http://www.cecill.info/licences/Licence_CeCILL_V2-fr.html
# eole@ac-dijon.fr 
#  
###########################################################################

"""
Module concernant les plages horaires
"""

import gtk, gtk.glade
import gobject
import re, time
from era.noyau.pool import library_store
from era.noyau.fwobjects import Range
from ihm_utils import *


class TimeRangeDialog:
    """Gère les plages horaires
    """


    def __init__(self, glade_file):
        """
        glade : l'objet XML glade
        """
        # traduction des jours
        self.days = {'Mon': _("Monday"),
                     'Tue': _("Tuesday"),
                     'Wed': _("Wednesday"),
                     'Thu': _("Thursday"),
                     'Fri': _("Friday"),
                     'Sat': _("Saturday"),
                     'Sun': _("Sunday")
                    }
                    
        self.glade_file = glade_file
        self.glade = gtk.glade.XML(self.glade_file,"ranges_dialog", "editeur")
        self.dlg = self.glade.get_widget("ranges_dialog")

        self.tree_view = self.glade.get_widget('range_treeview')

        handlers = {'on_ranges_dialog_response' : self.close,
                    'on_range_treeview_button_press_event':self.get_selected_clicked,
                    'on_add_range_button_clicked':self.add_timerange,
                    'on_button_edit_range_clicked':self.get_selected,
                    'on_button_del_range_clicked':self.del_timerange,
                   }
        self.glade.signal_autoconnect(handlers)

        store = gtk.ListStore(gobject.TYPE_PYOBJECT, gobject.TYPE_STRING)

        renderer = gtk.CellRendererText()

        column = gtk.TreeViewColumn(_('range description'), renderer, text=1)
        column.set_expand(True)

        self.tree_view.set_model(store)
        self.tree_view.append_column(column)
        tree_sel = self.tree_view.get_selection()
        tree_sel.connect('changed', self.selected_range_changed)

        for timerange in library_store.ranges.keys():
            store.append([library_store.ranges[timerange], library_store.ranges[timerange].name])

    def close(self, button, code, *args):
        self.dlg.hide()

    def save_tags(self, model, path, current_iter, data):
        timerange = model.get_value(current_iter, 0)
        library_store.ranges[timerange['name']] = timerange

    def selected_range_changed(self, selection, *args):
        # récupération de la plage sélectionnée
        store, iter_sel = selection.get_selected()
        if iter_sel is not None:
            selected_range = store.get_value(iter_sel,0)
            if selected_range is not None:
                self.selected_range = selected_range
                # texte de description de la plage horaire
                txt = []
                if self.selected_range.weekdays != '':
                    # traduction des jours
                    selected_days = self.selected_range.weekdays.split(',')
                    translated_days = [self.days[day] for day in selected_days]
                    txt.append(_("weekdays :"))
                    txt.append("  %s" % ",".join(translated_days))
                if self.selected_range.timestart != '':
                    txt.append(_("hours :"))
                    txt.append(_("  start : ") + self.selected_range.timestart)
                    txt.append(_("  end : ") + self.selected_range.timestop)
                if self.selected_range.datestart != '':
                    txt.append(_("dates :"))
                    txt.append(_("  start : ") + self.selected_range.datestart)
                    txt.append(_("  end : ") + self.selected_range.datestop)
                if self.selected_range.used > 0:
                    txt.append(_("used in %s directive(s)") % self.selected_range.used)
                    self.glade.get_widget('button_del_range').set_sensitive(False)
                else:
                    if self.selected_range.is_inherited():
                        self.glade.get_widget('button_del_range').set_sensitive(False)
                    else:
                        self.glade.get_widget('button_del_range').set_sensitive(True)

                buffer = self.glade.get_widget('range_textview').get_buffer()
                buffer.set_text(u'\n\n'.join(txt))

    def get_selected_clicked(self,widget,event,*args):
        """récupère le double click dans le treeview
        """
        # édition si double click
        if event.type == gtk.gdk._2BUTTON_PRESS:
            self.get_selected(widget)
        
    def get_selected(self,widget,*args):
        """édite la plage sélectionnée
        """
        store, iter_sel = self.tree_view.get_selection().get_selected()

        if self.selected_range is not None:
            # édition de la plage
            dial_edit = RangeEditDialog(self.glade_file, self.tree_view.get_model(),iter_sel,self.selected_range)
            dial_edit.show_dialog()

    def add_timerange(self,*args):
        """édite la plage sélectionnée
        """
        iter_add  = self.tree_view.get_model().append()
        # édition de la plage
        dial_edit = RangeEditDialog(self.glade_file, self.tree_view.get_model(),iter_add)
        dial_edit.show_dialog()

    def del_timerange(self, *args):
        """supprime la plage horaire sélectionnée
        """
        selection = self.tree_view.get_selection()
        store, iter_sel  = selection.get_selected()

        if iter_sel is not None:
            timerange=store.get_value(iter_sel,0)
            if self.selected_range.used > 0:
                # plage utilisée par des directives 
                dial_error = create_error_dialog(_("object still used"),self.dlg)
                dial_error.show()
            else:
                # suppression
                store.remove(iter_sel)
                del(library_store.ranges[self.selected_range.name])
                buffer = self.glade.get_widget('range_textview').get_buffer()
                buffer.set_text('')

                
    def show_dialog(self):
        self.dlg.show_all()

class RangeEditDialog:
    """édite une plage horaire
    """

    def __init__(self, glade_file, model, iter_sel=None, range_edit=None):
        """
        glade : l'objet XML glade
        """

        self.days = {'Mon': _("Monday"),
                     'Tue': _("Tuesday"),
                     'Wed': _("Wednesday"),
                     'Thu': _("Thursday"),
                     'Fri': _("Friday"),
                     'Sat': _("Saturday"),
                     'Sun': _("Sunday")
                     }

        self.glade_file = glade_file
        self.glade = gtk.glade.XML(self.glade_file,"range_edit_dialog", "editeur")
        self.dlg = self.glade.get_widget("range_edit_dialog")
        self.glade.get_widget("vbox_edit").set_sensitive(True)
        if range_edit != None:
            if not range_edit.is_editable():
                self.glade.get_widget("vbox_edit").set_sensitive(False)
        
        handlers = {'on_range_edit_dialog_response' : self.close,
                    'on_range_edit_dialog_delete_event' : self.close,
                    'on_button_add_day_clicked':self.add_day,
                    'on_button_remove_day_clicked':self.remove_day,
                   }
        self.glade.signal_autoconnect(handlers)
        
        self.order_days=['Mon','Tue','Wed','Thu','Fri','Sat','Sun']
        self.model = model
        self.iter_sel = iter_sel
        self.combo_box = self.glade.get_widget('combo_days')
        # remplissage du combo
        self.combo_model = gtk.ListStore(gobject.TYPE_STRING,gobject.TYPE_STRING)
        cell = gtk.CellRendererText()
        self.combo_box.pack_start(cell, True)
        self.combo_box.add_attribute(cell, 'text', 1)
        self.combo_box.set_model(self.combo_model)
        # récupération des paramètres de la plage horaire
        self.set_date('entry_timestart', ':')
        self.set_date('entry_timestop', ':')
        self.set_date('entry_datestart', '--', separator='-')
        self.set_date('entry_datestop', '--', separator='-')
        self.selected_days = []
        self.timerange = range_edit
        self.update_data()

    def update_data(self):
        """remplit les champs du dialogue
        """
        self.combo_model.clear()
        for day in self.order_days:
            iter = self.combo_model.append()
            self.combo_model.set_value(iter, 0, day)
            self.combo_model.set_value(iter, 1, self.days[day])
        iter_root = self.combo_model.get_iter_root()
        self.combo_box.set_active_iter(iter_root)
        if self.timerange != None:
            if self.timerange.weekdays != "":
                self.selected_days = self.timerange.weekdays.split(',')
            self.set_date('entry_timestart', self.timerange.timestart)
            self.set_date('entry_timestop', self.timerange.timestop)
            self.set_date('entry_datestart', self.timerange.datestart, separator='-')
            self.set_date('entry_datestop', self.timerange.datestop, separator='-')
            translated_days = [self.days[day] for day in self.selected_days]
            buff = self.glade.get_widget('textview_days').get_buffer()
            buff.set_text(", ".join(translated_days))
            self.glade.get_widget('entry_name').set_text(self.timerange.name)

    def set_date(self, basename, data, separator=':'):
        date_parts=data.split(separator)
        if date_parts != ['']:
            for i in range(len(date_parts)):
                widget = '%s%d'%(basename, i+1)
                self.glade.get_widget(widget).set_text(date_parts[i])

    def get_date(self, basename, separator=':'):
        date_parts = []
        for i in range(1,4):
            widget = 'entry_%s%d' % (basename, i)
            widget = self.glade.get_widget(widget)
            if widget != None:
                date_parts.append(widget.get_text())
                
        res = separator.join(date_parts)
        if res == separator + separator or res == separator:
            res = ''
        return res

    def add_day(self, button):
        """ajoute le jour sélectionné
        """
        self.combo_box = self.glade.get_widget('combo_days')
        iter_active = self.combo_box.get_active_iter()
        if iter_active != None:
            add_day = self.combo_model.get_value(iter_active, 0)
            if add_day not in self.selected_days:
                # on reconstruit pour garder l'ordre
                self.selected_days.append(add_day)
                new_selected_days = []
                for ordered_day in self.order_days:
                    if ordered_day in self.selected_days:
                        new_selected_days.append(ordered_day)
                self.selected_days = new_selected_days
                translated_days = [self.days[day] for day in self.selected_days]
                buff = self.glade.get_widget('textview_days').get_buffer()
                buff.set_text(", ".join(translated_days))

    def remove_day(self, button):
        """supprime le jour sélectionné
        """
        self.combo_box = self.glade.get_widget('combo_days')
        iter_active = self.combo_box.get_active_iter()
        if iter_active != None:
            del_day = self.combo_model.get_value(iter_active, 0)
            if del_day in self.selected_days:
                self.selected_days.remove(del_day)
                translated_days = [self.days[day] for day in self.selected_days]
                buff = self.glade.get_widget('textview_days').get_buffer()
                buff.set_text(", ".join(translated_days))

    def show_dialog(self):
        self.dlg.show_all()
    
    def save_range(self):
        """sauvegarde les modifs
        """
        name = self.glade.get_widget("entry_name").get_text()
        timestart = self.get_date('timestart')
        timestop = self.get_date('timestop')
        datestart = self.get_date('datestart', separator='-')
        datestop = self.get_date('datestop', separator='-')
        list_days = ','.join(self.selected_days)
        # vérification des données:
        return self.valid_saisie(name, timestart, timestop, datestart, datestop, list_days)

    def valid_saisie(self, name, timestart, timestop, datestart, datestop, list_days):
        errors = []
        nbdates = nbhours = 0
        msg = ""
        if name == '':
            msg = _("missing field name")
        else:
            for date in (datestart, datestop):
                if date != '':
                    if re.match('\d\d\d\d-\d+-\d+',date) == None:
                        errors.append(date)
                else:
                    nbdates += 1
            for data in (timestart, timestop):
                if data != '':
                    # format général
                    if re.match('\d+:\d+',data) == None:
                        errors.append(data)
                else:
                    nbhours += 1

            if nbdates == 2 and nbhours == 2 and self.selected_days == []:
                msg = _('missing fields')
            elif errors != []:
                msg = _('invalid format ') + ','.join(errors)
            elif nbdates == 1 or nbhours == 1:
                msg = _('start and stop needed')
            else:
                if nbhours == 0:
                    # validité de la plage horaire
                    try:
                        t1 = time.strptime('%s:%s' % tuple(timestart.split(':')) ,'%H:%M')
                        t2 = time.strptime('%s:%s' % tuple(timestop.split(':')) ,'%H:%M')
                        # assert t1 < t2
                    except AssertionError, ValueError:
                        msg = _('invalid timerange')
                if nbdates == 0:
                    # validité de la plage par date
                    try:
                        t1 = time.strptime('%s-%s-%s' % tuple(datestart.split('-')) ,'%Y-%m-%d')
                        t2 = time.strptime('%s-%s-%s' % tuple(datestop.split('-')) ,'%Y-%m-%d')
                        assert t1 < t2
                    except AssertionError, ValueError:
                        msg = _('invalid timerange')

        if msg != "":
            dial_err = create_error_dialog(msg, self.dlg)
            dial_err.show()
            return False
        # sauvegarde de l'objet
        dico_range = dict(name=name,
                          timestart=timestart,
                          timestop=timestop,
                          datestart=datestart,
                          datestop=datestop,
                          weekdays=list_days)
        library_store.ranges[name] = Range(dico_range)
        # update du treeview
        self.model.set_value(self.iter_sel,0,library_store.ranges[name])
        self.model.set_value(self.iter_sel,1,library_store.ranges[name].name)
        return True

    def close(self, button, code, *args):
        if code == gtk.RESPONSE_OK:
            # sauvegarde du range
            res=self.save_range()
            if res == False:
                return True
        self.dlg.hide()
        return True


