#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Generates actions and salt sls (states)
"""
from __future__ import unicode_literals
from os import mkdir, unlink, chmod, makedirs, symlink
from os.path import join, isfile, isdir, basename, islink
from shutil import rmtree, copy
from glob import glob
from json import dump

from creole.client import CreoleClient


CREOLE_ACTIONS = CreoleClient().get('.actions', {})
SLS_ACTION = '/srv/salt'
MODULE_ACTION = join(SLS_ACTION, '_modules')
GENERIC_DIR = '/usr/share/ewt/'
EXTRA_DIR = '/usr/share/eole/creole/extra'
CONFIG_DIR = '/etc/ewt/appdata'
INDEX_SLS_FILENAME = 'init.sls'

if not isdir(CONFIG_DIR):
    mkdir(CONFIG_DIR, 0755)


def retrieve_actions():
    """gets all actions from creoled and reorder informations
    """
    creole_dict = {}
    for key, val in CREOLE_ACTIONS.items():
        skey = key.split('.')
        if len(skey) == 2:
            # Family information, not action
            continue
        # Action is something like systeme.action_0.name
        # creole_dict should like {'action_0': {'name': xxxx}}
        if skey[1].startswith('action'):
            full_name = '.'.join(skey[0:2])
            if full_name not in creole_dict:
                family = skey[0]
                creole_dict[full_name] = {'family': family}
            creole_dict[full_name][skey[2]] = val
    return creole_dict


def delete_recreate_modules(creole_dict):
    """remove old actions and create module directory
    """
    ewtapps = []
    for key, val in creole_dict.items():
        for ewtapp in val.get('ewtapp', []):
            if ewtapp not in ewtapps:
                # delete old salt modules
                ewtapp_dir = join(MODULE_ACTION, ewtapp)
                if isdir(ewtapp_dir):
                    rmtree(ewtapp_dir)
                mkdir(ewtapp_dir, 0750)
                source_file = join(GENERIC_DIR, 'modules_init.py')
                dest_file = join(ewtapp_dir, '__init__.py')
                copy(source_file, dest_file)
                ewtapps.append(ewtapp)

def delete_recreate_customs(creole_dict):
    """remove old customs actions and create customs directory
    """
    ewtapps = []
    for key, val in creole_dict.items():
        for ewtapp in val.get('ewtapp', []):
            if ewtapp not in ewtapps:
                custom_data_dir = join(CONFIG_DIR, ewtapp, 'customs')
                if isdir(custom_data_dir):
                    rmtree(custom_data_dir)
                mkdir(custom_data_dir, 0755)

def generate_catalog():
    """
    {
      "packages": [
        {
          "name": "systeme",
          "title": "Système",
          "description": "Liste des actions pour gérer le système EOLE.",
          "color": "#ddc9e6",
          "image": "/images/system.svg"
        }
          ],
          "elements": [
        {
          "name": "backup",
          "title": "Sauvegarde",
          "package": "systeme",
          "description": "Gestion de la sauvegarde du serveur EOLE",
          "tags": ["sauvegarde", "bareos", "backup"],
          "image": "/images/backup.svg"
        }
      ]
    }
    """
    families = {}
    actions = {}
    ewtapps = {}
    forms = []
    readers = []
    apache = []
    for key, val in CREOLE_ACTIONS.items():
        skey = key.split('.')
        if len(skey) != 2:
            # full action name is family.action_<nb>
            full_name = '.'.join(skey[0:2])
            if skey[1].startswith('action'):
                if skey[2] in ['profile', 'input']:
                    continue
                elif skey[2] == 'type':
                    if val == 'form':
                        forms.append(full_name)
                    elif val == "reader":
                        readers.append(full_name)
                    elif val == 'apache':
                        apache.append(full_name)
                    elif val not in ['external', 'custom']:
                        print('action de type {} non supportée'.format(val))
                elif skey[2] == 'ewtapp':
                    for ewtapp in val:
                        ewtapps.setdefault(ewtapp, []).append(full_name)
                if full_name not in actions:
                    family = skey[0]
                    actions[full_name] = {'package': family}
                actions[full_name][skey[2]] = val
        else:
            if skey[0] not in families:
                families[skey[0]] = {'name': skey[0]}
            json_key = skey[1]
            if json_key == 'name':
                json_key = 'title'
            families[skey[0]][json_key] = val
    for form in forms:
        actions[form]['url'] = 'run_action.html?action={}'.format(actions[form]['name'])
    for reader in readers:
        actions[reader]['url'] = 'read_file.html?action={}'.format(actions[reader]['name'])
    for apa in apache:
        actions[apa]['url'] = '../{}'.format(actions[apa]['apache_path'])
        del(actions[apa]['apache_path'])

    if isdir(CONFIG_DIR):
        catalogs = glob(join(CONFIG_DIR, "*", "catalog.json"))
        for catalog in catalogs:
            unlink(catalog)
        chmod(CONFIG_DIR, 0755)

    for app, action_names in ewtapps.items():
        app_data = {'elements': [], 'packages': []}
        app_families = []
        action_names.sort(lambda x,y: int(x.rsplit('action',1)[-1]).__cmp__(int(y.rsplit('action',1)[-1])))
        for act_name in action_names:
            if actions[act_name].get('activate', False):
                action = actions[act_name]
                if action['package'] not in app_families:
                    app_families.append(action['package'])
                    app_data['packages'].append(families[action['package']])
                if action['type'] == 'external':
                    action['url'] = '/{}/{}'.format(app, action['name'])
                if action['type'] == 'custom':
                    action['url'] = 'customs/{}'.format(action['name'])
                app_data['elements'].append(action)
        app_data['ewtapp'] = app

        APPDIR = join(CONFIG_DIR, app)
        if isdir(APPDIR):
            chmod(APPDIR, 0755)
        else:
            makedirs(APPDIR, 0755)
        catalog_name = join(APPDIR, 'catalog.json')
        with file(catalog_name, 'w') as catalog:
            dump(app_data, catalog)

def generate_modules(creole_dict):
    """
    generate salt modules in module
    """
    for act_infos in creole_dict.values():
        for ewtapp in act_infos.get('ewtapp', []):
            if act_infos.get('activate', False):
                dir_action = join(MODULE_ACTION, ewtapp, act_infos['name'])
                # can't directly name a python file __init__.py because it means something to python...
                source_file = join(GENERIC_DIR, 'init_{}.py'.format(act_infos['type']))
                if isfile(source_file):
                    makedirs(dir_action, 0750)
                    dest_file = join(dir_action, '__init__.py')
                    copy(source_file, dest_file)
                    custom_file = join(EXTRA_DIR, act_infos['name'], 'salt', 'custom.py')
                    if isfile(custom_file):
                        copy(custom_file, join(dir_action, 'custom.py'))
                    if act_infos['type'] == 'form':
                        source_file = join(GENERIC_DIR, 'generic_{}.py'.format(act_infos['type']))
                        dest_file = join(dir_action, 'action.py')
                        copy(source_file, dest_file)
                        save = act_infos.get('save', False)
                        form = []
                        for input_ in act_infos.get('input', []):
                            form.append({'title': input_, 'type': 'submit'})
                        if form != [] or save is True:
                            with open(join(dir_action, 'form.py'), 'w') as fh:
                                fh.write("# -*- coding: utf-8 -*-\n")
                                if form != '[]':
                                    str_form = "form = {}\n".format(form)
                                    fh.write(str_form)
                                if save is True:
                                    fh.write("save = True\n")
                else:
                    custom_file = join(EXTRA_DIR, act_infos['name'], 'salt', 'custom.py')
                    if isfile(custom_file):
                        makedirs(dir_action, 0750)
                        copy(custom_file, join(dir_action, '__init__.py'))
                    # Link html content of custom actions in the static folder
                    # From /usr/share/eole/creole/extra/<action>/custom to /etc/ewt/appdata/ead/customs/<action>
                    custom_extra = join(EXTRA_DIR, act_infos['name'], 'custom')
                    custom_data = join(CONFIG_DIR, ewtapp, 'customs')
                    if not islink(join(custom_data, act_infos['name'])):
                        symlink(custom_extra, join(custom_data, act_infos['name']))

def copy_sls_files(creole_dict):
    """copies sls files
    """
    ewtapps = set()
    for act_infos in creole_dict.values():
        ewtapps = ewtapps.union(set(act_infos.get('ewtapp', [])))
    for app in ewtapps:
        salt_root_dir = join(SLS_ACTION, app)
        if isdir(salt_root_dir):
            rmtree(salt_root_dir)
        makedirs(salt_root_dir, 0750)
    for act_infos in creole_dict.values():
        for ewtapp in act_infos.get('ewtapp', []):
            # creole_local_slsdir = /usr/share/eole/creole/extra/shutdown/sls
            creole_local_slsdir = join(EXTRA_DIR, act_infos['name'], 'sls')
            # creole_eole_slsdir = /usr/share/eole/creole/extra/shutdown/sls/eole
            creole_eole_slsdir = join(creole_local_slsdir, 'eole')
            if isfile(join(creole_local_slsdir, INDEX_SLS_FILENAME)) or isfile(join(creole_eole_slsdir, INDEX_SLS_FILENAME)):
                salt_local_slsdir = join(SLS_ACTION, ewtapp, act_infos['name'])
                salt_eole_slsdir = join(salt_local_slsdir, 'eole')
                mkdir(salt_local_slsdir, 0750)
                slsfiles = set([basename(sls) for sls in glob(join(creole_local_slsdir, '*.sls'))])
                slsfiles.update(set([basename(sls) for sls in glob(join(creole_eole_slsdir, '*.sls'))]))
                for slsfile in slsfiles:
                    # creole_local_slsfile = /usr/share/eole/creole/extra/shutdown/sls/*.sls
                    creole_local_slsfile = join(creole_local_slsdir, slsfile)
                    # creole_eole_slsfile = /usr/share/eole/creole/extra/shutdown/sls/eole/*.sls
                    creole_eole_slsfile = join(creole_eole_slsdir, slsfile)
                    if isfile(creole_local_slsfile):
                        copy(creole_local_slsfile, salt_local_slsdir)
                        if isfile(creole_eole_slsfile):
                            if not isdir(salt_eole_slsdir):
                                mkdir(salt_eole_slsdir, 0750)
                            copy(creole_eole_slsfile, salt_eole_slsdir)
                    else:
                        copy(creole_eole_slsfile, salt_local_slsdir)

if __name__ == '__main__':
    creole_dict = retrieve_actions()
    delete_recreate_modules(creole_dict)
    delete_recreate_customs(creole_dict)
    generate_modules(creole_dict)
    copy_sls_files(creole_dict)
    generate_catalog()
