# -*- coding: utf-8 -*-
"""Provides default describe and submit methods
for the salt actions that use forms
"""
from json import dumps, loads
from os.path import isfile, dirname, abspath, join, basename

from creole.loader import creole_loader, config_save_values
from tiramisu_web import TiramisuWeb
from tiramisu.value import Multi
from tiramisu.error import PropertiesOptionError

SLS_ACTION = '/srv/salt'

try:
    from .form import save
except:
    save = False

try:
    from .form import form
except:
    form = {}

here = dirname(abspath(__file__))
modname = basename(here)
ewtapp = basename(dirname(here))


def describe(*args, **kwargs):
    """Retrieves the tiramisu's options informations
    and converts it in the tiramisu web format
    """
    config = creole_loader(load_extra=True, rw=True, owner=modname, mandatory_permissive=False)
    tiramisu = TiramisuWeb(getattr(config, modname), remotable='none', clearable='minimum')
    values = tiramisu.get_jsonform(form)
    return dumps(values).encode()


def _get_tiramisu_informations(args):
    config = creole_loader(load_extra=True, rw=True, owner=modname, mandatory_permissive=False)
    tiramisu = TiramisuWeb(getattr(config, modname), remotable='none', clearable='minimum')
    try:
        datas = loads(args[0].decode('utf8'), encoding='utf8')
    except:
        datas = {}
    values = tiramisu.set_updates(datas)
    # get parameters sent by form
    return getattr(config, modname).make_dict(fullpath=True), values, config


def submit(*args, **kwargs):
    """Retrieves the tiramisu web's informations

    - converts them in the tiramisu's options format
      in order to be validate
    - save these informations in the configeol file
    """
    if 'dico' in kwargs:
        dico, values, config = kwargs['dico']
    else:
        dico, values, config = _get_tiramisu_informations(args)
    if save:
        config_save_values(config, modname)
    # apply state if an sls exists for this action
    if isfile(join(SLS_ACTION, ewtapp, modname, 'init.sls')):
        for key, value in dico.items():
            if isinstance(value, Multi):
                new_value = []
                for val in value:
                    if isinstance(val, PropertiesOptionError):
                        new_value.append(None)
                    else:
                        new_value.append(val)
                dico[key] = new_value

        if dico != {}:
            results = __salt__['state.apply'](modname, saltenv=ewtapp, pillar=dico)
        else:
            results = __salt__['state.apply'](modname, saltenv=ewtapp)
        msg = []
        if isinstance(results, list):
            msg = results
        elif isinstance(results, dict):
            for result in results.values():
                if not result['result']:
                    # Action failed, try to extract error message
                    err_object = result['name']
                    if result.get('changes', {}):
                        # return cmd.run error ouput, or standard output if error is empty
                        err_output = result['changes'].get('stderr', '') or result['changes'].get('stdout', '')
                    else:
                        err_output = result['comment']
                        if not err_output.endswith('.'):
                            err_output += "."
                    # if output is a python Traceback, only return last line
                    if err_output.startswith('Traceback'):
                        err_output = err_output.split('\n')[-1].lstrip('Exception:')
                    comment = "{0} ({1})".format(err_output, err_object)
                    msg.append(comment)
        if msg != []:
            values['message'] = {'type': 'error', 'text': 'Erreur: ' + ' ; '.join(msg)}

    if 'message' not in values:
        values['message'] = {'type': 'info', 'text': "Les modifications sont appliquées"}
    return dumps(values).encode()
