#-*-coding:utf-8-*-
"""
    Configuration des #log. pour l'importation scribe
"""
from shutil import copy
from os import makedirs
from os.path import basename, isdir, join, expanduser
from pyeole.lock import acquire, release, is_locked
from pyeole.log import make_logger
from scribe.eoletools import format_current_date
from scribe.importation.config import LOGFILE, RAPPORTFILE, DEST_DIR, DEFAULT_USER

LINE = "#########################################"
LOGFMT = "%(asctime)s - %(levelname)s %(message)s"
DATEFMT = '%Y-%m-%d %H:%M:%S'
log = make_logger('scribe.importation', RAPPORTFILE, debugfile=LOGFILE,
                  loglevel='DEBUG', logfmt=LOGFMT, datefmt=DATEFMT)
LOCK_NAME = 'eoleimport'

def start_importation(mode):
    """
        nouvelle importation
        mode : console ou ead
    """
    # écrasement du rapport
    header = "** Importation du %s **\n" % format_current_date("%d/%m/%Y à %H:%M")
    write_header(header, RAPPORTFILE)
    log.info(LINE)
    log.info(" Début de l'importation en mode %s" % mode)
    log.info(LINE)

def debuglog(msg, title=False):
    """
        #log.personnalisé
        -> #log.er + print
    """
    if title:
        print('')
        print(msg)
        log.debug(msg)
    else:
        print(msg)
        log.debug(msg)

def infolog(msg, title=False):
    """
        log.personnalisé
        -> #log.info + print
    """
    if title:
        print('')
        print(msg)
        log.info("## %s ##" % msg)
    else:
        print(msg)
        log.info(msg)

def errorlog(msg):
    """
        log.personnalisé
        -> #log.error + print
    """
    print('')
    print(msg)
    log.error(msg)

def end_importation(mode, user=DEFAULT_USER):
    """
        fin de l'importation
        mode : console ou ead
    """
    log.info(LINE)
    log.info(" Fin de l'importation en mode %s" % mode)
    log.info(LINE)
    copy_info(RAPPORTFILE, user)

def write_info(line, filename):
    """
        écriture d'une ligne dans un fichier de comptes (type info.csv)
    """
    fic = open(filename, 'a')
    fic.write(line+'\n')
    fic.close()

def write_header(header, filename):
    """
        écrit l'en-tête d'un fichier (en écrasant le contenu précédant)
    """
    fic = open(filename, 'w')
    fic.write(header+'\n')
    fic.close()

def copy_info(src, user=DEFAULT_USER):
    """
        sauvegarde un fichier de comptes dans le perso d'admin
        et dans celui de l'utilisateur passé en paramètre
    """
    # perso d'admin
    dest_dirs = [DEST_DIR]
    if user != DEFAULT_USER:
        # perso de l'utilisateur != admin
        home_dir = expanduser("~{}".format(user))
        dest_dirs.append(join(home_dir, 'perso', 'importation'))
    date = format_current_date('%Y%m%d_%H%M%S')
    base, extension = basename(src).split('.')
    # on se base sur le fait que filename se termine en .xxx
    dname = "%s_%s.%s" % (base, date, extension)

    for dest_dir in dest_dirs:
        if not isdir(dest_dir):
            makedirs(dest_dir)
        dst = join(dest_dir, dname)
        try:
            copy(src, dst)
            if extension == 'csv':
                infolog("fichier des comptes copié dans :")
            else:
                infolog("fichier %s copié dans :" % dname)
            infolog(dst)
        except:
            errorlog("sauvegarde du fichier dans %s impossible !" % dst)

def test_lock():
    """ test des locks pour l'importation """
    if is_locked(LOCK_NAME, level='system'):
        msg = 'Le verrou (lock) système existe : abandon !'
        log.error(msg)
        raise Exception(msg)

def add_lock():
    """ mise en place du lock d'import """
    test_lock()
    debuglog('mise en place du lock %s' % LOCK_NAME)
    acquire(LOCK_NAME, level='system')

def del_lock():
    """ suppression du lock d'import """
    debuglog('suppression du lock %s' % LOCK_NAME)
    release(LOCK_NAME, level='system')
