# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2009
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################
"""
    Gestion des préférences d'importation
"""
from os.path import isfile
from scribe.ldapconf import PROFILS, SUPPORT_ETAB, FORCED_PASSWORD_MODIFICATION_ALLOWED, dico
from scribe.importation.config import PREF_FILES, CSV_PREFIX
import csv

if SUPPORT_ETAB:
    from scribe.eoleldap import get_etabs

class Preference:
    """
    Objet préférence
    """

    def __init__(self, name, _type, question, options, default=''):
        if _type not in ['liste', 'texte', 'entier', 'select']:
            raise Exception("Type de préférence invalide : %s" % _type)
        self._type = _type
        self.name = name
        self.question = question
        self.options = options
        if default == '' and _type == 'liste':
            self.default = options[0][0]
        else:
            self.default = default

    def set_default(self, value):
        """
        mise à jour de la valeur par défaut
        """
        if self._type == 'liste' or self._type == 'select':
            if value in [item[0] for item in self.options]:
                self.default = value
        elif self._type == 'entier':
            try:
                self.default = str(int(value))
            except:
                pass
        else:
            self.default = value

class Preferences:
    """
    Gestionnaire de préférences
    """
    def __init__(self, preferences, filepath):
        self.preferences = preferences
        self.filepath = filepath
        self._update_default()

    def _find_pref(self, name):
        """
        renvoit l'objet Preference recherché
        """
        for pref in self.preferences:
            if pref.name == name:
                return pref
        return None

    def _update_default(self):
        """
        récupération des valeurs par défaut depuis le fichier
        """
        if isfile(self.filepath):
            lines = open(self.filepath).read().splitlines()
            for line in lines:
                if len(line.split('##')) == 2:
                    pref, value = line.split('##')
                    self.set_default(pref, value)

    def get_default(self, name):
        """
        renvoie la valeur par défaut courante d'une préférence
        """
        pref = self._find_pref(name)
        if pref:
            return pref.default
        return None

    def set_default(self, name, value):
        """
        enregistrement de la valeur par defaut
        """
        pref = self._find_pref(name)
        if pref:
            pref.set_default(value)

    def save(self):
        """
        enregistrement des valeurs par défaut dans un fichier
        """
        lines = []
        for pref in self.preferences:
            lines.append("%s##%s" % (pref.name, pref.default))
        pref_file = open(self.filepath, 'w')
        pref_file.write("\n".join(lines))
        pref_file.close()

# -- préférences d'importation -- #

PREF_DOMAINE = ['domaine', 'liste',
        "Domaine de messagerie élève par défaut",
        (('restreint', 'restreint'), ('internet', 'Internet'))]
PREF_QUOTA = ['quota', 'entier',
            "Quota disque en Mo (0 pour inactif)",
            ""]
PREF_GEN_PWD = ['gen_pwd', 'liste',
            "Génération des mots de passe",
            (('alea', 'aléatoire'), )]
            #('naiss', 'date de naissance')
PREF_CHANGE_PWD = ['change_pwd', 'liste',
            "Changement du mot de passe à la première connexion"]

PREF_CHANGE_PWD.append((('oui', 'oui'), ('non', 'non')))

PREF_LOGIN = ['login', 'liste',
            "Génération des identifiants",
            (('standard', 'prenom.nom'), ('pnom', 'pnom'),
             ('nomp', 'nomp'), ('p.nnn', 'p.nnn (format Gibii)'),
             ('prenom.n', 'prenom.n'),
            )]
PREF_SHELL = ['shell', 'liste',
            "Activer le shell (clients Linux)",
            (('oui', 'oui'), ('non', 'non'))]
PREF_PROFIL = ['profil', 'liste',
            "Profil Windows",
            PROFILS]

PREF_MAIL = ['mail', 'liste',
             "Adresse mail",
             ( ('perso_restreint', 'adresse fournie ou domaine restreint'),
               ('perso_internet', 'adresse fournie ou domaine Internet'),
               ('perso_aucune', 'adresse fournie ou aucune'),
               ('restreint', 'adresse locale, domaine restreint'),
               ('internet', 'adresse locale, domaine Internet'),
               ('aucune', 'aucune adresse'),
            )]

PREF_PREFIX_ETAB = ['etab_prefix', 'texte',
        "Préfixe des groupes de l'établissement", ""]

if dico.get('activer_nfs', 'non') == 'oui' or 'interface_client_ltsp' in dico:
    DEFAULT_CHANGE_PWD = 'non'
    DEFAULT_SHELL = 'oui'
else:
    DEFAULT_CHANGE_PWD = 'oui'
    DEFAULT_SHELL = 'non'

def get_etab():
    """
    charge les établissements à chaque fois qu'on le demande
    """
    tetabs = get_etabs()
    etabs = [(tetabs[0], tetabs[0])]
    for etab in tetabs[1]:
        etabs.append((etab, etab))
    return ['etab', 'select', "Etablissement", etabs]

def get_etabprefix():
    """
    charge les établissements à chaque fois qu'on le demande
    """
    associations = {}
    etab_prefix = []
    if not isfile(CSV_PREFIX):
        open(CSV_PREFIX, 'w').close()
    with open(CSV_PREFIX, 'r+') as prefixcsv:
        read_prefix = csv.reader(prefixcsv, delimiter=';')
        for row in read_prefix:
            if len(row) > 1:
                associations[row[0]] = row[1]
    tetabs = get_etabs()
    # établissement racine
    etab_prefix.append((tetabs[0], associations.get(tetabs[0], '')))
    for etab in tetabs[1]:
        # autres établissements
        etab_prefix.append((etab, associations.get(etab, '')))
    return ['etabs_prefixes', 'select', "Prefixe Etablissement", etab_prefix]

def get_eleves_prefs():
    """ préférences pour les élèves """
    prefs_eleve = [ Preference(*PREF_DOMAINE),
                    Preference(default='50', *PREF_QUOTA),
                    Preference(*PREF_LOGIN),
                    Preference(default=DEFAULT_SHELL, *PREF_SHELL),
                    Preference(default='1', *PREF_PROFIL),
    ]
    if FORCED_PASSWORD_MODIFICATION_ALLOWED:
        prefs_eleve.insert(3, Preference(default=DEFAULT_CHANGE_PWD, *PREF_CHANGE_PWD))
    if SUPPORT_ETAB:
        prefs_eleve.insert(0, Preference(*get_etab()))
        prefs_eleve.insert(1, Preference(*PREF_PREFIX_ETAB))
        prefs_eleve.append(Preference(*get_etabprefix()))

    return Preferences(prefs_eleve, PREF_FILES['eleve'])

def get_responsables_prefs():
    """ préférences pour les responsables """
    prefs_resp = (Preference(*PREF_LOGIN),
                  Preference(*PREF_MAIL),
    )
    return Preferences(prefs_resp, PREF_FILES['responsable'])


def get_enseignants_prefs():
    """ préférences pour les enseignants """
    prefs_ens = [   Preference(default='0', *PREF_QUOTA),
                    Preference(*PREF_LOGIN),
                    Preference(default=DEFAULT_SHELL, *PREF_SHELL),
                    Preference(default='1', *PREF_PROFIL),
                    Preference(default='perso_aucune', *PREF_MAIL),
    ]
    if FORCED_PASSWORD_MODIFICATION_ALLOWED:
        prefs_ens.insert(2, Preference(default=DEFAULT_CHANGE_PWD, *PREF_CHANGE_PWD))
    if SUPPORT_ETAB:
        prefs_ens.insert(0, Preference(*get_etab()))
        prefs_ens.insert(1, Preference(*PREF_PREFIX_ETAB))
        prefs_ens.append(Preference(*get_etabprefix()))
    return Preferences(prefs_ens, PREF_FILES['enseignant'])

def get_administratifs_prefs():
    """ préférences pour les administratifs """
    prefs_admin = [ Preference(default='0', *PREF_QUOTA),
                    Preference(*PREF_LOGIN),
                    Preference(default=DEFAULT_SHELL, *PREF_SHELL),
                    Preference(default='1', *PREF_PROFIL),
                    Preference(default='perso_aucune', *PREF_MAIL),
    ]
    if FORCED_PASSWORD_MODIFICATION_ALLOWED:
        prefs_admin.insert(2, Preference(default=DEFAULT_CHANGE_PWD, *PREF_CHANGE_PWD))
    return Preferences(prefs_admin, PREF_FILES['administratif'])

def get_invites_prefs():
    """ préférences pour les comptes invités """
    prefs_autre = (Preference(*PREF_LOGIN),
                   Preference(*PREF_MAIL),
    )
    return Preferences(prefs_autre, PREF_FILES['invite'])
