# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

try: _ # localized string fetch function
except NameError: _ = str

import os, sys
from datetime import datetime, timedelta, timezone
from configparser import ConfigParser
from twisted.internet import defer
from twisted.python.failure import Failure
from twisted.web.microdom import *
from twisted.persisted import aot

from zephir.monitor.agentmanager import config as cfg, rrd, status, util
from zephir.monitor.agentmanager.data import *
from zephir.monitor.agentmanager.util import log
from zephir.lib_zephir import log as zephir_log, is_locked

action_map = {0:'action_unknown',1:'action_ok',2:'action_warn',3:'action_error',4:'action_dependant'}

class AgentData:
    """Persistance et accès aux données d'un agent.

    Cette classe contient la « charge utile » d'un agent ; avant envoi
    au serveur Zephir tous les agents sont convertis en une instance
    de C{L{AgentData}}.

    Attributs :

     - C{name} : nom de l'agent

     - C{period} : intervalle de mesures en secondes ; C{period=0}
       désactive les mesures automatiques.

     - C{description} : description de cette instance d'agent,
       apparaissant sur la page de l'agent.

     - C{max_status} : L{status<zephir.monitor.agentmanager.status>} le plus grave ; réinitialisé à chaque
       envoi au serveur Zephir ou sur demande via XML-RPC.

     - C{last_measure_date} : date de la dernière mesure.

     - C{data} : liste d'objets représentant les données associées à
       l'agent (cf. C{L{zephir.monitor.agentmanager.data}}).
    """

    def __init__(self, name, period, description, section,
                 max_status, max_status_date, last_status, last_measure_date, data, measure_data={}):
        self.name = name
        self.period = period
        self.description = description
        self.section = section
        self.max_status = max_status
        self.last_status = last_status
        self.max_status_date = max_status_date
        self.last_measure_date = last_measure_date
        self.data = data
        self.measure_data = measure_data


    def from_agent(self, agent):
        """I{Factory Method}

        @param agent: un agent concret

        @return: une copie de C{agent} ne contenant plus que les
          données utiles au serveur Zephir (instance de C{AgentData}
        """
        result = self(agent.name, agent.period, agent.description, agent.section,
                      agent.max_status, agent.max_status_date, agent.last_status, agent.last_measure_date, agent.data, agent.measure_data)
        return result
    from_agent = classmethod(from_agent)


    def from_archive(self, archive_dir):
        """I{Factory Method}

        @param archive_dir: le chemin d'un répertoire contenant les
          données d'un agent

        @return: une instance de C{AgentData} chargée depuis le
          système de fichiers
        """
        xml_filename = os.path.join(archive_dir, 'agent.xml')
        xml_file = open(xml_filename, 'r').read()
        xml_file = xml_file.replace('\\xc3\\x89', 'É').replace('\\xc3\\xa9', 'é').replace('\\xc3\\xa8', 'è').replace('\\xc3\\xb4', 'ô').replace('\\xc3\\xa0', 'à')
        try:
            result = aot.unjellyFromSource(xml_file)
        except Exception as e:
            error_agent = LoadErrorAgent(
                os.path.basename(archive_dir),
                title = _("Loading archive %s failed") % archive_dir,
                message = _("Loading agent from archive %s failed:\n%s") % (archive_dir, e))
            result = AgentData.from_agent(error_agent)
        return result
    from_archive = classmethod(from_archive)


    def archive(self, archive_dir):
        """Sérialise l'agent sur disque, cf. L{from_archive}"""
        xml_filename = os.path.join(archive_dir, 'agent.xml')
        xml_file = open(xml_filename, 'bw')
        # écriture avec aot
        aot.jellyToSource(self, xml_file)
        xml_file.close()

    def ensure_data_uptodate(self):
        """Met à jour les données de l'agent sur disque"""
        pass




STATUS_GRAPH_OPTIONS = [
    "-send-7days", "-l0", "-u1", "-g",
    "-w112", "-h10", "-xHOUR:6:DAY:1:DAY:1:0:%d",
    "CDEF:unknown=status,0,EQ",
    "CDEF:ok=status,1,EQ",
    "CDEF:warn=status,2,EQ",
    "CDEF:error=status,3,EQ",
    "AREA:unknown#666666",
    "AREA:ok#33BB33",
    "AREA:warn#CC6600",
    "AREA:error#BB3333",
    ]

STATUS_GRAPH_OPTIONS_MONTHLY = [
    "-send-1month", "-l0", "-u1", "-g",
    "-w300", "-h10", "-xDAY:1:WEEK:1:DAY:7:0:%d",
    "CDEF:unknown=status,0,EQ",
    "CDEF:ok=status,1,EQ",
    "CDEF:warn=status,2,EQ",
    "CDEF:error=status,3,EQ",
    "AREA:unknown#666666",
    "AREA:ok#33BB33",
    "AREA:warn#CC6600",
    "AREA:error#BB3333",
    ]

def no_action(agent, old_status, new_status):
    log.msg('%s : status changed to %s' % (agent.name, new_status))

class Agent(AgentData):
    """Classe abstraite des agents.

    Un agent concret est une sous-classe d'C{L{Agent}} implémentant
    (en particulier) la méthode C{L{measure()}}.
    """

    def __init__(self, name,
                 period = 60,
                 fields = ['value'],
                 description = None,
                 section = None,
                 modules = None,
                 requires = [],
                 **params):
        if description is None:
            description = self.__class__.__doc__
        AgentData.__init__(self, name, period, description, section,
                           status.Unknown(), "", status.Unknown(), "", [], {})
        self.fields = fields
        self.data_needs_update = False
        # those will be set by loader, eg UpdaterService
        self.archive_dir = None
        self.status_rrd = None
        self.manager = None
        # dépendances fonctionnelles
        self.requires = requires
        self.modules = modules
        self.last_measure = None

    def init_data(self, archive_dir):
        """Mémorise et initialise le répertoire d'archivage

        Cette méthode sera appelée par le framework après chargement
        de l'agent, afin de terminer les initialisations pour
        lesquelles l'agent a besoin de connaître l'emplacement de ses
        données sur disque.
        """
        self.archive_dir = archive_dir
        self.ensure_datadirs()
        if self.period != 0: # period = 0 => no agent measures scheduled
            status_period, xff = self.period, 0.75
        else:
            status_period, xff = 60, 1

        # on récupère le dernier problème et sa date dans l'archive si elle est présente
        xml_file = os.path.join(self.archive_dir, 'agent.xml')
        if os.path.exists(xml_file):
            try:
                a = AgentData("temp", 60, "agent temporaire",status.Unknown(), "", status.Unknown(), "", [], {})
                a = a.from_archive(self.archive_dir)
                self.max_status = a.max_status
                self.max_status_date = a.max_status_date
                del a
            except:
                pass

        statusname = os.path.join(self.archive_dir, 'status')
        self.status_rrd = rrd.Database(statusname + '.rrd',
                                       step = status_period)
        self.status_rrd.new_datasource( # status history
                name = "status",
                min_bound = 0, max_bound = len(status.STATUS_ORDER))
        if status_period > 3600:
            # 1 record per week on 1 year
            self.status_rrd.new_archive(
                rows = 52, steps = (3600*24*7) / status_period,
                consolidation = 'MAX', xfiles_factor = 0.75)
        else:
            # 1 record per 12 hours on 1 year
            self.status_rrd.new_archive(
                rows = 730, steps = (3600 * 12) / status_period,
                consolidation = 'MAX', xfiles_factor = 0.75)
            # 1 record per 2hour/pixel on 30 days
            self.status_rrd.new_archive(
                rows = 12*30, steps = 7200 / status_period,
                consolidation = 'MAX', xfiles_factor = 0.75)
            # 1 record per hour/pixel on 1 week
            self.status_rrd.new_archive(
                rows = 24*7, steps = 3600 / status_period,
                consolidation = 'MAX', xfiles_factor = 0.75)
            # 1 record per period on 1 day
            self.status_rrd.new_archive(
                rows = 24*3600/status_period, steps = 1,
                consolidation = 'MAX',  xfiles_factor = 0.75)
            self.status_rrd.new_graph(pngname = statusname + '.png',
                                      vnamedefs = { "status": ("status", 'MAX') },
                                      options = STATUS_GRAPH_OPTIONS)
            self.status_rrd.new_graph(pngname = statusname + '_long.png',
                                      vnamedefs = { "status": ("status", 'MAX') },
                                      options = STATUS_GRAPH_OPTIONS_MONTHLY)
        # vérification ajoutée pour recréer les anciennes archives (1 semaine de données seulement)
        try:
            rrd_info = self.status_rrd.info()
            if rrd_info:
                if 'rra[1].cf' in rrd_info and 'rra[2].cf' not in rrd_info:
                    # suppression du fichier si 2  sources
                    os.unlink(statusname + '.rrd')
        except:
            # fichier rrd illisible, on le supprime
            os.unlink(statusname + '.rrd')
        self.status_rrd.create()

    # Méthodes à spécialiser dans les agents concrets

    def measure(self):
        """Prend concrètement une mesure.

        Pour implémenter un agent, il faut implémenter au moins cette
        méthode.

        @return: Résultat de la mesure, un dictionnaire C{{champ:
          valeur}} ou un objet C{L{twisted.internet.defer.Deferred}}
          renvoyant ce dictionnaire.
        """
        raise NotImplementedError

    def check_status(self):
        """Renvoie le diagnostic de fonctionnement de l'agent.

        L'implémentation par défaut dans C{L{Agent}} renvoie un statut
        neutre. Les agents concrets doivent donc redéfinir cette
        méthode pour annoncer un diagnostic utile.
        """
        log.msg(_("Warning: agent class %s does not redefine method check_status()")
                % self.__class__.__name__)
        return status.Unknown()

    def update_status(self):
        new_status = None
        # si possible, on vérifie le dernier état des agents dont on dépend
        if self.manager is not None:
            for agent in self.requires:
                try:
                    if self.manager.agents[agent].last_status == status.Error():
                        new_status = status.Dependant()
                except:
                    # agent non chargé ou état inconnu
                    pass
        if new_status is None:
            new_status = self.check_status()
        # mise à jour de l'état
        self.set_status(new_status)

    def set_status(self, s, reset = False):
        """Mémorise le statut et met à jour C{statut_max}

        @param s: statut actuel
        @param reset: réinitialise C{max_status} à C{s} si C{reset==True}
        """
        # Si on détecte un nouveau problème, on met à jour max_status et max_status_date
        if  s > self.last_status and s not in [status.OK(),status.Dependant()]:
            self.max_status = s
            self.max_status_date = self.last_measure_date
        if self.last_status != s:
            # l'état a changé, on execute éventuellement des actions
            self.take_action(self.last_status, s)
        self.last_status = s

    def reset_max_status(self):
        """Réinitialise C{max_status} à la valeur courante du status
        """
        self.set_status(self.check_status(), reset = True)


    # Gestion des mesures
    def scheduled_measure(self):
        """Déclenche une mesure programmée.

        Prend une mesure et mémorise le résultat et l'heure.
        """
        # Si un lock indique qu'un reconfigure est en cours, on ne prend pas de mesure
        if not is_locked('actions'):
            now = datetime.now(tz=timezone.utc)
            try:
                m = self.measure()
            except Exception as e:
                import traceback
                traceback.print_exc()
                exc_data = sys.exc_info()
                self.handle_measure_exception(exc_data)
            else:
                if isinstance(m, defer.Deferred):
                    m.addCallbacks(
                        lambda m: self.save_measure(Measure(now, m)),
                        self.handle_measure_exception)
                else:
                    self.save_measure(Measure(now, m))

    def save_measure(self, measure):
        """Mémorise une mesure donnée.

        Méthode à redéfinir dans les sous-classes concrètes de C{L{Agent}}.
        (callback de succès pour C{L{scheduled_measure()}})
        """
        self.last_measure_date = measure.get_strdate()
        self.data_needs_update = True
        self.last_measure = measure
        self.update_status()
        self.status_rrd.update({'status': self.last_status.num_level()},
                               datetime.now(tz=timezone.utc))

    def check_action(self, action_name):
        """retourne la liste des actions autorisées/interdites pour cet agent
        """
        authorized = False
        if self.manager:
            for cfg_level in ["_eole", "_acad", ""]:
                cfg_file = os.path.join(self.manager.config['action_dir'], 'actions%s.cfg' % cfg_level)
                if os.path.isfile(cfg_file):
                    action_mngr = ConfigParser()
                    action_mngr.read(cfg_file)
                    try:
                        authorized = eval(action_mngr.get(self.name, action_name))
                    except:
                        pass
        return authorized

    def take_action(self, old_status, new_status):
        """exécute des actions en cas de changement de status
        """
        # si un fichier /var/run/actions existe, on n'exécute aurcune action
        if not is_locked('actions'):
            # recherche des actions prévues pour l'agent dans cet état
            action_name = action_map.get(new_status.num_level(), None)
            if action_name:
                if self.check_action(action_name):
                    action_func = getattr(self, action_name, no_action)
                    # exécution des actions si besoin
                    msg = action_func(self, old_status, new_status)
                    # si un message est renvoyé, on le loggue sur zephir
                    if msg:
                        # action,etat,msg
                        log.msg(msg)
                        zephir_log('SURVEILLANCE',0,msg)
            else:
                log.msg('status not defined in action_map')

    def handle_measure_exception(self, exc):
        """Callback d'erreur pour C{L{scheduled_measure()}}
        """
        if isinstance(exc, Failure):
            if exc.tb:
                err_line = " (ligne %s)" % exc.tb.tb_lineno
            else:
                err_line = ""
            err_message = "%s%s" % (exc.getErrorMessage(), err_line)
        else:
            err_line = exc[2].tb_lineno
            err_message = "%s (ligne %s)" % (str(exc[1]), str(err_line))
        log.msg(_("/!\\ Agent %s, exception during measure: %s".encode())
                % (self.name.encode(), err_message.encode()))
        self.set_status(status.Error(err_message))


    def archive(self):
        """Crée l'archive de l'agent sur disque
        """
        self.ensure_data_uptodate()
        AgentData.from_agent(self).archive(self.archive_dir)


    def ensure_data_uptodate(self):
        self.ensure_datadirs()
        if self.data_needs_update:
            # would need locking here if multithreaded (archive must have
            # up-to-date datas)
            # for d in self.data:
            #     d.ensure_uptodate()
            self.write_data()
        self.update_status()
        self.data_needs_update = False;

    def write_data(self):
        """Écrit les données générées par l'agent sur disque

        Méthode à redéfinir si nécessaire dans les sous-classes.
        """
        additional_args = []
        this_morning = datetime.now(tz=timezone.utc).replace(hour = 0, minute = 0, second = 0, microsecond = 0)
        for i in range(7):
            t = this_morning - i*timedelta(days = 1)
            additional_args += "VRULE:%s#000000" % rrd.rrd_date_from_datetime(t)
        self.status_rrd.graph_all()

    # convenience method
    def ensure_datadirs(self):
        """Méthode de convenance, cf C{L{zephir.monitor.agentmanager.util.ensure_dir}}
        """
        assert self.archive_dir is not None
        util.ensure_dirs(self.archive_dir)


class TableAgent(Agent):
    """Agent concret mémorisant ses mesures dans une table.

    Les valeurs mesurées peuvent être non-numériques.
    """

    def __init__(self, name,
                 max_measures=100,
                 **params):
        """
        @param max_measures: nombre maximal de mesures mémorisées
        """
        Agent.__init__(self, name, **params)
        assert max_measures >= 1
        self.max_measures = max_measures
        self.measures = []
        self.data = [MeasuresData(self.measures, self.fields)]


    def save_measure(self, measure):
        """Maintient la table de mesures triée et en dessous de la taille
        maximale (cf. C{Agent.save_measure}).
        """
        Agent.save_measure(self, measure)
        # drop old measures to keep list size under max_measures
        # can't slice because the list must be modified in place
        for x in range(max(0, len(self.measures) - self.max_measures +1)):
            self.measures.pop(0)
        self.measures.append(measure)
        self.measures.sort()




class MultiRRDAgent(Agent):
    """Classe abstraite pour les agents utilisant RRDtool.

    Les valeurs mesurées étant visualisées sous forme de graphes,
    elles doivent être numériques.

    Les agents de cette classe maintiennent plusieurs bases de données RRD et
    génèrent des graphes au format PNG de leurs données.
    """

    def __init__(self, name,
                 datasources, archives, graphs,
                 **params):
        """
        Les paramètres C{datasources}, C{archives} et C{graphs} sont
        des listes de paramètres pour la configuration d'une L{base
        RRD<agentmanager.rrd.Database>}.
        """
        Agent.__init__(self, name, **params)
        self.datasources = datasources
        self.archives = archives
        self.graphs = graphs
        self.data = []


    def init_data(self, archive_dir):
        """Crée et initialise la base RRD dans C{archive_dir}.
        """
        Agent.init_data(self, archive_dir)
        self.rrd = {}
        arch_names = list(self.archives.keys())
        arch_names.sort()
        for name in arch_names:
            rrdname = name + '.rrd'
            self.rrd[name] = rrd.Database(os.path.join(self.archive_dir, rrdname),
                                          step = self.period)
            self.data.append(RRDFileData(rrdname))
            for ds in self.datasources[name]: self.rrd[name].new_datasource(**ds)
            for rra in self.archives[name]: self.rrd[name].new_archive(**rra)
            for g in self.graphs[name]:
                self.data.append(ImgFileData(g['pngname']))
                self.data.append(HTMLData('<br>'))
                g['pngname'] = os.path.join(self.archive_dir, g['pngname'])
                self.rrd[name].new_graph(**g)
            self.rrd[name].create()

    def save_measure(self, measure):
        Agent.save_measure(self, measure)
        # on ne prend que les mesures déclarées comme datasources
        for name in list(self.archives.keys()):
            rrd_values = {}
            for datasource in self.datasources[name]:
                if datasource['name'] in measure.value:
                    rrd_values[datasource['name']] = measure.value[datasource['name']]
            # update des bases rrd
            self.rrd[name].update(rrd_values, measure.get_date())

    def write_data(self):
        Agent.write_data(self)
        for name in list(self.archives.keys()):
            self.rrd[name].graph_all()


class RRDAgent(Agent):
    """Classe abstraite pour les agents utilisant RRDtool.

    Les valeurs mesurées étant visualisées sous forme de graphes,
    elles doivent être numériques.

    Les agents de cette classe maintiennent une base de données RRD et
    génèrent des graphes au format PNG de leurs données.
    """

    def __init__(self, name,
                 datasources, archives, graphs,
                 **params):
        """
        Les paramètres C{datasources}, C{archives} et C{graphs} sont
        des listes de paramètres pour la configuration d'une L{base
        RRD<agentmanager.rrd.Database>}.
        """
        Agent.__init__(self, name, **params)
        self.datasources = datasources
        self.archives = archives
        self.graphs = graphs
        self.data = []


    def init_data(self, archive_dir):
        """Crée et initialise la base RRD dans C{archive_dir}.
        """
        Agent.init_data(self, archive_dir)
        rrdname = self.name + '.rrd'
        self.rrd = rrd.Database(os.path.join(self.archive_dir, rrdname),
                                step = self.period)
        self.data.append(RRDFileData(rrdname))
        for ds in self.datasources: self.rrd.new_datasource(**ds)
        for rra in self.archives: self.rrd.new_archive(**rra)
        for g in self.graphs:
            self.data.append(ImgFileData(g['pngname']))
            self.data.append(HTMLData('<br>'))
            g['pngname'] = os.path.join(self.archive_dir, g['pngname'])
            self.rrd.new_graph(**g)
        self.rrd.create()


    def save_measure(self, measure):
        Agent.save_measure(self, measure)
        # on ne prend que les mesures déclarées comme datasources
        rrd_values = {}
        for datasource in self.datasources:
            if measure.value is not None:
                rrd_values[datasource['name']] = measure.value[datasource['name']]
            else:
                log.msg("/!\\ measure return empty value: %s" % self.name)
                rrd_values[datasource['name']] = None
        # update des bases rrd
        self.rrd.update(rrd_values, measure.get_date())

    def write_data(self):
        Agent.write_data(self)
        self.rrd.graph_all()


class LoadErrorAgent(Agent):
    """Pseudo-agent représentant une erreur de chargement d'un fichier
    de configuration.
    """

    HTML = """<p class="error">%s</p>"""

    def __init__(self, name,
                 title=_("Error while loading agent"),
                 message="",
                 **params):
        Agent.__init__(self, name, **params)
        self.max_status = status.Error(title)
        m = '<br />'.join(message.splitlines())
        self.html = HTMLData(LoadErrorAgent.HTML % m)
        self.data = [self.html]

    def check_status(self):
        return self.max_status()



# def test_main():
#     test_support.run_unittest(UserStringTest)

# if __name__ == "__main__":
#     test_main()
