# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent Zéphir pour l'étude des Statistiques Exim
"""

from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import HTMLData, TableData
from twisted.internet.utils import getProcessOutput
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.util import log


def format_quantity(quantity):
    """
    L'option "-nvr" d'eximstats permet de toujours avoir le résulat en bytes
    Il n'y a plus qu'à le transformer en ko ou en Mo
    """
    quantity = float(quantity)/1024
    if quantity < 1000:
        return '%.1f ko' % quantity
    else:
        return '%.1f Mo' % (quantity/1024)


class EximStats(Agent):

    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.status = status.OK()
        title1 = HTMLData("<h3>Etat du trafic<h3>")
        self.table = TableData([
            ('label', '', {'align':'center'}, None),
            ('quantity', 'Volume de données', {'align':'center'}, None),
            ('number', 'Nombre de messages', {'align':'center'}, None),
        ])
        title2 = HTMLData("<h3>Erreurs<h3>")
        self.table2 = TableData([
            ('label', '', {'align':'center'}, None),
            ('value', 'Valeur', {'align':'center'}, None),
        ])
        self.tdate = TableData([
            ('date', 'Période de mesure', {'align':'center'}, None),
        ])
        self.data = [self.tdate, title1, self.table, title2, self.table2]

    def measure(self):
        """
            Utilisation de la commande eximstats :
            eximstats -nt -h0 -tnl -t0 -nr -nvr -q60 /var/log/exim4/mainlog
            2011 : utilisation d'un script externe pour la gestion du mode conteneur
        """
        res = getProcessOutput("/usr/share/zephir/monitor/bin/eximstats.sh",
                               env = {'LC_ALL': 'C'}, errortoo=1)
        res.addCallbacks(self.measure_process, self.measure_error)
        return res

    def measure_error(self, failure):
        """
        erreur lors du lancement du script de mesure
        """
        log.msg(u"Erreur remontée par le script eximstat.sh :")
        log.msg(failure.getErrorMessage())
        self.status = status.Error('eximstats.sh a remonté des erreurs (voir logs)')
        return self.no_measure()

    def measure_process(self, result):
        """
        traitement du résultat
        """
        result = result.decode()
        # étude du résultat
        res1 = []
        res2 = []
        date = []
        Queue = False
        if result == '' or result.startswith('****'):
            # No valid log lines read
            return self.no_measure()
        for ligne in result.splitlines():
            # dernier !
            if ligne.startswith('Errors encountered:') :
                # en première ligne
                # L.insert(index, object) -- insert object before index
                res2.insert (0, {'label' : 'Nombre d\'erreurs', 'value' : ligne[20:]} )
            if ligne.startswith('Exim statistics from') :
                l = ligne.split()
                deb = l[3].split('-')
                fin = l[6].split('-')
                date.append ({'date' : 'du %s/%s/%s au %s/%s/%s' % (deb[2], deb[1], deb[0],
                                                                    fin[2], fin[1], fin[0])})
            # 1er
            elif ligne.startswith('  Received '):
                l = ligne.split()
                res1.append ({'label' : 'Réception', 'quantity' : format_quantity(l[1]), 'number' : l[2] })
                res2.append ({'label' : 'Pourcentage de mails en attente', 'value' : l[5] })
                res2.append ({'label' : 'Pourcentage de mails en erreur', 'value' : l[7] })

            # 2ème
            elif ligne.startswith('  Delivered'):
                l = ligne.split()
                res1.append ({'label' : 'Envoi', 'quantity' : format_quantity(l[1]), 'number' : l[2] })

            # 3ème
            elif ligne.startswith('Under   1m') and Queue==False:
                # tour de passe-passe car la ligne y est 2 fois
                # on prend la première !
                Queue = True
                l = ligne.split()
                res2.insert (0, {'label' : 'Mails ayant passé plus de 1 minute en queue',
                              'value' : '%.1f%%' % ( float(100) - float(l[3][:-1]) ) })

        if res1 == []:
            # rien trouvé ?
            return self.no_measure()

        return { 'statistics'  : res1,
                 'statistics2' : res2,
                 'statdate' : date
               }

    def no_measure(self):
        """
        affichage élégant si pas d'erreur
        """
        return { 'statistics'  : [{'label' : '', 'quantity' : '0', 'number' : '0'}],
                 'statistics2' : [{'label' : '', 'value' : '---'}],
                 'statdate'    : [{'date'  : '---' }],
               }

    def write_data(self):
        """
        écriture du résultat
        """
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['statistics']
            self.table2.table_data = self.last_measure.value['statistics2']
            self.tdate.table_data = self.last_measure.value['statdate']

    def check_status(self):
        """
        calcul du statut
        """
        return self.status

