#!/bin/sh
#
# preconfig.sh
# Copyright 2024 - abc-nix (antiX Linux)
#
# Script to configure Debian package build environment

# Check if required arguments are provided
if [ "$#" -ne 2 ]; then
    echo "Usage: $0 <template-name> <selected-build>"
    exit 1
fi

# Assign input arguments to variables
TEMPLATE=$1
SELECTED_BUILD=$2

# Define paths
KERNEL_MAKEFILE="./Makefile"
TEMPLATE_FILE="debian/template/$TEMPLATE"
CONTROL_SCRIPT="debian/scripts/gen-control.sh"
CHANGELOG_SCRIPT="debian/scripts/gen-changelog.sh"
CUSTOM_CONFIG_MK="debian/rules.d/custom-config.mk"
ARCH_FILE="debian/arch"
COMPAT_FILE="debian/compat"

# Check if kernel Makefile exists
if [ ! -f "$KERNEL_MAKEFILE" ]; then
    echo "Error: $KERNEL_MAKEFILE file not found!"
    exit 1
fi

# Extract VERSION, PATCHLEVEL, and SUBLEVEL from the kernel Makefile
VERSION=$(grep '^VERSION =' "$KERNEL_MAKEFILE" | awk '{print $3}')
PATCHLEVEL=$(grep '^PATCHLEVEL =' "$KERNEL_MAKEFILE" | awk '{print $3}')
SUBLEVEL=$(grep '^SUBLEVEL =' "$KERNEL_MAKEFILE" | awk '{print $3}')

# Compute KERNELVER
KERNELVER="${VERSION}.${PATCHLEVEL}.${SUBLEVEL}"

# Check if template file exists
if [ ! -f "$TEMPLATE_FILE" ]; then
    echo "Error: $TEMPLATE_FILE file not found!"
    exit 1
fi

# Source the template file
. "$TEMPLATE_FILE"

# Set DEB_HOST_ARCH and KERNELVER_REL based on SELECTED_BUILD
KERNELVER_SUB="${EXTRAVERSION}-${SELECTED_BUILD}-smp"
case "$SELECTED_BUILD" in
    amd64)
        DEB_HOST_ARCH=amd64
        KERNEL_ARCH=x86_64
        ;;
    486)
        DEB_HOST_ARCH=i386
        KERNEL_ARCH=i386
        ;;
    686)
        DEB_HOST_ARCH=i386
        KERNEL_ARCH=i386
        KERNELVER_SUB="${KERNELVER_SUB}-pae"
        ;;
    *)
        echo "Error: SELECTED_BUILD must be either amd64, 486, or 686"
        exit 1
        ;;
esac

KERNELVER_REL="${KERNELVER}-${KERNELVER_SUB}"

# Update debian/arch file
echo "$DEB_HOST_ARCH" > "$ARCH_FILE"

# Detect Debian release version number
case "$DEBIAN_RELEASE" in
    11|11.*|oldstable|Bullseye|bullseye)
        DEBIAN_RELEASE="bullseye"
        DEBIAN_RELEASE_VERSION=11
        ;;
    12|12.*|stable|Bookworm|bookworm)
        DEBIAN_RELEASE="bookworm"
        DEBIAN_RELEASE_VERSION=12
        ;;
    13|13.*|testing|Testing|trixie|Trixie)
        DEBIAN_RELEASE="testing"
        DEBIAN_RELEASE_VERSION=13
        ;;
    unstable|Unstable|sid|SID)
        DEBIAN_RELEASE="unstable"
        DEBIAN_RELEASE_VERSION=13
        ;;
    *)
        DEBIAN_RELEASE="UNRELEASED"
        DEBIAN_RELEASE_VERSION=12
        ;;
esac

# Update debian/compat
echo $DEBIAN_RELEASE_VERSION > "$COMPAT_FILE"

# Generate control file
$CONTROL_SCRIPT "$DEB_HOST_ARCH" "$DEB_PACKAGE_MAINT" "$KERNELVER_REL" "$BUILD_DEBUG_PACKAGE"

# Generate changelog file
$CHANGELOG_SCRIPT "$DEB_PACKAGE_MAINT" "$KERNELVER" "${BUILD_SUBVERSION}${EXTRAVERSION}${SELECTED_BUILD}" "$CHANGELOG_TEMPLATE" "$DEBIAN_RELEASE"


# Create custom-config.mk file with variables from the template
{
    echo "KERNELVER=$KERNELVER" 
    echo "SELECTED_BUILD=$SELECTED_BUILD"
    echo "DEB_HOST_ARCH=$DEB_HOST_ARCH"
    echo "KERNEL_ARCH=$KERNEL_ARCH"
    echo "KERNELVER_REL=$KERNELVER_REL"
    echo "CUSTOM_KERNEL_CONFIG=$CUSTOM_KERNEL_CONFIG"
} > "$CUSTOM_CONFIG_MK"

echo "Configuration complete for $SELECTED_BUILD kernel version $KERNELVER-$EXTRAVERSION"
