#!/usr/bin/env bash
set -x
set -e
set -o pipefail

USAGE="epd_encryption_test.sh <path_to_encryptcli> <tested_file>"
if [ $# -lt 2 ]
then
    echo $USAGE
    exit 1
fi

ENCRYPT_CLI="$1"
FILE="$2"
TMP_DIR="./tmp"
PASSPHRASE_FILE="passphrase.txt"
KEY_FILE="x2_key/test_x2_key.key"

mkdir -p $TMP_DIR

RESULT=0

############
# Passphrase only
############
echo "Passphrase only"

cat $PASSPHRASE_FILE | $ENCRYPT_CLI -e --pwd-fd 0 -o $TMP_DIR/out_file.gpg $FILE
if ! [ -r $TMP_DIR/out_file.gpg ]
then
    echo encrypted file does not exist
    exit -1;
fi
$GPG -d --batch --passphrase-file $PASSPHRASE_FILE -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.gpg

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.gpg
rm $TMP_DIR/out_file.txt

############
# Key only
############
echo "Key only"

$ENCRYPT_CLI -e --key-only --plain-text-key -k $KEY_FILE -o $TMP_DIR/out_file.epd $FILE
$ENCRYPT_CLI -d --key-only --plain-text-key -k $KEY_FILE -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.epd

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.epd
rm $TMP_DIR/out_file.txt

############
# Key only when encrypting a .gpg file
############
echo "Key only when encrypting a .gpg file"
$ENCRYPT_CLI -e --key-only --plain-text-key -k $KEY_FILE -o $TMP_DIR/out_file.gpg $FILE
$GPG -d --batch --batch --passphrase-file $KEY_FILE -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.gpg

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.gpg
rm $TMP_DIR/out_file.txt

############
# Key only when decrypting a .gpg file
############
echo "Key only when decrytping a .gpg file"
$ENCRYPT_CLI -e --key-only --plain-text-key -k $KEY_FILE -o $TMP_DIR/out_file.gpg $FILE
$ENCRYPT_CLI -d --key-only --plain-text-key -k $KEY_FILE -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.gpg

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.gpg
rm $TMP_DIR/out_file.txt

############
# Key only (persisted)
############
echo "Key only (persisted)"

$ENCRYPT_CLI -e --key-only --plain-text-key --persist-key -k $KEY_FILE -o $TMP_DIR/out_file.epd $FILE
$ENCRYPT_CLI -d --key-only --plain-text-key -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.epd

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.epd
rm $TMP_DIR/out_file.txt

############
# Key and passphrase
############
echo "Key and passphrase"

cat $PASSPHRASE_FILE | $ENCRYPT_CLI -e -k $KEY_FILE --plain-text-key --pwd-fd 0 -o $TMP_DIR/out_file.epd $FILE
cat $PASSPHRASE_FILE | $ENCRYPT_CLI -d -k $KEY_FILE --plain-text-key --pwd-fd 0 -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.epd

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.epd
rm $TMP_DIR/out_file.txt

############
# Key (persisted) and passphrase
############
echo "Key (persisted) and passphrase"

cat $PASSPHRASE_FILE | $ENCRYPT_CLI -e --persist-key --plain-text-key -k $KEY_FILE --pwd-fd 0 -o $TMP_DIR/out_file.epd $FILE
cat $PASSPHRASE_FILE | $ENCRYPT_CLI -d --pwd-fd 0 --plain-text-key -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.epd

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi
rm $TMP_DIR/out_file.epd
rm $TMP_DIR/out_file.txt

############
# Encrypted key file generated by EncryptPad and decrypted by GPG
############
echo "Encrypted key file generated by EncryptPad and decrypted by GPG"

cat $PASSPHRASE_FILE | $ENCRYPT_CLI --generate-key --key-pwd-fd 0 $TMP_DIR/encrypted_key.key
$GPG -d --batch --passphrase-file $PASSPHRASE_FILE -o $TMP_DIR/out_file.txt $TMP_DIR/encrypted_key.key

if [ "$?" = "0" ]
then
    echo OK
else
    RESULT=1
    echo FAIL
fi

rm $TMP_DIR/encrypted_key.key
rm $TMP_DIR/out_file.txt

############
# EncryptPad encrypts with an encrypted key file
############
echo "EncryptPad encrypts with an encrypted key file"

cat $PASSPHRASE_FILE | $ENCRYPT_CLI --generate-key --key-pwd-fd 0 $TMP_DIR/encrypted_key.key
cat $PASSPHRASE_FILE | $ENCRYPT_CLI -e --key-only --key-pwd-fd 0 -k $TMP_DIR/encrypted_key.key -o $TMP_DIR/out_file.gpg $FILE

$GPG -d --batch --passphrase-file $PASSPHRASE_FILE -o $TMP_DIR/unencrypted_key.key $TMP_DIR/encrypted_key.key
$GPG -d --batch --passphrase-file $TMP_DIR/unencrypted_key.key -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.gpg

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi

cat $PASSPHRASE_FILE | $ENCRYPT_CLI -d --key-only --key-pwd-fd 0 -k $TMP_DIR/encrypted_key.key -o $TMP_DIR/epad_out_file.txt $TMP_DIR/out_file.gpg

if diff $FILE $TMP_DIR/epad_out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi

rm $TMP_DIR/encrypted_key.key
rm $TMP_DIR/out_file.gpg
rm $TMP_DIR/unencrypted_key.key
rm $TMP_DIR/out_file.txt
rm $TMP_DIR/epad_out_file.txt

###########
# EncryptPad encrypts and decrypts with an encrypted key in double protection mode
###########
echo "EncryptPad encrypts and decrypts with an encrypted key in double protection mode"

cat $PASSPHRASE_FILE | $ENCRYPT_CLI --generate-key --key-pwd-fd 0 $TMP_DIR/encrypted_key.key
cat $PASSPHRASE_FILE | $ENCRYPT_CLI -e -k $TMP_DIR/encrypted_key.key --key-pwd-file $PASSPHRASE_FILE --pwd-fd 0 -o $TMP_DIR/out_file.epd $FILE
cat $PASSPHRASE_FILE | $ENCRYPT_CLI -d -k $TMP_DIR/encrypted_key.key --key-pwd-file $PASSPHRASE_FILE --pwd-fd 0 -o $TMP_DIR/out_file.txt $TMP_DIR/out_file.epd

if diff $FILE $TMP_DIR/out_file.txt
then
    echo OK
else
    RESULT=1
    echo FAIL
fi

rm $TMP_DIR/out_file.epd
rm $TMP_DIR/out_file.txt
rm $TMP_DIR/encrypted_key.key

###########
# Conclusion
###########

exit $RESULT
